/*
 * py_gpio.c
 *
 * Copyright (c) 2016 Nuvoton technology corporation.
 *
 * This code is based on RPi.GPIO of Ben Croston
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation;version 2 of the License.
 *
 */
#include "Python.h"
#include "event_gpio.h"
#include "constants.h"
#include "common.h"

static int gpio_warnings = 1;

// python function setup(channel(s), direction, initial=None)
static PyObject *py_setup_channel(PyObject *self, PyObject *args)
{
   unsigned int gpio;
   int channel = -1;
   int direction;
   int initial=-1;
   int func;
   
   if (!PyArg_ParseTuple(args, "ii|i", &channel, &direction, &initial))
     return NULL;
 
   if (direction != INPUT && direction != OUTPUT) {
      PyErr_SetString(PyExc_ValueError, "An invalid direction was passed to setup()");
      return 0;
   }
   
   if (direction == INPUT && initial != -1) {
      PyErr_SetString(PyExc_ValueError, "initial parameter is not valid for inputs");
      return 0;
   }

   if (get_gpio_number(channel, &gpio))
      return 0;

   func = gpio_get_direction(gpio);

   if (func != -1) {
       if (gpio_warnings &&                             // warnings enabled and
           ((func != 0 && func != 1) ||                 // (already one of the alt functions or
           (gpio_direction[gpio] == -1 && func == 1)))  // already an output not set from this program)
       {
          PyErr_WarnEx(NULL, "This channel is already in use, continuing anyway.  Use GPIO.setwarnings(False) to disable warnings.", 1);
       }
   }
 
   if (gpio_export(gpio) != 0) {
       return NULL;
   }
    
   if (direction == OUTPUT && (initial == LOW || initial == HIGH)) {
      gpio_set_value(gpio, initial);
   }

   gpio_set_direction(gpio, direction);
   gpio_direction[gpio] = direction; 

   Py_RETURN_NONE;
}

// python function output(channel(s), value(s))
static PyObject *py_output_gpio(PyObject *self, PyObject *args)
{
   unsigned int gpio;
   int channel = -1;
   int value = -1;
   int i;
   
   if (!PyArg_ParseTuple(args, "ii", &channel, &value))
      return NULL;
  
   if (get_gpio_number(channel, &gpio))
     return 0;

   if (gpio_direction[gpio] != OUTPUT)
   {
     PyErr_SetString(PyExc_RuntimeError, "The GPIO channel has not been set up as an OUTPUT");
     return 0;
   }

   if (check_gpio_priv())
     return 0;

   gpio_set_value(gpio, value);
      
   Py_RETURN_NONE;
}

// python function value = input(channel)
static PyObject *py_input_gpio(PyObject *self, PyObject *args)
{
   unsigned int gpio;
   int channel;
   PyObject *value;

   if (!PyArg_ParseTuple(args, "i", &channel))
      return NULL;

   if (get_gpio_number(channel, &gpio))
       return NULL;

   // check channel is set up as an input or output
   if (gpio_direction[gpio] != INPUT && gpio_direction[gpio] != OUTPUT)
   {
      PyErr_SetString(PyExc_RuntimeError, "You must setup() the GPIO channel first");
      return NULL;
   }

   if (check_gpio_priv())
      return NULL;

   if (gpio_get_value(gpio)) {
      value = Py_BuildValue("i", HIGH);
   } else {
      value = Py_BuildValue("i", LOW);
   }
   return value;
}

// python function setmode(mode)
static PyObject *py_setmode(PyObject *self, PyObject *args)
{
   int new_mode;

   if (!PyArg_ParseTuple(args, "i", &new_mode))
      return NULL;

   if (gpio_mode != MODE_UNKNOWN && new_mode != gpio_mode)
   {
      PyErr_SetString(PyExc_ValueError, "A different mode has already been set!");
      return NULL;
   }

   if (setup_error)
   {
      PyErr_SetString(PyExc_RuntimeError, "Module not imported correctly!");
      return NULL;
   }   

   gpio_mode = new_mode;
   
    if (gpio_mode == TOMATO_BOARD)
         pin_to_gpio = &pin_to_gpio_beta;
    else if (gpio_mode == TOMATOV1_BOARD)
         pin_to_gpio = &pin_to_gpio_v1;
     
   Py_RETURN_NONE;
}

// python function getmode()
static PyObject *py_getmode(PyObject *self, PyObject *args)
{
   PyObject *value;

   if (setup_error)
   {
      PyErr_SetString(PyExc_RuntimeError, "Module not imported correctly!");
      return NULL;
   }

   if (gpio_mode == MODE_UNKNOWN)
      Py_RETURN_NONE;

   value = Py_BuildValue("i", gpio_mode);
   return value;
}

// python function setwarnings(state)
static PyObject *py_setwarnings(PyObject *self, PyObject *args)
{
   if (!PyArg_ParseTuple(args, "i", &gpio_warnings))
      return NULL;

   if (setup_error)
   {
      PyErr_SetString(PyExc_RuntimeError, "Module not imported correctly!");
      return NULL;
   }

   Py_RETURN_NONE;
}

// python function cleanup(channel=None)
static PyObject *py_cleanup(PyObject *self, PyObject *args)
{
   int i;
   int found = 0;
   int channel = -666;
   unsigned int gpio;

   if (!PyArg_ParseTuple(args, "i", &channel))
      return NULL;
  
   if (get_gpio_number(channel, &gpio))
     return 0;
 
   // set everything back to input
   if (gpio_direction[gpio] != -1) {
        gpio_set_direction(gpio, INPUT);
        gpio_direction[gpio] = -1;
        found = 1;
   }
   
   if( gpio_unexport(gpio) < 0)
       PyErr_WarnEx(NULL, "Un export error!", 1);

   // check if any channels set up - if not warn about misuse of GPIO.cleanup()
   if (!found && gpio_warnings) {
      PyErr_WarnEx(NULL, "No channels have been set up yet - nothing to clean up!  Try cleaning up at the end of your program instead!", 1);
   }

   Py_RETURN_NONE;
}

static const char moduledocstring[] = "GPIO functionality of a Tomato using Python";

PyMethodDef nvt_gpio_methods[] = {
   {"setup", (PyCFunction)py_setup_channel, METH_VARARGS | METH_KEYWORDS, "Set up a GPIO channel or list of channels with a direction and (optional) pull/up down control\nchannel        - either board pin number or BCM number depending on which mode is set.\ndirection      - IN or OUT\n[pull_up_down] - PUD_OFF (default), PUD_UP or PUD_DOWN\n[initial]      - Initial value for an output channel"},
   {"cleanup", (PyCFunction)py_cleanup, METH_VARARGS | METH_KEYWORDS, "Clean up by resetting all GPIO channels that have been used by this program to INPUT with no pullup/pulldown and no event detection\n[channel] - individual channel or list/tuple of channels to clean up.  Default - clean every channel that has been used."},
   {"output", py_output_gpio, METH_VARARGS, "Output to a GPIO channel or list of channels\nchannel - either board pin number or BCM number depending on which mode is set.\nvalue   - 0/1 or False/True or LOW/HIGH"},
   {"input", py_input_gpio, METH_VARARGS, "Input from a GPIO channel.  Returns HIGH=1=True or LOW=0=False\nchannel - either board pin number or BCM number depending on which mode is set."},
   {"setmode", py_setmode, METH_VARARGS, "Set up numbering mode to use for channels.\nBOARD - Use Raspberry Pi board numbers\nBCM   - Use Broadcom GPIO 00..nn numbers"},
   {"setwarnings", py_setwarnings, METH_VARARGS, "Enable or disable warning messages"},
   {NULL, NULL, 0, NULL}
};

#if PY_MAJOR_VERSION >= 3
static struct PyModuleDef nvtgpiomodule = {
   PyModuleDef_HEAD_INIT,
   "gpio",      // name of module
   moduledocstring,  // module documentation, may be NULL
   -1,               // size of per-interpreter state of the module, or -1 if the module keeps state in global variables.
   nvt_gpio_methods
};
#endif

#if PY_MAJOR_VERSION >= 3
PyMODINIT_FUNC PyInit__GPIO(void)
#else
PyMODINIT_FUNC initgpio(void)
#endif
{
   int i;
   PyObject *module = NULL;

#if PY_MAJOR_VERSION >= 3
   if ((module = PyModule_Create(&nvtgpiomodule)) == NULL)
      return NULL;
#else
   if ((module = Py_InitModule3("gpio", nvt_gpio_methods, moduledocstring)) == NULL)
      return;
#endif

   define_constants(module);

   for (i=0; i<320; i++)
      gpio_direction[i] = -1;
     
   pin_to_gpio = &pin_to_gpio_beta;
   
#if PY_MAJOR_VERSION >= 3
   return module;
#else
   return;
#endif
}
