/**************************************************************************//**
 * @file     dac.c
 * @version  V2.00
 * $Revision: 5 $
 * $Date: 14/06/06 8:38p $
 * @brief    M451 series DAC driver source file
 *
 * @note
 * Copyright (C) 2014 Nuvoton Technology Corp. All rights reserved.
*****************************************************************************/
#include "M451Series.h"

/** @addtogroup M451_Device_Driver M451 Device Driver
  @{
*/

/** @addtogroup M451_DAC_Driver DAC Driver
  @{
*/

/** @addtogroup M451_DAC_EXPORTED_FUNCTIONS DAC Exported Functions
  @{
*/

/**
  * @brief This function make DAC module be ready to convert.
  * @param[in] dac Base address of DAC module.
  * @param[in] u32Ch Not used in M451 Series DAC.
  * @param[in] u32TrgSrc Decides the trigger source. Valid values are:
  *                      - \ref DAC_WRITE_DAT_TRIGGER             :Write DAC_DAT trigger
  *                      - \ref DAC_SOFTWARE_TRIGGER              :Software trigger
  *                      - \ref DAC_LOW_LEVEL_TRIGGER             :STDAC pin low level trigger
  *                      - \ref DAC_HIGH_LEVEL_TRIGGER            :STDAC pin high level trigger
  *                      - \ref DAC_FALLING_EDGE_TRIGGER          :STDAC pin falling edge trigger
  *                      - \ref DAC_RISING_EDGE_TRIGGER           :STDAC pin rising edge trigger
  *                      - \ref DAC_TIMER0_TRIGGER                :Timer 0 trigger
  *                      - \ref DAC_TIMER1_TRIGGER                :Timer 1 trigger
  *                      - \ref DAC_TIMER2_TRIGGER                :Timer 2 trigger
  *                      - \ref DAC_TIMER3_TRIGGER                :Timer 3 trigger
  *                      - \ref DAC_PWM0_TRIGGER                  :PWM0 trigger
  *                      - \ref DAC_PWM1_TRIGGER                  :PWM1 trigger
  * @return None
  * @details The DAC conversion can be started by writing DAC_DAT, software trigger or hardware trigger.
  *         When TRGEN (DAC_CTL[4]) is 0, the data conversion is started by writing DAC_DAT register.
  *         When TRGEN (DAC_CTL[4]) is 1, the data conversion is started by SWTRG (DAC_SWTRG[0]) is set to 1,
  *         external STDAC pin, timer event, or PWM timer event.
  */
void DAC_Open(DAC_T *dac,
              uint32_t u32Ch,
              uint32_t u32TrgSrc)
{
    dac->CTL &= ~(DAC_CTL_ETRGSEL_Msk | DAC_CTL_TRGSEL_Msk | DAC_CTL_TRGEN_Msk);

    dac->CTL |= (u32TrgSrc | DAC_CTL_DACEN_Msk);
}

/**
  * @brief Disable DAC analog power.
  * @param[in] dac Base address of DAC module.
  * @param[in] u32Ch Not used in M451 Series DAC.
  * @return None
  * @details Disable DAC analog power for saving power consumption.
  */
void DAC_Close(DAC_T *dac, uint32_t u32Ch)
{
    dac->CTL &= (~DAC_CTL_DACEN_Msk);
}

/**
  * @brief Set delay time for DAC to become stable.
  * @param[in] dac Base address of DAC module.
  * @param[in] u32Delay Decides the DAC conversion settling time, the range is from 0~(1023/PCLK*1000000) micro seconds.
  * @return Real DAC conversion settling time (micro second).
  * @details For example, DAC controller clock speed is 72MHz and DAC conversion setting time is 1 us, SETTLET (DAC_TCTL[9:0]) value must be greater than 0x48.
  * @note User needs to write appropriate value to meet DAC conversion settling time base on PCLK (APB clock) speed.
  */
float DAC_SetDelayTime(DAC_T *dac, uint32_t u32Delay)
{
    SystemCoreClockUpdate();

    dac->TCTL = ((SystemCoreClock * u32Delay / 1000000) & 0x3FFF);

    return ((dac->TCTL) * 1000000 / SystemCoreClock);
}

/*@}*/ /* end of group M451_DAC_EXPORTED_FUNCTIONS */

/*@}*/ /* end of group M451_DAC_Driver */

/*@}*/ /* end of group M451_Device_Driver */

/*** (C) COPYRIGHT 2014 Nuvoton Technology Corp. ***/
