/**************************************************************************//**
 * @file     NUC505Series.h
 * @version  V1.00
 * $Revision: 56 $
 * $Date: 14/04/25 3:00p $
 * @brief    NUC505 series peripheral access layer header file.
 *           This file contains all the peripheral register's definitions,
 *           bits definitions and memory mapping for NuMicro NUC505 series MCU.
 * @note
 * Copyright (C) 2014 Nuvoton Technology Corp. All rights reserved.
 *****************************************************************************/
/**
   \mainpage NuMicro NUC505 Series MCU Driver Reference Guide
   *
   * <b>Introduction</b>
   *
   * This user manual describes the usage of NUC505 Series MCU device driver
   *
   * <b>Disclaimer</b>
   *
   * The Software is furnished "AS IS", without warranty as to performance or results, and
   * the entire risk as to performance or results is assumed by YOU. Nuvoton disclaims all
   * warranties, express, implied or otherwise, with regard to the Software, its use, or
   * operation, including without limitation any and all warranties of merchantability, fitness
   * for a particular purpose, and non-infringement of intellectual property rights.
   *
   * <b>Important Notice</b>
   *
   * Nuvoton Products are neither intended nor warranted for usage in systems or equipment,
   * any malfunction or failure of which may cause loss of human life, bodily injury or severe
   * property damage. Such applications are deemed, "Insecure Usage".
   *
   * Insecure usage includes, but is not limited to: equipment for surgical implementation,
   * atomic energy control instruments, airplane or spaceship instruments, the control or
   * operation of dynamic, brake or safety systems designed for vehicular use, traffic signal
   * instruments, all types of safety devices, and other applications intended to support or
   * sustain life.
   *
   * All Insecure Usage shall be made at customer's risk, and in the event that third parties
   * lay claims to Nuvoton as a result of customer's Insecure Usage, customer shall indemnify
   * the damages and liabilities thus incurred by Nuvoton.
   *
   * Please note that all data and specifications are subject to change without notice. All the
   * trademarks of products and companies mentioned in this datasheet belong to their respective
   * owners.
   *
   * <b>Copyright Notice</b>
   *
   * Copyright (C) 2014 Nuvoton Technology Corp. All rights reserved.
   */
/**
  * \page pg1 NuMicro NUC505 BSP Directory Structure
  * Please refer to Readme.pdf under BSP root directory for the BSP directory structure
  *
  * \page pg2 Revision History
  * Please refer to NuMicro NUC505 Series CMSIS BSP Revision History.pdf under BSP\\Document directory for the BSP Revision History
  *
*/
#ifndef __NUC505SERIES_H__
#define __NUC505SERIES_H__

#ifdef __cplusplus
extern "C" {
#endif


/******************************************************************************/
/*                Processor and Core Peripherals                              */
/******************************************************************************/
/** @addtogroup NUC505_CMSIS NUC505 Device CMSIS Definitions
  Configuration of the Cortex-M4 Processor and Core Peripherals
  @{
*/

/**
 * @details  Interrupt Number Definition.
 */
typedef enum IRQn
{
    /******  Cortex-M4 Processor Exceptions Numbers ***************************************************/
    NonMaskableInt_IRQn           = -14,      /*!<  2 Non Maskable Interrupt                        */
    MemoryManagement_IRQn         = -12,      /*!<  4 Memory Management Interrupt                   */
    BusFault_IRQn                 = -11,      /*!<  5 Bus Fault Interrupt                           */
    UsageFault_IRQn               = -10,      /*!<  6 Usage Fault Interrupt                         */
    SVCall_IRQn                   = -5,       /*!< 11 SV Call Interrupt                             */
    DebugMonitor_IRQn             = -4,       /*!< 12 Debug Monitor Interrupt                       */
    PendSV_IRQn                   = -2,       /*!< 14 Pend SV Interrupt                             */
    SysTick_IRQn                  = -1,       /*!< 15 System Tick Interrupt                         */

    /******  NUC505 Specific Interrupt Numbers ********************************************************/

    PWR_IRQn                      = 0,        /*!< Power On Interrupt                                           */
    WDT_IRQn                      = 1,        /*!< Watch Dog Timer Interrupt                        */
    APU_IRQn                      = 2,        /*!< Audio Process Unit Interrupt                     */
    I2S_IRQn                      = 3,        /*!< I2S  Interrupt   */
    EINT0_IRQn                    = 4,        /*!< External GPIO Group 0 Interrupt                                                  */
    EINT1_IRQn                    = 5,        /*!< External GPIO Group 1 Interrupt                                                                  */
    EINT2_IRQn                    = 6,        /*!< External GPIO Group 2 Interrupt                                          */
    EINT3_IRQn                    = 7,        /*!< External GPIO Group 3 Interrupt                                          */
    SPIM_IRQn                     = 8,        /*!< SPI_ROM Interrupt                                        */
    USBD_IRQn                     = 9,        /*!< USB Device 2.0 Interrupt                         */
    TMR0_IRQn                       = 10,       /*!< Timer0 Interrupt                                               */
    TMR1_IRQn                       = 11,       /*!< Timer1 Interrupt                                           */
    TMR2_IRQn                       = 12,       /*!< Timer2 Interrupt                                           */
    TMR3_IRQn                       = 13,       /*!< Timer3 Interrupt                                           */
    SDH_IRQn                      = 14,       /*!< SDH Interrupt                                            */
    PWM0_IRQn                     = 15,       /*!< PWM0 Interrupt                                               */
    PWM1_IRQn                     = 16,       /*!< PWM1 Interrupt                                           */
    PWM2_IRQn                     = 17,       /*!< PWM2 Interrupt                                           */
    PWM3_IRQn                     = 18,       /*!< PWM3 Interrupt                                           */
    RTC_IRQn                      = 19,       /*!< Real Time Clock Interrupt                        */
    SPI0_IRQn                     = 20,       /*!< SPI0 Interrupt                                           */
    I2C1_IRQn                     = 21,       /*!< I2C1 Interrupt                                           */
    I2C0_IRQn                     = 22,       /*!< I2C0 Interrupt                                           */
    UART0_IRQn                    = 23,       /*!< UART0 Interrupt                                      */
    UART1_IRQn                    = 24,       /*!< UART1 Interrupt                                      */
    ADC_IRQn                      = 25,       /*!< ADC Interrupt                                            */
    WWDT_IRQn                     = 26,       /*!< Window Watch Dog Timer Interrupt                 */
    USBH_IRQn                     = 27,       /*!< USB Host 1.1 Interrupt                           */
    UART2_IRQn                    = 28,       /*!< UART1 Interrupt                           */
    LVD_IRQn                      = 29,       /*!< LVD Interrupt                           */
    SPI1_IRQn                     = 30,       /*!< SPI1 Interrupt                                     */
}
IRQn_Type;


/*
 * ==========================================================================
 * ----------- Processor and Core Peripheral Section ------------------------
 * ==========================================================================
 */

/* Configuration of the Cortex-M# Processor and Core Peripherals */
#define __CM4_REV                 0x0201    /*!< Core Revision r2p1                               */
#define __NVIC_PRIO_BITS          4         /*!< Number of Bits used for Priority Levels          */
#define __Vendor_SysTickConfig    0         /*!< Set to 1 if different SysTick Config is used     */
#define __MPU_PRESENT             1         /*!< MPU present or not                               */
#define __FPU_PRESENT             1         /*!< FPU present or not                               */

/*@}*/ /* end of group NUC505_CMSIS */


#include "core_cm4.h"                       /* Cortex-M4 processor and core peripherals           */
#include "system_NUC505Series.h"            /* NUC505 System include file                         */
#include <stdint.h>

/******************************************************************************/
/*                Device Specific Peripheral registers structures             */
/******************************************************************************/
/** @addtogroup NUC505_Peripherals NUC505 Control Register
  NUC505 Device Specific Peripheral registers structures
  @{
*/

#if defined ( __CC_ARM   )
#pragma anon_unions
#endif

/*---------------------- Analog to Digital Converter -------------------------*/
/**
    @addtogroup ADC Analog to Digital Converter(ADC)
    Memory Mapped Structure for ADC Controller
@{ */

typedef struct
{


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x00  ADC Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SWTRG     |A/D Conversion Start
     * |        |          |A trigger to start one A/D conversion process.
     * |        |          |0 = A/D conversion enters idle state.
     * |        |          |1 = Start conversion.
     * |        |          |Note: This bit will be cleared to '0' automatically.
     * |[13]    |PD        |Power Down ADC
     * |        |          |0 = ADC is in normal state.
     * |        |          |1 = ADC is in power down state.
     * |        |          |Note1: ADC power must be enabled before a trigger to get the data.
     * |        |          |Note2: It needs 100ms to wait analog block stable when setting PD from 1 to 0.
     * |[15]    |PDKEY     |Power Down Keypad Detection
     * |        |          |0 = Power down keypad detection Disabled.
     * |        |          |1 = Power down keypad detection Enabled.
     * |[18:16] |CHSEL     |Analog Input Selection Signals
     * |        |          |000 = ADC_CH0.
     * |        |          |001 = ADC_CH1.
     * |        |          |010 = ADC_CH2.
     * |        |          |011 = ADC_CH3.
     * |        |          |100 = ADC_CH4.
     * |        |          |101 = ADC_CH5.
     * |        |          |110 = ADC_CH6.
     * |        |          |111 = ADC_CH7.
     * |        |          |Note1: ADC_CH0 is used for battery voltage detection.
     * |        |          |It includes an inherent resistor divider and a switc.
     * |        |          |Note2: User needs to pay attention to electric leakage with ADC_CH0 because the default CHSEL is selected to ADC_CH0, and ADC_CH0 has an internal resistor divider.
     * |        |          |If ADC_CH0 is connected to battery, there is a leakage path.
     * |        |          |Therefore, user can change CHSEL to other channel to cut off this path, when finishing battery detectio.
     * |[31:24] |EXTSMPT   |ADC Extend Sampling Time
     * |        |          |When A/D converting at high conversion rate, the sampling time of analog input voltage may not enough if the input channel loading is heavy, software can extend A/D sampling time after trigger source is coming to get enough sampling time.
     * |        |          |Note: The unit is ADC clock.
    */
    __IO uint32_t CTL;

    /**
     * INTCTL
     * ===================================================================================================
     * Offset: 0x04  ADC Interrupt State
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ADCIF     |ADC Conversion Done Interrupt Flag
     * |        |          |When finishing the sample process, the ADCIF bit will be set.
     * |        |          |If the ADCIEN is set, the interrupt will be transferred to NVI.
     * |        |          |0 = ADC conversion done flag is not set.
     * |        |          |1 = ADC conversion done flag is set.
     * |[1]     |KEYIF     |Keypad Interrupt Flag
     * |        |          |In the process of checking keypad, the KEYIF shows the state.
     * |        |          |ADC_CH2 is used for keypad.
     * |        |          |When ADC_CH2 is not equal to AVDDADC, the interrupt flag will be raise.
     * |        |          |0 = Keypad is not pressing state.
     * |        |          |1 = keypad is pressing state.
     * |[8]     |ADCIEN    |ADC Interrupt Enable Control
     * |        |          |0 = ADC conversion done interrupt Disabled.
     * |        |          |1 = ADC conversion done interrupt Enabled.
     * |[9]     |KEYIEN    |Keypad Interrupt Enable Control
     * |        |          |0 = Keypad down interrupt Disabled.
     * |        |          |1 = Keypad down interrupt Enabled.
    */
    __IO uint32_t INTCTL;

    /**
     * DAT
     * ===================================================================================================
     * Offset: 0x08  ADC Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |RESULT    |A/D Conversion Result
     * |        |          |This field contains conversion result of ADC.
     * |        |          |When A/D conversion done, 12-bit ADC conversion result with unsigned format will be filled in RESULT[11:0.
    */
    __I  uint32_t DAT;

} ADC_T;

/**
    @addtogroup ADC_CONST ADC Bit Field Definition
    Constant Definitions for ADC Controller
@{ */

#define ADC_CTL_SWTRG_Pos                (0)                                               /*!< ADC_T::CTL: SWTRG Position                */
#define ADC_CTL_SWTRG_Msk                (0x1ul << ADC_CTL_SWTRG_Pos)                      /*!< ADC_T::CTL: SWTRG Mask                    */

#define ADC_CTL_PD_Pos                   (13)                                              /*!< ADC_T::CTL: PD Position                   */
#define ADC_CTL_PD_Msk                   (0x1ul << ADC_CTL_PD_Pos)                         /*!< ADC_T::CTL: PD Mask                       */

#define ADC_CTL_PDKEY_Pos                (15)                                              /*!< ADC_T::CTL: PDKEY Position                */
#define ADC_CTL_PDKEY_Msk                (0x1ul << ADC_CTL_PDKEY_Pos)                      /*!< ADC_T::CTL: PDKEY Mask                    */

#define ADC_CTL_CHSEL_Pos                (16)                                              /*!< ADC_T::CTL: CHSEL Position                */
#define ADC_CTL_CHSEL_Msk                (0x7ul << ADC_CTL_CHSEL_Pos)                      /*!< ADC_T::CTL: CHSEL Mask                    */

#define ADC_CTL_EXTSMPT_Pos              (24)                                              /*!< ADC_T::CTL: EXTSMPT Position              */
#define ADC_CTL_EXTSMPT_Msk              (0xfful << ADC_CTL_EXTSMPT_Pos)                   /*!< ADC_T::CTL: EXTSMPT Mask                  */

#define ADC_INTCTL_ADCIF_Pos             (0)                                               /*!< ADC_T::INTCTL: ADCIF Position             */
#define ADC_INTCTL_ADCIF_Msk             (0x1ul << ADC_INTCTL_ADCIF_Pos)                   /*!< ADC_T::INTCTL: ADCIF Mask                 */

#define ADC_INTCTL_KEYIF_Pos             (1)                                               /*!< ADC_T::INTCTL: KEYIF Position             */
#define ADC_INTCTL_KEYIF_Msk             (0x1ul << ADC_INTCTL_KEYIF_Pos)                   /*!< ADC_T::INTCTL: KEYIF Mask                 */

#define ADC_INTCTL_ADCIEN_Pos            (8)                                               /*!< ADC_T::INTCTL: ADCIEN Position            */
#define ADC_INTCTL_ADCIEN_Msk            (0x1ul << ADC_INTCTL_ADCIEN_Pos)                  /*!< ADC_T::INTCTL: ADCIEN Mask                */

#define ADC_INTCTL_KEYIEN_Pos            (9)                                               /*!< ADC_T::INTCTL: KEYIEN Position            */
#define ADC_INTCTL_KEYIEN_Msk            (0x1ul << ADC_INTCTL_KEYIEN_Pos)                  /*!< ADC_T::INTCTL: KEYIEN Mask                */

#define ADC_DAT_RESULT_Pos               (0)                                               /*!< ADC_T::DAT: RESULT Position               */
#define ADC_DAT_RESULT_Msk               (0xffful << ADC_DAT_RESULT_Pos)                   /*!< ADC_T::DAT: RESULT Mask                   */

/**@}*/ /* ADC_CONST */
/**@}*/ /* end of ADC register group */


/*---------------------- System Clock Controller -------------------------*/
/**
    @addtogroup CLK System Clock Controller(CLK)
    Memory Mapped Structure for CLK Controller
@{ */

typedef struct
{


    /**
     * PWRCTL
     * ===================================================================================================
     * Offset: 0x00  System Power-down Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |HXTEN     |Crystal (Power Down) Control
     * |        |          |0 = Crystal off (Power down).
     * |        |          |1 = Crystal on (Normal operation).
     * |[1]     |HXTCTL    |Power-down Mode Wake-up Pre-divider Counter Enable Control
     * |        |          |The HXT pre-divider controls wake-up time from Power-down mode
     * |        |          |The chip will delay (PDWKPSC*256) cycles to wait until the HXT is stable after the reset signal.
     * |        |          |0 = PDWKPSC counter Disabled.
     * |        |          |1 = PDWKPSC counter Enabled.
     * |[2]     |PDWKIF    |Power-down Mode Wake-up Interrupt Flag
     * |        |          |Set by "power down wake-up event" indicates that resume from Power-down mode
     * |        |          |The flag is set if the GPIO, USBH, USBD, UART, TIMER, WDT, RTC or I2C wake-up occurred.
     * |        |          |Note1: Write 1 to clear the bit to zero.
     * |        |          |Note2: This bit works only if PDWKIEN (CLK_PWRCTL[24]) set to 1
     * |[3]     |PDWKIEN   |Power-down Mode Wake-up Interrupt Enable Control
     * |        |          |0 = Power-down Mode Wake-up Interrupt Disabled.
     * |        |          |1 = Power-down Mode Wake-up Interrupt Enabled.
     * |        |          |Note: The interrupt will occur when both PDWKIF and PDWKIEN are set
     * |[23:8]  |PDWKPSC   |PDWKPSC Counter
     * |        |          |Assuming the HXT is stable after the PDWKPSC x 256 HXT cycles, Clock controller would not output clock to system before the counter reaches (PDWKPSC x 256).
     * |[24]    |PDWTCPU   |Control Power-down Entry Condition
     * |        |          |0 = Chip enters Power-down mode when the HXTEN bit is set to 1.
     * |        |          |1 = Chip enters Power-down mode when the both PDWTCPU and HXTEN bits are set to 1 and CPU run WFI instruction.
    */
    __IO uint32_t PWRCTL;

    /**
     * AHBCLK
     * ===================================================================================================
     * Offset: 0x04  AHB Devices Clock Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SRAM01CKEN|SRAM#1 Clock Enable Control
     * |        |          |0 = SRAM#1 Clock Disabled.
     * |        |          |1 = SRAM#1 Clock Enabled.
     * |[1]     |SRAM23CKEN|SRAM#2 Clock Enable Control
     * |        |          |0 = SRAM#2 Clock Disabled.
     * |        |          |1 = SRAM#2 Clock Enabled.
     * |[2]     |ROMCKEN   |ROM Clock Enable Control
     * |        |          |0 = ROM Clock Disabled.
     * |        |          |1 = ROM Clock Enabled.
     * |[3]     |SPIMCKEN  |SPIM Clock Enable Control
     * |        |          |0 = SPIM Clock Disabled.
     * |        |          |1 = SPIM Clock Enabled.
     * |[5]     |SDHCKEN   |SDH Clock Enable Control
     * |        |          |0 = SDH Clock Disabled.
     * |        |          |1 = SDH Clock Enabled.
     * |[6]     |USBDCKEN  |USB Device Clock Enable Control
     * |        |          |0 = USB Device Clock Disabled.
     * |        |          |1 = USB Device Clock Enabled.
     * |[9]     |USBHCKEN  |USB Host Clock Enable Control
     * |        |          |0 = USB Host Clock Disabled.
     * |        |          |1 = USB Host Clock Enabled.
    */
    __IO uint32_t AHBCLK;

    /**
     * APBCLK
     * ===================================================================================================
     * Offset: 0x08  APB Devices Clock Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TMR0CKEN  |TIMER0 Clock Enable Control
     * |        |          |0 = TIMER0 Clock Disabled.
     * |        |          |1 = TIMER0 Clock Enabled.
     * |[1]     |TMR1CKEN  |TIMER1 Clock Enable Control
     * |        |          |0 = TIMER1 Clock Disabled.
     * |        |          |1 = TIMER1 Clock Enabled.
     * |[2]     |TMR2CKEN  |TIMER2 Clock Enable Control
     * |        |          |0 = TIMER2 Clock Disabled.
     * |        |          |1 = TIMER2 Clock Enabled.
     * |[3]     |TMR3CKEN  |TIMER3 Clock Enable Control
     * |        |          |0 = TIMER3 Clock Disabled.
     * |        |          |1 = TIMER3 Clock Enabled.
     * |[4]     |WDTCKEN   |Watchdog Timer Clock Enable Control
     * |        |          |0 = Watchdog Timer Clock Disabled.
     * |        |          |1 = Watchdog Timer Clock Enabled.
     * |[5]     |I2C0CKEN  |I2C0 Clock Enable Control
     * |        |          |0 = I2C0 Clock Disabled.
     * |        |          |1 = I2C0 Clock Enabled.
     * |[6]     |I2C1CKEN  |I2C1 Clock Enable Control
     * |        |          |0 = I2C1 Clock Disabled.
     * |        |          |1 = I2C1 Clock Enabled.
     * |[7]     |RTCCKEN   |RTC Clock Enable Control
     * |        |          |0 = RTC Clock Disabled.
     * |        |          |1 = RTC Clock Enabled.
     * |[8]     |PWMCKEN   |PWM Clock Enable Control
     * |        |          |0 = PWM Clock Disabled.
     * |        |          |1 = PWM Clock Enabled.
     * |[9]     |SPI0CKEN  |SPI0 Clock Enable Control
     * |        |          |0 = SPI0 Clock Disabled.
     * |        |          |1 = SPI0 Clock Enabled.
     * |[10]    |SPI1CKEN  |SPI1 Clock Enable Control
     * |        |          |0 = SPI1 Clock Disabled.
     * |        |          |1 = SPI1 Clock Enabled.
     * |[11]    |UART0CKEN |UART0 Clock Enable Control
     * |        |          |0 = UART0 Clock Disabled.
     * |        |          |1 = UART0 Clock Enabled.
     * |[12]    |UART1CKEN |UART1 Clock Enable Control
     * |        |          |0 = UART1 Clock Disabled.
     * |        |          |1 = UART1 Clock Enabled.
     * |[13]    |UART2CKEN |UART2 Clock Enable Control
     * |        |          |0 = UART2 Clock Disabled.
     * |        |          |1 = UART2 Clock Enabled.
     * |[14]    |I2SCKEN   |I2S Clock Enable Control
     * |        |          |0 = I2S Clock Disabled.
     * |        |          |1 = I2S Clock Enabled.
     * |[15]    |ADCCKEN   |ADC Clock Enable Control
     * |        |          |0 = ADC Clock Disabled.
     * |        |          |1 = ADC Clock Enabled.
    */
    __IO uint32_t APBCLK;
    uint32_t RESERVE0[1];


    /**
     * CLKDIV0
     * ===================================================================================================
     * Offset: 0x10  Clock Divider Number Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |HCLKDIV   |Defines the Clock Divider Number for SYS_CLK
     * |        |          |The actual clock divider number is (HCLKDIV+1). So,
     * |        |          |SYS_CLK = SYS_SrcCLK / (HCLKDIV+1).
     * |[7]     |HCLKSEL   |System Source Clock Select (SYS_SrcCLK)
     * |        |          |0 = System Clock source from HXT.
     * |        |          |1 = System Clock source from PLL_FOUT.
     * |[11:8]  |PCLKDIV   |Defines the Clock Divider Number for APB_CLK
     * |        |          |The actual clock divider number is (PCLKDIV+1). So,
     * |        |          |APB_CLK = HCLK / (PCLKDIV+1).
     * |[20:16] |USBDDIV   |Defines the Clock Divider Number for USB Device
     * |        |          |The actual clock divider number is (USBDDIV+1). So,
     * |        |          |USBD_CLK = USBD_SrcCLK / (USBDDIV+1).
     * |[23]    |USBDSEL   |USB Device Source Clock Select (USBD_SrcCLK)
     * |        |          |0 = USB Device Clock source from HXT.
     * |        |          |1 = USB Device Clock source from PLL_FOUT.
     * |[27:24] |USBHDIV   |Defines the Clock Divider Number for USB Host
     * |        |          |The actual clock divider number is (USBHDIV+1). So,
     * |        |          |USBH_CLK = USBH_SrcCLK / (USBHDIV+1).
     * |[31]    |USBHSEL   |USB Host Clock Source Select (USBH_SrcCLK)
     * |        |          |0 = USB Host Clock source from HXT.
     * |        |          |1 = USB Host Clock source from PLL_FOUT.
    */
    __IO uint32_t CLKDIV0;

    /**
     * CLKDIV1
     * ===================================================================================================
     * Offset: 0x14  Clock Divider Number Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |ADCDIV    |Defines the Clock Divider Number for ADC
     * |        |          |The actual clock divider number is (ADCDIV+1). So,
     * |        |          |ADC_CLK = ADC_SrcCLK / (ADCDIV+1).
     * |[15:8]  |STICKDIV  |Defines the Clock Divider Number for SYS_TICK
     * |        |          |The actual clock divider number is (STICKDIV+1). So,
     * |        |          |SYS_TICK_CLK = SYS_TICK_SrcCLK / (STICKDIV+1).
     * |[26:16] |SDHDIV    |Defines the Clock Divider Number for SDH
     * |        |          |The actual clock divider number is (SDHDIV+1). So,
     * |        |          |SDH_CLK = SDH_SrcCLK / (SDHDIV+1).
     * |[28]    |ADCSEL    |ADC Clock Select (ADC_SrcCLK)
     * |        |          |0 = ADC Clock source from HXT.
     * |        |          |1 = ADC Clock source from PLL_FOUT.
     * |[30]    |SDHSEL    |SDH Clock Select (SDH_SrcCLK)
     * |        |          |0 = SDH Clock source from HXT.
     * |        |          |1 = SDH Clock source from PLL_FOUT.
    */
    __IO uint32_t CLKDIV1;

    /**
     * CLKDIV2
     * ===================================================================================================
     * Offset: 0x18  Clock Divider Number Control Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |I2SDIV    |Defines the Clock Divider Number for I2S
     * |        |          |The actual clock divider number is (I2SDIV+1). So,
     * |        |          |I2S_CLK = I2S_SrcCLK / (I2SDIV+1).
     * |[25:24] |I2SSEL    |I2S Clock Select (I2S_SrcCLK)
     * |        |          |00 = I2S Clock source from HXT.
     * |        |          |01 = I2S Clock source from PLL_FOUT.
     * |        |          |10 = Reserved.
     * |        |          |11 = I2S Clock source from APLL_FOUT.
     * |[28]    |SPI0SEL   |SPI0 Engine Clock Select (SPI0SEL)
     * |        |          |0 = SPI0 Engine Clock source from HXT.
     * |        |          |1 = SPI0 Engine Clock source from PLL_FOUT.
     * |[29]    |SPI1SEL   |SPI1 Engine Clock Select (SPI1SEL)
     * |        |          |0 = SPI1 Engine Clock source from HXT.
     * |        |          |1 = SPI1 Engine Clock source from PLL_FOUT.
    */
    __IO uint32_t CLKDIV2;

    /**
     * CLKDIV3
     * ===================================================================================================
     * Offset: 0x1C  Clock Divider Number Control Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |UART0DIV  |Defines the Clock Divider Number for UART0
     * |        |          |The actual clock divider number is (UART0DIV +1). So,
     * |        |          |UART0_CLK = UART0_SrcCLK / (UART0DIV +1).
     * |[4]     |UART0SEL  |UART0 Source Clock Select (UART0_SrcCLK)
     * |        |          |0 = UART0 Source Clock source from HXT.
     * |        |          |1 = UART0 Source Clock source from PLL_FOUT.
     * |[11:8]  |UART1DIV  |Defines the Clock Divider Number for UART1
     * |        |          |The actual clock divider number is (UART1DIV +1). So,
     * |        |          |UART1_CLK = UART1_SrcCLK / (UART1DIV +1).
     * |[12]    |UART1SEL  |UART1 Source Clock Select (UART1_SrcCLK)
     * |        |          |0 = UART1 Source Clock source from HXT.
     * |        |          |1 = UART1 Source Clock source from PLL_FOUT.
     * |[19:16] |UART2DIV  |Defines the Clock Divider Number for UART2
     * |        |          |The actual clock divider number is (UART2DIV +1). So,
     * |        |          |UART2_CLK = UART2_SrcCLK / (UART2DIV +1).
     * |[20]    |UART2SEL  |UART2 Source Clock Select (UART2_SrcCLK)
     * |        |          |0 = UART2 Source Clock source from HXT.
     * |        |          |1 = UART2 Source Clock source from PLL_FOUT.
    */
    __IO uint32_t CLKDIV3;

    /**
     * PLLCTL
     * ===================================================================================================
     * Offset: 0x20  PLL Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6:0]   |FBDIV     |Feedback Divider Control (N)
     * |        |          |Set the feedback divider factor from 1 to 128
     * |[12:7]  |INDIV     |Reference Input Divider (M)
     * |        |          |Set the input reference clock divider factor from 1 to 64.
     * |[15:13] |OUTDIV    |Output Divider Control (P)
     * |        |          |Set the output divider factor from 1 to 8.
     * |[16]    |BP        |PLL Bypass Control
     * |        |          |0 = PLL at Normal mode.
     * |        |          |1 = Bypass Fin (i.e. Fout = XIN).
     * |[17]    |PD        |Power-down Mode
     * |        |          |0 = PLL in Normal mode.
     * |        |          |1 = PLL in Power-down mode (Default).
    */
    __IO uint32_t PLLCTL;
    uint32_t RESERVE1[1];


    /**
     * APLLCTL
     * ===================================================================================================
     * Offset: 0x28  APLL Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |INDIV     |Reference Input Divider Control
     * |        |          |Set the reference divider factor from 1 to 64.
     * |        |          |M = INDIV[5:0]+1.
     * |[12:6]  |FBDIV     |Feedback Divider Control
     * |        |          |Set the Feedback divider factor from 1 to 128.
     * |        |          |N = FBDIV[6:0]+1.
     * |[15:13] |OUTDIV    |Output Divider Control
     * |        |          |Set the Output divider factor from 1 to 8.
     * |        |          |P = OUTDIV[2:0]+1.
     * |[16]    |PD        |Power Down Enable Control
     * |        |          |0 = Power down Disabled.
     * |        |          |1 = Power down Enabled.
     * |[18]    |MODE      |Mode Select
     * |        |          |0 = Integer mode;.
     * |        |          |1 = Fraction mode.
     * |[31:20] |FRAC      |Sigma-delta Modulator Control Pins
     * |        |          |Set the fractional number of the Feedback divider.
    */
    __IO uint32_t APLLCTL;
    uint32_t RESERVE2[1];


    /**
     * CLKDIV4
     * ===================================================================================================
     * Offset: 0x30  Clock Divider Number Control Register 4
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |TMR0DIV   |Defines the Clock Divider Number for TMR0
     * |        |          |The actual clock divider number is (TMR0DIV+1). So,
     * |        |          |TMR0_CLK = TMR0_SrcCLK / (TMR0DIV+1).
     * |[15:8]  |TMR1DIV   |Defines the Clock Divider Number for TMR1
     * |        |          |The actual clock divider number is (TMR1DIV+1). So,
     * |        |          |TMR1_CLK = TMR1_SrcCLK / (TMR1DIV+1).
     * |[23:16] |TMR2DIV   |Defines the Clock Divider Number for TMR2
     * |        |          |The actual clock divider number is (TMR2DIV+1). So,
     * |        |          |TMR2_CLK = TMR2_SrcCLK / (TMR2CLK_N+1).
     * |[24]    |TMR0SEL   |Timer0 Engine Clock Select (TMR0_SrcCLK)
     * |        |          |0 = Timer0 Engine Clock source from RTC_CLK.
     * |        |          |1 = Timer0 Engine Clock source from HXT.
     * |[25]    |TMR1SEL   |Timer1 Engine Clock Select (TMR1_SrcCLK)
     * |        |          |0 = Timer1 Engine Clock source from RTC_CLK.
     * |        |          |1 = Timer1 Engine Clock source from HXT.
     * |[26]    |TMR2SEL   |Timer2 Engine Clock Select (TMR2_SrcCLK)
     * |        |          |0 = Timer2 Engine Clock source from RTC_CLK.
     * |        |          |1 = Timer2 Engine Clock source from HXT.
    */
    __IO uint32_t CLKDIV4;

    /**
     * CLKDIV5
     * ===================================================================================================
     * Offset: 0x34  Clock Divider Number Control Register 5
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |TMR3DIV   |Define the Clock Divider Number for TMR3
     * |        |          |The actual clock divider number is (TM3DIV+1). So,
     * |        |          |TMR3_CLK = TMR3_SrcCLK / (TMR3DIV+1).
     * |[15:8]  |WDTDIV    |Define the Clock Divider Number for WDT
     * |        |          |The actual clock divider number is (WDTDIV+1). So,
     * |        |          |WDT_CLK = WDT_SrcCLK / (WDTDIV+1).
     * |[23:16] |PWMDIV    |Define the Clock Divider Number for PWM
     * |        |          |The actual clock divider number is (PWMDIV+1). So,
     * |        |          |PWM_CLK = PWM_SrcCLK / (PWMDIV+1).
     * |[24]    |TMR3SEL   |Timer3 Engine Clock Select (TMR3_SrcCLK)
     * |        |          |0 = Timer3 Engine Clock source from RTC_CLK.
     * |        |          |1 = Timer3 Engine Clock source from HXT.
     * |[25]    |WDTSEL    |WDT Engine Clock Select (WDT_SrcCLK)
     * |        |          |0 = WDT Engine Clock source from RTC_CLK.
     * |        |          |1 = WDT Engine Clock source from HXT.
     * |[26]    |PWMSEL    |PWM Engine Clock Select (PWM_SrcCLK)
     * |        |          |0 = PWM Engine Clock source from HXT.
     * |        |          |1 = PWM Engine Clock source from PLL_FOUT.
    */
    __IO uint32_t CLKDIV5;

} CLK_T;

/**
    @addtogroup CLK_CONST CLK Bit Field Definition
    Constant Definitions for CLK Controller
@{ */

#define CLK_PWRCTL_HXTEN_Pos             (0)                                               /*!< CLK_T::PWRCTL: HXTEN Position             */
#define CLK_PWRCTL_HXTEN_Msk             (0x1ul << CLK_PWRCTL_HXTEN_Pos)                   /*!< CLK_T::PWRCTL: HXTEN Mask                 */

#define CLK_PWRCTL_HXTCTL_Pos            (1)                                               /*!< CLK_T::PWRCTL: HXTCTL Position            */
#define CLK_PWRCTL_HXTCTL_Msk            (0x1ul << CLK_PWRCTL_HXTCTL_Pos)                  /*!< CLK_T::PWRCTL: HXTCTL Mask                */

#define CLK_PWRCTL_PDWKIF_Pos            (2)                                               /*!< CLK_T::PWRCTL: PDWKIF Position            */
#define CLK_PWRCTL_PDWKIF_Msk            (0x1ul << CLK_PWRCTL_PDWKIF_Pos)                  /*!< CLK_T::PWRCTL: PDWKIF Mask                */

#define CLK_PWRCTL_PDWKIEN_Pos           (3)                                               /*!< CLK_T::PWRCTL: PDWKIEN Position           */
#define CLK_PWRCTL_PDWKIEN_Msk           (0x1ul << CLK_PWRCTL_PDWKIEN_Pos)                 /*!< CLK_T::PWRCTL: PDWKIEN Mask               */

#define CLK_PWRCTL_PDWKPSC_Pos           (8)                                               /*!< CLK_T::PWRCTL: PDWKPSC Position           */
#define CLK_PWRCTL_PDWKPSC_Msk           (0xfffful << CLK_PWRCTL_PDWKPSC_Pos)              /*!< CLK_T::PWRCTL: PDWKPSC Mask               */

#define CLK_PWRCTL_PDWTCPU_Pos           (24)                                              /*!< CLK_T::PWRCTL: PDWTCPU Position           */
#define CLK_PWRCTL_PDWTCPU_Msk           (0x1ul << CLK_PWRCTL_PDWTCPU_Pos)                 /*!< CLK_T::PWRCTL: PDWTCPU Mask               */

#define CLK_AHBCLK_SRAM01CKEN_Pos        (0)                                               /*!< CLK_T::AHBCLK: SRAM01CKEN Position        */
#define CLK_AHBCLK_SRAM01CKEN_Msk        (0x1ul << CLK_AHBCLK_SRAM01CKEN_Pos)              /*!< CLK_T::AHBCLK: SRAM01CKEN Mask            */

#define CLK_AHBCLK_SRAM23CKEN_Pos        (1)                                               /*!< CLK_T::AHBCLK: SRAM23CKEN Position        */
#define CLK_AHBCLK_SRAM23CKEN_Msk        (0x1ul << CLK_AHBCLK_SRAM23CKEN_Pos)              /*!< CLK_T::AHBCLK: SRAM23CKEN Mask            */

#define CLK_AHBCLK_ROMCKEN_Pos           (2)                                               /*!< CLK_T::AHBCLK: ROMCKEN Position           */
#define CLK_AHBCLK_ROMCKEN_Msk           (0x1ul << CLK_AHBCLK_ROMCKEN_Pos)                 /*!< CLK_T::AHBCLK: ROMCKEN Mask               */

#define CLK_AHBCLK_SPIMCKEN_Pos          (3)                                               /*!< CLK_T::AHBCLK: SPIMCKEN Position          */
#define CLK_AHBCLK_SPIMCKEN_Msk          (0x1ul << CLK_AHBCLK_SPIMCKEN_Pos)                /*!< CLK_T::AHBCLK: SPIMCKEN Mask              */

#define CLK_AHBCLK_SDHCKEN_Pos           (5)                                               /*!< CLK_T::AHBCLK: SDHCKEN Position           */
#define CLK_AHBCLK_SDHCKEN_Msk           (0x1ul << CLK_AHBCLK_SDHCKEN_Pos)                 /*!< CLK_T::AHBCLK: SDHCKEN Mask               */

#define CLK_AHBCLK_USBDCKEN_Pos          (6)                                               /*!< CLK_T::AHBCLK: USBDCKEN Position          */
#define CLK_AHBCLK_USBDCKEN_Msk          (0x1ul << CLK_AHBCLK_USBDCKEN_Pos)                /*!< CLK_T::AHBCLK: USBDCKEN Mask              */

#define CLK_AHBCLK_USBHCKEN_Pos          (9)                                               /*!< CLK_T::AHBCLK: USBHCKEN Position          */
#define CLK_AHBCLK_USBHCKEN_Msk          (0x1ul << CLK_AHBCLK_USBHCKEN_Pos)                /*!< CLK_T::AHBCLK: USBHCKEN Mask              */

#define CLK_APBCLK_TMR0CKEN_Pos          (0)                                               /*!< CLK_T::APBCLK: TMR0CKEN Position          */
#define CLK_APBCLK_TMR0CKEN_Msk          (0x1ul << CLK_APBCLK_TMR0CKEN_Pos)                /*!< CLK_T::APBCLK: TMR0CKEN Mask              */

#define CLK_APBCLK_TMR1CKEN_Pos          (1)                                               /*!< CLK_T::APBCLK: TMR1CKEN Position          */
#define CLK_APBCLK_TMR1CKEN_Msk          (0x1ul << CLK_APBCLK_TMR1CKEN_Pos)                /*!< CLK_T::APBCLK: TMR1CKEN Mask              */

#define CLK_APBCLK_TMR2CKEN_Pos          (2)                                               /*!< CLK_T::APBCLK: TMR2CKEN Position          */
#define CLK_APBCLK_TMR2CKEN_Msk          (0x1ul << CLK_APBCLK_TMR2CKEN_Pos)                /*!< CLK_T::APBCLK: TMR2CKEN Mask              */

#define CLK_APBCLK_TMR3CKEN_Pos          (3)                                               /*!< CLK_T::APBCLK: TMR3CKEN Position          */
#define CLK_APBCLK_TMR3CKEN_Msk          (0x1ul << CLK_APBCLK_TMR3CKEN_Pos)                /*!< CLK_T::APBCLK: TMR3CKEN Mask              */

#define CLK_APBCLK_WDTCKEN_Pos           (4)                                               /*!< CLK_T::APBCLK: WDTCKEN Position           */
#define CLK_APBCLK_WDTCKEN_Msk           (0x1ul << CLK_APBCLK_WDTCKEN_Pos)                 /*!< CLK_T::APBCLK: WDTCKEN Mask               */

#define CLK_APBCLK_I2C0CKEN_Pos          (5)                                               /*!< CLK_T::APBCLK: I2C0CKEN Position          */
#define CLK_APBCLK_I2C0CKEN_Msk          (0x1ul << CLK_APBCLK_I2C0CKEN_Pos)                /*!< CLK_T::APBCLK: I2C0CKEN Mask              */

#define CLK_APBCLK_I2C1CKEN_Pos          (6)                                               /*!< CLK_T::APBCLK: I2C1CKEN Position          */
#define CLK_APBCLK_I2C1CKEN_Msk          (0x1ul << CLK_APBCLK_I2C1CKEN_Pos)                /*!< CLK_T::APBCLK: I2C1CKEN Mask              */

#define CLK_APBCLK_RTCCKEN_Pos           (7)                                               /*!< CLK_T::APBCLK: RTCCKEN Position           */
#define CLK_APBCLK_RTCCKEN_Msk           (0x1ul << CLK_APBCLK_RTCCKEN_Pos)                 /*!< CLK_T::APBCLK: RTCCKEN Mask               */

#define CLK_APBCLK_PWMCKEN_Pos           (8)                                               /*!< CLK_T::APBCLK: PWMCKEN Position           */
#define CLK_APBCLK_PWMCKEN_Msk           (0x1ul << CLK_APBCLK_PWMCKEN_Pos)                 /*!< CLK_T::APBCLK: PWMCKEN Mask               */

#define CLK_APBCLK_SPI0CKEN_Pos          (9)                                               /*!< CLK_T::APBCLK: SPI0CKEN Position          */
#define CLK_APBCLK_SPI0CKEN_Msk          (0x1ul << CLK_APBCLK_SPI0CKEN_Pos)                /*!< CLK_T::APBCLK: SPI0CKEN Mask              */

#define CLK_APBCLK_SPI1CKEN_Pos          (10)                                              /*!< CLK_T::APBCLK: SPI1CKEN Position          */
#define CLK_APBCLK_SPI1CKEN_Msk          (0x1ul << CLK_APBCLK_SPI1CKEN_Pos)                /*!< CLK_T::APBCLK: SPI1CKEN Mask              */

#define CLK_APBCLK_UART0CKEN_Pos         (11)                                              /*!< CLK_T::APBCLK: UART0CKEN Position         */
#define CLK_APBCLK_UART0CKEN_Msk         (0x1ul << CLK_APBCLK_UART0CKEN_Pos)               /*!< CLK_T::APBCLK: UART0CKEN Mask             */

#define CLK_APBCLK_UART1CKEN_Pos         (12)                                              /*!< CLK_T::APBCLK: UART1CKEN Position         */
#define CLK_APBCLK_UART1CKEN_Msk         (0x1ul << CLK_APBCLK_UART1CKEN_Pos)               /*!< CLK_T::APBCLK: UART1CKEN Mask             */

#define CLK_APBCLK_UART2CKEN_Pos         (13)                                              /*!< CLK_T::APBCLK: UART2CKEN Position         */
#define CLK_APBCLK_UART2CKEN_Msk         (0x1ul << CLK_APBCLK_UART2CKEN_Pos)               /*!< CLK_T::APBCLK: UART2CKEN Mask             */

#define CLK_APBCLK_I2SCKEN_Pos           (14)                                              /*!< CLK_T::APBCLK: I2SCKEN Position           */
#define CLK_APBCLK_I2SCKEN_Msk           (0x1ul << CLK_APBCLK_I2SCKEN_Pos)                 /*!< CLK_T::APBCLK: I2SCKEN Mask               */

#define CLK_APBCLK_ADCCKEN_Pos           (15)                                              /*!< CLK_T::APBCLK: ADCCKEN Position           */
#define CLK_APBCLK_ADCCKEN_Msk           (0x1ul << CLK_APBCLK_ADCCKEN_Pos)                 /*!< CLK_T::APBCLK: ADCCKEN Mask               */

#define CLK_CLKDIV0_HCLKDIV_Pos          (0)                                               /*!< CLK_T::CLKDIV0: HCLKDIV Position          */
#define CLK_CLKDIV0_HCLKDIV_Msk          (0xful << CLK_CLKDIV0_HCLKDIV_Pos)                /*!< CLK_T::CLKDIV0: HCLKDIV Mask              */

#define CLK_CLKDIV0_HCLKSEL_Pos          (7)                                               /*!< CLK_T::CLKDIV0: HCLKSEL Position          */
#define CLK_CLKDIV0_HCLKSEL_Msk          (0x1ul << CLK_CLKDIV0_HCLKSEL_Pos)                /*!< CLK_T::CLKDIV0: HCLKSEL Mask              */

#define CLK_CLKDIV0_PCLKDIV_Pos          (8)                                               /*!< CLK_T::CLKDIV0: PCLKDIV Position          */
#define CLK_CLKDIV0_PCLKDIV_Msk          (0xful << CLK_CLKDIV0_PCLKDIV_Pos)                /*!< CLK_T::CLKDIV0: PCLKDIV Mask              */

#define CLK_CLKDIV0_USBDDIV_Pos          (16)                                              /*!< CLK_T::CLKDIV0: USBDDIV Position          */
#define CLK_CLKDIV0_USBDDIV_Msk          (0x1ful << CLK_CLKDIV0_USBDDIV_Pos)               /*!< CLK_T::CLKDIV0: USBDDIV Mask              */

#define CLK_CLKDIV0_USBDSEL_Pos          (23)                                              /*!< CLK_T::CLKDIV0: USBDSEL Position          */
#define CLK_CLKDIV0_USBDSEL_Msk          (0x1ul << CLK_CLKDIV0_USBDSEL_Pos)                /*!< CLK_T::CLKDIV0: USBDSEL Mask              */

#define CLK_CLKDIV0_USBHDIV_Pos          (24)                                              /*!< CLK_T::CLKDIV0: USBHDIV Position          */
#define CLK_CLKDIV0_USBHDIV_Msk          (0xful << CLK_CLKDIV0_USBHDIV_Pos)                /*!< CLK_T::CLKDIV0: USBHDIV Mask              */

#define CLK_CLKDIV0_USBHSEL_Pos          (31)                                              /*!< CLK_T::CLKDIV0: USBHSEL Position          */
#define CLK_CLKDIV0_USBHSEL_Msk          (0x1ul << CLK_CLKDIV0_USBHSEL_Pos)                /*!< CLK_T::CLKDIV0: USBHSEL Mask              */

#define CLK_CLKDIV1_ADCDIV_Pos           (0)                                               /*!< CLK_T::CLKDIV1: ADCDIV Position           */
#define CLK_CLKDIV1_ADCDIV_Msk           (0xfful << CLK_CLKDIV1_ADCDIV_Pos)                /*!< CLK_T::CLKDIV1: ADCDIV Mask               */

#define CLK_CLKDIV1_STICKDIV_Pos         (8)                                               /*!< CLK_T::CLKDIV1: STICKDIV Position         */
#define CLK_CLKDIV1_STICKDIV_Msk         (0xfful << CLK_CLKDIV1_STICKDIV_Pos)              /*!< CLK_T::CLKDIV1: STICKDIV Mask             */

#define CLK_CLKDIV1_SDHDIV_Pos           (16)                                              /*!< CLK_T::CLKDIV1: SDHDIV Position           */
#define CLK_CLKDIV1_SDHDIV_Msk           (0x7fful << CLK_CLKDIV1_SDHDIV_Pos)               /*!< CLK_T::CLKDIV1: SDHDIV Mask               */

#define CLK_CLKDIV1_ADCSEL_Pos           (28)                                              /*!< CLK_T::CLKDIV1: ADCSEL Position           */
#define CLK_CLKDIV1_ADCSEL_Msk           (0x1ul << CLK_CLKDIV1_ADCSEL_Pos)                 /*!< CLK_T::CLKDIV1: ADCSEL Mask               */

#define CLK_CLKDIV1_SDHSEL_Pos           (30)                                              /*!< CLK_T::CLKDIV1: SDHSEL Position           */
#define CLK_CLKDIV1_SDHSEL_Msk           (0x1ul << CLK_CLKDIV1_SDHSEL_Pos)                 /*!< CLK_T::CLKDIV1: SDHSEL Mask               */

#define CLK_CLKDIV2_I2SDIV_Pos           (0)                                               /*!< CLK_T::CLKDIV2: I2SDIV Position           */
#define CLK_CLKDIV2_I2SDIV_Msk           (0xfful << CLK_CLKDIV2_I2SDIV_Pos)                /*!< CLK_T::CLKDIV2: I2SDIV Mask               */

#define CLK_CLKDIV2_I2SSEL_Pos           (24)                                              /*!< CLK_T::CLKDIV2: I2SSEL Position           */
#define CLK_CLKDIV2_I2SSEL_Msk           (0x3ul << CLK_CLKDIV2_I2SSEL_Pos)                 /*!< CLK_T::CLKDIV2: I2SSEL Mask               */

#define CLK_CLKDIV2_SPI0SEL_Pos          (28)                                              /*!< CLK_T::CLKDIV2: SPI0SEL Position          */
#define CLK_CLKDIV2_SPI0SEL_Msk          (0x1ul << CLK_CLKDIV2_SPI0SEL_Pos)                /*!< CLK_T::CLKDIV2: SPI0SEL Mask              */

#define CLK_CLKDIV2_SPI1SEL_Pos          (29)                                              /*!< CLK_T::CLKDIV2: SPI1SEL Position          */
#define CLK_CLKDIV2_SPI1SEL_Msk          (0x1ul << CLK_CLKDIV2_SPI1SEL_Pos)                /*!< CLK_T::CLKDIV2: SPI1SEL Mask              */

#define CLK_CLKDIV3_UART0DIV_Pos         (0)                                               /*!< CLK_T::CLKDIV3: UART0DIV Position         */
#define CLK_CLKDIV3_UART0DIV_Msk         (0xful << CLK_CLKDIV3_UART0DIV_Pos)               /*!< CLK_T::CLKDIV3: UART0DIV Mask             */

#define CLK_CLKDIV3_UART0SEL_Pos         (4)                                               /*!< CLK_T::CLKDIV3: UART0SEL Position         */
#define CLK_CLKDIV3_UART0SEL_Msk         (0x1ul << CLK_CLKDIV3_UART0SEL_Pos)               /*!< CLK_T::CLKDIV3: UART0SEL Mask             */

#define CLK_CLKDIV3_UART1DIV_Pos         (8)                                               /*!< CLK_T::CLKDIV3: UART1DIV Position         */
#define CLK_CLKDIV3_UART1DIV_Msk         (0xful << CLK_CLKDIV3_UART1DIV_Pos)               /*!< CLK_T::CLKDIV3: UART1DIV Mask             */

#define CLK_CLKDIV3_UART1SEL_Pos         (12)                                              /*!< CLK_T::CLKDIV3: UART1SEL Position         */
#define CLK_CLKDIV3_UART1SEL_Msk         (0x1ul << CLK_CLKDIV3_UART1SEL_Pos)               /*!< CLK_T::CLKDIV3: UART1SEL Mask             */

#define CLK_CLKDIV3_UART2DIV_Pos         (16)                                              /*!< CLK_T::CLKDIV3: UART2DIV Position         */
#define CLK_CLKDIV3_UART2DIV_Msk         (0xful << CLK_CLKDIV3_UART2DIV_Pos)               /*!< CLK_T::CLKDIV3: UART2DIV Mask             */

#define CLK_CLKDIV3_UART2SEL_Pos         (20)                                              /*!< CLK_T::CLKDIV3: UART2SEL Position         */
#define CLK_CLKDIV3_UART2SEL_Msk         (0x1ul << CLK_CLKDIV3_UART2SEL_Pos)               /*!< CLK_T::CLKDIV3: UART2SEL Mask             */

#define CLK_PLLCTL_FBDIV_Pos             (0)                                               /*!< CLK_T::PLLCTL: FBDIV Position             */
#define CLK_PLLCTL_FBDIV_Msk             (0x7ful << CLK_PLLCTL_FBDIV_Pos)                  /*!< CLK_T::PLLCTL: FBDIV Mask                 */

#define CLK_PLLCTL_INDIV_Pos             (7)                                               /*!< CLK_T::PLLCTL: INDIV Position             */
#define CLK_PLLCTL_INDIV_Msk             (0x3ful << CLK_PLLCTL_INDIV_Pos)                  /*!< CLK_T::PLLCTL: INDIV Mask                 */

#define CLK_PLLCTL_OUTDIV_Pos            (13)                                              /*!< CLK_T::PLLCTL: OUTDIV Position            */
#define CLK_PLLCTL_OUTDIV_Msk            (0x7ul << CLK_PLLCTL_OUTDIV_Pos)                  /*!< CLK_T::PLLCTL: OUTDIV Mask                */

#define CLK_PLLCTL_BP_Pos                (16)                                              /*!< CLK_T::PLLCTL: BP Position                */
#define CLK_PLLCTL_BP_Msk                (0x1ul << CLK_PLLCTL_BP_Pos)                      /*!< CLK_T::PLLCTL: BP Mask                    */

#define CLK_PLLCTL_PD_Pos                (17)                                              /*!< CLK_T::PLLCTL: PD Position                */
#define CLK_PLLCTL_PD_Msk                (0x1ul << CLK_PLLCTL_PD_Pos)                      /*!< CLK_T::PLLCTL: PD Mask                    */

#define CLK_APLLCTL_INDIV_Pos            (0)                                               /*!< CLK_T::APLLCTL: INDIV Position            */
#define CLK_APLLCTL_INDIV_Msk            (0x3ful << CLK_APLLCTL_INDIV_Pos)                 /*!< CLK_T::APLLCTL: INDIV Mask                */

#define CLK_APLLCTL_FBDIV_Pos            (6)                                               /*!< CLK_T::APLLCTL: FBDIV Position            */
#define CLK_APLLCTL_FBDIV_Msk            (0x7ful << CLK_APLLCTL_FBDIV_Pos)                 /*!< CLK_T::APLLCTL: FBDIV Mask                */

#define CLK_APLLCTL_OUTDIV_Pos           (13)                                              /*!< CLK_T::APLLCTL: OUTDIV Position           */
#define CLK_APLLCTL_OUTDIV_Msk           (0x7ul << CLK_APLLCTL_OUTDIV_Pos)                 /*!< CLK_T::APLLCTL: OUTDIV Mask               */

#define CLK_APLLCTL_PD_Pos               (16)                                              /*!< CLK_T::APLLCTL: PD Position               */
#define CLK_APLLCTL_PD_Msk               (0x1ul << CLK_APLLCTL_PD_Pos)                     /*!< CLK_T::APLLCTL: PD Mask                   */

#define CLK_APLLCTL_MODE_Pos             (18)                                              /*!< CLK_T::APLLCTL: MODE Position             */
#define CLK_APLLCTL_MODE_Msk             (0x1ul << CLK_APLLCTL_MODE_Pos)                   /*!< CLK_T::APLLCTL: MODE Mask                 */

#define CLK_APLLCTL_FRAC_Pos             (20)                                              /*!< CLK_T::APLLCTL: FRAC Position             */
#define CLK_APLLCTL_FRAC_Msk             (0xffful << CLK_APLLCTL_FRAC_Pos)                 /*!< CLK_T::APLLCTL: FRAC Mask                 */

#define CLK_CLKDIV4_TMR0DIV_Pos          (0)                                               /*!< CLK_T::CLKDIV4: TMR0DIV Position          */
#define CLK_CLKDIV4_TMR0DIV_Msk          (0xfful << CLK_CLKDIV4_TMR0DIV_Pos)               /*!< CLK_T::CLKDIV4: TMR0DIV Mask              */

#define CLK_CLKDIV4_TMR1DIV_Pos          (8)                                               /*!< CLK_T::CLKDIV4: TMR1DIV Position          */
#define CLK_CLKDIV4_TMR1DIV_Msk          (0xfful << CLK_CLKDIV4_TMR1DIV_Pos)               /*!< CLK_T::CLKDIV4: TMR1DIV Mask              */

#define CLK_CLKDIV4_TMR2DIV_Pos          (16)                                              /*!< CLK_T::CLKDIV4: TMR2DIV Position          */
#define CLK_CLKDIV4_TMR2DIV_Msk          (0xfful << CLK_CLKDIV4_TMR2DIV_Pos)               /*!< CLK_T::CLKDIV4: TMR2DIV Mask              */

#define CLK_CLKDIV4_TMR0SEL_Pos          (24)                                              /*!< CLK_T::CLKDIV4: TMR0SEL Position          */
#define CLK_CLKDIV4_TMR0SEL_Msk          (0x1ul << CLK_CLKDIV4_TMR0SEL_Pos)                /*!< CLK_T::CLKDIV4: TMR0SEL Mask              */

#define CLK_CLKDIV4_TMR1SEL_Pos          (25)                                              /*!< CLK_T::CLKDIV4: TMR1SEL Position          */
#define CLK_CLKDIV4_TMR1SEL_Msk          (0x1ul << CLK_CLKDIV4_TMR1SEL_Pos)                /*!< CLK_T::CLKDIV4: TMR1SEL Mask              */

#define CLK_CLKDIV4_TMR2SEL_Pos          (26)                                              /*!< CLK_T::CLKDIV4: TMR2SEL Position          */
#define CLK_CLKDIV4_TMR2SEL_Msk          (0x1ul << CLK_CLKDIV4_TMR2SEL_Pos)                /*!< CLK_T::CLKDIV4: TMR2SEL Mask              */

#define CLK_CLKDIV5_TMR3DIV_Pos          (0)                                               /*!< CLK_T::CLKDIV5: TMR3DIV Position          */
#define CLK_CLKDIV5_TMR3DIV_Msk          (0xfful << CLK_CLKDIV5_TMR3DIV_Pos)               /*!< CLK_T::CLKDIV5: TMR3DIV Mask              */

#define CLK_CLKDIV5_WDTDIV_Pos           (8)                                               /*!< CLK_T::CLKDIV5: WDTDIV Position           */
#define CLK_CLKDIV5_WDTDIV_Msk           (0xfful << CLK_CLKDIV5_WDTDIV_Pos)                /*!< CLK_T::CLKDIV5: WDTDIV Mask               */

#define CLK_CLKDIV5_PWMDIV_Pos           (16)                                              /*!< CLK_T::CLKDIV5: PWMDIV Position           */
#define CLK_CLKDIV5_PWMDIV_Msk           (0xfful << CLK_CLKDIV5_PWMDIV_Pos)                /*!< CLK_T::CLKDIV5: PWMDIV Mask               */

#define CLK_CLKDIV5_TMR3SEL_Pos          (24)                                              /*!< CLK_T::CLKDIV5: TMR3SEL Position          */
#define CLK_CLKDIV5_TMR3SEL_Msk          (0x1ul << CLK_CLKDIV5_TMR3SEL_Pos)                /*!< CLK_T::CLKDIV5: TMR3SEL Mask              */

#define CLK_CLKDIV5_WDTSEL_Pos           (25)                                              /*!< CLK_T::CLKDIV5: WDTSEL Position           */
#define CLK_CLKDIV5_WDTSEL_Msk           (0x1ul << CLK_CLKDIV5_WDTSEL_Pos)                 /*!< CLK_T::CLKDIV5: WDTSEL Mask               */

#define CLK_CLKDIV5_PWMSEL_Pos           (26)                                              /*!< CLK_T::CLKDIV5: PWMSEL Position           */
#define CLK_CLKDIV5_PWMSEL_Msk           (0x1ul << CLK_CLKDIV5_PWMSEL_Pos)                 /*!< CLK_T::CLKDIV5: PWMSEL Mask               */

/**@}*/ /* CLK_CONST */
/**@}*/ /* end of CLK register group */


/*---------------------- System Manager Controller -------------------------*/
/**
    @addtogroup SYS System Manager Controller (SYS)
    Memory Mapped Structure for SYS Controller
@{ */

typedef struct
{


    /**
     * PDID
     * ===================================================================================================
     * Offset: 0x00  Part Device Identification Number Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |PDID      |Part Device Identification Number (Read Only)
     * |        |          |This register reflects device part number code.
     * |        |          |Software can read this register to identify which device is use.
    */
    __I  uint32_t PDID;

    /**
     * BOOTSET
     * ===================================================================================================
     * Offset: 0x04  System Power-on Configuration Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |BOOTSET   |System Mode Configuration
     * |        |          |0110 = Boot from ICE Mode with external SPI Flash.
     * |        |          |0111 = Boot from ICE Mode with SPI Flash.
     * |        |          |1011 = Boot from ICP Mode.
     * |        |          |1101 = Boot from external SPI Flash.
     * |        |          |1110 = Boot from USB.
     * |        |          |1111 = Boot from SPI Flash.
     * |        |          |Note: If BOOTSET is equal to ICE Mode, the software cannot change BOOTSET to other mode.
     * |        |          |But other modes don't have this limitation.
    */
    __IO uint32_t BOOTSET;

    /**
     * IPRST0
     * ===================================================================================================
     * Offset: 0x08  Peripheral Reset Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CPURST    |Processor Core One-shot Reset
     * |        |          |Setting this bit will only reset the processor core, and this bit will automatically return to 0 after the 2 clock cycles.
     * |        |          |0 = Processor core normal operation.
     * |        |          |1 = Processor core one-shot reset.
     * |[1]     |CHIPRST   |Chip One-shot Reset
     * |        |          |Setting this bit will reset the whole chip, including processor core and all peripherals, and this bit will automatically return to 0 after the 2 clock cycles.
     * |        |          |The CHIPRST is same as the POR reset, all the chip controllers is reset and the chip setting from flash are also reload.
     * |        |          |0 = Chip normal operation.
     * |        |          |1 = Chip one-shot reset.
    */
    __IO uint32_t IPRST0;

    /**
     * LVDCTL
     * ===================================================================================================
     * Offset: 0x0C  Low Voltage Detection Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LVDIF     |Low Voltage Detect Flag
     * |        |          |0 = Low voltage Period.
     * |        |          |1 = Normal voltage Period.
     * |        |          |Note: This bit is useful when LVDEN is enabled.
     * |[1]     |LVDSEL    |Low Voltage Detection Level Selection
     * |        |          |0 = The threshold level is 2.6V.
     * |        |          |1 = The threshold level is 2.8V.
     * |[2]     |LVDEN     |Low Voltage Detection Enable Control
     * |        |          |0 = Detection Disabled.
     * |        |          |1 = Detection Enabled.
     * |[3]     |LVREN     |Low Voltage Reset Enable Control
     * |        |          |0 = Low Voltage Reset Disabled.
     * |        |          |1 = Low Voltage Reset Enabled. (default)
     * |        |          |Note: The voltage threshold level is 2.4V
     * |[4]     |PORENB    |Power on Reset Enable Control
     * |        |          |0 = Function Enabled. (Default)
     * |        |          |1 = Function Disabled.
    */
    __IO uint32_t LVDCTL;

    /**
     * WAKEUP
     * ===================================================================================================
     * Offset: 0x10  Wake-Up Control and Status Resister
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |I2C0WE    |I2C0 Wake-up Enable Control
     * |        |          |0 = I2C0 wake-up Disabled.
     * |        |          |1 = I2C0 wake-up Enabled.
     * |[1]     |I2C1WE    |I2C1 Wake-up Enable Control
     * |        |          |0 = I2C1 wake-up Disabled.
     * |        |          |1 = I2C1 wake-up Enabled.
     * |[2]     |GPIOWE    |GPIO Wake-up Enable Control
     * |        |          |0 = GPIO wake-up Disabled.
     * |        |          |1 = GPIO wake-up Enabled.
     * |[3]     |RTCWE     |RTC Wake-up Enable Control
     * |        |          |0 = RTC wake-up Disabled.
     * |        |          |1 = RTC wake-up Enabled.
     * |[4]     |WDTWE     |WDT Wake-up Enable Control
     * |        |          |0 = WDT wake-up Disabled.
     * |        |          |1 = WDT wake-up Enabled.
     * |[5]     |TMR0WE    |Timer0 Wake-up Enable Control
     * |        |          |0 = Timer0 wake-up Disabled.
     * |        |          |1 = Timer0 wake-up Enabled.
     * |[6]     |TMR1WE    |Timer1 Wake-up Enable Control
     * |        |          |0 = Timer1 wake-up Disabled.
     * |        |          |1 = Timer1 wake-up Enabled.
     * |[7]     |TMR2WE    |Timer2 Wake-up Enable Control
     * |        |          |0 = Timer2 wake-up Disabled.
     * |        |          |1 = Timer2 wake-up Enabled.
     * |[8]     |TMR3WE    |Timer3 Wake-up Enable Control
     * |        |          |0 = Timer3 wake-up Disabled.
     * |        |          |1 = Timer3 wake-up Enabled.
     * |[9]     |UART0WE   |UART0 Wake-up Enable Control
     * |        |          |0 = UART0 wake-up Disabled.
     * |        |          |1 = UART0 wake-up Enabled.
     * |[10]    |UART1WE   |UART1 Wake-up Enable Control
     * |        |          |0 = UART1 wake-up Disabled.
     * |        |          |1 = UART1 wake-up Enabled.
     * |[11]    |UART2WE   |UART2 Wake-up Enable Control
     * |        |          |0 = UART2 wake-up Disabled.
     * |        |          |1 = UART2 wake-up Enabled.
     * |[12]    |USBDWE    |USB Device Wake-up Enable Control
     * |        |          |0 = USB Device wake-up Disabled.
     * |        |          |1 = USB Device wake-up Enabled.
     * |[13]    |USBHWE    |USB Host Wake-up Enable Control
     * |        |          |0 = USB Host wake-up Disabled.
     * |        |          |1 = USB Host wake-up Enabled.
     * |[16]    |I2C0WF    |I2C0 Wake-up Flag
     * |        |          |0 = I2C0 wake-up source is deasserted.
     * |        |          |1 = I2C0 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[17]    |I2C1WF    |I2C1 Wake-up Flag
     * |        |          |0 = I2C1 wake-up source is deasserted.
     * |        |          |1 = I2C1 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[18]    |GPIOWF    |GPIO Wake-up Flag
     * |        |          |0 = GPIO wake-up source is deasserted.
     * |        |          |1 = GPIO wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[19]    |RTCWF     |RTC Wake-up Flag
     * |        |          |0 = RTC wake-up source is deasserted.
     * |        |          |1 = RTC wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[20]    |WDTWF     |WDT Wake-up Flag
     * |        |          |0 = WDT wake-up source is deasserted.
     * |        |          |1 = WDT wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[21]    |TMR0WF    |Timer0 Wake-up Flag
     * |        |          |0 = Timer0 wake-up source is deasserted.
     * |        |          |1 = Timer0 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[22]    |TMR1WF    |Timer1 Wake-up Flag
     * |        |          |0 = Timer1 wake-up source is deasserted.
     * |        |          |1 = Timer1 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[23]    |TMR2WF    |Timer2 Wake-up Flag
     * |        |          |0 = Timer2 wake-up source is deasserted.
     * |        |          |1 = Timer2 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[24]    |TMR3WF    |Timer3 Wake-up Flag
     * |        |          |0 = Timer3 wake-up source is deasserted.
     * |        |          |1 = Timer3 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[25]    |UART0WF   |UART0 Wake-up Flag
     * |        |          |0 = UART0 wake-up source is deasserted.
     * |        |          |1 = UART0 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[26]    |UART1WF   |UART1 Wake-up Flag
     * |        |          |0 = UART1 wake-up source is deasserted.
     * |        |          |1 = UART1 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[27]    |UART2WF   |UART2 Wake-up Flag
     * |        |          |0 = UART2 wake-up source is deasserted.
     * |        |          |1 = UART2 wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[28]    |USBDWF    |USB Device Wake-up Flag
     * |        |          |0 = USB Device wake-up source is deasserted.
     * |        |          |1 = USB Device wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
     * |[29]    |USBHWF    |USB Host Wake-up Flag
     * |        |          |0 = USB Host wake-up source is deasserted.
     * |        |          |1 = USB Host wake-up source is asserted.
     * |        |          |Note: Write 1 to clear this bit
    */
    __IO uint32_t WAKEUP;

    /**
     * IPRST1
     * ===================================================================================================
     * Offset: 0x14  Peripheral Reset Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |UART0RST  |UART0 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = UART0 controller normal operation.
     * |        |          |1 = UART0 controller reset.
     * |[1]     |UART1RST  |UART1 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = UART1 controller normal operation.
     * |        |          |1 = UART1 controller reset.
     * |[2]     |TMR0RST   |Timer0 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = Timer0 controller normal operation.
     * |        |          |1 = Timer0 controller reset.
     * |[3]     |TMR1RST   |Timer1 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = Timer1 controller normal operation.
     * |        |          |1 = Timer1 controller reset.
     * |[4]     |WDTFRST   |WDT Hardware Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = WDT controller normal operation.
     * |        |          |1 = WDT controller reset.
     * |[5]     |TMR2RST   |Timer2 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = Timer2 controller normal operation.
     * |        |          |1 = Timer2 controller reset.
     * |[6]     |UART2RST  |UART2 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = UART2 controller normal operation.
     * |        |          |1 = UART2 controller reset.
     * |[7]     |PWMRST    |PWM Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = PWM controller normal operation.
     * |        |          |1 = PWM controller reset.
     * |[8]     |I2C0RST   |I2C0 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = I2C0 controller normal operation.
     * |        |          |1 = I2C0 controller reset.
     * |[9]     |I2C1RST   |I2C1 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = I2C1 controller normal operation.
     * |        |          |1 = I2C1 controller reset.
     * |[10]    |SPIMRST   |SPIM Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = SPIM controller normal operation.
     * |        |          |1 = SPIM controller reset.
     * |[11]    |USBDRST   |USB Device Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = USB Device controller normal operation.
     * |        |          |1 = USB Device controller reset.
     * |[12]    |TMR3RST   |Timer3 Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = Timer3 controller normal operation.
     * |        |          |1 = Timer3 controller reset.
     * |[13]    |WDTPRST   |WDT Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = WDT controller normal operation.
     * |        |          |1 = WDT controller reset.
     * |[17]    |I2SRST    |I2S Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = I2S controller normal operation.
     * |        |          |1 = I2S controller reset.
     * |[18]    |USBHRST   |USB Host Controller Reset
     * |        |          |Setting this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = USB Host controller normal operation.
     * |        |          |1 = USB Host controller reset.
     * |[24]    |SDHRST    |SDH Controller Reset
     * |        |          |Set this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = SDH controller normal operation.
     * |        |          |1 = SDH controller reset.
     * |[25]    |SRAMRST   |SRAM Controller Reset
     * |        |          |Set this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = SRAM controller normal operation.
     * |        |          |1 = SRAM controller reset.
     * |[27]    |GPIORST   |GPIO Controller Reset
     * |        |          |Set this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = GPIO controller normal operation.
     * |        |          |1 = GPIO controller reset.
     * |[28]    |ADCRST    |ADC Controller Reset
     * |        |          |Set this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = ADC controller normal operation.
     * |        |          |1 = ADC controller reset.
     * |[30]    |SPI0RST   |SPI0 Controller Reset
     * |        |          |Set this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = SPI0 controller normal operation.
     * |        |          |1 = SPI0 controller reset.
     * |[31]    |SPI1RST   |SPI1 Controller Reset
     * |        |          |Set this bit to 1 will generate a reset signal to the USB host controller.
     * |        |          |User needs to set this bit to 0 to release from the reset stat.
     * |        |          |0 = SPI1 controller normal operation.
     * |        |          |1 = SPI1 controller reset.
    */
    __IO uint32_t IPRST1;

    /**
     * NMICTL
     * ===================================================================================================
     * Offset: 0x18  Non Maskable Interrupt Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RTCIEN    |RTC Interrupt NMI Source Enable Control
     * |        |          |0 = RTC NMI source Disabled.
     * |        |          |1 = RTC NMI source Enabled.
     * |[1]     |WDTIEN    |WDT Interrupt NMI Source Enable Control
     * |        |          |0 = WDT NMI source Disabled.
     * |        |          |1 = WDT NMI source Enabled.
     * |[2]     |PORIEN    |Power on Interrupt NMI Source Enable Control
     * |        |          |0 = POR NMI source Disabled.
     * |        |          |1 = POR NMI source Enabled.
     * |[3]     |EINT0IEN  |External GPIO Group 0 NMI Source Enable Control
     * |        |          |0 = External GPIO group 0 NMI source Disabled.
     * |        |          |1 = External GPIO group 0 NMI source Enabled.
     * |[4]     |EINT1IEN  |External GPIO Group 1 NMI Source Enable Control
     * |        |          |0 = External GPIO group 1 NMI source Disabled.
     * |        |          |1 = External GPIO group 1 NMI source Enabled.
     * |[5]     |EINT2IEN  |External GPIO Group 2 NMI Source Enable Control
     * |        |          |0 = External GPIO group 2 NMI source Disabled.
     * |        |          |1 = External GPIO group 2 NMI source Enabled.
     * |[6]     |EINT3IEN  |External GPIO Group 3 NMI Source Enable Control
     * |        |          |0 = External GPIO group 3 NMI source Disabled.
     * |        |          |1 = External GPIO group 3 NMI source Enabled.
     * |[7]     |LVDIEN    |Low Voltage Detect NMI Source Enable Control
     * |        |          |0 = LVD NMI source Disabled.
     * |        |          |1 = LVD NMI source Enabled.
     * |[16]    |RTCIF     |RTC Interrupt Flag (Read Only)
     * |        |          |0 = RTC interrupt is deasserted.
     * |        |          |1 = RTC interrupt is asserted.
     * |[17]    |WDTIF     |Watch Dog Timer (WDT) Interrupt Flag (Read Only)
     * |        |          |0 = WDT interrupt is deasserted.
     * |        |          |1 = WDT interrupt is asserted.
     * |[18]    |PORIF     |Power on Reset (POR) Interrupt Flag (Read Only)
     * |        |          |0 = POR interrupt is deasserted.
     * |        |          |1 = POR interrupt is asserted.
     * |[19]    |EINT0IF   |External GPIO Group 0 Interrupt Flag (Read Only)
     * |        |          |0 = External GPIO group 0 interrupt is deasserted.
     * |        |          |1 = External GPIO group 0 interrupt is asserted.
     * |[20]    |EINT1IF   |External GPIO Group 1 Interrupt Flag (Read Only)
     * |        |          |0 = External GPIO group 1 interrupt is deasserted.
     * |        |          |1 = External GPIO group 1 interrupt is asserted.
     * |[21]    |EINT2IF   |External GPIO Group 2 Interrupt Flag (Read Only)
     * |        |          |0 = External GPIO group 2 interrupt is deasserted.
     * |        |          |1 = External GPIO group 2 interrupt is asserted.
     * |[22]    |EINT3IF   |External GPIO Group 3 Interrupt Flag (Read Only)
     * |        |          |0 = External GPIO group 3 interrupt is deasserted.
     * |        |          |1 = External GPIO group 3 interrupt is asserted.
     * |[23]    |LVDIF     |Low Voltage Detect (LVD) Interrupt Flag (Read Only)
     * |        |          |0 = LVD interrupt is deasserted.
     * |        |          |1 = LVD interrupt is asserted.
    */
    __IO uint32_t NMICTL;

    /**
     * RSTSTS
     * ===================================================================================================
     * Offset: 0x1C  Reset Status Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PINRF     |Reset Status for External nReset Pin
     * |        |          |0 = No effect.
     * |        |          |1 = nRESET pin had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[1]     |LVRF      |Reset Status for Low Voltage Reset
     * |        |          |0 = No effect.
     * |        |          |1 = LVR controller had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[2]     |WDTRF     |Reset Status for Watch Dog Reset
     * |        |          |0 = No effect.
     * |        |          |1 = The watchdog timer or window watchdog timer had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[3]     |CPURF     |Reset Status for Software Setting
     * |        |          |0 = No effect.
     * |        |          |1 = The CPURST had be triggered to reset the CPU or the CHIPREST had be triggered to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[4]     |PORF      |Reset Status for POR Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Power-on Reset (POR) had issued the reset signal to reset the system.
     * |        |          |Note: Write 1 to clear this bit to 0.
    */
    __IO uint32_t RSTSTS;

    /**
     * AHBCTL
     * ===================================================================================================
     * Offset: 0x20  AHB Bus Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PRISEL    |AHB Bus Arbitration Mode Control
     * |        |          |0 = Fixed priority mode.
     * |        |          |1 = Round-robin priority mode (rotate).
     * |        |          |The priority mode for fixed priority mode is I2S > SDH > USBH > USBD > SPIM > M4(S) > M4(D) > M4(I)
     * |[4]     |CPUHPRI   |Enable Raising the Priority of CPU in IRQ Period
     * |        |          |It can be used to reduce the interrupt latency in a real-time system
     * |        |          |Setting this bit, the CPU will have the highest AHB priority.
     * |        |          |0 = No effect.
     * |        |          |1 = The function that CPU has the highest AHB bus priority in IRQ period Enabled.
     * |[5]     |PRISTS    |Interrupt Active Status in CPUHPRI Enabled Mode
     * |        |          |If it is high, the CPU has the highest AHB bus priority
     * |        |          |It is set when the CPUHPRI is enabled and the external IRQ is active
     * |        |          |This bit is cleared by writing 1 to it
     * |        |          |Therefore, if exiting from the IRQ when CPUHPRI is enabled, PRISTS must be cleared
     * |        |          |Otherwise, the CPU always has the highest AHB bus priority.
     * |        |          |0 = No effect.
     * |        |          |1 = The highest AHB bus priority for CPU is active.
    */
    __IO uint32_t AHBCTL;
    uint32_t RESERVE0[3];


    /**
     * GPA_MFPL
     * ===================================================================================================
     * Offset: 0x30  GPIOA Low Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PA0MFP    |PA.0 Multi-function Pin Selection
     * |[6:4]   |PA1MFP    |PA.1 Multi-function Pin Selection
     * |[10:8]  |PA2MFP    |PA.2 Multi-function Pin Selection
     * |[14:12] |PA3MFP    |PA.3 Multi-function Pin Selection
     * |[18:16] |PA4MFP    |PA.4 Multi-function Pin Selection
     * |[22:20] |PA5MFP    |PA.5 Multi-function Pin Selection
     * |[26:24] |PA6MFP    |PA.6 Multi-function Pin Selection
     * |[30:28] |PA7MFP    |PA.7 Multi-function Pin Selection
    */
    __IO uint32_t GPA_MFPL;

    /**
     * GPA_MFPH
     * ===================================================================================================
     * Offset: 0x34  GPIOA High Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PA8MFP    |PA.8 Multi-function Pin Selection
     * |[6:4]   |PA9MFP    |PA.9 Multi-function Pin Selection
     * |[10:8]  |PA10MFP   |PA.10 Multi-function Pin Selection
     * |[14:12] |PA11MFP   |PA.11 Multi-function Pin Selection
     * |[18:16] |PA12MFP   |PA.12 Multi-function Pin Selection
     * |[22:20] |PA13MFP   |PA.13 Multi-function Pin Selection
     * |[26:24] |PA14MFP   |PA.14 Multi-function Pin Selection
     * |[30:28] |PA15MFP   |PA.15 Multi-function Pin Selection
    */
    __IO uint32_t GPA_MFPH;

    /**
     * GPB_MFPL
     * ===================================================================================================
     * Offset: 0x38  GPIOB Low Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PB0MFP    |PB.0 Multi-function Pin Selection
     * |[6:4]   |PB1MFP    |PB.1 Multi-function Pin Selection
     * |[10:8]  |PB2MFP    |PB.2 Multi-function Pin Selection
     * |[14:12] |PB3MFP    |PB.3 Multi-function Pin Selection
     * |[18:16] |PB4MFP    |PB.4 Multi-function Pin Selection
     * |[22:20] |PB5MFP    |PB.5 Multi-function Pin Selection
     * |[26:24] |PB6MFP    |PB.6 Multi-function Pin Selection
     * |[30:28] |PB7MFP    |PB.7 Multi-function Pin Selection
    */
    __IO uint32_t GPB_MFPL;

    /**
     * GPB_MFPH
     * ===================================================================================================
     * Offset: 0x3C  GPIOB High Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PB8MFP    |PB.8 Multi-function Pin Selection
     * |[6:4]   |PB9MFP    |PB.9 Multi-function Pin Selection
     * |[10:8]  |PB10MFP   |PB.10 Multi-function Pin Selection
     * |[14:12] |PB11MFP   |PB.11 Multi-function Pin Selection
     * |[18:16] |PB12MFP   |PB.12 Multi-function Pin Selection
     * |[22:20] |PB13MFP   |PB.13 Multi-function Pin Selection
     * |[26:24] |PB14MFP   |PB.14 Multi-function Pin Selection
     * |[30:28] |PB15MFP   |PB.15 Multi-function Pin Selection
    */
    __IO uint32_t GPB_MFPH;

    /**
     * GPC_MFPL
     * ===================================================================================================
     * Offset: 0x40  GPIOC Low Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PC0MFP    |PC.0 Multi-function Pin Selection
     * |[6:4]   |PC1MFP    |PC.1 Multi-function Pin Selection
     * |[10:8]  |PC2MFP    |PC.2 Multi-function Pin Selection
     * |[14:12] |PC3MFP    |PC.3 Multi-function Pin Selection
     * |[18:16] |PC4MFP    |PC.4 Multi-function Pin Selection
     * |[22:20] |PC5MFP    |PC.5 Multi-function Pin Selection
     * |[26:24] |PC6MFP    |PC.6 Multi-function Pin Selection
     * |[30:28] |PC7MFP    |PC.7 Multi-function Pin Selection
    */
    __IO uint32_t GPC_MFPL;

    /**
     * GPC_MFPH
     * ===================================================================================================
     * Offset: 0x44  GPIOC High Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PC8MFP    |PC.8 Multi-function Pin Selection
     * |[6:4]   |PC9MFP    |PC.9 Multi-function Pin Selection
     * |[10:8]  |PC10MFP   |PC.10 Multi-function Pin Selection
     * |[14:12] |PC11MFP   |PC.11 Multi-function Pin Selection
     * |[18:16] |PC12MFP   |PC.12 Multi-function Pin Selection
     * |[22:20] |PC13MFP   |PC.13 Multi-function Pin Selection
     * |[26:24] |PC14MFP   |PC.14 Multi-function Pin Selection
    */
    __IO uint32_t GPC_MFPH;

    /**
     * GPD_MFPL
     * ===================================================================================================
     * Offset: 0x48  GPIOD Low Byte Multi-function Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PD0MFP    |PD.0 Multi-function Pin Selection
     * |[6:4]   |PD1MFP    |PD.1 Multi-function Pin Selection
     * |[10:8]  |PD2MFP    |PD.2 Multi-function Pin Selection
     * |[14:12] |PD3MFP    |PD.3 Multi-function Pin Selection
     * |[18:16] |PD4MFP    |PD.4 Multi-function Pin Selection
    */
    __IO uint32_t GPD_MFPL;
    uint32_t RESERVE1[1];


    /**
     * LVMPADDR
     * ===================================================================================================
     * Offset: 0x50  Load VECMAP Address Parameter Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |Load VECMAP Address Register
     * |        |          |This is the start address for mapping to the address 0x0000_0000 in VECMAP function.
     * |        |          |Only when CPU reset or setting RLDVMP to 1, the loading signal will be loaded to the SYS_RVMPADDR registe.
     * |        |          |Note: This register can only be reset by CHIPRST and HW Reset.
    */
    __IO uint32_t LVMPADDR;

    /**
     * LVMPLEN
     * ===================================================================================================
     * Offset: 0x54  Load VECMAP Length Parameter Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |LEN       |LD_VECMAP Length
     * |        |          |This is the memory length loading signal for mapping to the address 0x0000_0000 in VECMAP function.
     * |        |          |Only when OCPU reset or setting RLDVMP to 1, the loading signal will be loaded to the SYS_RVMPLEN registe.
     * |        |          |Mapping memory length = VMP_LEN * 1K (bytes).
     * |        |          |Note1: The maximum mapping length is 128K bytes
     * |        |          |Note2: This register only can be reset by CHIPRST and HW Reset.
    */
    __IO uint32_t LVMPLEN;

    /**
     * RVMPADDR
     * ===================================================================================================
     * Offset: 0x58  Real VECMAP Address Parameter Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |Real VECMAP Address Register Parameter
     * |        |          |This is the real start address parameter for mapping to the address 0x0000_0000 in VECMAP function.
     * |        |          |(Default value is mapping to IBR_ROM start address.).
     * |        |          |Note: This register is only loaded from ADDR during CPU Reset process (SYS_IPRST0[0]) or setting RLDVMP.
    */
    __I  uint32_t RVMPADDR;

    /**
     * RVMPLEN
     * ===================================================================================================
     * Offset: 0x5C  Real VECMAP Length Parameter Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RLDVMP    |Load VECMAP Parameter Signal
     * |        |          |0 = No effect.
     * |        |          |1 = Load VECMAP Address and Length.
     * |        |          |Note: This bit is auto cleared to 0
     * |[31:24] |LEN       |Real VECMAP Length
     * |        |          |This is the real memory length for mapping to the address 0x0000_0000 in VECMAP function.
     * |        |          |Mapping memory length = VMP_LEN * 1K (bytes).
     * |        |          |Note1: Read Only
     * |        |          |Note2: These bits are only loaded from LEN when setting CPURST (SYS_IPRST0[0]) or setting RLDVMP.
    */
    __IO uint32_t RVMPLEN;
    uint32_t RESERVE2[3];


    /**
     * EPADPUEN
     * ===================================================================================================
     * Offset: 0x6C  Embedded SPI Flash Pad Pull-up Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[4:0]   |EPADPUEN  |Embedded SPI Flash Pad Pull-up Enable Control
     * |        |          |00000 = All embedded pads are pull-up Disabled.
     * |        |          |10010 = Only SPI Flash MISO pads are pull-up Enabled.
     * |        |          |11111 = All embedded pads are pull-up Enabled.
     * |        |          |Other = Reserved.
     * |        |          |Note: In Power-down mode, user should set EPADPUEN[4:0] to 0x12.
    */
    __IO uint32_t EPADPUEN;

    /**
     * GPADS
     * ===================================================================================================
     * Offset: 0x70  GPIOA Driving Strength Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PA0DS     |PA.0 Driving Strength Control
     * |        |          |Setting driving strength for PA.0 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[6:4]   |PA1DS     |PA.1 Driving Strength Control
     * |        |          |Setting driving strength for PA.1 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[10:8]  |PA2DS     |PA.2 Driving Strength Control
     * |        |          |Setting driving strength for PA.2 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[14:12] |PA3DS     |PA.3 Driving Strength Control
     * |        |          |Setting driving strength for PA.3 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[18:16] |PA4DS     |PA.4 Driving Strength Control
     * |        |          |Setting driving strength for PA.4 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[22:20] |PA5DS     |PA.5 Driving Strength Control
     * |        |          |Setting driving strength for PA.5 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[26:24] |PA6DS     |PA.6 Driving Strength Control
     * |        |          |Setting driving strength for PA.6 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[30:28] |PA7DS     |PA.7 Driving Strength Control
     * |        |          |Setting driving strength for PA.7 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
    */
    __IO uint32_t GPADS;

    /**
     * GPAIBE
     * ===================================================================================================
     * Offset: 0x74  GPIOA Input Buffer Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |IBSELx    |IBSELx (x = 0,1,...,7) Input Buffer Select from Schmitt Trigger or CMOS.
     * |        |          |0 = PA.x CMOS Input Buffer (Default).
     * |        |          |1 = PA.x Schmitt Trigger Input Buffer.
     * |[15:8]  |CMOSENx   |CMOSENx (x = 8,9,...,15)CMOS Input Buffer Enable Control.
     * |        |          |0 = PA.x CMOS Input Buffer Disabled. (Default)
     * |        |          |1 = PA.x CMOS Input Buffer Enabled.
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
     * |[23:16] |DINONx    |DINONx (x = 0,1,...,7) Input Buffer Enable Control.
     * |        |          |0 = PA.x Input Buffer Disabled.
     * |        |          |1 = PA.x Input Buffer Enabled (Default).
     * |        |          |Note1: If setting to 0, the input signal from PAD will always be zero.
     * |        |          |Note2: If using PA.0~PA.7 as analog pads, remember to disable input buffer.
     * |[30:24] |SMTENx    |SMTENx (x = 8,9,...,15) Schmitt Trigger Input Buffer Enable Control.
     * |        |          |0 = PA.x schmitt Trigger Input Buffer Disabled.
     * |        |          |1 = PA.x schmitt Trigger Input Buffer Enabled (Default).
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
    */
    __IO uint32_t GPAIBE;

    /**
     * GPBIBE
     * ===================================================================================================
     * Offset: 0x78  GPIOB Input Buffer Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMOSENx   |CMOSENx (x = 0,1,...,15)CMOS Input Buffer Enable Control.
     * |        |          |0 = PB.x CMOS Input Buffer Disabled. (Default)
     * |        |          |1 =PB.x CMOS Input Buffer Enabled.
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
     * |[30:16] |SMTENx    |SMTENx (x = 0,1,...,15) Schmitt Trigger Input Buffer Enable Control.
     * |        |          |0 = PB.x Schmitt Trigger Input Buffer Disabled.
     * |        |          |1 = PB.x Schmitt Trigger Input Buffer Enabled (Default).
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
    */
    __IO uint32_t GPBIBE;

    /**
     * GPCIBE
     * ===================================================================================================
     * Offset: 0x7C  GPIOC Input Buffer Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[14:0]  |CMOSENx   |CMOSENx (x = 0,1,...,14)CMOS Input Buffer Enable Control.
     * |        |          |0 = PC.x CMOS Input Buffer Disabled. (Default)
     * |        |          |1 =PC.x CMOS Input Buffer Enabled.
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
     * |[30:16] |SMTENx    |SMTENx (x = 0,1,...,14) Schmitt Trigger Input Buffer Enable Control.
     * |        |          |0 = PC.x Schmitt Trigger Input Buffer Disabled.
     * |        |          |1 = PC.x Schmitt Trigger Input Buffer Enabled (Default).
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
    */
    __IO uint32_t GPCIBE;

    /**
     * GPDIBE
     * ===================================================================================================
     * Offset: 0x80  GPIOD Input Buffer Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |CMOSENx   |CMOSENx (x = 0,1)CMOS Input Buffer Enable Control.
     * |        |          |0 = PD.x CMOS Input Buffer Disabled (Default).
     * |        |          |1 = PD.x CMOS Input Buffer Enabled.
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
     * |[4:2]   |IBSELx    |IBSELx (x = 2,3,4) Input Buffer Select from Schmitt Trigger or CMOS.
     * |        |          |0 = PD.x CMOS Input Buffer (Default).
     * |        |          |1 = PD.x Schmitt Trigger Input Buffer.
     * |[17:16] |SMTENx    |SMTENx (x = 0,1) Schmitt Trigger Input Buffer Enable Control.
     * |        |          |0 = PD.x Schmitt Trigger Input Buffer Disabled.
     * |        |          |1 = PD.x Schmitt Trigger Input Buffer Enabled (Default).
     * |        |          |Note: If both Schmitt Trigger and CMOS input buffer are set to 0, the input signal from PAD will always be zero.
     * |[20:18] |DINONx    |DINONx (x = 2,3,4) Input Buffer Enable Control.
     * |        |          |0 = PD.x Input Buffer Disabled.
     * |        |          |1 = PD.x Input Buffer Enabled (Default).
     * |        |          |Note1: If setting to 0, the input signal from PAD will always be zero.
     * |        |          |Note2: If using PD.2, PD.3, and PD.4 as analog pads, user must disable PD.2, PD.3, and PD.4 input buffer.
    */
    __IO uint32_t GPDIBE;

    /**
     * GPDDS
     * ===================================================================================================
     * Offset: 0x84  GPIOD Driving Strength Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |PD2DS     |PD.2 Driving Strength Control
     * |        |          |Setting driving strength for PD.2 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[6:4]   |PD3DS     |PD.3 Driving Strength Control
     * |        |          |Setting driving strength for PD.3 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
     * |[10:8]  |PD4DS     |PD.4 Driving Strength Control
     * |        |          |Setting driving strength for PD.4 analog / digital combo pin.
     * |        |          |000 = 2.0 mA (Default).
     * |        |          |001 = 6.5 mA.
     * |        |          |010 = 8.7 mA.
     * |        |          |011 = 13.0 mA.
     * |        |          |100 = 15.2 mA.
     * |        |          |101 = 19.5 mA.
     * |        |          |110 = 21.7 mA.
     * |        |          |111 = 26.1 mA.
    */
    __IO uint32_t GPDDS;
    uint32_t RESERVE3[30];


    /**
     * RSTDBCNT
     * ===================================================================================================
     * Offset: 0x100  External nRESET Pin De-bounce Counter Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RSTDBCNT  |External nRESET De-bounce Counter
     * |        |          |This 16-bit external nRESET pin de-bounce counter can specify the external nRESET pin de-bounce time up to around 5.46ms (0xFFFF) at XIN=12 MHz.
     * |        |          |Note: The default external nRESET pin de-bounce time is 0.1ms (0x04B0) at XIN = 12 MHz.
    */
    __IO uint32_t RSTDBCNT;

    /**
     * RSTDBEN
     * ===================================================================================================
     * Offset: 0x104  External nRESET Pin De-bounce Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RSTDBEN   |External nRESET De-bounce Control
     * |        |          |This bit is to enable or disable the external nRESET pin de-bounce process.
     * |        |          |0 = De-bounce function Disabled.
     * |        |          |1 = De-bounce function Enabled.
    */
    __IO uint32_t RSTDBEN;

} SYS_T;

/**
    @addtogroup SYS_CONST SYS Bit Field Definition
    Constant Definitions for SYS Controller
@{ */

#define SYS_PDID_PDID_Pos                (0)                                               /*!< SYS_T::PDID: PDID Position                */
#define SYS_PDID_PDID_Msk                (0xfffffful << SYS_PDID_PDID_Pos)                 /*!< SYS_T::PDID: PDID Mask                    */

#define SYS_BOOTSET_BOOTSET_Pos          (0)                                               /*!< SYS_T::BOOTSET: BOOTSET Position          */
#define SYS_BOOTSET_BOOTSET_Msk          (0xful << SYS_BOOTSET_BOOTSET_Pos)                /*!< SYS_T::BOOTSET: BOOTSET Mask              */

#define SYS_IPRST0_CPURST_Pos            (0)                                               /*!< SYS_T::IPRST0: CPURST Position            */
#define SYS_IPRST0_CPURST_Msk            (0x1ul << SYS_IPRST0_CPURST_Pos)                  /*!< SYS_T::IPRST0: CPURST Mask                */

#define SYS_IPRST0_CHIPRST_Pos           (1)                                               /*!< SYS_T::IPRST0: CHIPRST Position           */
#define SYS_IPRST0_CHIPRST_Msk           (0x1ul << SYS_IPRST0_CHIPRST_Pos)                 /*!< SYS_T::IPRST0: CHIPRST Mask               */

#define SYS_LVDCTL_LVDIF_Pos             (0)                                               /*!< SYS_T::LVDCTL: LVDIF Position             */
#define SYS_LVDCTL_LVDIF_Msk             (0x1ul << SYS_LVDCTL_LVDIF_Pos)                   /*!< SYS_T::LVDCTL: LVDIF Mask                 */

#define SYS_LVDCTL_LVDSEL_Pos            (1)                                               /*!< SYS_T::LVDCTL: LVDSEL Position            */
#define SYS_LVDCTL_LVDSEL_Msk            (0x1ul << SYS_LVDCTL_LVDSEL_Pos)                  /*!< SYS_T::LVDCTL: LVDSEL Mask                */

#define SYS_LVDCTL_LVDEN_Pos             (2)                                               /*!< SYS_T::LVDCTL: LVDEN Position             */
#define SYS_LVDCTL_LVDEN_Msk             (0x1ul << SYS_LVDCTL_LVDEN_Pos)                   /*!< SYS_T::LVDCTL: LVDEN Mask                 */

#define SYS_LVDCTL_LVREN_Pos             (3)                                               /*!< SYS_T::LVDCTL: LVREN Position             */
#define SYS_LVDCTL_LVREN_Msk             (0x1ul << SYS_LVDCTL_LVREN_Pos)                   /*!< SYS_T::LVDCTL: LVREN Mask                 */

#define SYS_LVDCTL_PORENB_Pos            (4)                                               /*!< SYS_T::LVDCTL: PORENB Position            */
#define SYS_LVDCTL_PORENB_Msk            (0x1ul << SYS_LVDCTL_PORENB_Pos)                  /*!< SYS_T::LVDCTL: PORENB Mask                */

#define SYS_WAKEUP_I2C0WE_Pos            (0)                                               /*!< SYS_T::WAKEUP: I2C0WE Position            */
#define SYS_WAKEUP_I2C0WE_Msk            (0x1ul << SYS_WAKEUP_I2C0WE_Pos)                  /*!< SYS_T::WAKEUP: I2C0WE Mask                */

#define SYS_WAKEUP_I2C1WE_Pos            (1)                                               /*!< SYS_T::WAKEUP: I2C1WE Position            */
#define SYS_WAKEUP_I2C1WE_Msk            (0x1ul << SYS_WAKEUP_I2C1WE_Pos)                  /*!< SYS_T::WAKEUP: I2C1WE Mask                */

#define SYS_WAKEUP_GPIOWE_Pos            (2)                                               /*!< SYS_T::WAKEUP: GPIOWE Position            */
#define SYS_WAKEUP_GPIOWE_Msk            (0x1ul << SYS_WAKEUP_GPIOWE_Pos)                  /*!< SYS_T::WAKEUP: GPIOWE Mask                */

#define SYS_WAKEUP_RTCWE_Pos             (3)                                               /*!< SYS_T::WAKEUP: RTCWE Position             */
#define SYS_WAKEUP_RTCWE_Msk             (0x1ul << SYS_WAKEUP_RTCWE_Pos)                   /*!< SYS_T::WAKEUP: RTCWE Mask                 */

#define SYS_WAKEUP_WDTWE_Pos             (4)                                               /*!< SYS_T::WAKEUP: WDTWE Position             */
#define SYS_WAKEUP_WDTWE_Msk             (0x1ul << SYS_WAKEUP_WDTWE_Pos)                   /*!< SYS_T::WAKEUP: WDTWE Mask                 */

#define SYS_WAKEUP_TMR0WE_Pos            (5)                                               /*!< SYS_T::WAKEUP: TMR0WE Position            */
#define SYS_WAKEUP_TMR0WE_Msk            (0x1ul << SYS_WAKEUP_TMR0WE_Pos)                  /*!< SYS_T::WAKEUP: TMR0WE Mask                */

#define SYS_WAKEUP_TMR1WE_Pos            (6)                                               /*!< SYS_T::WAKEUP: TMR1WE Position            */
#define SYS_WAKEUP_TMR1WE_Msk            (0x1ul << SYS_WAKEUP_TMR1WE_Pos)                  /*!< SYS_T::WAKEUP: TMR1WE Mask                */

#define SYS_WAKEUP_TMR2WE_Pos            (7)                                               /*!< SYS_T::WAKEUP: TMR2WE Position            */
#define SYS_WAKEUP_TMR2WE_Msk            (0x1ul << SYS_WAKEUP_TMR2WE_Pos)                  /*!< SYS_T::WAKEUP: TMR2WE Mask                */

#define SYS_WAKEUP_TMR3WE_Pos            (8)                                               /*!< SYS_T::WAKEUP: TMR3WE Position            */
#define SYS_WAKEUP_TMR3WE_Msk            (0x1ul << SYS_WAKEUP_TMR3WE_Pos)                  /*!< SYS_T::WAKEUP: TMR3WE Mask                */

#define SYS_WAKEUP_UART0WE_Pos           (9)                                               /*!< SYS_T::WAKEUP: UART0WE Position           */
#define SYS_WAKEUP_UART0WE_Msk           (0x1ul << SYS_WAKEUP_UART0WE_Pos)                 /*!< SYS_T::WAKEUP: UART0WE Mask               */

#define SYS_WAKEUP_UART1WE_Pos           (10)                                              /*!< SYS_T::WAKEUP: UART1WE Position           */
#define SYS_WAKEUP_UART1WE_Msk           (0x1ul << SYS_WAKEUP_UART1WE_Pos)                 /*!< SYS_T::WAKEUP: UART1WE Mask               */

#define SYS_WAKEUP_UART2WE_Pos           (11)                                              /*!< SYS_T::WAKEUP: UART2WE Position           */
#define SYS_WAKEUP_UART2WE_Msk           (0x1ul << SYS_WAKEUP_UART2WE_Pos)                 /*!< SYS_T::WAKEUP: UART2WE Mask               */

#define SYS_WAKEUP_USBDWE_Pos            (12)                                              /*!< SYS_T::WAKEUP: USBDWE Position            */
#define SYS_WAKEUP_USBDWE_Msk            (0x1ul << SYS_WAKEUP_USBDWE_Pos)                  /*!< SYS_T::WAKEUP: USBDWE Mask                */

#define SYS_WAKEUP_USBHWE_Pos            (13)                                              /*!< SYS_T::WAKEUP: USBHWE Position            */
#define SYS_WAKEUP_USBHWE_Msk            (0x1ul << SYS_WAKEUP_USBHWE_Pos)                  /*!< SYS_T::WAKEUP: USBHWE Mask                */

#define SYS_WAKEUP_I2C0WF_Pos            (16)                                              /*!< SYS_T::WAKEUP: I2C0WF Position            */
#define SYS_WAKEUP_I2C0WF_Msk            (0x1ul << SYS_WAKEUP_I2C0WF_Pos)                  /*!< SYS_T::WAKEUP: I2C0WF Mask                */

#define SYS_WAKEUP_I2C1WF_Pos            (17)                                              /*!< SYS_T::WAKEUP: I2C1WF Position            */
#define SYS_WAKEUP_I2C1WF_Msk            (0x1ul << SYS_WAKEUP_I2C1WF_Pos)                  /*!< SYS_T::WAKEUP: I2C1WF Mask                */

#define SYS_WAKEUP_GPIOWF_Pos            (18)                                              /*!< SYS_T::WAKEUP: GPIOWF Position            */
#define SYS_WAKEUP_GPIOWF_Msk            (0x1ul << SYS_WAKEUP_GPIOWF_Pos)                  /*!< SYS_T::WAKEUP: GPIOWF Mask                */

#define SYS_WAKEUP_RTCWF_Pos             (19)                                              /*!< SYS_T::WAKEUP: RTCWF Position             */
#define SYS_WAKEUP_RTCWF_Msk             (0x1ul << SYS_WAKEUP_RTCWF_Pos)                   /*!< SYS_T::WAKEUP: RTCWF Mask                 */

#define SYS_WAKEUP_WDTWF_Pos             (20)                                              /*!< SYS_T::WAKEUP: WDTWF Position             */
#define SYS_WAKEUP_WDTWF_Msk             (0x1ul << SYS_WAKEUP_WDTWF_Pos)                   /*!< SYS_T::WAKEUP: WDTWF Mask                 */

#define SYS_WAKEUP_TMR0WF_Pos            (21)                                              /*!< SYS_T::WAKEUP: TMR0WF Position            */
#define SYS_WAKEUP_TMR0WF_Msk            (0x1ul << SYS_WAKEUP_TMR0WF_Pos)                  /*!< SYS_T::WAKEUP: TMR0WF Mask                */

#define SYS_WAKEUP_TMR1WF_Pos            (22)                                              /*!< SYS_T::WAKEUP: TMR1WF Position            */
#define SYS_WAKEUP_TMR1WF_Msk            (0x1ul << SYS_WAKEUP_TMR1WF_Pos)                  /*!< SYS_T::WAKEUP: TMR1WF Mask                */

#define SYS_WAKEUP_TMR2WF_Pos            (23)                                              /*!< SYS_T::WAKEUP: TMR2WF Position            */
#define SYS_WAKEUP_TMR2WF_Msk            (0x1ul << SYS_WAKEUP_TMR2WF_Pos)                  /*!< SYS_T::WAKEUP: TMR2WF Mask                */

#define SYS_WAKEUP_TMR3WF_Pos            (24)                                              /*!< SYS_T::WAKEUP: TMR3WF Position            */
#define SYS_WAKEUP_TMR3WF_Msk            (0x1ul << SYS_WAKEUP_TMR3WF_Pos)                  /*!< SYS_T::WAKEUP: TMR3WF Mask                */

#define SYS_WAKEUP_UART0WF_Pos           (25)                                              /*!< SYS_T::WAKEUP: UART0WF Position           */
#define SYS_WAKEUP_UART0WF_Msk           (0x1ul << SYS_WAKEUP_UART0WF_Pos)                 /*!< SYS_T::WAKEUP: UART0WF Mask               */

#define SYS_WAKEUP_UART1WF_Pos           (26)                                              /*!< SYS_T::WAKEUP: UART1WF Position           */
#define SYS_WAKEUP_UART1WF_Msk           (0x1ul << SYS_WAKEUP_UART1WF_Pos)                 /*!< SYS_T::WAKEUP: UART1WF Mask               */

#define SYS_WAKEUP_UART2WF_Pos           (27)                                              /*!< SYS_T::WAKEUP: UART2WF Position           */
#define SYS_WAKEUP_UART2WF_Msk           (0x1ul << SYS_WAKEUP_UART2WF_Pos)                 /*!< SYS_T::WAKEUP: UART2WF Mask               */

#define SYS_WAKEUP_USBDWF_Pos            (28)                                              /*!< SYS_T::WAKEUP: USBDWF Position            */
#define SYS_WAKEUP_USBDWF_Msk            (0x1ul << SYS_WAKEUP_USBDWF_Pos)                  /*!< SYS_T::WAKEUP: USBDWF Mask                */

#define SYS_WAKEUP_USBHWF_Pos            (29)                                              /*!< SYS_T::WAKEUP: USBHWF Position            */
#define SYS_WAKEUP_USBHWF_Msk            (0x1ul << SYS_WAKEUP_USBHWF_Pos)                  /*!< SYS_T::WAKEUP: USBHWF Mask                */

#define SYS_IPRST1_UART0RST_Pos          (0)                                               /*!< SYS_T::IPRST1: UART0RST Position          */
#define SYS_IPRST1_UART0RST_Msk          (0x1ul << SYS_IPRST1_UART0RST_Pos)                /*!< SYS_T::IPRST1: UART0RST Mask              */

#define SYS_IPRST1_UART1RST_Pos          (1)                                               /*!< SYS_T::IPRST1: UART1RST Position          */
#define SYS_IPRST1_UART1RST_Msk          (0x1ul << SYS_IPRST1_UART1RST_Pos)                /*!< SYS_T::IPRST1: UART1RST Mask              */

#define SYS_IPRST1_TMR0RST_Pos           (2)                                               /*!< SYS_T::IPRST1: TMR0RST Position           */
#define SYS_IPRST1_TMR0RST_Msk           (0x1ul << SYS_IPRST1_TMR0RST_Pos)                 /*!< SYS_T::IPRST1: TMR0RST Mask               */

#define SYS_IPRST1_TMR1RST_Pos           (3)                                               /*!< SYS_T::IPRST1: TMR1RST Position           */
#define SYS_IPRST1_TMR1RST_Msk           (0x1ul << SYS_IPRST1_TMR1RST_Pos)                 /*!< SYS_T::IPRST1: TMR1RST Mask               */

#define SYS_IPRST1_WDTFRST_Pos           (4)                                               /*!< SYS_T::IPRST1: WDTFRST Position           */
#define SYS_IPRST1_WDTFRST_Msk           (0x1ul << SYS_IPRST1_WDTFRST_Pos)                 /*!< SYS_T::IPRST1: WDTFRST Mask               */

#define SYS_IPRST1_TMR2RST_Pos           (5)                                               /*!< SYS_T::IPRST1: TMR2RST Position           */
#define SYS_IPRST1_TMR2RST_Msk           (0x1ul << SYS_IPRST1_TMR2RST_Pos)                 /*!< SYS_T::IPRST1: TMR2RST Mask               */

#define SYS_IPRST1_UART2RST_Pos          (6)                                               /*!< SYS_T::IPRST1: UART2RST Position          */
#define SYS_IPRST1_UART2RST_Msk          (0x1ul << SYS_IPRST1_UART2RST_Pos)                /*!< SYS_T::IPRST1: UART2RST Mask              */

#define SYS_IPRST1_PWMRST_Pos            (7)                                               /*!< SYS_T::IPRST1: PWMRST Position            */
#define SYS_IPRST1_PWMRST_Msk            (0x1ul << SYS_IPRST1_PWMRST_Pos)                  /*!< SYS_T::IPRST1: PWMRST Mask                */

#define SYS_IPRST1_I2C0RST_Pos           (8)                                               /*!< SYS_T::IPRST1: I2C0RST Position           */
#define SYS_IPRST1_I2C0RST_Msk           (0x1ul << SYS_IPRST1_I2C0RST_Pos)                 /*!< SYS_T::IPRST1: I2C0RST Mask               */

#define SYS_IPRST1_I2C1RST_Pos           (9)                                               /*!< SYS_T::IPRST1: I2C1RST Position           */
#define SYS_IPRST1_I2C1RST_Msk           (0x1ul << SYS_IPRST1_I2C1RST_Pos)                 /*!< SYS_T::IPRST1: I2C1RST Mask               */

#define SYS_IPRST1_SPIMRST_Pos           (10)                                              /*!< SYS_T::IPRST1: SPIMRST Position           */
#define SYS_IPRST1_SPIMRST_Msk           (0x1ul << SYS_IPRST1_SPIMRST_Pos)                 /*!< SYS_T::IPRST1: SPIMRST Mask               */

#define SYS_IPRST1_USBDRST_Pos           (11)                                              /*!< SYS_T::IPRST1: USBDRST Position           */
#define SYS_IPRST1_USBDRST_Msk           (0x1ul << SYS_IPRST1_USBDRST_Pos)                 /*!< SYS_T::IPRST1: USBDRST Mask               */

#define SYS_IPRST1_TMR3RST_Pos           (12)                                              /*!< SYS_T::IPRST1: TMR3RST Position           */
#define SYS_IPRST1_TMR3RST_Msk           (0x1ul << SYS_IPRST1_TMR3RST_Pos)                 /*!< SYS_T::IPRST1: TMR3RST Mask               */

#define SYS_IPRST1_WDTPRST_Pos           (13)                                              /*!< SYS_T::IPRST1: WDTPRST Position           */
#define SYS_IPRST1_WDTPRST_Msk           (0x1ul << SYS_IPRST1_WDTPRST_Pos)                 /*!< SYS_T::IPRST1: WDTPRST Mask               */

#define SYS_IPRST1_I2SRST_Pos            (17)                                              /*!< SYS_T::IPRST1: I2SRST Position            */
#define SYS_IPRST1_I2SRST_Msk            (0x1ul << SYS_IPRST1_I2SRST_Pos)                  /*!< SYS_T::IPRST1: I2SRST Mask                */

#define SYS_IPRST1_USBHRST_Pos           (18)                                              /*!< SYS_T::IPRST1: USBHRST Position           */
#define SYS_IPRST1_USBHRST_Msk           (0x1ul << SYS_IPRST1_USBHRST_Pos)                 /*!< SYS_T::IPRST1: USBHRST Mask               */

#define SYS_IPRST1_SDHRST_Pos            (24)                                              /*!< SYS_T::IPRST1: SDHRST Position            */
#define SYS_IPRST1_SDHRST_Msk            (0x1ul << SYS_IPRST1_SDHRST_Pos)                  /*!< SYS_T::IPRST1: SDHRST Mask                */

#define SYS_IPRST1_SRAMRST_Pos           (25)                                              /*!< SYS_T::IPRST1: SRAMRST Position           */
#define SYS_IPRST1_SRAMRST_Msk           (0x1ul << SYS_IPRST1_SRAMRST_Pos)                 /*!< SYS_T::IPRST1: SRAMRST Mask               */

#define SYS_IPRST1_GPIORST_Pos           (27)                                              /*!< SYS_T::IPRST1: GPIORST Position           */
#define SYS_IPRST1_GPIORST_Msk           (0x1ul << SYS_IPRST1_GPIORST_Pos)                 /*!< SYS_T::IPRST1: GPIORST Mask               */

#define SYS_IPRST1_ADCRST_Pos            (28)                                              /*!< SYS_T::IPRST1: ADCRST Position            */
#define SYS_IPRST1_ADCRST_Msk            (0x1ul << SYS_IPRST1_ADCRST_Pos)                  /*!< SYS_T::IPRST1: ADCRST Mask                */

#define SYS_IPRST1_SPI0RST_Pos           (30)                                              /*!< SYS_T::IPRST1: SPI0RST Position           */
#define SYS_IPRST1_SPI0RST_Msk           (0x1ul << SYS_IPRST1_SPI0RST_Pos)                 /*!< SYS_T::IPRST1: SPI0RST Mask               */

#define SYS_IPRST1_SPI1RST_Pos           (31)                                              /*!< SYS_T::IPRST1: SPI1RST Position           */
#define SYS_IPRST1_SPI1RST_Msk           (0x1ul << SYS_IPRST1_SPI1RST_Pos)                 /*!< SYS_T::IPRST1: SPI1RST Mask               */

#define SYS_NMICTL_RTCIEN_Pos            (0)                                               /*!< SYS_T::NMICTL: RTCIEN Position            */
#define SYS_NMICTL_RTCIEN_Msk            (0x1ul << SYS_NMICTL_RTCIEN_Pos)                  /*!< SYS_T::NMICTL: RTCIEN Mask                */

#define SYS_NMICTL_WDTIEN_Pos            (1)                                               /*!< SYS_T::NMICTL: WDTIEN Position            */
#define SYS_NMICTL_WDTIEN_Msk            (0x1ul << SYS_NMICTL_WDTIEN_Pos)                  /*!< SYS_T::NMICTL: WDTIEN Mask                */

#define SYS_NMICTL_PORIEN_Pos            (2)                                               /*!< SYS_T::NMICTL: PORIEN Position            */
#define SYS_NMICTL_PORIEN_Msk            (0x1ul << SYS_NMICTL_PORIEN_Pos)                  /*!< SYS_T::NMICTL: PORIEN Mask                */

#define SYS_NMICTL_EINT0IEN_Pos          (3)                                               /*!< SYS_T::NMICTL: EINT0IEN Position          */
#define SYS_NMICTL_EINT0IEN_Msk          (0x1ul << SYS_NMICTL_EINT0IEN_Pos)                /*!< SYS_T::NMICTL: EINT0IEN Mask              */

#define SYS_NMICTL_EINT1IEN_Pos          (4)                                               /*!< SYS_T::NMICTL: EINT1IEN Position          */
#define SYS_NMICTL_EINT1IEN_Msk          (0x1ul << SYS_NMICTL_EINT1IEN_Pos)                /*!< SYS_T::NMICTL: EINT1IEN Mask              */

#define SYS_NMICTL_EINT2IEN_Pos          (5)                                               /*!< SYS_T::NMICTL: EINT2IEN Position          */
#define SYS_NMICTL_EINT2IEN_Msk          (0x1ul << SYS_NMICTL_EINT2IEN_Pos)                /*!< SYS_T::NMICTL: EINT2IEN Mask              */

#define SYS_NMICTL_EINT3IEN_Pos          (6)                                               /*!< SYS_T::NMICTL: EINT3IEN Position          */
#define SYS_NMICTL_EINT3IEN_Msk          (0x1ul << SYS_NMICTL_EINT3IEN_Pos)                /*!< SYS_T::NMICTL: EINT3IEN Mask              */

#define SYS_NMICTL_LVDIEN_Pos            (7)                                               /*!< SYS_T::NMICTL: LVDIEN Position            */
#define SYS_NMICTL_LVDIEN_Msk            (0x1ul << SYS_NMICTL_LVDIEN_Pos)                  /*!< SYS_T::NMICTL: LVDIEN Mask                */

#define SYS_NMICTL_RTCIF_Pos             (16)                                              /*!< SYS_T::NMICTL: RTCIF Position             */
#define SYS_NMICTL_RTCIF_Msk             (0x1ul << SYS_NMICTL_RTCIF_Pos)                   /*!< SYS_T::NMICTL: RTCIF Mask                 */

#define SYS_NMICTL_WDTIF_Pos             (17)                                              /*!< SYS_T::NMICTL: WDTIF Position             */
#define SYS_NMICTL_WDTIF_Msk             (0x1ul << SYS_NMICTL_WDTIF_Pos)                   /*!< SYS_T::NMICTL: WDTIF Mask                 */

#define SYS_NMICTL_PORIF_Pos             (18)                                              /*!< SYS_T::NMICTL: PORIF Position             */
#define SYS_NMICTL_PORIF_Msk             (0x1ul << SYS_NMICTL_PORIF_Pos)                   /*!< SYS_T::NMICTL: PORIF Mask                 */

#define SYS_NMICTL_EINT0IF_Pos           (19)                                              /*!< SYS_T::NMICTL: EINT0IF Position           */
#define SYS_NMICTL_EINT0IF_Msk           (0x1ul << SYS_NMICTL_EINT0IF_Pos)                 /*!< SYS_T::NMICTL: EINT0IF Mask               */

#define SYS_NMICTL_EINT1IF_Pos           (20)                                              /*!< SYS_T::NMICTL: EINT1IF Position           */
#define SYS_NMICTL_EINT1IF_Msk           (0x1ul << SYS_NMICTL_EINT1IF_Pos)                 /*!< SYS_T::NMICTL: EINT1IF Mask               */

#define SYS_NMICTL_EINT2IF_Pos           (21)                                              /*!< SYS_T::NMICTL: EINT2IF Position           */
#define SYS_NMICTL_EINT2IF_Msk           (0x1ul << SYS_NMICTL_EINT2IF_Pos)                 /*!< SYS_T::NMICTL: EINT2IF Mask               */

#define SYS_NMICTL_EINT3IF_Pos           (22)                                              /*!< SYS_T::NMICTL: EINT3IF Position           */
#define SYS_NMICTL_EINT3IF_Msk           (0x1ul << SYS_NMICTL_EINT3IF_Pos)                 /*!< SYS_T::NMICTL: EINT3IF Mask               */

#define SYS_NMICTL_LVDIF_Pos             (23)                                              /*!< SYS_T::NMICTL: LVDIF Position             */
#define SYS_NMICTL_LVDIF_Msk             (0x1ul << SYS_NMICTL_LVDIF_Pos)                   /*!< SYS_T::NMICTL: LVDIF Mask                 */

#define SYS_RSTSTS_PINRF_Pos             (0)                                               /*!< SYS_T::RSTSTS: PINRF Position             */
#define SYS_RSTSTS_PINRF_Msk             (0x1ul << SYS_RSTSTS_PINRF_Pos)                   /*!< SYS_T::RSTSTS: PINRF Mask                 */

#define SYS_RSTSTS_LVRF_Pos              (1)                                               /*!< SYS_T::RSTSTS: LVRF Position              */
#define SYS_RSTSTS_LVRF_Msk              (0x1ul << SYS_RSTSTS_LVRF_Pos)                    /*!< SYS_T::RSTSTS: LVRF Mask                  */

#define SYS_RSTSTS_WDTRF_Pos             (2)                                               /*!< SYS_T::RSTSTS: WDTRF Position             */
#define SYS_RSTSTS_WDTRF_Msk             (0x1ul << SYS_RSTSTS_WDTRF_Pos)                   /*!< SYS_T::RSTSTS: WDTRF Mask                 */

#define SYS_RSTSTS_CPURF_Pos             (3)                                               /*!< SYS_T::RSTSTS: CPURF Position             */
#define SYS_RSTSTS_CPURF_Msk             (0x1ul << SYS_RSTSTS_CPURF_Pos)                   /*!< SYS_T::RSTSTS: CPURF Mask                 */

#define SYS_RSTSTS_PORF_Pos              (4)                                               /*!< SYS_T::RSTSTS: PORF Position              */
#define SYS_RSTSTS_PORF_Msk              (0x1ul << SYS_RSTSTS_PORF_Pos)                    /*!< SYS_T::RSTSTS: PORF Mask                  */

#define SYS_AHBCTL_PRISEL_Pos            (0)                                               /*!< SYS_T::AHBCTL: PRISEL Position            */
#define SYS_AHBCTL_PRISEL_Msk            (0x1ul << SYS_AHBCTL_PRISEL_Pos)                  /*!< SYS_T::AHBCTL: PRISEL Mask                */

#define SYS_AHBCTL_CPUHPRI_Pos           (4)                                               /*!< SYS_T::AHBCTL: CPUHPRI Position           */
#define SYS_AHBCTL_CPUHPRI_Msk           (0x1ul << SYS_AHBCTL_CPUHPRI_Pos)                 /*!< SYS_T::AHBCTL: CPUHPRI Mask               */

#define SYS_AHBCTL_PRISTS_Pos            (5)                                               /*!< SYS_T::AHBCTL: PRISTS Position            */
#define SYS_AHBCTL_PRISTS_Msk            (0x1ul << SYS_AHBCTL_PRISTS_Pos)                  /*!< SYS_T::AHBCTL: PRISTS Mask                */

#define SYS_GPA_MFPL_PA0MFP_Pos          (0)                                               /*!< SYS_T::GPA_MFPL: PA0MFP Position          */
#define SYS_GPA_MFPL_PA0MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA0MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA0MFP Mask              */

#define SYS_GPA_MFPL_PA1MFP_Pos          (4)                                               /*!< SYS_T::GPA_MFPL: PA1MFP Position          */
#define SYS_GPA_MFPL_PA1MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA1MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA1MFP Mask              */

#define SYS_GPA_MFPL_PA2MFP_Pos          (8)                                               /*!< SYS_T::GPA_MFPL: PA2MFP Position          */
#define SYS_GPA_MFPL_PA2MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA2MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA2MFP Mask              */

#define SYS_GPA_MFPL_PA3MFP_Pos          (12)                                              /*!< SYS_T::GPA_MFPL: PA3MFP Position          */
#define SYS_GPA_MFPL_PA3MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA3MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA3MFP Mask              */

#define SYS_GPA_MFPL_PA4MFP_Pos          (16)                                              /*!< SYS_T::GPA_MFPL: PA4MFP Position          */
#define SYS_GPA_MFPL_PA4MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA4MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA4MFP Mask              */

#define SYS_GPA_MFPL_PA5MFP_Pos          (20)                                              /*!< SYS_T::GPA_MFPL: PA5MFP Position          */
#define SYS_GPA_MFPL_PA5MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA5MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA5MFP Mask              */

#define SYS_GPA_MFPL_PA6MFP_Pos          (24)                                              /*!< SYS_T::GPA_MFPL: PA6MFP Position          */
#define SYS_GPA_MFPL_PA6MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA6MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA6MFP Mask              */

#define SYS_GPA_MFPL_PA7MFP_Pos          (28)                                              /*!< SYS_T::GPA_MFPL: PA7MFP Position          */
#define SYS_GPA_MFPL_PA7MFP_Msk          (0x7ul << SYS_GPA_MFPL_PA7MFP_Pos)                /*!< SYS_T::GPA_MFPL: PA7MFP Mask              */

#define SYS_GPA_MFPH_PA8MFP_Pos          (0)                                               /*!< SYS_T::GPA_MFPH: PA8MFP Position          */
#define SYS_GPA_MFPH_PA8MFP_Msk          (0x7ul << SYS_GPA_MFPH_PA8MFP_Pos)                /*!< SYS_T::GPA_MFPH: PA8MFP Mask              */

#define SYS_GPA_MFPH_PA9MFP_Pos          (4)                                               /*!< SYS_T::GPA_MFPH: PA9MFP Position          */
#define SYS_GPA_MFPH_PA9MFP_Msk          (0x7ul << SYS_GPA_MFPH_PA9MFP_Pos)                /*!< SYS_T::GPA_MFPH: PA9MFP Mask              */

#define SYS_GPA_MFPH_PA10MFP_Pos         (8)                                               /*!< SYS_T::GPA_MFPH: PA10MFP Position         */
#define SYS_GPA_MFPH_PA10MFP_Msk         (0x7ul << SYS_GPA_MFPH_PA10MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA10MFP Mask             */

#define SYS_GPA_MFPH_PA11MFP_Pos         (12)                                              /*!< SYS_T::GPA_MFPH: PA11MFP Position         */
#define SYS_GPA_MFPH_PA11MFP_Msk         (0x7ul << SYS_GPA_MFPH_PA11MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA11MFP Mask             */

#define SYS_GPA_MFPH_PA12MFP_Pos         (16)                                              /*!< SYS_T::GPA_MFPH: PA12MFP Position         */
#define SYS_GPA_MFPH_PA12MFP_Msk         (0x7ul << SYS_GPA_MFPH_PA12MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA12MFP Mask             */

#define SYS_GPA_MFPH_PA13MFP_Pos         (20)                                              /*!< SYS_T::GPA_MFPH: PA13MFP Position         */
#define SYS_GPA_MFPH_PA13MFP_Msk         (0x7ul << SYS_GPA_MFPH_PA13MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA13MFP Mask             */

#define SYS_GPA_MFPH_PA14MFP_Pos         (24)                                              /*!< SYS_T::GPA_MFPH: PA14MFP Position         */
#define SYS_GPA_MFPH_PA14MFP_Msk         (0x7ul << SYS_GPA_MFPH_PA14MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA14MFP Mask             */

#define SYS_GPA_MFPH_PA15MFP_Pos         (28)                                              /*!< SYS_T::GPA_MFPH: PA15MFP Position         */
#define SYS_GPA_MFPH_PA15MFP_Msk         (0x7ul << SYS_GPA_MFPH_PA15MFP_Pos)               /*!< SYS_T::GPA_MFPH: PA15MFP Mask             */

#define SYS_GPB_MFPL_PB0MFP_Pos          (0)                                               /*!< SYS_T::GPB_MFPL: PB0MFP Position          */
#define SYS_GPB_MFPL_PB0MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB0MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB0MFP Mask              */

#define SYS_GPB_MFPL_PB1MFP_Pos          (4)                                               /*!< SYS_T::GPB_MFPL: PB1MFP Position          */
#define SYS_GPB_MFPL_PB1MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB1MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB1MFP Mask              */

#define SYS_GPB_MFPL_PB2MFP_Pos          (8)                                               /*!< SYS_T::GPB_MFPL: PB2MFP Position          */
#define SYS_GPB_MFPL_PB2MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB2MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB2MFP Mask              */

#define SYS_GPB_MFPL_PB3MFP_Pos          (12)                                              /*!< SYS_T::GPB_MFPL: PB3MFP Position          */
#define SYS_GPB_MFPL_PB3MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB3MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB3MFP Mask              */

#define SYS_GPB_MFPL_PB4MFP_Pos          (16)                                              /*!< SYS_T::GPB_MFPL: PB4MFP Position          */
#define SYS_GPB_MFPL_PB4MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB4MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB4MFP Mask              */

#define SYS_GPB_MFPL_PB5MFP_Pos          (20)                                              /*!< SYS_T::GPB_MFPL: PB5MFP Position          */
#define SYS_GPB_MFPL_PB5MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB5MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB5MFP Mask              */

#define SYS_GPB_MFPL_PB6MFP_Pos          (24)                                              /*!< SYS_T::GPB_MFPL: PB6MFP Position          */
#define SYS_GPB_MFPL_PB6MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB6MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB6MFP Mask              */

#define SYS_GPB_MFPL_PB7MFP_Pos          (28)                                              /*!< SYS_T::GPB_MFPL: PB7MFP Position          */
#define SYS_GPB_MFPL_PB7MFP_Msk          (0x7ul << SYS_GPB_MFPL_PB7MFP_Pos)                /*!< SYS_T::GPB_MFPL: PB7MFP Mask              */

#define SYS_GPB_MFPH_PB8MFP_Pos          (0)                                               /*!< SYS_T::GPB_MFPH: PB8MFP Position          */
#define SYS_GPB_MFPH_PB8MFP_Msk          (0x7ul << SYS_GPB_MFPH_PB8MFP_Pos)                /*!< SYS_T::GPB_MFPH: PB8MFP Mask              */

#define SYS_GPB_MFPH_PB9MFP_Pos          (4)                                               /*!< SYS_T::GPB_MFPH: PB9MFP Position          */
#define SYS_GPB_MFPH_PB9MFP_Msk          (0x7ul << SYS_GPB_MFPH_PB9MFP_Pos)                /*!< SYS_T::GPB_MFPH: PB9MFP Mask              */

#define SYS_GPB_MFPH_PB10MFP_Pos         (8)                                               /*!< SYS_T::GPB_MFPH: PB10MFP Position         */
#define SYS_GPB_MFPH_PB10MFP_Msk         (0x7ul << SYS_GPB_MFPH_PB10MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB10MFP Mask             */

#define SYS_GPB_MFPH_PB11MFP_Pos         (12)                                              /*!< SYS_T::GPB_MFPH: PB11MFP Position         */
#define SYS_GPB_MFPH_PB11MFP_Msk         (0x7ul << SYS_GPB_MFPH_PB11MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB11MFP Mask             */

#define SYS_GPB_MFPH_PB12MFP_Pos         (16)                                              /*!< SYS_T::GPB_MFPH: PB12MFP Position         */
#define SYS_GPB_MFPH_PB12MFP_Msk         (0x7ul << SYS_GPB_MFPH_PB12MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB12MFP Mask             */

#define SYS_GPB_MFPH_PB13MFP_Pos         (20)                                              /*!< SYS_T::GPB_MFPH: PB13MFP Position         */
#define SYS_GPB_MFPH_PB13MFP_Msk         (0x7ul << SYS_GPB_MFPH_PB13MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB13MFP Mask             */

#define SYS_GPB_MFPH_PB14MFP_Pos         (24)                                              /*!< SYS_T::GPB_MFPH: PB14MFP Position         */
#define SYS_GPB_MFPH_PB14MFP_Msk         (0x7ul << SYS_GPB_MFPH_PB14MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB14MFP Mask             */

#define SYS_GPB_MFPH_PB15MFP_Pos         (28)                                              /*!< SYS_T::GPB_MFPH: PB15MFP Position         */
#define SYS_GPB_MFPH_PB15MFP_Msk         (0x7ul << SYS_GPB_MFPH_PB15MFP_Pos)               /*!< SYS_T::GPB_MFPH: PB15MFP Mask             */

#define SYS_GPC_MFPL_PC0MFP_Pos          (0)                                               /*!< SYS_T::GPC_MFPL: PC0MFP Position          */
#define SYS_GPC_MFPL_PC0MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC0MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC0MFP Mask              */

#define SYS_GPC_MFPL_PC1MFP_Pos          (4)                                               /*!< SYS_T::GPC_MFPL: PC1MFP Position          */
#define SYS_GPC_MFPL_PC1MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC1MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC1MFP Mask              */

#define SYS_GPC_MFPL_PC2MFP_Pos          (8)                                               /*!< SYS_T::GPC_MFPL: PC2MFP Position          */
#define SYS_GPC_MFPL_PC2MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC2MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC2MFP Mask              */

#define SYS_GPC_MFPL_PC3MFP_Pos          (12)                                              /*!< SYS_T::GPC_MFPL: PC3MFP Position          */
#define SYS_GPC_MFPL_PC3MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC3MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC3MFP Mask              */

#define SYS_GPC_MFPL_PC4MFP_Pos          (16)                                              /*!< SYS_T::GPC_MFPL: PC4MFP Position          */
#define SYS_GPC_MFPL_PC4MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC4MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC4MFP Mask              */

#define SYS_GPC_MFPL_PC5MFP_Pos          (20)                                              /*!< SYS_T::GPC_MFPL: PC5MFP Position          */
#define SYS_GPC_MFPL_PC5MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC5MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC5MFP Mask              */

#define SYS_GPC_MFPL_PC6MFP_Pos          (24)                                              /*!< SYS_T::GPC_MFPL: PC6MFP Position          */
#define SYS_GPC_MFPL_PC6MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC6MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC6MFP Mask              */

#define SYS_GPC_MFPL_PC7MFP_Pos          (28)                                              /*!< SYS_T::GPC_MFPL: PC7MFP Position          */
#define SYS_GPC_MFPL_PC7MFP_Msk          (0x7ul << SYS_GPC_MFPL_PC7MFP_Pos)                /*!< SYS_T::GPC_MFPL: PC7MFP Mask              */

#define SYS_GPC_MFPH_PC8MFP_Pos          (0)                                               /*!< SYS_T::GPC_MFPH: PC8MFP Position          */
#define SYS_GPC_MFPH_PC8MFP_Msk          (0x7ul << SYS_GPC_MFPH_PC8MFP_Pos)                /*!< SYS_T::GPC_MFPH: PC8MFP Mask              */

#define SYS_GPC_MFPH_PC9MFP_Pos          (4)                                               /*!< SYS_T::GPC_MFPH: PC9MFP Position          */
#define SYS_GPC_MFPH_PC9MFP_Msk          (0x7ul << SYS_GPC_MFPH_PC9MFP_Pos)                /*!< SYS_T::GPC_MFPH: PC9MFP Mask              */

#define SYS_GPC_MFPH_PC10MFP_Pos         (8)                                               /*!< SYS_T::GPC_MFPH: PC10MFP Position         */
#define SYS_GPC_MFPH_PC10MFP_Msk         (0x7ul << SYS_GPC_MFPH_PC10MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC10MFP Mask             */

#define SYS_GPC_MFPH_PC11MFP_Pos         (12)                                              /*!< SYS_T::GPC_MFPH: PC11MFP Position         */
#define SYS_GPC_MFPH_PC11MFP_Msk         (0x7ul << SYS_GPC_MFPH_PC11MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC11MFP Mask             */

#define SYS_GPC_MFPH_PC12MFP_Pos         (16)                                              /*!< SYS_T::GPC_MFPH: PC12MFP Position         */
#define SYS_GPC_MFPH_PC12MFP_Msk         (0x7ul << SYS_GPC_MFPH_PC12MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC12MFP Mask             */

#define SYS_GPC_MFPH_PC13MFP_Pos         (20)                                              /*!< SYS_T::GPC_MFPH: PC13MFP Position         */
#define SYS_GPC_MFPH_PC13MFP_Msk         (0x7ul << SYS_GPC_MFPH_PC13MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC13MFP Mask             */

#define SYS_GPC_MFPH_PC14MFP_Pos         (24)                                              /*!< SYS_T::GPC_MFPH: PC14MFP Position         */
#define SYS_GPC_MFPH_PC14MFP_Msk         (0x7ul << SYS_GPC_MFPH_PC14MFP_Pos)               /*!< SYS_T::GPC_MFPH: PC14MFP Mask             */

#define SYS_GPD_MFPL_PD0MFP_Pos          (0)                                               /*!< SYS_T::GPD_MFPL: PD0MFP Position          */
#define SYS_GPD_MFPL_PD0MFP_Msk          (0x7ul << SYS_GPD_MFPL_PD0MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD0MFP Mask              */

#define SYS_GPD_MFPL_PD1MFP_Pos          (4)                                               /*!< SYS_T::GPD_MFPL: PD1MFP Position          */
#define SYS_GPD_MFPL_PD1MFP_Msk          (0x7ul << SYS_GPD_MFPL_PD1MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD1MFP Mask              */

#define SYS_GPD_MFPL_PD2MFP_Pos          (8)                                               /*!< SYS_T::GPD_MFPL: PD2MFP Position          */
#define SYS_GPD_MFPL_PD2MFP_Msk          (0x7ul << SYS_GPD_MFPL_PD2MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD2MFP Mask              */

#define SYS_GPD_MFPL_PD3MFP_Pos          (12)                                              /*!< SYS_T::GPD_MFPL: PD3MFP Position          */
#define SYS_GPD_MFPL_PD3MFP_Msk          (0x7ul << SYS_GPD_MFPL_PD3MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD3MFP Mask              */

#define SYS_GPD_MFPL_PD4MFP_Pos          (16)                                              /*!< SYS_T::GPD_MFPL: PD4MFP Position          */
#define SYS_GPD_MFPL_PD4MFP_Msk          (0x7ul << SYS_GPD_MFPL_PD4MFP_Pos)                /*!< SYS_T::GPD_MFPL: PD4MFP Mask              */

#define SYS_LVMPADDR_ADDR_Pos            (0)                                               /*!< SYS_T::LVMPADDR: ADDR Position            */
#define SYS_LVMPADDR_ADDR_Msk            (0xfffffffful << SYS_LVMPADDR_ADDR_Pos)           /*!< SYS_T::LVMPADDR: ADDR Mask                */

#define SYS_LVMPLEN_LEN_Pos              (0)                                               /*!< SYS_T::LVMPLEN: LEN Position              */
#define SYS_LVMPLEN_LEN_Msk              (0xfful << SYS_LVMPLEN_LEN_Pos)                   /*!< SYS_T::LVMPLEN: LEN Mask                  */

#define SYS_RVMPADDR_ADDR_Pos            (0)                                               /*!< SYS_T::RVMPADDR: ADDR Position            */
#define SYS_RVMPADDR_ADDR_Msk            (0xfffffffful << SYS_RVMPADDR_ADDR_Pos)           /*!< SYS_T::RVMPADDR: ADDR Mask                */

#define SYS_RVMPLEN_RLDVMP_Pos           (0)                                               /*!< SYS_T::RVMPLEN: RLDVMP Position           */
#define SYS_RVMPLEN_RLDVMP_Msk           (0x1ul << SYS_RVMPLEN_RLDVMP_Pos)                 /*!< SYS_T::RVMPLEN: RLDVMP Mask               */

#define SYS_RVMPLEN_LEN_Pos              (24)                                              /*!< SYS_T::RVMPLEN: LEN Position              */
#define SYS_RVMPLEN_LEN_Msk              (0xfful << SYS_RVMPLEN_LEN_Pos)                   /*!< SYS_T::RVMPLEN: LEN Mask                  */

#define SYS_EPADPUEN_EPADPUEN_Pos        (0)                                               /*!< SYS_T::EPADPUEN: EPADPUEN Position        */
#define SYS_EPADPUEN_EPADPUEN_Msk        (0x1ful << SYS_EPADPUEN_EPADPUEN_Pos)             /*!< SYS_T::EPADPUEN: EPADPUEN Mask            */

#define SYS_GPADS_PA0DS_Pos              (0)                                               /*!< SYS_T::GPADS: PA0DS Position              */
#define SYS_GPADS_PA0DS_Msk              (0x7ul << SYS_GPADS_PA0DS_Pos)                    /*!< SYS_T::GPADS: PA0DS Mask                  */

#define SYS_GPADS_PA1DS_Pos              (4)                                               /*!< SYS_T::GPADS: PA1DS Position              */
#define SYS_GPADS_PA1DS_Msk              (0x7ul << SYS_GPADS_PA1DS_Pos)                    /*!< SYS_T::GPADS: PA1DS Mask                  */

#define SYS_GPADS_PA2DS_Pos              (8)                                               /*!< SYS_T::GPADS: PA2DS Position              */
#define SYS_GPADS_PA2DS_Msk              (0x7ul << SYS_GPADS_PA2DS_Pos)                    /*!< SYS_T::GPADS: PA2DS Mask                  */

#define SYS_GPADS_PA3DS_Pos              (12)                                              /*!< SYS_T::GPADS: PA3DS Position              */
#define SYS_GPADS_PA3DS_Msk              (0x7ul << SYS_GPADS_PA3DS_Pos)                    /*!< SYS_T::GPADS: PA3DS Mask                  */

#define SYS_GPADS_PA4DS_Pos              (16)                                              /*!< SYS_T::GPADS: PA4DS Position              */
#define SYS_GPADS_PA4DS_Msk              (0x7ul << SYS_GPADS_PA4DS_Pos)                    /*!< SYS_T::GPADS: PA4DS Mask                  */

#define SYS_GPADS_PA5DS_Pos              (20)                                              /*!< SYS_T::GPADS: PA5DS Position              */
#define SYS_GPADS_PA5DS_Msk              (0x7ul << SYS_GPADS_PA5DS_Pos)                    /*!< SYS_T::GPADS: PA5DS Mask                  */

#define SYS_GPADS_PA6DS_Pos              (24)                                              /*!< SYS_T::GPADS: PA6DS Position              */
#define SYS_GPADS_PA6DS_Msk              (0x7ul << SYS_GPADS_PA6DS_Pos)                    /*!< SYS_T::GPADS: PA6DS Mask                  */

#define SYS_GPADS_PA7DS_Pos              (28)                                              /*!< SYS_T::GPADS: PA7DS Position              */
#define SYS_GPADS_PA7DS_Msk              (0x7ul << SYS_GPADS_PA7DS_Pos)                    /*!< SYS_T::GPADS: PA7DS Mask                  */

#define SYS_GPAIBE_IBSELx_Pos            (0)                                               /*!< SYS_T::GPAIBE: IBSELx Position            */
#define SYS_GPAIBE_IBSELx_Msk            (0xfful << SYS_GPAIBE_IBSELx_Pos)                 /*!< SYS_T::GPAIBE: IBSELx Mask                */

#define SYS_GPAIBE_CMOSENx_Pos           (8)                                               /*!< SYS_T::GPAIBE: CMOSENx Position           */
#define SYS_GPAIBE_CMOSENx_Msk           (0xfful << SYS_GPAIBE_CMOSENx_Pos)                /*!< SYS_T::GPAIBE: CMOSENx Mask               */

#define SYS_GPAIBE_DINONx_Pos            (16)                                              /*!< SYS_T::GPAIBE: DINONx Position            */
#define SYS_GPAIBE_DINONx_Msk            (0xfful << SYS_GPAIBE_DINONx_Pos)                 /*!< SYS_T::GPAIBE: DINONx Mask                */

#define SYS_GPAIBE_SMTENx_Pos            (24)                                              /*!< SYS_T::GPAIBE: SMTENx Position            */
#define SYS_GPAIBE_SMTENx_Msk            (0x7ful << SYS_GPAIBE_SMTENx_Pos)                 /*!< SYS_T::GPAIBE: SMTENx Mask                */

#define SYS_GPBIBE_CMOSENx_Pos           (0)                                               /*!< SYS_T::GPBIBE: CMOSENx Position           */
#define SYS_GPBIBE_CMOSENx_Msk           (0xfffful << SYS_GPBIBE_CMOSENx_Pos)              /*!< SYS_T::GPBIBE: CMOSENx Mask               */

#define SYS_GPBIBE_SMTENx_Pos            (16)                                              /*!< SYS_T::GPBIBE: SMTENx Position            */
#define SYS_GPBIBE_SMTENx_Msk            (0x7ffful << SYS_GPBIBE_SMTENx_Pos)               /*!< SYS_T::GPBIBE: SMTENx Mask                */

#define SYS_GPCIBE_CMOSENx_Pos           (0)                                               /*!< SYS_T::GPCIBE: CMOSENx Position           */
#define SYS_GPCIBE_CMOSENx_Msk           (0x7ffful << SYS_GPCIBE_CMOSENx_Pos)              /*!< SYS_T::GPCIBE: CMOSENx Mask               */

#define SYS_GPCIBE_SMTENx_Pos            (16)                                              /*!< SYS_T::GPCIBE: SMTENx Position            */
#define SYS_GPCIBE_SMTENx_Msk            (0x7ffful << SYS_GPCIBE_SMTENx_Pos)               /*!< SYS_T::GPCIBE: SMTENx Mask                */

#define SYS_GPDIBE_CMOSENx_Pos           (0)                                               /*!< SYS_T::GPDIBE: CMOSENx Position           */
#define SYS_GPDIBE_CMOSENx_Msk           (0x3ul << SYS_GPDIBE_CMOSENx_Pos)                 /*!< SYS_T::GPDIBE: CMOSENx Mask               */

#define SYS_GPDIBE_IBSELx_Pos            (2)                                               /*!< SYS_T::GPDIBE: IBSELx Position            */
#define SYS_GPDIBE_IBSELx_Msk            (0x7ul << SYS_GPDIBE_IBSELx_Pos)                  /*!< SYS_T::GPDIBE: IBSELx Mask                */

#define SYS_GPDIBE_SMTENx_Pos            (16)                                              /*!< SYS_T::GPDIBE: SMTENx Position            */
#define SYS_GPDIBE_SMTENx_Msk            (0x3ul << SYS_GPDIBE_SMTENx_Pos)                  /*!< SYS_T::GPDIBE: SMTENx Mask                */

#define SYS_GPDIBE_DINONx_Pos            (18)                                              /*!< SYS_T::GPDIBE: DINONx Position            */
#define SYS_GPDIBE_DINONx_Msk            (0x7ul << SYS_GPDIBE_DINONx_Pos)                  /*!< SYS_T::GPDIBE: DINONx Mask                */

#define SYS_GPDDS_PD2DS_Pos              (0)                                               /*!< SYS_T::GPDDS: PD2DS Position              */
#define SYS_GPDDS_PD2DS_Msk              (0x7ul << SYS_GPDDS_PD2DS_Pos)                    /*!< SYS_T::GPDDS: PD2DS Mask                  */

#define SYS_GPDDS_PD3DS_Pos              (4)                                               /*!< SYS_T::GPDDS: PD3DS Position              */
#define SYS_GPDDS_PD3DS_Msk              (0x7ul << SYS_GPDDS_PD3DS_Pos)                    /*!< SYS_T::GPDDS: PD3DS Mask                  */

#define SYS_GPDDS_PD4DS_Pos              (8)                                               /*!< SYS_T::GPDDS: PD4DS Position              */
#define SYS_GPDDS_PD4DS_Msk              (0x7ul << SYS_GPDDS_PD4DS_Pos)                    /*!< SYS_T::GPDDS: PD4DS Mask                  */

#define SYS_RSTDBCNT_RSTDBCNT_Pos        (0)                                               /*!< SYS_T::RSTDBCNT: RSTDBCNT Position        */
#define SYS_RSTDBCNT_RSTDBCNT_Msk        (0xfffful << SYS_RSTDBCNT_RSTDBCNT_Pos)           /*!< SYS_T::RSTDBCNT: RSTDBCNT Mask            */

#define SYS_RSTDBEN_RSTDBEN_Pos          (0)                                               /*!< SYS_T::RSTDBEN: RSTDBEN Position          */
#define SYS_RSTDBEN_RSTDBEN_Msk          (0x1ul << SYS_RSTDBEN_RSTDBEN_Pos)                /*!< SYS_T::RSTDBEN: RSTDBEN Mask              */

/**@}*/ /* SYS_CONST */
/**@}*/ /* end of SYS register group */


/*---------------------- General Purpose Input/Output Controller -------------------------*/
/**
    @addtogroup GPIO General Purpose Input/Output Controller(GPIO)
    Memory Mapped Structure for GPIO Controller
@{ */

typedef struct
{


    /**
     * Px_MODE
     * ===================================================================================================
     * Offset: 0x00  Px I/O Mode Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |MODE0     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[1]     |MODE1     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[2]     |MODE2     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[3]     |MODE3     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[4]     |MODE4     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[5]     |MODE5     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[6]     |MODE6     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[7]     |MODE7     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[8]     |MODE8     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[9]     |MODE9     |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[10]    |MODE10    |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[11]    |MODE11    |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[12]    |MODE12    |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[13]    |MODE13    |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[14]    |MODE14    |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[15]    |MODE15    |Port A-D I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |0 = Px.n is in Input mode.
     * |        |          |1 = Px.n is in Push-pull Output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
    */
    __IO uint32_t MODE;

    /**
     * Px_PUEN
     * ===================================================================================================
     * Offset: 0x04  Px I/O Pull-up/down Resistor Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |PULLSEL0  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[3:2]   |PULLSEL1  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[5:4]   |PULLSEL2  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[7:6]   |PULLSEL3  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[9:8]   |PULLSEL4  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[11:10] |PULLSEL5  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[13:12] |PULLSEL6  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[15:14] |PULLSEL7  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[17:16] |PULLSEL8  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[19:18] |PULLSEL9  |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[21:20] |PULLSEL10 |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[23:22] |PULLSEL11 |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[25:24] |PULLSEL12 |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[27:26] |PULLSEL13 |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[29:28] |PULLSEL14 |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
     * |[31:30] |PULLSEL15 |Port A-D Pin Pull-up Resistor or Pull-down Resistor   Enable Control
     * |        |          |00 = Px.n   pull-up or pull-down resistors are all Disabled.
     * |        |          |01 = Px.n pull-up   resistor Enabled.
     * |        |          |10 = Px.n pull-down   resistor Enabled.
     * |        |          |11 = Reserved.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |        |          |Note2: Refer to Figure 6.4-1
    */
    __IO uint32_t PUEN;

    /**
     * Px_DOUT
     * ===================================================================================================
     * Offset: 0x08  Px Data Output Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DOUT0     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[1]     |DOUT1     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[2]     |DOUT2     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[3]     |DOUT3     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[4]     |DOUT4     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[5]     |DOUT5     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[6]     |DOUT6     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[7]     |DOUT7     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[8]     |DOUT8     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[9]     |DOUT9     |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[10]    |DOUT10    |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[11]    |DOUT11    |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[12]    |DOUT12    |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[13]    |DOUT13    |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[14]    |DOUT14    |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
     * |[15]    |DOUT15    |Port A-D Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output mode.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=7 for port D.
    */
    __IO uint32_t DOUT;

    /**
     * Px_PIN
     * ===================================================================================================
     * Offset: 0x0C  Px Pin Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PIN0      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[1]     |PIN1      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[2]     |PIN2      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[3]     |PIN3      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[4]     |PIN4      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[5]     |PIN5      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[6]     |PIN6      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[7]     |PIN7      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[8]     |PIN8      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[9]     |PIN9      |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[10]    |PIN10     |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[11]    |PIN11     |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[12]    |PIN12     |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[13]    |PIN13     |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[14]    |PIN14     |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[15]    |PIN15     |Port A-D Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is lo.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
    */
    __I  uint32_t PIN;
    uint32_t RESERVE1[28];

    /**
     * Px_INTSRCGP
     * ===================================================================================================
     * Offset: 0x80  Px Interrupt Event (EINT) Source Grouping
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |INTSEL0   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[3:2]   |INTSEL1   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[5:4]   |INTSEL2   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[7:6]   |INTSEL3   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[9:8]   |INTSEL4   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[11:10] |INTSEL5   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[13:12] |INTSEL6   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[15:14] |INTSEL7   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[17:16] |INTSEL8   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[19:18] |INTSEL9   |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[21:20] |INTSEL10  |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[23:22] |INTSEL11  |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[25:24] |INTSEL12  |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[27:26] |INTSEL13  |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[29:28] |INTSEL14  |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[31:30] |INTSEL15  |Selection For Px.N As One Of Interrupt Sources To EINT0, EINT1, EINT2, Or EINT3
     * |        |          |00 = Px.n pin is selected as one of interrupt sources to EINT0.
     * |        |          |01 = Px.n pin is selected as one of interrupt sources to EINT1.
     * |        |          |10 = Px.n pin is selected as one of interrupt sources to EINT2.
     * |        |          |11 = Px.n pin is selected as one of interrupt sources to EINT3.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
    */
    __IO uint32_t INTSRCGP;
    uint32_t RESERVE2[3];

    /**
     * Px_INTEN
     * ===================================================================================================
     * Offset: 0x90  Px Interrupt Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FEIEN0    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[1]     |FEIEN1    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[2]     |FEIEN2    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[3]     |FEIEN3    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[4]     |FEIEN4    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[5]     |FEIEN5    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[6]     |FEIEN6    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[7]     |FEIEN7    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[8]     |FEIEN8    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[9]     |FEIEN9    |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[10]    |FEIEN10   |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[11]    |FEIEN11   |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[12]    |FEIEN12   |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[13]    |FEIEN13   |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[14]    |FEIEN14   |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[15]    |FEIEN15   |Port A-D Pin[N] Control Falling Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n high to low interrupt Disabled.
     * |        |          |1 = Px.n high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[16]    |REIEN0    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[17]    |REIEN1    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[18]    |REIEN2    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[19]    |REIEN3    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[20]    |REIEN4    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[21]    |REIEN5    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[22]    |REIEN6    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[23]    |REIEN7    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[24]    |REIEN8    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[25]    |REIEN9    |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[26]    |REIEN10   |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[27]    |REIEN11   |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[28]    |REIEN12   |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[29]    |REIEN13   |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[30]    |REIEN14   |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
     * |[31]    |REIEN15   |Port A-D Pin[N] Control Rising Edge Of Input Px.N Pin To Trigger The Interrupt
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1 :
     * |        |          |The input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n low to high interrupt Disabled.
     * |        |          |1 = Px.n low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=15 for port A/B.
     * |        |          |Max. n=14 for port C.
     * |        |          |Max. n=4 for port D.
    */
    __IO uint32_t INTEN;
    uint32_t RESERVE3[4];

    /**
     * Px_LATCHDAT
     * ===================================================================================================
     * Offset: 0xA4  Px Interrupt Latch Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DAT0      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[1]     |DAT1      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[2]     |DAT2      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[3]     |DAT3      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[4]     |DAT4      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[5]     |DAT5      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[6]     |DAT6      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[7]     |DAT7      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[8]     |DAT8      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[9]     |DAT9      |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[10]    |DAT10     |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[11]    |DAT11     |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[12]    |DAT12     |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[13]    |DAT13     |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[14]    |DAT14     |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
     * |[15]    |DAT15     |Latch   Px.N Interrupt Value
     * |        |          |Latched   value of Px.n while the EINT (EINT0~EINT3) selected by GPIO_INTCTL is active.
     * |        |          |Note:
     * |        |          |Max.   n=15 for port A/B.
     * |        |          |Max.   n=14 for port C.
     * |        |          |Max. n=4   for port D.
    */
    __IO uint32_t LATCHDAT;

} GPIO_PA_T;


typedef struct
{
    __IO uint32_t MODE;
    __IO uint32_t PUEN;
    __IO uint32_t DOUT;
    __I  uint32_t PIN;
    uint32_t RESERVE1[25];
    __IO uint32_t INTSRCGP;
    uint32_t RESERVE2[3];
    __IO uint32_t INTEN;
    uint32_t RESERVE3[4];
    __IO uint32_t LATCHDAT;
} GPIO_PB_T;


typedef struct
{
    __IO uint32_t MODE;
    __IO uint32_t PUEN;
    __IO uint32_t DOUT;
    __I  uint32_t PIN;
    uint32_t RESERVE1[22];
    __IO uint32_t INTSRCGP;
    uint32_t RESERVE2[3];
    __IO uint32_t INTEN;
    uint32_t RESERVE3[4];
    __IO uint32_t LATCHDAT;
} GPIO_PC_T;


typedef struct
{
    __IO uint32_t MODE;
    __IO uint32_t PUEN;
    __IO uint32_t DOUT;
    __I  uint32_t PIN;
    uint32_t RESERVE1[19];
    __IO uint32_t INTSRCGP;
    uint32_t RESERVE2[3];
    __IO uint32_t INTEN;
    uint32_t RESERVE3[4];
    __IO uint32_t LATCHDAT;
} GPIO_PD_T;


typedef struct
{


    /**
     * GPIO_DBCTL
     * ===================================================================================================
     * Offset: 0x70  Interrupt Event (EINT) De-bounce Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DBEN0     |EINT (EINT0~EINT3) De-Bounce Enable Control
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding EINT.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de- bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [7:4].
     * |        |          |0 = EINTn de-bounce function Disabled.
     * |        |          |1 = EINTn de-bounce function Enabled.
     * |[1]     |DBEN1     |EINT (EINT0~EINT3) De-Bounce Enable Control
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding EINT.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de- bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [7:4].
     * |        |          |0 = EINTn de-bounce function Disabled.
     * |        |          |1 = EINTn de-bounce function Enabled.
     * |[2]     |DBEN2     |EINT (EINT0~EINT3) De-Bounce Enable Control
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding EINT.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de- bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [7:4].
     * |        |          |0 = EINTn de-bounce function Disabled.
     * |        |          |1 = EINTn de-bounce function Enabled.
     * |[3]     |DBEN3     |EINT (EINT0~EINT3) De-Bounce Enable Control
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding EINT.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de- bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [7:4].
     * |        |          |0 = EINTn de-bounce function Disabled.
     * |        |          |1 = EINTn de-bounce function Enabled.
     * |[7:4]   |DBCLKSEL  |De-Bounce Sampling Cycle Selection
     * |        |          |0000 = Sample interrupt input once per 1 APB clocks.
     * |        |          |0001 = Sample interrupt input once per 2 APB clocks.
     * |        |          |0010 = Sample interrupt input once per 4 APB clocks.
     * |        |          |0011 = Sample interrupt input once per 8 APB clocks.
     * |        |          |0100 = Sample interrupt input once per 16 APB clocks.
     * |        |          |0101 = Sample interrupt input once per 32 APB clocks.
     * |        |          |0110 = Sample interrupt input once per 64 APB clocks.
     * |        |          |0111 = Sample interrupt input once per 128 APB clocks.
     * |        |          |1000 = Sample interrupt input once per 256 APB clocks.
     * |        |          |1001 = Sample interrupt input once per 512 APB clocks.
     * |        |          |1010 = Sample interrupt input once per 1024 APB clocks.
     * |        |          |1011 = Sample interrupt input once per 2048 APB clocks.
     * |        |          |1100 = Sample interrupt input once per 4096 APB clocks.
     * |        |          |1101 = Sample interrupt input once per 8192 APB clocks.
     * |        |          |1110 = Sample interrupt input once per 16384 APB clocks.
     * |        |          |1111 = Sample interrupt input once per 32768 APB clocks.
    */
    __IO uint32_t DBCTL;
    uint32_t RESERVE1[11];

    /**
     * GPIO_INTCTL
     * ===================================================================================================
     * Offset: 0xA0  Interrupt Latch Trigger Selection Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INTLHEN0  |Enable Latch PA.N/PB.N/PC.N/PD.N Value When EINTn Happen
     * |        |          |0 = No effect.
     * |        |          |1 = When EINTn interrupt has happened, PA_LATCHDAT, PB_LATCHDAT, PC_LATCHDAT, and PD_LATCHDAT registers will latch PA, PB, PC, and PD port values within EINTn group.
     * |[1]     |INTLHEN1  |Enable Latch PA.N/PB.N/PC.N/PD.N Value When EINTn Happen
     * |        |          |0 = No effect.
     * |        |          |1 = When EINTn interrupt has happened, PA_LATCHDAT, PB_LATCHDAT, PC_LATCHDAT, and PD_LATCHDAT registers will latch PA, PB, PC, and PD port values within EINTn group.
     * |[2]     |INTLHEN2  |Enable Latch PA.N/PB.N/PC.N/PD.N Value When EINTn Happen
     * |        |          |0 = No effect.
     * |        |          |1 = When EINTn interrupt has happened, PA_LATCHDAT, PB_LATCHDAT, PC_LATCHDAT, and PD_LATCHDAT registers will latch PA, PB, PC, and PD port values within EINTn group.
     * |[3]     |INTLHEN3  |Enable Latch PA.N/PB.N/PC.N/PD.N Value When EINTn Happen
     * |        |          |0 = No effect.
     * |        |          |1 = When EINTn interrupt has happened, PA_LATCHDAT, PB_LATCHDAT, PC_LATCHDAT, and PD_LATCHDAT registers will latch PA, PB, PC, and PD port values within EINTn group.
     * |[4]     |WKEN0     |GPIO Interrupt Wake Up System Enable Control
     * |        |          |0 = No effect.
     * |        |          |1 = EINTn can wake up the chip from Idle and Power-down mode.
     * |[5]     |WKEN1     |GPIO Interrupt Wake Up System Enable Control
     * |        |          |0 = No effect.
     * |        |          |1 = EINTn can wake up the chip from Idle and Power-down mode.
     * |[6]     |WKEN2     |GPIO Interrupt Wake Up System Enable Control
     * |        |          |0 = No effect.
     * |        |          |1 = EINTn can wake up the chip from Idle and Power-down mode.
     * |[7]     |WKEN3     |GPIO Interrupt Wake Up System Enable Control
     * |        |          |0 = No effect.
     * |        |          |1 = EINTn can wake up the chip from Idle and Power-down mode.
     * |[8]     |INTCTL    |Interrupt Request Source Control
     * |        |          |0 = When the GPIO interrupt occurs, the GPIO interrupt controller generates 1 APB clock pulse to the NVIC.
     * |        |          |1 = When the GPIO interrupt occurs, the interrupt from GPIO to NVIC will keep till the CPU clear the interrupt trigger source.
     * |        |          |(GPIO_INTSTSA_B, GPIO_INTSTSC_D).
    */
    __IO uint32_t INTCTL;
    uint32_t RESERVE2[4];

    /**
     * GPIO_INTSTSA_B
     * ===================================================================================================
     * Offset: 0xB4  EINT0~3 Interrupt Trigger Source Indicator from PA and PB
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PAIF0     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[1]     |PAIF1     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[2]     |PAIF2     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[3]     |PAIF3     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[4]     |PAIF4     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[5]     |PAIF5     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[6]     |PAIF6     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[7]     |PAIF7     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[8]     |PAIF8     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[9]     |PAIF9     |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[10]    |PAIF10    |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[11]    |PAIF11    |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[12]    |PAIF12    |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[13]    |PAIF13    |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[14]    |PAIF14    |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[15]    |PAIF15    |Port A Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PA.n.
     * |        |          |1 = PA.n generates an interrupt.
     * |        |          |When this bit is read as "1", it indicates that PA.n is a trigger source to generate the interrupt.
     * |        |          |Note: Write 1 to clear the correspond interrupt source
     * |[16]    |PBIF0     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[17]    |PBIF1     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[18]    |PBIF2     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[19]    |PBIF3     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[20]    |PBIF4     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[21]    |PBIF5     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[22]    |PBIF6     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[23]    |PBIF7     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[24]    |PBIF8     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[25]    |PBIF9     |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[26]    |PBIF10    |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[27]    |PBIF11    |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[28]    |PBIF12    |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[29]    |PBIF13    |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[30]    |PBIF14    |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
     * |[31]    |PBIF15    |Port B Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PB.n1 = PB.n generates an interrupt.
    */
    __IO uint32_t INTSTSA_B;

    /**
     * GPIO_INTSTSC_D
     * ===================================================================================================
     * Offset: 0xB8  EINT0~3 Interrupt Trigger Source Indicator from PC and PD
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PCIF0     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[1]     |PCIF1     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[2]     |PCIF2     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[3]     |PCIF3     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[4]     |PCIF4     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[5]     |PCIF5     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[6]     |PCIF6     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[7]     |PCIF7     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[8]     |PCIF8     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[9]     |PCIF9     |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[10]    |PCIF10    |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[11]    |PCIF11    |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[12]    |PCIF12    |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[13]    |PCIF13    |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[14]    |PCIF14    |Port C Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PC.n.
     * |        |          |1 = PC.n generates an interrupt.
     * |[16]    |PDIF0     |Port D Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PD.n.
     * |        |          |1 = PD.n generates an interrupt.
     * |[17]    |PDIF1     |Port D Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PD.n.
     * |        |          |1 = PD.n generates an interrupt.
     * |[18]    |PDIF2     |Port D Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PD.n.
     * |        |          |1 = PD.n generates an interrupt.
     * |[19]    |PDIF3     |Port D Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PD.n.
     * |        |          |1 = PD.n generates an interrupt.
     * |[20]    |PDIF4     |Port D Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at PD.n.
     * |        |          |1 = PD.n generates an interrupt.
    */
    __IO uint32_t INTSTSC_D;

} GPIO_T;

/**
    @addtogroup GPIO_CONST GPIO Bit Field Definition
    Constant Definitions for GPIO Controller
@{ */

#define GPIO_MODE_MODE0_Pos              (0)                                               /*!< GPIO_PA_T::MODE: MODE0 Position              */
#define GPIO_MODE_MODE0_Msk              (0x1ul << GPIO_MODE_MODE0_Pos)                    /*!< GPIO_PA_T::MODE: MODE0 Mask                  */

#define GPIO_MODE_MODE1_Pos              (1)                                               /*!< GPIO_PA_T::MODE: MODE1 Position              */
#define GPIO_MODE_MODE1_Msk              (0x1ul << GPIO_MODE_MODE1_Pos)                    /*!< GPIO_PA_T::MODE: MODE1 Mask                  */

#define GPIO_MODE_MODE2_Pos              (2)                                               /*!< GPIO_PA_T::MODE: MODE2 Position              */
#define GPIO_MODE_MODE2_Msk              (0x1ul << GPIO_MODE_MODE2_Pos)                    /*!< GPIO_PA_T::MODE: MODE2 Mask                  */

#define GPIO_MODE_MODE3_Pos              (3)                                               /*!< GPIO_PA_T::MODE: MODE3 Position              */
#define GPIO_MODE_MODE3_Msk              (0x1ul << GPIO_MODE_MODE3_Pos)                    /*!< GPIO_PA_T::MODE: MODE3 Mask                  */

#define GPIO_MODE_MODE4_Pos              (4)                                               /*!< GPIO_PA_T::MODE: MODE4 Position              */
#define GPIO_MODE_MODE4_Msk              (0x1ul << GPIO_MODE_MODE4_Pos)                    /*!< GPIO_PA_T::MODE: MODE4 Mask                  */

#define GPIO_MODE_MODE5_Pos              (5)                                               /*!< GPIO_PA_T::MODE: MODE5 Position              */
#define GPIO_MODE_MODE5_Msk              (0x1ul << GPIO_MODE_MODE5_Pos)                    /*!< GPIO_PA_T::MODE: MODE5 Mask                  */

#define GPIO_MODE_MODE6_Pos              (6)                                               /*!< GPIO_PA_T::MODE: MODE6 Position              */
#define GPIO_MODE_MODE6_Msk              (0x1ul << GPIO_MODE_MODE6_Pos)                    /*!< GPIO_PA_T::MODE: MODE6 Mask                  */

#define GPIO_MODE_MODE7_Pos              (7)                                               /*!< GPIO_PA_T::MODE: MODE7 Position              */
#define GPIO_MODE_MODE7_Msk              (0x1ul << GPIO_MODE_MODE7_Pos)                    /*!< GPIO_PA_T::MODE: MODE7 Mask                  */

#define GPIO_MODE_MODE8_Pos              (8)                                               /*!< GPIO_PA_T::MODE: MODE8 Position              */
#define GPIO_MODE_MODE8_Msk              (0x1ul << GPIO_MODE_MODE8_Pos)                    /*!< GPIO_PA_T::MODE: MODE8 Mask                  */

#define GPIO_MODE_MODE9_Pos              (9)                                               /*!< GPIO_PA_T::MODE: MODE9 Position              */
#define GPIO_MODE_MODE9_Msk              (0x1ul << GPIO_MODE_MODE9_Pos)                    /*!< GPIO_PA_T::MODE: MODE9 Mask                  */

#define GPIO_MODE_MODE10_Pos              (10)                                               /*!< GPIO_PA_T::MODE: MODE10 Position              */
#define GPIO_MODE_MODE10_Msk              (0x1ul << GPIO_MODE_MODE10_Pos)                    /*!< GPIO_PA_T::MODE: MODE10 Mask                  */

#define GPIO_MODE_MODE11_Pos              (11)                                               /*!< GPIO_PA_T::MODE: MODE11 Position              */
#define GPIO_MODE_MODE11_Msk              (0x1ul << GPIO_MODE_MODE11_Pos)                    /*!< GPIO_PA_T::MODE: MODE11 Mask                  */

#define GPIO_MODE_MODE12_Pos              (12)                                               /*!< GPIO_PA_T::MODE: MODE12 Position              */
#define GPIO_MODE_MODE12_Msk              (0x1ul << GPIO_MODE_MODE12_Pos)                    /*!< GPIO_PA_T::MODE: MODE12 Mask                  */

#define GPIO_MODE_MODE13_Pos              (13)                                               /*!< GPIO_PA_T::MODE: MODE13 Position              */
#define GPIO_MODE_MODE13_Msk              (0x1ul << GPIO_MODE_MODE13_Pos)                    /*!< GPIO_PA_T::MODE: MODE13 Mask                  */

#define GPIO_MODE_MODE14_Pos              (14)                                               /*!< GPIO_PA_T::MODE: MODE14 Position              */
#define GPIO_MODE_MODE14_Msk              (0x1ul << GPIO_MODE_MODE14_Pos)                    /*!< GPIO_PA_T::MODE: MODE14 Mask                  */

#define GPIO_MODE_MODE15_Pos              (15)                                               /*!< GPIO_PA_T::MODE: MODE15 Position              */
#define GPIO_MODE_MODE15_Msk              (0x1ul << GPIO_MODE_MODE15_Pos)                    /*!< GPIO_PA_T::MODE: MODE15 Mask                  */

#define GPIO_PUEN_PULLSEL0_Pos           (0)                                               /*!< GPIO_PA_T::PUEN: PULLSEL0 Position           */
#define GPIO_PUEN_PULLSEL0_Msk           (0x3ul << GPIO_PUEN_PULLSEL0_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL0 Mask               */

#define GPIO_PUEN_PULLSEL1_Pos           (2)                                               /*!< GPIO_PA_T::PUEN: PULLSEL1 Position           */
#define GPIO_PUEN_PULLSEL1_Msk           (0x3ul << GPIO_PUEN_PULLSEL1_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL1 Mask               */

#define GPIO_PUEN_PULLSEL2_Pos           (4)                                               /*!< GPIO_PA_T::PUEN: PULLSEL2 Position           */
#define GPIO_PUEN_PULLSEL2_Msk           (0x3ul << GPIO_PUEN_PULLSEL2_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL2 Mask               */

#define GPIO_PUEN_PULLSEL3_Pos           (6)                                               /*!< GPIO_PA_T::PUEN: PULLSEL3 Position           */
#define GPIO_PUEN_PULLSEL3_Msk           (0x3ul << GPIO_PUEN_PULLSEL3_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL3 Mask               */

#define GPIO_PUEN_PULLSEL4_Pos           (8)                                               /*!< GPIO_PA_T::PUEN: PULLSEL4 Position           */
#define GPIO_PUEN_PULLSEL4_Msk           (0x3ul << GPIO_PUEN_PULLSEL4_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL4 Mask               */

#define GPIO_PUEN_PULLSEL5_Pos           (10)                                              /*!< GPIO_PA_T::PUEN: PULLSEL5 Position           */
#define GPIO_PUEN_PULLSEL5_Msk           (0x3ul << GPIO_PUEN_PULLSEL5_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL5 Mask               */

#define GPIO_PUEN_PULLSEL6_Pos           (12)                                              /*!< GPIO_PA_T::PUEN: PULLSEL6 Position           */
#define GPIO_PUEN_PULLSEL6_Msk           (0x3ul << GPIO_PUEN_PULLSEL6_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL6 Mask               */

#define GPIO_PUEN_PULLSEL7_Pos           (14)                                              /*!< GPIO_PA_T::PUEN: PULLSEL7 Position           */
#define GPIO_PUEN_PULLSEL7_Msk           (0x3ul << GPIO_PUEN_PULLSEL7_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL7 Mask               */

#define GPIO_PUEN_PULLSEL8_Pos           (16)                                              /*!< GPIO_PA_T::PUEN: PULLSEL8 Position           */
#define GPIO_PUEN_PULLSEL8_Msk           (0x3ul << GPIO_PUEN_PULLSEL8_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL8 Mask               */

#define GPIO_PUEN_PULLSEL9_Pos           (18)                                              /*!< GPIO_PA_T::PUEN: PULLSEL9 Position           */
#define GPIO_PUEN_PULLSEL9_Msk           (0x3ul << GPIO_PUEN_PULLSEL9_Pos)                 /*!< GPIO_PA_T::PUEN: PULLSEL9 Mask               */

#define GPIO_PUEN_PULLSEL10_Pos          (20)                                              /*!< GPIO_PA_T::PUEN: PULLSEL10 Position          */
#define GPIO_PUEN_PULLSEL10_Msk          (0x3ul << GPIO_PUEN_PULLSEL10_Pos)                /*!< GPIO_PA_T::PUEN: PULLSEL10 Mask              */

#define GPIO_PUEN_PULLSEL11_Pos          (22)                                              /*!< GPIO_PA_T::PUEN: PULLSEL11 Position          */
#define GPIO_PUEN_PULLSEL11_Msk          (0x3ul << GPIO_PUEN_PULLSEL11_Pos)                /*!< GPIO_PA_T::PUEN: PULLSEL11 Mask              */

#define GPIO_PUEN_PULLSEL12_Pos          (24)                                              /*!< GPIO_PA_T::PUEN: PULLSEL12 Position          */
#define GPIO_PUEN_PULLSEL12_Msk          (0x3ul << GPIO_PUEN_PULLSEL12_Pos)                /*!< GPIO_PA_T::PUEN: PULLSEL12 Mask              */

#define GPIO_PUEN_PULLSEL13_Pos          (26)                                              /*!< GPIO_PA_T::PUEN: PULLSEL13 Position          */
#define GPIO_PUEN_PULLSEL13_Msk          (0x3ul << GPIO_PUEN_PULLSEL13_Pos)                /*!< GPIO_PA_T::PUEN: PULLSEL13 Mask              */

#define GPIO_PUEN_PULLSEL14_Pos          (28)                                              /*!< GPIO_PA_T::PUEN: PULLSEL14 Position          */
#define GPIO_PUEN_PULLSEL14_Msk          (0x3ul << GPIO_PUEN_PULLSEL14_Pos)                /*!< GPIO_PA_T::PUEN: PULLSEL14 Mask              */

#define GPIO_PUEN_PULLSEL15_Pos          (30)                                              /*!< GPIO_PA_T::PUEN: PULLSEL15 Position          */
#define GPIO_PUEN_PULLSEL15_Msk          (0x3ul << GPIO_PUEN_PULLSEL15_Pos)                /*!< GPIO_PA_T::PUEN: PULLSEL15 Mask              */

#define GPIO_DOUT_DOUT0_Pos              (0)                                               /*!< GPIO_PA_T::DOUT: DOUT0 Position              */
#define GPIO_DOUT_DOUT0_Msk              (0x1ul << GPIO_DOUT_DOUT0_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT0 Mask                  */

#define GPIO_DOUT_DOUT1_Pos              (1)                                               /*!< GPIO_PA_T::DOUT: DOUT1 Position              */
#define GPIO_DOUT_DOUT1_Msk              (0x1ul << GPIO_DOUT_DOUT1_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT1 Mask                  */

#define GPIO_DOUT_DOUT2_Pos              (2)                                               /*!< GPIO_PA_T::DOUT: DOUT2 Position              */
#define GPIO_DOUT_DOUT2_Msk              (0x1ul << GPIO_DOUT_DOUT2_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT2 Mask                  */

#define GPIO_DOUT_DOUT3_Pos              (3)                                               /*!< GPIO_PA_T::DOUT: DOUT3 Position              */
#define GPIO_DOUT_DOUT3_Msk              (0x1ul << GPIO_DOUT_DOUT3_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT3 Mask                  */

#define GPIO_DOUT_DOUT4_Pos              (4)                                               /*!< GPIO_PA_T::DOUT: DOUT4 Position              */
#define GPIO_DOUT_DOUT4_Msk              (0x1ul << GPIO_DOUT_DOUT4_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT4 Mask                  */

#define GPIO_DOUT_DOUT5_Pos              (5)                                               /*!< GPIO_PA_T::DOUT: DOUT5 Position              */
#define GPIO_DOUT_DOUT5_Msk              (0x1ul << GPIO_DOUT_DOUT5_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT5 Mask                  */

#define GPIO_DOUT_DOUT6_Pos              (6)                                               /*!< GPIO_PA_T::DOUT: DOUT6 Position              */
#define GPIO_DOUT_DOUT6_Msk              (0x1ul << GPIO_DOUT_DOUT6_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT6 Mask                  */

#define GPIO_DOUT_DOUT7_Pos              (7)                                               /*!< GPIO_PA_T::DOUT: DOUT7 Position              */
#define GPIO_DOUT_DOUT7_Msk              (0x1ul << GPIO_DOUT_DOUT7_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT7 Mask                  */

#define GPIO_DOUT_DOUT8_Pos              (8)                                               /*!< GPIO_PA_T::DOUT: DOUT8 Position              */
#define GPIO_DOUT_DOUT8_Msk              (0x1ul << GPIO_DOUT_DOUT8_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT8 Mask                  */

#define GPIO_DOUT_DOUT9_Pos              (9)                                               /*!< GPIO_PA_T::DOUT: DOUT9 Position              */
#define GPIO_DOUT_DOUT9_Msk              (0x1ul << GPIO_DOUT_DOUT9_Pos)                    /*!< GPIO_PA_T::DOUT: DOUT9 Mask                  */

#define GPIO_DOUT_DOUT10_Pos             (10)                                              /*!< GPIO_PA_T::DOUT: DOUT10 Position             */
#define GPIO_DOUT_DOUT10_Msk             (0x1ul << GPIO_DOUT_DOUT10_Pos)                   /*!< GPIO_PA_T::DOUT: DOUT10 Mask                 */

#define GPIO_DOUT_DOUT11_Pos             (11)                                              /*!< GPIO_PA_T::DOUT: DOUT11 Position             */
#define GPIO_DOUT_DOUT11_Msk             (0x1ul << GPIO_DOUT_DOUT11_Pos)                   /*!< GPIO_PA_T::DOUT: DOUT11 Mask                 */

#define GPIO_DOUT_DOUT12_Pos             (12)                                              /*!< GPIO_PA_T::DOUT: DOUT12 Position             */
#define GPIO_DOUT_DOUT12_Msk             (0x1ul << GPIO_DOUT_DOUT12_Pos)                   /*!< GPIO_PA_T::DOUT: DOUT12 Mask                 */

#define GPIO_DOUT_DOUT13_Pos             (13)                                              /*!< GPIO_PA_T::DOUT: DOUT13 Position             */
#define GPIO_DOUT_DOUT13_Msk             (0x1ul << GPIO_DOUT_DOUT13_Pos)                   /*!< GPIO_PA_T::DOUT: DOUT13 Mask                 */

#define GPIO_DOUT_DOUT14_Pos             (14)                                              /*!< GPIO_PA_T::DOUT: DOUT14 Position             */
#define GPIO_DOUT_DOUT14_Msk             (0x1ul << GPIO_DOUT_DOUT14_Pos)                   /*!< GPIO_PA_T::DOUT: DOUT14 Mask                 */

#define GPIO_DOUT_DOUT15_Pos             (15)                                              /*!< GPIO_PA_T::DOUT: DOUT15 Position             */
#define GPIO_DOUT_DOUT15_Msk             (0x1ul << GPIO_DOUT_DOUT15_Pos)                   /*!< GPIO_PA_T::DOUT: DOUT15 Mask                 */

#define GPIO_PIN_PIN0_Pos                (0)                                               /*!< GPIO_PA_T::PIN: PIN0 Position                */
#define GPIO_PIN_PIN0_Msk                (0x1ul << GPIO_PIN_PIN0_Pos)                      /*!< GPIO_PA_T::PIN: PIN0 Mask                    */

#define GPIO_PIN_PIN1_Pos                (1)                                               /*!< GPIO_PA_T::PIN: PIN1 Position                */
#define GPIO_PIN_PIN1_Msk                (0x1ul << GPIO_PIN_PIN1_Pos)                      /*!< GPIO_PA_T::PIN: PIN1 Mask                    */

#define GPIO_PIN_PIN2_Pos                (2)                                               /*!< GPIO_PA_T::PIN: PIN2 Position                */
#define GPIO_PIN_PIN2_Msk                (0x1ul << GPIO_PIN_PIN2_Pos)                      /*!< GPIO_PA_T::PIN: PIN2 Mask                    */

#define GPIO_PIN_PIN3_Pos                (3)                                               /*!< GPIO_PA_T::PIN: PIN3 Position                */
#define GPIO_PIN_PIN3_Msk                (0x1ul << GPIO_PIN_PIN3_Pos)                      /*!< GPIO_PA_T::PIN: PIN3 Mask                    */

#define GPIO_PIN_PIN4_Pos                (4)                                               /*!< GPIO_PA_T::PIN: PIN4 Position                */
#define GPIO_PIN_PIN4_Msk                (0x1ul << GPIO_PIN_PIN4_Pos)                      /*!< GPIO_PA_T::PIN: PIN4 Mask                    */

#define GPIO_PIN_PIN5_Pos                (5)                                               /*!< GPIO_PA_T::PIN: PIN5 Position                */
#define GPIO_PIN_PIN5_Msk                (0x1ul << GPIO_PIN_PIN5_Pos)                      /*!< GPIO_PA_T::PIN: PIN5 Mask                    */

#define GPIO_PIN_PIN6_Pos                (6)                                               /*!< GPIO_PA_T::PIN: PIN6 Position                */
#define GPIO_PIN_PIN6_Msk                (0x1ul << GPIO_PIN_PIN6_Pos)                      /*!< GPIO_PA_T::PIN: PIN6 Mask                    */

#define GPIO_PIN_PIN7_Pos                (7)                                               /*!< GPIO_PA_T::PIN: PIN7 Position                */
#define GPIO_PIN_PIN7_Msk                (0x1ul << GPIO_PIN_PIN7_Pos)                      /*!< GPIO_PA_T::PIN: PIN7 Mask                    */

#define GPIO_PIN_PIN8_Pos                (8)                                               /*!< GPIO_PA_T::PIN: PIN8 Position                */
#define GPIO_PIN_PIN8_Msk                (0x1ul << GPIO_PIN_PIN8_Pos)                      /*!< GPIO_PA_T::PIN: PIN8 Mask                    */

#define GPIO_PIN_PIN9_Pos                (9)                                               /*!< GPIO_PA_T::PIN: PIN9 Position                */
#define GPIO_PIN_PIN9_Msk                (0x1ul << GPIO_PIN_PIN9_Pos)                      /*!< GPIO_PA_T::PIN: PIN9 Mask                    */

#define GPIO_PIN_PIN10_Pos               (10)                                              /*!< GPIO_PA_T::PIN: PIN10 Position               */
#define GPIO_PIN_PIN10_Msk               (0x1ul << GPIO_PIN_PIN10_Pos)                     /*!< GPIO_PA_T::PIN: PIN10 Mask                   */

#define GPIO_PIN_PIN11_Pos               (11)                                              /*!< GPIO_PA_T::PIN: PIN11 Position               */
#define GPIO_PIN_PIN11_Msk               (0x1ul << GPIO_PIN_PIN11_Pos)                     /*!< GPIO_PA_T::PIN: PIN11 Mask                   */

#define GPIO_PIN_PIN12_Pos               (12)                                              /*!< GPIO_PA_T::PIN: PIN12 Position               */
#define GPIO_PIN_PIN12_Msk               (0x1ul << GPIO_PIN_PIN12_Pos)                     /*!< GPIO_PA_T::PIN: PIN12 Mask                   */

#define GPIO_PIN_PIN13_Pos               (13)                                              /*!< GPIO_PA_T::PIN: PIN13 Position               */
#define GPIO_PIN_PIN13_Msk               (0x1ul << GPIO_PIN_PIN13_Pos)                     /*!< GPIO_PA_T::PIN: PIN13 Mask                   */

#define GPIO_PIN_PIN14_Pos               (14)                                              /*!< GPIO_PA_T::PIN: PIN14 Position               */
#define GPIO_PIN_PIN14_Msk               (0x1ul << GPIO_PIN_PIN14_Pos)                     /*!< GPIO_PA_T::PIN: PIN14 Mask                   */

#define GPIO_PIN_PIN15_Pos               (15)                                              /*!< GPIO_PA_T::PIN: PIN15 Position               */
#define GPIO_PIN_PIN15_Msk               (0x1ul << GPIO_PIN_PIN15_Pos)                     /*!< GPIO_PA_T::PIN: PIN15 Mask                   */

#define GPIO_DBCTL_DBEN0_Pos             (0)                                               /*!< GPIO_T::DBCTL: DBEN0 Position             */
#define GPIO_DBCTL_DBEN0_Msk             (0x1ul << GPIO_DBCTL_DBEN0_Pos)                   /*!< GPIO_T::DBCTL: DBEN0 Mask                 */

#define GPIO_DBCTL_DBEN1_Pos             (1)                                               /*!< GPIO_T::DBCTL: DBEN1 Position             */
#define GPIO_DBCTL_DBEN1_Msk             (0x1ul << GPIO_DBCTL_DBEN1_Pos)                   /*!< GPIO_T::DBCTL: DBEN1 Mask                 */

#define GPIO_DBCTL_DBEN2_Pos             (2)                                               /*!< GPIO_T::DBCTL: DBEN2 Position             */
#define GPIO_DBCTL_DBEN2_Msk             (0x1ul << GPIO_DBCTL_DBEN2_Pos)                   /*!< GPIO_T::DBCTL: DBEN2 Mask                 */

#define GPIO_DBCTL_DBEN3_Pos             (3)                                               /*!< GPIO_T::DBCTL: DBEN3 Position             */
#define GPIO_DBCTL_DBEN3_Msk             (0x1ul << GPIO_DBCTL_DBEN3_Pos)                   /*!< GPIO_T::DBCTL: DBEN3 Mask                 */

#define GPIO_DBCTL_DBCLKSEL_Pos          (4)                                               /*!< GPIO_T::DBCTL: DBCLKSEL Position          */
#define GPIO_DBCTL_DBCLKSEL_Msk          (0xful << GPIO_DBCTL_DBCLKSEL_Pos)                /*!< GPIO_T::DBCTL: DBCLKSEL Mask              */

#define GPIO_INTSRCGP_INTSEL0_Pos        (0)                                               /*!< GPIO_PA_T::INTSRCGP: INTSEL0 Position        */
#define GPIO_INTSRCGP_INTSEL0_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL0_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL0 Mask            */

#define GPIO_INTSRCGP_INTSEL1_Pos        (2)                                               /*!< GPIO_PA_T::INTSRCGP: INTSEL1 Position        */
#define GPIO_INTSRCGP_INTSEL1_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL1_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL1 Mask            */

#define GPIO_INTSRCGP_INTSEL2_Pos        (4)                                               /*!< GPIO_PA_T::INTSRCGP: INTSEL2 Position        */
#define GPIO_INTSRCGP_INTSEL2_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL2_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL2 Mask            */

#define GPIO_INTSRCGP_INTSEL3_Pos        (6)                                               /*!< GPIO_PA_T::INTSRCGP: INTSEL3 Position        */
#define GPIO_INTSRCGP_INTSEL3_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL3_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL3 Mask            */

#define GPIO_INTSRCGP_INTSEL4_Pos        (8)                                               /*!< GPIO_PA_T::INTSRCGP: INTSEL4 Position        */
#define GPIO_INTSRCGP_INTSEL4_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL4_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL4 Mask            */

#define GPIO_INTSRCGP_INTSEL5_Pos        (10)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL5 Position        */
#define GPIO_INTSRCGP_INTSEL5_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL5_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL5 Mask            */

#define GPIO_INTSRCGP_INTSEL6_Pos        (12)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL6 Position        */
#define GPIO_INTSRCGP_INTSEL6_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL6_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL6 Mask            */

#define GPIO_INTSRCGP_INTSEL7_Pos        (14)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL7 Position        */
#define GPIO_INTSRCGP_INTSEL7_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL7_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL7 Mask            */

#define GPIO_INTSRCGP_INTSEL8_Pos        (16)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL8 Position        */
#define GPIO_INTSRCGP_INTSEL8_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL8_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL8 Mask            */

#define GPIO_INTSRCGP_INTSEL9_Pos        (18)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL9 Position        */
#define GPIO_INTSRCGP_INTSEL9_Msk        (0x3ul << GPIO_INTSRCGP_INTSEL9_Pos)              /*!< GPIO_PA_T::INTSRCGP: INTSEL9 Mask            */

#define GPIO_INTSRCGP_INTSEL10_Pos       (20)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL10 Position       */
#define GPIO_INTSRCGP_INTSEL10_Msk       (0x3ul << GPIO_INTSRCGP_INTSEL10_Pos)             /*!< GPIO_PA_T::INTSRCGP: INTSEL10 Mask           */

#define GPIO_INTSRCGP_INTSEL11_Pos       (22)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL11 Position       */
#define GPIO_INTSRCGP_INTSEL11_Msk       (0x3ul << GPIO_INTSRCGP_INTSEL11_Pos)             /*!< GPIO_PA_T::INTSRCGP: INTSEL11 Mask           */

#define GPIO_INTSRCGP_INTSEL12_Pos       (24)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL12 Position       */
#define GPIO_INTSRCGP_INTSEL12_Msk       (0x3ul << GPIO_INTSRCGP_INTSEL12_Pos)             /*!< GPIO_PA_T::INTSRCGP: INTSEL12 Mask           */

#define GPIO_INTSRCGP_INTSEL13_Pos       (26)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL13 Position       */
#define GPIO_INTSRCGP_INTSEL13_Msk       (0x3ul << GPIO_INTSRCGP_INTSEL13_Pos)             /*!< GPIO_PA_T::INTSRCGP: INTSEL13 Mask           */

#define GPIO_INTSRCGP_INTSEL14_Pos       (28)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL14 Position       */
#define GPIO_INTSRCGP_INTSEL14_Msk       (0x3ul << GPIO_INTSRCGP_INTSEL14_Pos)             /*!< GPIO_PA_T::INTSRCGP: INTSEL14 Mask           */

#define GPIO_INTSRCGP_INTSEL15_Pos       (30)                                              /*!< GPIO_PA_T::INTSRCGP: INTSEL15 Position       */
#define GPIO_INTSRCGP_INTSEL15_Msk       (0x3ul << GPIO_INTSRCGP_INTSEL15_Pos)             /*!< GPIO_PA_T::INTSRCGP: INTSEL15 Mask           */

#define GPIO_INTEN_FEIEN0_Pos            (0)                                               /*!< GPIO_PA_T::INTEN: FEIEN0 Position            */
#define GPIO_INTEN_FEIEN0_Msk            (0x1ul << GPIO_INTEN_FEIEN0_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN0 Mask                */

#define GPIO_INTEN_FEIEN1_Pos            (1)                                               /*!< GPIO_PA_T::INTEN: FEIEN1 Position            */
#define GPIO_INTEN_FEIEN1_Msk            (0x1ul << GPIO_INTEN_FEIEN1_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN1 Mask                */

#define GPIO_INTEN_FEIEN2_Pos            (2)                                               /*!< GPIO_PA_T::INTEN: FEIEN2 Position            */
#define GPIO_INTEN_FEIEN2_Msk            (0x1ul << GPIO_INTEN_FEIEN2_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN2 Mask                */

#define GPIO_INTEN_FEIEN3_Pos            (3)                                               /*!< GPIO_PA_T::INTEN: FEIEN3 Position            */
#define GPIO_INTEN_FEIEN3_Msk            (0x1ul << GPIO_INTEN_FEIEN3_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN3 Mask                */

#define GPIO_INTEN_FEIEN4_Pos            (4)                                               /*!< GPIO_PA_T::INTEN: FEIEN4 Position            */
#define GPIO_INTEN_FEIEN4_Msk            (0x1ul << GPIO_INTEN_FEIEN4_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN4 Mask                */

#define GPIO_INTEN_FEIEN5_Pos            (5)                                               /*!< GPIO_PA_T::INTEN: FEIEN5 Position            */
#define GPIO_INTEN_FEIEN5_Msk            (0x1ul << GPIO_INTEN_FEIEN5_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN5 Mask                */

#define GPIO_INTEN_FEIEN6_Pos            (6)                                               /*!< GPIO_PA_T::INTEN: FEIEN6 Position            */
#define GPIO_INTEN_FEIEN6_Msk            (0x1ul << GPIO_INTEN_FEIEN6_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN6 Mask                */

#define GPIO_INTEN_FEIEN7_Pos            (7)                                               /*!< GPIO_PA_T::INTEN: FEIEN7 Position            */
#define GPIO_INTEN_FEIEN7_Msk            (0x1ul << GPIO_INTEN_FEIEN7_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN7 Mask                */

#define GPIO_INTEN_FEIEN8_Pos            (8)                                               /*!< GPIO_PA_T::INTEN: FEIEN8 Position            */
#define GPIO_INTEN_FEIEN8_Msk            (0x1ul << GPIO_INTEN_FEIEN8_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN8 Mask                */

#define GPIO_INTEN_FEIEN9_Pos            (9)                                               /*!< GPIO_PA_T::INTEN: FEIEN9 Position            */
#define GPIO_INTEN_FEIEN9_Msk            (0x1ul << GPIO_INTEN_FEIEN9_Pos)                  /*!< GPIO_PA_T::INTEN: FEIEN9 Mask                */

#define GPIO_INTEN_FEIEN10_Pos           (10)                                              /*!< GPIO_PA_T::INTEN: FEIEN10 Position           */
#define GPIO_INTEN_FEIEN10_Msk           (0x1ul << GPIO_INTEN_FEIEN10_Pos)                 /*!< GPIO_PA_T::INTEN: FEIEN10 Mask               */

#define GPIO_INTEN_FEIEN11_Pos           (11)                                              /*!< GPIO_PA_T::INTEN: FEIEN11 Position           */
#define GPIO_INTEN_FEIEN11_Msk           (0x1ul << GPIO_INTEN_FEIEN11_Pos)                 /*!< GPIO_PA_T::INTEN: FEIEN11 Mask               */

#define GPIO_INTEN_FEIEN12_Pos           (12)                                              /*!< GPIO_PA_T::INTEN: FEIEN12 Position           */
#define GPIO_INTEN_FEIEN12_Msk           (0x1ul << GPIO_INTEN_FEIEN12_Pos)                 /*!< GPIO_PA_T::INTEN: FEIEN12 Mask               */

#define GPIO_INTEN_FEIEN13_Pos           (13)                                              /*!< GPIO_PA_T::INTEN: FEIEN13 Position           */
#define GPIO_INTEN_FEIEN13_Msk           (0x1ul << GPIO_INTEN_FEIEN13_Pos)                 /*!< GPIO_PA_T::INTEN: FEIEN13 Mask               */

#define GPIO_INTEN_FEIEN14_Pos           (14)                                              /*!< GPIO_PA_T::INTEN: FEIEN14 Position           */
#define GPIO_INTEN_FEIEN14_Msk           (0x1ul << GPIO_INTEN_FEIEN14_Pos)                 /*!< GPIO_PA_T::INTEN: FEIEN14 Mask               */

#define GPIO_INTEN_FEIEN15_Pos           (15)                                              /*!< GPIO_PA_T::INTEN: FEIEN15 Position           */
#define GPIO_INTEN_FEIEN15_Msk           (0x1ul << GPIO_INTEN_FEIEN15_Pos)                 /*!< GPIO_PA_T::INTEN: FEIEN15 Mask               */

#define GPIO_INTEN_REIEN0_Pos            (16)                                              /*!< GPIO_PA_T::INTEN: REIEN0 Position            */
#define GPIO_INTEN_REIEN0_Msk            (0x1ul << GPIO_INTEN_REIEN0_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN0 Mask                */

#define GPIO_INTEN_REIEN1_Pos            (17)                                              /*!< GPIO_PA_T::INTEN: REIEN1 Position            */
#define GPIO_INTEN_REIEN1_Msk            (0x1ul << GPIO_INTEN_REIEN1_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN1 Mask                */

#define GPIO_INTEN_REIEN2_Pos            (18)                                              /*!< GPIO_PA_T::INTEN: REIEN2 Position            */
#define GPIO_INTEN_REIEN2_Msk            (0x1ul << GPIO_INTEN_REIEN2_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN2 Mask                */

#define GPIO_INTEN_REIEN3_Pos            (19)                                              /*!< GPIO_PA_T::INTEN: REIEN3 Position            */
#define GPIO_INTEN_REIEN3_Msk            (0x1ul << GPIO_INTEN_REIEN3_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN3 Mask                */

#define GPIO_INTEN_REIEN4_Pos            (20)                                              /*!< GPIO_PA_T::INTEN: REIEN4 Position            */
#define GPIO_INTEN_REIEN4_Msk            (0x1ul << GPIO_INTEN_REIEN4_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN4 Mask                */

#define GPIO_INTEN_REIEN5_Pos            (21)                                              /*!< GPIO_PA_T::INTEN: REIEN5 Position            */
#define GPIO_INTEN_REIEN5_Msk            (0x1ul << GPIO_INTEN_REIEN5_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN5 Mask                */

#define GPIO_INTEN_REIEN6_Pos            (22)                                              /*!< GPIO_PA_T::INTEN: REIEN6 Position            */
#define GPIO_INTEN_REIEN6_Msk            (0x1ul << GPIO_INTEN_REIEN6_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN6 Mask                */

#define GPIO_INTEN_REIEN7_Pos            (23)                                              /*!< GPIO_PA_T::INTEN: REIEN7 Position            */
#define GPIO_INTEN_REIEN7_Msk            (0x1ul << GPIO_INTEN_REIEN7_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN7 Mask                */

#define GPIO_INTEN_REIEN8_Pos            (24)                                              /*!< GPIO_PA_T::INTEN: REIEN8 Position            */
#define GPIO_INTEN_REIEN8_Msk            (0x1ul << GPIO_INTEN_REIEN8_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN8 Mask                */

#define GPIO_INTEN_REIEN9_Pos            (25)                                              /*!< GPIO_PA_T::INTEN: REIEN9 Position            */
#define GPIO_INTEN_REIEN9_Msk            (0x1ul << GPIO_INTEN_REIEN9_Pos)                  /*!< GPIO_PA_T::INTEN: REIEN9 Mask                */

#define GPIO_INTEN_REIEN10_Pos           (26)                                              /*!< GPIO_PA_T::INTEN: REIEN10 Position           */
#define GPIO_INTEN_REIEN10_Msk           (0x1ul << GPIO_INTEN_REIEN10_Pos)                 /*!< GPIO_PA_T::INTEN: REIEN10 Mask               */

#define GPIO_INTEN_REIEN11_Pos           (27)                                              /*!< GPIO_PA_T::INTEN: REIEN11 Position           */
#define GPIO_INTEN_REIEN11_Msk           (0x1ul << GPIO_INTEN_REIEN11_Pos)                 /*!< GPIO_PA_T::INTEN: REIEN11 Mask               */

#define GPIO_INTEN_REIEN12_Pos           (28)                                              /*!< GPIO_PA_T::INTEN: REIEN12 Position           */
#define GPIO_INTEN_REIEN12_Msk           (0x1ul << GPIO_INTEN_REIEN12_Pos)                 /*!< GPIO_PA_T::INTEN: REIEN12 Mask               */

#define GPIO_INTEN_REIEN13_Pos           (29)                                              /*!< GPIO_PA_T::INTEN: REIEN13 Position           */
#define GPIO_INTEN_REIEN13_Msk           (0x1ul << GPIO_INTEN_REIEN13_Pos)                 /*!< GPIO_PA_T::INTEN: REIEN13 Mask               */

#define GPIO_INTEN_REIEN14_Pos           (30)                                              /*!< GPIO_PA_T::INTEN: REIEN14 Position           */
#define GPIO_INTEN_REIEN14_Msk           (0x1ul << GPIO_INTEN_REIEN14_Pos)                 /*!< GPIO_PA_T::INTEN: REIEN14 Mask               */

#define GPIO_INTEN_REIEN15_Pos           (31)                                              /*!< GPIO_PA_T::INTEN: REIEN15 Position           */
#define GPIO_INTEN_REIEN15_Msk           (0x1ul << GPIO_INTEN_REIEN15_Pos)                 /*!< GPIO_PA_T::INTEN: REIEN15 Mask               */

#define GPIO_INTCTL_INTLHEN0_Pos         (0)                                               /*!< GPIO_T::INTCTL: INTLHEN0 Position         */
#define GPIO_INTCTL_INTLHEN0_Msk         (0x1ul << GPIO_INTCTL_INTLHEN0_Pos)               /*!< GPIO_T::INTCTL: INTLHEN0 Mask             */

#define GPIO_INTCTL_INTLHEN1_Pos         (1)                                               /*!< GPIO_T::INTCTL: INTLHEN1 Position         */
#define GPIO_INTCTL_INTLHEN1_Msk         (0x1ul << GPIO_INTCTL_INTLHEN1_Pos)               /*!< GPIO_T::INTCTL: INTLHEN1 Mask             */

#define GPIO_INTCTL_INTLHEN2_Pos         (2)                                               /*!< GPIO_T::INTCTL: INTLHEN2 Position         */
#define GPIO_INTCTL_INTLHEN2_Msk         (0x1ul << GPIO_INTCTL_INTLHEN2_Pos)               /*!< GPIO_T::INTCTL: INTLHEN2 Mask             */

#define GPIO_INTCTL_INTLHEN3_Pos         (3)                                               /*!< GPIO_T::INTCTL: INTLHEN3 Position         */
#define GPIO_INTCTL_INTLHEN3_Msk         (0x1ul << GPIO_INTCTL_INTLHEN3_Pos)               /*!< GPIO_T::INTCTL: INTLHEN3 Mask             */

#define GPIO_INTCTL_WKEN0_Pos            (4)                                               /*!< GPIO_T::INTCTL: WKEN0 Position            */
#define GPIO_INTCTL_WKEN0_Msk            (0x1ul << GPIO_INTCTL_WKEN0_Pos)                  /*!< GPIO_T::INTCTL: WKEN0 Mask                */

#define GPIO_INTCTL_WKEN1_Pos            (5)                                               /*!< GPIO_T::INTCTL: WKEN1 Position            */
#define GPIO_INTCTL_WKEN1_Msk            (0x1ul << GPIO_INTCTL_WKEN1_Pos)                  /*!< GPIO_T::INTCTL: WKEN1 Mask                */

#define GPIO_INTCTL_WKEN2_Pos            (6)                                               /*!< GPIO_T::INTCTL: WKEN2 Position            */
#define GPIO_INTCTL_WKEN2_Msk            (0x1ul << GPIO_INTCTL_WKEN2_Pos)                  /*!< GPIO_T::INTCTL: WKEN2 Mask                */

#define GPIO_INTCTL_WKEN3_Pos            (7)                                               /*!< GPIO_T::INTCTL: WKEN3 Position            */
#define GPIO_INTCTL_WKEN3_Msk            (0x1ul << GPIO_INTCTL_WKEN3_Pos)                  /*!< GPIO_T::INTCTL: WKEN3 Mask                */

#define GPIO_INTCTL_INTCTL_Pos           (8)                                               /*!< GPIO_T::INTCTL: INTCTL Position           */
#define GPIO_INTCTL_INTCTL_Msk           (0x1ul << GPIO_INTCTL_INTCTL_Pos)                 /*!< GPIO_T::INTCTL: INTCTL Mask               */

#define GPIO_LATCHDAT_DAT0_Pos           (0)                                               /*!< GPIO_PA_T::LATCHDAT: DAT0 Position           */
#define GPIO_LATCHDAT_DAT0_Msk           (0x1ul << GPIO_LATCHDAT_DAT0_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT0 Mask               */

#define GPIO_LATCHDAT_DAT1_Pos           (1)                                               /*!< GPIO_PA_T::LATCHDAT: DAT1 Position           */
#define GPIO_LATCHDAT_DAT1_Msk           (0x1ul << GPIO_LATCHDAT_DAT1_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT1 Mask               */

#define GPIO_LATCHDAT_DAT2_Pos           (2)                                               /*!< GPIO_PA_T::LATCHDAT: DAT2 Position           */
#define GPIO_LATCHDAT_DAT2_Msk           (0x1ul << GPIO_LATCHDAT_DAT2_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT2 Mask               */

#define GPIO_LATCHDAT_DAT3_Pos           (3)                                               /*!< GPIO_PA_T::LATCHDAT: DAT3 Position           */
#define GPIO_LATCHDAT_DAT3_Msk           (0x1ul << GPIO_LATCHDAT_DAT3_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT3 Mask               */

#define GPIO_LATCHDAT_DAT4_Pos           (4)                                               /*!< GPIO_PA_T::LATCHDAT: DAT4 Position           */
#define GPIO_LATCHDAT_DAT4_Msk           (0x1ul << GPIO_LATCHDAT_DAT4_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT4 Mask               */

#define GPIO_LATCHDAT_DAT5_Pos           (5)                                               /*!< GPIO_PA_T::LATCHDAT: DAT5 Position           */
#define GPIO_LATCHDAT_DAT5_Msk           (0x1ul << GPIO_LATCHDAT_DAT5_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT5 Mask               */

#define GPIO_LATCHDAT_DAT6_Pos           (6)                                               /*!< GPIO_PA_T::LATCHDAT: DAT6 Position           */
#define GPIO_LATCHDAT_DAT6_Msk           (0x1ul << GPIO_LATCHDAT_DAT6_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT6 Mask               */

#define GPIO_LATCHDAT_DAT7_Pos           (7)                                               /*!< GPIO_PA_T::LATCHDAT: DAT7 Position           */
#define GPIO_LATCHDAT_DAT7_Msk           (0x1ul << GPIO_LATCHDAT_DAT7_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT7 Mask               */

#define GPIO_LATCHDAT_DAT8_Pos           (8)                                               /*!< GPIO_PA_T::LATCHDAT: DAT8 Position           */
#define GPIO_LATCHDAT_DAT8_Msk           (0x1ul << GPIO_LATCHDAT_DAT8_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT8 Mask               */

#define GPIO_LATCHDAT_DAT9_Pos           (9)                                               /*!< GPIO_PA_T::LATCHDAT: DAT9 Position           */
#define GPIO_LATCHDAT_DAT9_Msk           (0x1ul << GPIO_LATCHDAT_DAT9_Pos)                 /*!< GPIO_PA_T::LATCHDAT: DAT9 Mask               */

#define GPIO_LATCHDAT_DAT10_Pos          (10)                                              /*!< GPIO_PA_T::LATCHDAT: DAT10 Position          */
#define GPIO_LATCHDAT_DAT10_Msk          (0x1ul << GPIO_LATCHDAT_DAT10_Pos)                /*!< GPIO_PA_T::LATCHDAT: DAT10 Mask              */

#define GPIO_LATCHDAT_DAT11_Pos          (11)                                              /*!< GPIO_PA_T::LATCHDAT: DAT11 Position          */
#define GPIO_LATCHDAT_DAT11_Msk          (0x1ul << GPIO_LATCHDAT_DAT11_Pos)                /*!< GPIO_PA_T::LATCHDAT: DAT11 Mask              */

#define GPIO_LATCHDAT_DAT12_Pos          (12)                                              /*!< GPIO_PA_T::LATCHDAT: DAT12 Position          */
#define GPIO_LATCHDAT_DAT12_Msk          (0x1ul << GPIO_LATCHDAT_DAT12_Pos)                /*!< GPIO_PA_T::LATCHDAT: DAT12 Mask              */

#define GPIO_LATCHDAT_DAT13_Pos          (13)                                              /*!< GPIO_PA_T::LATCHDAT: DAT13 Position          */
#define GPIO_LATCHDAT_DAT13_Msk          (0x1ul << GPIO_LATCHDAT_DAT13_Pos)                /*!< GPIO_PA_T::LATCHDAT: DAT13 Mask              */

#define GPIO_LATCHDAT_DAT14_Pos          (14)                                              /*!< GPIO_PA_T::LATCHDAT: DAT14 Position          */
#define GPIO_LATCHDAT_DAT14_Msk          (0x1ul << GPIO_LATCHDAT_DAT14_Pos)                /*!< GPIO_PA_T::LATCHDAT: DAT14 Mask              */

#define GPIO_LATCHDAT_DAT15_Pos          (15)                                              /*!< GPIO_PA_T::LATCHDAT: DAT15 Position          */
#define GPIO_LATCHDAT_DAT15_Msk          (0x1ul << GPIO_LATCHDAT_DAT15_Pos)                /*!< GPIO_PA_T::LATCHDAT: DAT15 Mask              */

#define GPIO_INTSTSA_B_PAIF0_Pos         (0)                                               /*!< GPIO_T::INTSTSA_B: PAIF0 Position         */
#define GPIO_INTSTSA_B_PAIF0_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF0_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF0 Mask             */

#define GPIO_INTSTSA_B_PAIF1_Pos         (1)                                               /*!< GPIO_T::INTSTSA_B: PAIF1 Position         */
#define GPIO_INTSTSA_B_PAIF1_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF1_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF1 Mask             */

#define GPIO_INTSTSA_B_PAIF2_Pos         (2)                                               /*!< GPIO_T::INTSTSA_B: PAIF2 Position         */
#define GPIO_INTSTSA_B_PAIF2_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF2_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF2 Mask             */

#define GPIO_INTSTSA_B_PAIF3_Pos         (3)                                               /*!< GPIO_T::INTSTSA_B: PAIF3 Position         */
#define GPIO_INTSTSA_B_PAIF3_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF3_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF3 Mask             */

#define GPIO_INTSTSA_B_PAIF4_Pos         (4)                                               /*!< GPIO_T::INTSTSA_B: PAIF4 Position         */
#define GPIO_INTSTSA_B_PAIF4_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF4_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF4 Mask             */

#define GPIO_INTSTSA_B_PAIF5_Pos         (5)                                               /*!< GPIO_T::INTSTSA_B: PAIF5 Position         */
#define GPIO_INTSTSA_B_PAIF5_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF5_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF5 Mask             */

#define GPIO_INTSTSA_B_PAIF6_Pos         (6)                                               /*!< GPIO_T::INTSTSA_B: PAIF6 Position         */
#define GPIO_INTSTSA_B_PAIF6_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF6_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF6 Mask             */

#define GPIO_INTSTSA_B_PAIF7_Pos         (7)                                               /*!< GPIO_T::INTSTSA_B: PAIF7 Position         */
#define GPIO_INTSTSA_B_PAIF7_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF7_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF7 Mask             */

#define GPIO_INTSTSA_B_PAIF8_Pos         (8)                                               /*!< GPIO_T::INTSTSA_B: PAIF8 Position         */
#define GPIO_INTSTSA_B_PAIF8_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF8_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF8 Mask             */

#define GPIO_INTSTSA_B_PAIF9_Pos         (9)                                               /*!< GPIO_T::INTSTSA_B: PAIF9 Position         */
#define GPIO_INTSTSA_B_PAIF9_Msk         (0x1ul << GPIO_INTSTSA_B_PAIF9_Pos)               /*!< GPIO_T::INTSTSA_B: PAIF9 Mask             */

#define GPIO_INTSTSA_B_PAIF10_Pos        (10)                                              /*!< GPIO_T::INTSTSA_B: PAIF10 Position        */
#define GPIO_INTSTSA_B_PAIF10_Msk        (0x1ul << GPIO_INTSTSA_B_PAIF10_Pos)              /*!< GPIO_T::INTSTSA_B: PAIF10 Mask            */

#define GPIO_INTSTSA_B_PAIF11_Pos        (11)                                              /*!< GPIO_T::INTSTSA_B: PAIF11 Position        */
#define GPIO_INTSTSA_B_PAIF11_Msk        (0x1ul << GPIO_INTSTSA_B_PAIF11_Pos)              /*!< GPIO_T::INTSTSA_B: PAIF11 Mask            */

#define GPIO_INTSTSA_B_PAIF12_Pos        (12)                                              /*!< GPIO_T::INTSTSA_B: PAIF12 Position        */
#define GPIO_INTSTSA_B_PAIF12_Msk        (0x1ul << GPIO_INTSTSA_B_PAIF12_Pos)              /*!< GPIO_T::INTSTSA_B: PAIF12 Mask            */

#define GPIO_INTSTSA_B_PAIF13_Pos        (13)                                              /*!< GPIO_T::INTSTSA_B: PAIF13 Position        */
#define GPIO_INTSTSA_B_PAIF13_Msk        (0x1ul << GPIO_INTSTSA_B_PAIF13_Pos)              /*!< GPIO_T::INTSTSA_B: PAIF13 Mask            */

#define GPIO_INTSTSA_B_PAIF14_Pos        (14)                                              /*!< GPIO_T::INTSTSA_B: PAIF14 Position        */
#define GPIO_INTSTSA_B_PAIF14_Msk        (0x1ul << GPIO_INTSTSA_B_PAIF14_Pos)              /*!< GPIO_T::INTSTSA_B: PAIF14 Mask            */

#define GPIO_INTSTSA_B_PAIF15_Pos        (15)                                              /*!< GPIO_T::INTSTSA_B: PAIF15 Position        */
#define GPIO_INTSTSA_B_PAIF15_Msk        (0x1ul << GPIO_INTSTSA_B_PAIF15_Pos)              /*!< GPIO_T::INTSTSA_B: PAIF15 Mask            */

#define GPIO_INTSTSA_B_PBIF0_Pos         (16)                                              /*!< GPIO_T::INTSTSA_B: PBIF0 Position         */
#define GPIO_INTSTSA_B_PBIF0_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF0_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF0 Mask             */

#define GPIO_INTSTSA_B_PBIF1_Pos         (17)                                              /*!< GPIO_T::INTSTSA_B: PBIF1 Position         */
#define GPIO_INTSTSA_B_PBIF1_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF1_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF1 Mask             */

#define GPIO_INTSTSA_B_PBIF2_Pos         (18)                                              /*!< GPIO_T::INTSTSA_B: PBIF2 Position         */
#define GPIO_INTSTSA_B_PBIF2_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF2_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF2 Mask             */

#define GPIO_INTSTSA_B_PBIF3_Pos         (19)                                              /*!< GPIO_T::INTSTSA_B: PBIF3 Position         */
#define GPIO_INTSTSA_B_PBIF3_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF3_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF3 Mask             */

#define GPIO_INTSTSA_B_PBIF4_Pos         (20)                                              /*!< GPIO_T::INTSTSA_B: PBIF4 Position         */
#define GPIO_INTSTSA_B_PBIF4_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF4_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF4 Mask             */

#define GPIO_INTSTSA_B_PBIF5_Pos         (21)                                              /*!< GPIO_T::INTSTSA_B: PBIF5 Position         */
#define GPIO_INTSTSA_B_PBIF5_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF5_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF5 Mask             */

#define GPIO_INTSTSA_B_PBIF6_Pos         (22)                                              /*!< GPIO_T::INTSTSA_B: PBIF6 Position         */
#define GPIO_INTSTSA_B_PBIF6_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF6_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF6 Mask             */

#define GPIO_INTSTSA_B_PBIF7_Pos         (23)                                              /*!< GPIO_T::INTSTSA_B: PBIF7 Position         */
#define GPIO_INTSTSA_B_PBIF7_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF7_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF7 Mask             */

#define GPIO_INTSTSA_B_PBIF8_Pos         (24)                                              /*!< GPIO_T::INTSTSA_B: PBIF8 Position         */
#define GPIO_INTSTSA_B_PBIF8_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF8_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF8 Mask             */

#define GPIO_INTSTSA_B_PBIF9_Pos         (25)                                              /*!< GPIO_T::INTSTSA_B: PBIF9 Position         */
#define GPIO_INTSTSA_B_PBIF9_Msk         (0x1ul << GPIO_INTSTSA_B_PBIF9_Pos)               /*!< GPIO_T::INTSTSA_B: PBIF9 Mask             */

#define GPIO_INTSTSA_B_PBIF10_Pos        (26)                                              /*!< GPIO_T::INTSTSA_B: PBIF10 Position        */
#define GPIO_INTSTSA_B_PBIF10_Msk        (0x1ul << GPIO_INTSTSA_B_PBIF10_Pos)              /*!< GPIO_T::INTSTSA_B: PBIF10 Mask            */

#define GPIO_INTSTSA_B_PBIF11_Pos        (27)                                              /*!< GPIO_T::INTSTSA_B: PBIF11 Position        */
#define GPIO_INTSTSA_B_PBIF11_Msk        (0x1ul << GPIO_INTSTSA_B_PBIF11_Pos)              /*!< GPIO_T::INTSTSA_B: PBIF11 Mask            */

#define GPIO_INTSTSA_B_PBIF12_Pos        (28)                                              /*!< GPIO_T::INTSTSA_B: PBIF12 Position        */
#define GPIO_INTSTSA_B_PBIF12_Msk        (0x1ul << GPIO_INTSTSA_B_PBIF12_Pos)              /*!< GPIO_T::INTSTSA_B: PBIF12 Mask            */

#define GPIO_INTSTSA_B_PBIF13_Pos        (29)                                              /*!< GPIO_T::INTSTSA_B: PBIF13 Position        */
#define GPIO_INTSTSA_B_PBIF13_Msk        (0x1ul << GPIO_INTSTSA_B_PBIF13_Pos)              /*!< GPIO_T::INTSTSA_B: PBIF13 Mask            */

#define GPIO_INTSTSA_B_PBIF14_Pos        (30)                                              /*!< GPIO_T::INTSTSA_B: PBIF14 Position        */
#define GPIO_INTSTSA_B_PBIF14_Msk        (0x1ul << GPIO_INTSTSA_B_PBIF14_Pos)              /*!< GPIO_T::INTSTSA_B: PBIF14 Mask            */

#define GPIO_INTSTSA_B_PBIF15_Pos        (31)                                              /*!< GPIO_T::INTSTSA_B: PBIF15 Position        */
#define GPIO_INTSTSA_B_PBIF15_Msk        (0x1ul << GPIO_INTSTSA_B_PBIF15_Pos)              /*!< GPIO_T::INTSTSA_B: PBIF15 Mask            */

#define GPIO_INTSTSC_D_PCIF0_Pos         (0)                                               /*!< GPIO_T::INTSTSC_D: PCIF0 Position         */
#define GPIO_INTSTSC_D_PCIF0_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF0_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF0 Mask             */

#define GPIO_INTSTSC_D_PCIF1_Pos         (1)                                               /*!< GPIO_T::INTSTSC_D: PCIF1 Position         */
#define GPIO_INTSTSC_D_PCIF1_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF1_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF1 Mask             */

#define GPIO_INTSTSC_D_PCIF2_Pos         (2)                                               /*!< GPIO_T::INTSTSC_D: PCIF2 Position         */
#define GPIO_INTSTSC_D_PCIF2_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF2_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF2 Mask             */

#define GPIO_INTSTSC_D_PCIF3_Pos         (3)                                               /*!< GPIO_T::INTSTSC_D: PCIF3 Position         */
#define GPIO_INTSTSC_D_PCIF3_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF3_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF3 Mask             */

#define GPIO_INTSTSC_D_PCIF4_Pos         (4)                                               /*!< GPIO_T::INTSTSC_D: PCIF4 Position         */
#define GPIO_INTSTSC_D_PCIF4_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF4_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF4 Mask             */

#define GPIO_INTSTSC_D_PCIF5_Pos         (5)                                               /*!< GPIO_T::INTSTSC_D: PCIF5 Position         */
#define GPIO_INTSTSC_D_PCIF5_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF5_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF5 Mask             */

#define GPIO_INTSTSC_D_PCIF6_Pos         (6)                                               /*!< GPIO_T::INTSTSC_D: PCIF6 Position         */
#define GPIO_INTSTSC_D_PCIF6_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF6_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF6 Mask             */

#define GPIO_INTSTSC_D_PCIF7_Pos         (7)                                               /*!< GPIO_T::INTSTSC_D: PCIF7 Position         */
#define GPIO_INTSTSC_D_PCIF7_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF7_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF7 Mask             */

#define GPIO_INTSTSC_D_PCIF8_Pos         (8)                                               /*!< GPIO_T::INTSTSC_D: PCIF8 Position         */
#define GPIO_INTSTSC_D_PCIF8_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF8_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF8 Mask             */

#define GPIO_INTSTSC_D_PCIF9_Pos         (9)                                               /*!< GPIO_T::INTSTSC_D: PCIF9 Position         */
#define GPIO_INTSTSC_D_PCIF9_Msk         (0x1ul << GPIO_INTSTSC_D_PCIF9_Pos)               /*!< GPIO_T::INTSTSC_D: PCIF9 Mask             */

#define GPIO_INTSTSC_D_PCIF10_Pos        (10)                                              /*!< GPIO_T::INTSTSC_D: PCIF10 Position        */
#define GPIO_INTSTSC_D_PCIF10_Msk        (0x1ul << GPIO_INTSTSC_D_PCIF10_Pos)              /*!< GPIO_T::INTSTSC_D: PCIF10 Mask            */

#define GPIO_INTSTSC_D_PCIF11_Pos        (11)                                              /*!< GPIO_T::INTSTSC_D: PCIF11 Position        */
#define GPIO_INTSTSC_D_PCIF11_Msk        (0x1ul << GPIO_INTSTSC_D_PCIF11_Pos)              /*!< GPIO_T::INTSTSC_D: PCIF11 Mask            */

#define GPIO_INTSTSC_D_PCIF12_Pos        (12)                                              /*!< GPIO_T::INTSTSC_D: PCIF12 Position        */
#define GPIO_INTSTSC_D_PCIF12_Msk        (0x1ul << GPIO_INTSTSC_D_PCIF12_Pos)              /*!< GPIO_T::INTSTSC_D: PCIF12 Mask            */

#define GPIO_INTSTSC_D_PCIF13_Pos        (13)                                              /*!< GPIO_T::INTSTSC_D: PCIF13 Position        */
#define GPIO_INTSTSC_D_PCIF13_Msk        (0x1ul << GPIO_INTSTSC_D_PCIF13_Pos)              /*!< GPIO_T::INTSTSC_D: PCIF13 Mask            */

#define GPIO_INTSTSC_D_PCIF14_Pos        (14)                                              /*!< GPIO_T::INTSTSC_D: PCIF14 Position        */
#define GPIO_INTSTSC_D_PCIF14_Msk        (0x1ul << GPIO_INTSTSC_D_PCIF14_Pos)              /*!< GPIO_T::INTSTSC_D: PCIF14 Mask            */

#define GPIO_INTSTSC_D_PDIF0_Pos         (16)                                              /*!< GPIO_T::INTSTSC_D: PDIF0 Position         */
#define GPIO_INTSTSC_D_PDIF0_Msk         (0x1ul << GPIO_INTSTSC_D_PDIF0_Pos)               /*!< GPIO_T::INTSTSC_D: PDIF0 Mask             */

#define GPIO_INTSTSC_D_PDIF1_Pos         (17)                                              /*!< GPIO_T::INTSTSC_D: PDIF1 Position         */
#define GPIO_INTSTSC_D_PDIF1_Msk         (0x1ul << GPIO_INTSTSC_D_PDIF1_Pos)               /*!< GPIO_T::INTSTSC_D: PDIF1 Mask             */

#define GPIO_INTSTSC_D_PDIF2_Pos         (18)                                              /*!< GPIO_T::INTSTSC_D: PDIF2 Position         */
#define GPIO_INTSTSC_D_PDIF2_Msk         (0x1ul << GPIO_INTSTSC_D_PDIF2_Pos)               /*!< GPIO_T::INTSTSC_D: PDIF2 Mask             */

#define GPIO_INTSTSC_D_PDIF3_Pos         (19)                                              /*!< GPIO_T::INTSTSC_D: PDIF3 Position         */
#define GPIO_INTSTSC_D_PDIF3_Msk         (0x1ul << GPIO_INTSTSC_D_PDIF3_Pos)               /*!< GPIO_T::INTSTSC_D: PDIF3 Mask             */

#define GPIO_INTSTSC_D_PDIF4_Pos         (20)                                              /*!< GPIO_T::INTSTSC_D: PDIF4 Position         */
#define GPIO_INTSTSC_D_PDIF4_Msk         (0x1ul << GPIO_INTSTSC_D_PDIF4_Pos)               /*!< GPIO_T::INTSTSC_D: PDIF4 Mask             */

/**@}*/ /* GPIO_CONST */
/**@}*/ /* end of GPIO register group */


/*---------------------- Inter-IC Bus Controller -------------------------*/
/**
    @addtogroup I2C Inter-IC Bus Controller(I2C)
    Memory Mapped Structure for I2C Controller
@{ */

typedef struct
{


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x00  I2C Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2]     |AA        |Assert Acknowledge Control
     * |        |          |When AA =1 prior to address or data received, an acknowledged (low level to SDA) will be returned during the acknowledge clock pulse on the SCL line when 1.).
     * |        |          |A slave is acknowledging the address sent from master, 2.) The receiver devices are acknowledging the data sent by transmitter.
     * |        |          |When AA=0 prior to address or data received, a Not acknowledged (high level to SDA) will be returned during the acknowledge clock pulse on the SCL line.
     * |[3]     |SI        |I2C Interrupt Flag
     * |        |          |When a new I2C state is present in the I2C_STATUS register, the SI flag is set by hardware, and if bit INTEN (I2C_CTL [7]) is set, the I2C interrupt is requested.
     * |        |          |SI must be cleared by software.
     * |        |          |Clear SI by writing 1 to this bi.
     * |[4]     |STO       |I2C STOP Control
     * |        |          |In Master mode, setting STO to transmit a STOP condition to bus then I2C hardware will check the bus condition if a STOP condition is detected this bit will be cleared by hardware automatically.
     * |        |          |In Slave mode, setting STO resets I2C hardware to the defined "not addressed" Slave mode.
     * |        |          |This means it is NO LONGER in the slave receiver mode to receive data from the master transmit devic.
     * |[5]     |STA       |I2C START Control
     * |        |          |Setting STA to logic 1 to enter Master mode, the I2C hardware sends a START or repeat START condition to bus when the bus is free.
     * |[6]     |I2CEN     |I2C Controller Enable Control
     * |        |          |0 = Disabled.
     * |        |          |1 = Enabled.
     * |        |          |Set to enable I2C serial function controller.
     * |        |          |When I2CEN=1 the I2C serial function enables.
     * |        |          |The multi-function pin function of SDA and SCL must set to I2C function firs.
     * |[7]     |INTEN     |I2C Interrupt Enable Control
     * |        |          |0 = I2C interrupt Disabled.
     * |        |          |1 = I2C interrupt Enabled.
    */
    __IO uint32_t CTL;

    /**
     * ADDR0
     * ===================================================================================================
     * Offset: 0x04  I2C Slave Address Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address Bits
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In Slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matche.
    */
    __IO uint32_t ADDR0;

    /**
     * DAT
     * ===================================================================================================
     * Offset: 0x08  I2C Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |I2C Data Bits
     * |        |          |Bit [7:0] is located with the 8-bit transferred data of I2C serial port.
    */
    __IO uint32_t DAT;

    /**
     * STATUS
     * ===================================================================================================
     * Offset: 0x0C  I2C Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |STATUS    |I2C Status Bits
     * |        |          |The status register of I2C:
     * |        |          |The three least significant bits are always 0.
     * |        |          |The five most significant bits contain the status code.
     * |        |          |Refer to section 6.11.6.4 for detail descriptio.
    */
    __I  uint32_t STATUS;

    /**
     * CLKDIV
     * ===================================================================================================
     * Offset: 0x10  I2C Clock Divided Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DIVIDER   |I2C Clock Divided Bits
     * |        |          |The I2C clock rate bits: Data Baud Rate of I2C = (PCLK) / (4x (DIVIDER+1)).
     * |        |          |The minimum value of DIVIDER is 4.
    */
    __IO uint32_t CLKDIV;

    /**
     * TOCTL
     * ===================================================================================================
     * Offset: 0x14  I2C Time-out Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TOIF      |Time-out Flag
     * |        |          |This bit is set by H/W when I2C time-out happened and it can interrupt CPU if I2C interrupt enable bit (INTEN (I2C_CTL[7])) is set to 1.
     * |        |          |Write 1 to clear this bit.
     * |[1]     |TOCDIV4   |Time-out Counter Input Clock Divided By 4
     * |        |          |0 = Disabled.
     * |        |          |1 = Enabled.
     * |        |          |Note When Enabled, The time-out period is extend 4 times.
     * |[2]     |TOCEN     |Time-out Counter Enable Control
     * |        |          |0 = Disabled.
     * |        |          |1 = Enabled.
     * |        |          |Note: When Enabled, the 14-bit time-out counter will start counting when SI (I2C_CTL[3]) is cleared.
     * |        |          |Setting flag SI to high will reset counter and re-start up counting after SI is cleare.
    */
    __IO uint32_t TOCTL;

    /**
     * ADDR1
     * ===================================================================================================
     * Offset: 0x18  I2C Slave Address Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address Bits
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In Slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matche.
    */
    __IO uint32_t ADDR1;

    /**
     * ADDR2
     * ===================================================================================================
     * Offset: 0x1C  I2C Slave Address Register2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address Bits
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In Slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matche.
    */
    __IO uint32_t ADDR2;

    /**
     * ADDR3
     * ===================================================================================================
     * Offset: 0x20  I2C Slave Address Register3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GC        |General Call Function
     * |        |          |0 = General Call Function Disabled.
     * |        |          |1 = General Call Function Enabled.
     * |[7:1]   |ADDR      |I2C Address Bits
     * |        |          |The content of this register is irrelevant when I2C is in Master mode.
     * |        |          |In Slave mode, the seven most significant bits must be loaded with the chip's own address.
     * |        |          |The I2C hardware will react if either of the address is matche.
    */
    __IO uint32_t ADDR3;

    /**
     * ADDRMSK0
     * ===================================================================================================
     * Offset: 0x24  I2C Slave Address Mask Register0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask Bits
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address registe.
    */
    __IO uint32_t ADDRMSK0;

    /**
     * ADDRMSK1
     * ===================================================================================================
     * Offset: 0x28  I2C Slave Address Mask Register1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask Bits
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address registe.
    */
    __IO uint32_t ADDRMSK1;

    /**
     * ADDRMSK2
     * ===================================================================================================
     * Offset: 0x2C  I2C Slave Address Mask Register2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask Bits
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address registe.
    */
    __IO uint32_t ADDRMSK2;

    /**
     * ADDRMSK3
     * ===================================================================================================
     * Offset: 0x30  I2C Slave Address Mask Register3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:1]   |ADDRMSK   |I2C Address Mask Bits
     * |        |          |0 = Mask Disabled (the received corresponding register bit should be exact the same as address register.).
     * |        |          |1 = Mask Enabled (the received corresponding address bit is don't care.).
     * |        |          |I2C bus controllers support multiple address recognition with four address mask register.
     * |        |          |When the bit in the address mask register is set to one, it means the received corresponding address bit is don't-care.
     * |        |          |If the bit is set to zero, that means the received corresponding register bit should be exact the same as address registe.
    */
    __IO uint32_t ADDRMSK3;
    uint32_t RESERVE0[2];


    /**
     * WKCTL
     * ===================================================================================================
     * Offset: 0x3C  I2C Wake-up Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKEN      |I2C Wake-Up Enable Control
     * |        |          |0 = I2C wake-up function Disabled.
     * |        |          |1 = I2C wake-up function Enabled.
    */
    __IO uint32_t WKCTL;

    /**
     * WKSTS
     * ===================================================================================================
     * Offset: 0x40  I2C Wake-up Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |WKIF      |I2C Wake-Up Flag
     * |        |          |0 = No wake up occurred.
     * |        |          |1 = Wake up from Power-down mode.
     * |        |          |Note: Software can write 1 to clear this bit.
    */
    __IO uint32_t WKSTS;

} I2C_T;

/**
    @addtogroup I2C_CONST I2C Bit Field Definition
    Constant Definitions for I2C Controller
@{ */

#define I2C_CTL_AA_Pos                   (2)                                               /*!< I2C_T::CTL: AA Position                   */
#define I2C_CTL_AA_Msk                   (0x1ul << I2C_CTL_AA_Pos)                         /*!< I2C_T::CTL: AA Mask                       */

#define I2C_CTL_SI_Pos                   (3)                                               /*!< I2C_T::CTL: SI Position                   */
#define I2C_CTL_SI_Msk                   (0x1ul << I2C_CTL_SI_Pos)                         /*!< I2C_T::CTL: SI Mask                       */

#define I2C_CTL_STO_Pos                  (4)                                               /*!< I2C_T::CTL: STO Position                  */
#define I2C_CTL_STO_Msk                  (0x1ul << I2C_CTL_STO_Pos)                        /*!< I2C_T::CTL: STO Mask                      */

#define I2C_CTL_STA_Pos                  (5)                                               /*!< I2C_T::CTL: STA Position                  */
#define I2C_CTL_STA_Msk                  (0x1ul << I2C_CTL_STA_Pos)                        /*!< I2C_T::CTL: STA Mask                      */

#define I2C_CTL_I2CEN_Pos                (6)                                               /*!< I2C_T::CTL: I2CEN Position                */
#define I2C_CTL_I2CEN_Msk                (0x1ul << I2C_CTL_I2CEN_Pos)                      /*!< I2C_T::CTL: I2CEN Mask                    */

#define I2C_CTL_INTEN_Pos                (7)                                               /*!< I2C_T::CTL: INTEN Position                */
#define I2C_CTL_INTEN_Msk                (0x1ul << I2C_CTL_INTEN_Pos)                      /*!< I2C_T::CTL: INTEN Mask                    */

#define I2C_ADDR0_GC_Pos                 (0)                                               /*!< I2C_T::ADDR0: GC Position                 */
#define I2C_ADDR0_GC_Msk                 (0x1ul << I2C_ADDR0_GC_Pos)                       /*!< I2C_T::ADDR0: GC Mask                     */

#define I2C_ADDR0_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR0: ADDR Position               */
#define I2C_ADDR0_ADDR_Msk               (0x7ful << I2C_ADDR0_ADDR_Pos)                    /*!< I2C_T::ADDR0: ADDR Mask                   */

#define I2C_DAT_DAT_Pos                  (0)                                               /*!< I2C_T::DAT: DAT Position                  */
#define I2C_DAT_DAT_Msk                  (0xfful << I2C_DAT_DAT_Pos)                       /*!< I2C_T::DAT: DAT Mask                      */

#define I2C_STATUS_STATUS_Pos            (0)                                               /*!< I2C_T::STATUS: STATUS Position            */
#define I2C_STATUS_STATUS_Msk            (0xfful << I2C_STATUS_STATUS_Pos)                 /*!< I2C_T::STATUS: STATUS Mask                */

#define I2C_CLKDIV_DIVIDER_Pos           (0)                                               /*!< I2C_T::CLKDIV: DIVIDER Position           */
#define I2C_CLKDIV_DIVIDER_Msk           (0xfful << I2C_CLKDIV_DIVIDER_Pos)                /*!< I2C_T::CLKDIV: DIVIDER Mask               */

#define I2C_TOCTL_TOIF_Pos               (0)                                               /*!< I2C_T::TOCTL: TOIF Position               */
#define I2C_TOCTL_TOIF_Msk               (0x1ul << I2C_TOCTL_TOIF_Pos)                     /*!< I2C_T::TOCTL: TOIF Mask                   */

#define I2C_TOCTL_TOCDIV4_Pos            (1)                                               /*!< I2C_T::TOCTL: TOCDIV4 Position            */
#define I2C_TOCTL_TOCDIV4_Msk            (0x1ul << I2C_TOCTL_TOCDIV4_Pos)                  /*!< I2C_T::TOCTL: TOCDIV4 Mask                */

#define I2C_TOCTL_TOCEN_Pos              (2)                                               /*!< I2C_T::TOCTL: TOCEN Position              */
#define I2C_TOCTL_TOCEN_Msk              (0x1ul << I2C_TOCTL_TOCEN_Pos)                    /*!< I2C_T::TOCTL: TOCEN Mask                  */

#define I2C_ADDR1_GC_Pos                 (0)                                               /*!< I2C_T::ADDR1: GC Position                 */
#define I2C_ADDR1_GC_Msk                 (0x1ul << I2C_ADDR1_GC_Pos)                       /*!< I2C_T::ADDR1: GC Mask                     */

#define I2C_ADDR1_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR1: ADDR Position               */
#define I2C_ADDR1_ADDR_Msk               (0x7ful << I2C_ADDR1_ADDR_Pos)                    /*!< I2C_T::ADDR1: ADDR Mask                   */

#define I2C_ADDR2_GC_Pos                 (0)                                               /*!< I2C_T::ADDR2: GC Position                 */
#define I2C_ADDR2_GC_Msk                 (0x1ul << I2C_ADDR2_GC_Pos)                       /*!< I2C_T::ADDR2: GC Mask                     */

#define I2C_ADDR2_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR2: ADDR Position               */
#define I2C_ADDR2_ADDR_Msk               (0x7ful << I2C_ADDR2_ADDR_Pos)                    /*!< I2C_T::ADDR2: ADDR Mask                   */

#define I2C_ADDR3_GC_Pos                 (0)                                               /*!< I2C_T::ADDR3: GC Position                 */
#define I2C_ADDR3_GC_Msk                 (0x1ul << I2C_ADDR3_GC_Pos)                       /*!< I2C_T::ADDR3: GC Mask                     */

#define I2C_ADDR3_ADDR_Pos               (1)                                               /*!< I2C_T::ADDR3: ADDR Position               */
#define I2C_ADDR3_ADDR_Msk               (0x7ful << I2C_ADDR3_ADDR_Pos)                    /*!< I2C_T::ADDR3: ADDR Mask                   */

#define I2C_ADDRMSK0_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK0: ADDRMSK Position         */
#define I2C_ADDRMSK0_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK0_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK0: ADDRMSK Mask             */

#define I2C_ADDRMSK1_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK1: ADDRMSK Position         */
#define I2C_ADDRMSK1_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK1_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK1: ADDRMSK Mask             */

#define I2C_ADDRMSK2_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK2: ADDRMSK Position         */
#define I2C_ADDRMSK2_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK2_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK2: ADDRMSK Mask             */

#define I2C_ADDRMSK3_ADDRMSK_Pos         (1)                                               /*!< I2C_T::ADDRMSK3: ADDRMSK Position         */
#define I2C_ADDRMSK3_ADDRMSK_Msk         (0x7ful << I2C_ADDRMSK3_ADDRMSK_Pos)              /*!< I2C_T::ADDRMSK3: ADDRMSK Mask             */

#define I2C_WKCTL_WKEN_Pos               (0)                                               /*!< I2C_T::WKCTL: WKEN Position               */
#define I2C_WKCTL_WKEN_Msk               (0x1ul << I2C_WKCTL_WKEN_Pos)                     /*!< I2C_T::WKCTL: WKEN Mask                   */

#define I2C_WKSTS_WKIF_Pos               (0)                                               /*!< I2C_T::WKSTS: WKIF Position               */
#define I2C_WKSTS_WKIF_Msk               (0x1ul << I2C_WKSTS_WKIF_Pos)                     /*!< I2C_T::WKSTS: WKIF Mask                   */

/**@}*/ /* I2C_CONST */
/**@}*/ /* end of I2C register group */


/*---------------------- I2S Interface Controller -------------------------*/
/**
    @addtogroup I2S I2S Interface Controller(I2S)
    Memory Mapped Structure for I2S Controller
@{ */

typedef struct
{


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x00  I2S Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |I2SEN     |I2S Controller Enable Control
     * |        |          |0 = Disabled.
     * |        |          |1 = Enabled.
     * |[1]     |TXEN      |Transmit Enable Control
     * |        |          |0 = Data transmission Disabled.
     * |        |          |1 = Data transmission Enabled.
     * |[2]     |RXEN      |Receive Enable Control
     * |        |          |0 = Data receiving Disabled.
     * |        |          |1 = Data receiving Enabled.
     * |[3]     |MUTE      |Transmit Mute Enable Control
     * |        |          |0 = Transmit data is shifted from buffer.
     * |        |          |1 = Transmit data is fixed to zero.
     * |[5:4]   |WDWIDTH   |Word Width
     * |        |          |00 = Data is 8-bit.
     * |        |          |01 = Data is 16-bit.
     * |        |          |10 = Data is 24-bit.
     * |        |          |11 = Data is 32-bit.
     * |[6]     |MONO      |Monaural Data
     * |        |          |0 = Data is stereo format.
     * |        |          |1 = Data is monaural format.
     * |        |          |Note: When chip records data and MONO = 1, RXLCH (I2S_CTL[23]) will control the selection for recording right channel data or left channel data.
     * |[7]     |FORMAT    |Data Format Selection
     * |        |          |If PCMEN=0,.
     * |        |          |0 = I2S data format.
     * |        |          |1 = MSB justified data format.
     * |        |          |If PCMEN=1,.
     * |        |          |0 = PCM mode A.
     * |        |          |1 = PCM mode B.
     * |[11:8]  |TXTH      |Transmit FIFO Threshold Level
     * |        |          |If remain data word (32 bits) in transmit FIFO is the same or less than threshold level then TXTHIF flag is set.
     * |        |          |0000 = 0 word data in transmit FIFO.
     * |        |          |0001 = 1 word data in transmit FIFO.
     * |        |          |0010 = 2 words data in transmit FIFO.
     * |        |          |0011 = 3 words data in transmit FIFO.
     * |        |          |0100 = 4 words data in transmit FIFO.
     * |        |          |0101 = 5 words data in transmit FIFO.
     * |        |          |0110 = 6 words data in transmit FIFO.
     * |        |          |0111 = 7 words data in transmit FIFO.
     * |        |          |1000 = 8 word data in transmit FIFO.
     * |        |          |1001 = 9 word data in transmit FIFO.
     * |        |          |1010 = 10 words data in transmit FIFO.
     * |        |          |1011 = 11 words data in transmit FIFO.
     * |        |          |1100 = 12 words data in transmit FIFO.
     * |        |          |1101 = 13 words data in transmit FIFO.
     * |        |          |1110 = 14 words data in transmit FIFO.
     * |        |          |1111 = 15 words data in transmit FIFO.
     * |[15:12] |RXTH      |Receive FIFO Threshold Level
     * |        |          |When received data word(s) in buffer is equal to or higher than threshold level then RXTHIF flag is set.
     * |        |          |0000 = 1 word data in receive FIFO.
     * |        |          |0001 = 2 word data in receive FIFO.
     * |        |          |0010 = 3 word data in receive FIFO.
     * |        |          |0011 = 4 word data in receive FIFO.
     * |        |          |0100 = 5 word data in receive FIFO.
     * |        |          |0101 = 6 word data in receive FIFO.
     * |        |          |0110 = 7 word data in receive FIFO.
     * |        |          |0111 = 8 word data in receive FIFO.
     * |        |          |1000 = 9 word data in receive FIFO.
     * |        |          |1001 = 10 word data in receive FIFO.
     * |        |          |1010 = 11 word data in receive FIFO.
     * |        |          |1011 = 12 word data in receive FIFO.
     * |        |          |1100 = 13 word data in receive FIFO.
     * |        |          |1101 = 14 word data in receive FIFO.
     * |        |          |1110 = 15 word data in receive FIFO.
     * |        |          |1111 = 16 word data in receive FIFO.
     * |[16]    |RZCEN     |Right Channel Zero-Cross Detection Enable Control
     * |        |          |If this bit is set to 1, when right channel data sign bit change or next shift data bits are all zero then RZCIF flag in I2S_STATUS register is set to 1.
     * |        |          |0 = Right channel zero-cross detect Disabled.
     * |        |          |1 = Right channel zero-cross detect Enabled.
     * |[17]    |LZCEN     |Left Channel Zero-Cross Detect Enable Control
     * |        |          |If this bit is set to 1, when left channel data sign bit change or next shift data bits are all zero then LZCIF flag in I2S_STATUS register is set to 1.
     * |        |          |0 = Left channel zero-cross detect Disabled.
     * |        |          |1 = Left channel zero-cross detect Enabled.
     * |[18]    |TXCLR     |Clear Transmit FIFO
     * |        |          |0 = No effect.
     * |        |          |1 = Transmit FIFO will be cleared.
     * |        |          |Note: This bit will be cleared to 0 automatically.
     * |        |          |Note2: If clearing the transmit FIFO, TXCNT (I2S_STATUS[31:28]) returns to 0x0 and transmit FIFO becomes empty.
     * |[19]    |RXCLR     |Clear Receive FIFO
     * |        |          |0 = No effect.
     * |        |          |1 = Receiver FIFO will be cleared.
     * |        |          |Note: This bit will be cleared to 0 automatically.
     * |        |          |Note2: If clearing the receiver FIFO, RXCNT (I2S_STATUS[27:24]) returns to 0x0 and receiver FIFO becomes empty.
     * |[20]    |TXDMAEN   |TX DMA Enable Control (Transmit Path)
     * |        |          |0 = TX DMA mode Disabled.
     * |        |          |1 = TX DMA mode Enabled.
     * |        |          |Note: The I2S_TXSTADDR will be updated to new setting only when TXDMAEN is from low to high.
     * |        |          |Therefore, if you want to change I2S_TXSTADDR, you should confirm TXDMAEN is disable.
     * |[21]    |RXDMAEN   |RX DMA Enable Control (Record Path)
     * |        |          |0 = RX DMA mode Disabled.
     * |        |          |1 = RX DMA mode Enabled.
     * |        |          |Note: The I2S_RXSTADDR will be updated to new setting only when RXDMAEN is from low to high.
     * |        |          |Therefore, if you want to change I2S_RXSTADDR, you should confirm RXDMAEN is disable.
     * |[23]    |RXLCH     |Receive Left Channel Enable Control
     * |        |          |When monaural format is selected (MONO = 1), I2S will receive right channel data if.
     * |        |          |RXLCH is set to 0, and receive left channel data if RXLCH is set to 1.
     * |        |          |0 = Receives right channel data when monaural format is selected.
     * |        |          |1 = Receives left channel data when monaural format is selected.
     * |[24]    |PCMEN     |PCM Interface Enable Control
     * |        |          |0 = I2S Interface.
     * |        |          |1 = PCM Interface.
     * |[25]    |MCLKEN    |Master Clock Enable Control
     * |        |          |0 = I2S master clock output Disabled.
     * |        |          |1 = I2S master clock output Enabled.
     * |        |          |Note1: I2S_MCLK is always output.
     * |        |          |Note2: I2S_MCLK frequency is controlled by MCLKDIV[5:0].
     * |[26]    |SLAVE     |Slave Mode
     * |        |          |I2S can operate as master or slave.
     * |        |          |For Master mode, I2S_BCLK and I2S_LRCLK signals are output to CODEC.
     * |        |          |In Slave mode, I2S_BCLK and I2S_LRCLK pins are received from CODE.
     * |        |          |0 = Master mode.
     * |        |          |1 = Slave mode.
     * |        |          |Note: If using internal codec, the I2S must be master mode.
     * |[28]    |CODECSEL  |Internal Codec Or External Codec Selection
     * |        |          |0 = I2S interface connect to internal codec.
     * |        |          |1 = I2S interface connect to external codec.
     * |[29]    |CODECRST  |Internal Codec Hardware Reset Control
     * |        |          |0 = Reset Operation.
     * |        |          |1 = Normal Operation.
    */
    __IO uint32_t CTL;

    /**
     * CLKDIV
     * ===================================================================================================
     * Offset: 0x04  I2S Clock Divider Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |MCLKDIV   |Master Clock Divider
     * |        |          |If F_I2SCLK is (2*MCLKDIV)*256*F_LRCLK then software can program these bits to generate 256*F_LRCLK clock frequency as master clock to audio codec.
     * |        |          |But if MCLKDIV is set to 0, MCLK is the same as I2SCLK inpu.
     * |        |          |For example, if sampling rate is 24 kHz (F_LRCLK = 24 kHz) and F_I2SCLK is 12.288 MHz, MCLKDIV should be set to 1 to get 256*F_LRCLK frequency.
     * |        |          |(12.288 MHz= (2*1*256*24) kHz.
     * |        |          |F_MCLK = F_I2SCLK / (2*MCLKDIV) (When MCLKDIV >= 1).
     * |        |          |F_MCLK = F_I2SCLK (When MCLKDIV = 0).
     * |[16:8]  |BCLKDIV   |Bit Clock Divider
     * |        |          |User can program these bits to generate the frequency of BCLK, when I2S operates in master mode.
     * |        |          |In Slave mode, the frequency of BCLK is controlled by master devic.
     * |        |          |F_BCLK = F_I2SCLK / (2*(BCLKDIV+1)).
    */
    __IO uint32_t CLKDIV;

    /**
     * IEN
     * ===================================================================================================
     * Offset: 0x08  I2S Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXUDIEN   |Receive FIFO Underflow Interrupt Enable Control
     * |        |          |If software reads receive FIFO when it is empty the RXUDIF flag in I2S_STATUS register is set to 1.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[1]     |RXOVIEN   |Receive FIFO Overflow Interrupt Enable Control
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[2]     |RXTHIEN   |Receive FIFO Threshold Level Interrupt Enable Control
     * |        |          |When data word in receive FIFO is equal to or higher then RXTH[3:0] and the RXTHIF bit is set to 1.
     * |        |          |If RXTHIEN bit is enabled, interrupt will occu.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[4]     |RDMAEIEN  |RX DMA End Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and DMA current address is equal to I2S_RXEADDR register
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[5]     |RDMATIEN  |RX DMA Threshold Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and DMA current address is equal to I2S_RXTHADDR register
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[6]     |TDMAEIEN  |TX DMA End Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and DMA current address is equal to I2S_TXEADDR register
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[7]     |TDMATIEN  |TX DMA Threshold Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and DMA current address is equal to I2S_TXTHADDR register
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[8]     |TXUDIEN   |Transmit FIFO Underflow Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and transmit FIFO underflow flag is set to 1.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[9]     |TXOVIEN   |Transmit FIFO Overflow Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and transmit FIFO overflow flag is set to 1.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[10]    |TXTHIEN   |Transmit FIFO Threshold Level Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and data words in transmit FIFO is less than TXTH[3:0].
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[11]    |RZCIEN    |Right Channel Zero-Cross Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and right channel zero-cross is detected.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
     * |[12]    |LZCIEN    |Left Channel Zero-Cross Interrupt Enable Control
     * |        |          |Interrupt occurs if this bit is set to 1 and left channel zero-cross is detected.
     * |        |          |0 = Interrupt Disabled.
     * |        |          |1 = Interrupt Enabled.
    */
    __IO uint32_t IEN;

    /**
     * STATUS
     * ===================================================================================================
     * Offset: 0x0C  I2S Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |I2SIF     |I2S Interrupt Flag
     * |        |          |0 = No I2S interrupt.
     * |        |          |1 = I2S interrupt.
     * |        |          |Note1: This flag is triggered if any of TXIF and RXIF bits are enabled.
     * |        |          |Note2: This bit is read only.
     * |[1]     |RXIF      |I2S Receive Interrupt
     * |        |          |0 = No receive interrupt.
     * |        |          |1 = Receive interrupt.
     * |        |          |Note1: This flag is triggered if any of RXTHIF, RXOVIF, RXUDIF, RDMATIF, and RDMAEIF occurs.
     * |        |          |Note2: This bit is read only.
     * |[2]     |TXIF      |I2S Transmit Interrupt
     * |        |          |0 = No transmit interrupt.
     * |        |          |1 = Transmit interrupt.
     * |        |          |Note1: This flag is triggered if any of LZCIF, RZCIF, TXTHIF, TXOIF, TXUDIF, TDMATIF, and TDMAEIF occurs.
     * |        |          |Note2: This bit is read only.
     * |[3]     |RIGHT     |Right Channel
     * |        |          |Indicates that the current transmit data belongs to right channel
     * |        |          |0 = Left channel.
     * |        |          |1 = Right channel.
     * |        |          |Note: This bit is read only,
     * |[4]     |RDMAEIF   |RX DMA Equal End Address Interrupt Flag
     * |        |          |If RX DMA current address is equal to I2S_RXEADDR register, this interrupt flag will be set.
     * |        |          |If the RDMAEIEN is set, an interrupt to NVIC will occu.
     * |        |          |0 = No RX End Interrupt.
     * |        |          |1 = RX End Interrupt.
     * |        |          |Note: Write 1 to clear this bit to zero
     * |[5]     |RDMATIF   |RX DMA Equal Threshold Address Interrupt Flag
     * |        |          |If RX DMA current address is equal to I2S_RXTHADDR register, this interrupt flag will be set.
     * |        |          |If the RDMATIEN is set, an interrupt to NVIC will occu.
     * |        |          |0 = No RX Threshold Interrupt.
     * |        |          |1 = RX Threshold Interrupt.
     * |        |          |Note: Write 1 to clear this bit to zero
     * |[6]     |TDMAEIF   |TX DMA Equal End Address Interrupt Flag
     * |        |          |If TX DMA current address is equal to I2S_TXEADDR register, this interrupt flag will be set.
     * |        |          |If the TDMAEIEN is set, an interrupt to NVIC will occu.
     * |        |          |0 = No TX End Interrupt.
     * |        |          |1 = TX End Interrupt.
     * |        |          |Note: Write 1 to clear this bit to zero
     * |[7]     |TDMATIF   |TX DMA Equal Threshold Address Interrupt Flag
     * |        |          |If TX DMA current address is equal to I2S_TXTHADDR register, this interrupt flag will be set.
     * |        |          |If the TDMATIEN is set, an interrupt to NVIC will occu.
     * |        |          |0 = No TX Threshold Interrupt.
     * |        |          |1 = TX Threshold Interrupt.
     * |        |          |Note: Write 1 to clear this bit to zero
     * |[8]     |RXUDIF    |Receive FIFO Underflow Flag
     * |        |          |Read receive FIFO when it is empty. Setting this bit to 1 indicates underflow occurred.
     * |        |          |0 = No underflow occurred.
     * |        |          |1 = Underflow occurred.
     * |        |          |Note: Write 1 to clear this bit to zero
     * |[9]     |RXOVIF    |Receive FIFO Overflow Flag
     * |        |          |When receive FIFO is full and receive hardware attempt write to data into receive FIFO this bit is set to 1, and data in 1st buffer is overwritten.
     * |        |          |0 = No overflow occurred.
     * |        |          |1 = Overflow occurred.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[10]    |RXTHIF    |Receive FIFO Threshold Flag
     * |        |          |When data word(s) in receive FIFO is equal to or higher than the threshold value set in RXTH[3:0] the RXTHIF bit becomes to 1.
     * |        |          |It keeps at 1 till RXCNT[3:0] is less than RXTH[3:0] after software reads the I2S_RX registe.
     * |        |          |0 = Data word(s) in FIFO is lower than threshold level.
     * |        |          |1 = Data word(s) in FIFO is equal or higher than threshold level.
     * |        |          |Note: This bit is read only.
     * |[11]    |RXFULL    |Receive FIFO Full
     * |        |          |This bit reflect data words number in receive FIFO is 16
     * |        |          |0 = FIFO not full.
     * |        |          |1 = FIFO full.
     * |        |          |Note: This bit is read only.
     * |[12]    |RXEMPTY   |Receive FIFO Empty
     * |        |          |This bit reflects data words number in receive FIFO is zero
     * |        |          |0 = FIFO not empty.
     * |        |          |1 = FIFO empty.
     * |        |          |Note: This bit is read only.
     * |[16]    |TXUDIF    |Transmit FIFO Underflow Flag
     * |        |          |When transmit FIFO is empty and shift logic hardware read data from data FIFO causes this set to 1.
     * |        |          |0 = No underflow.
     * |        |          |1 = Underflow.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[17]    |TXOVIF    |Transmit FIFO Overflow Flag
     * |        |          |Write data to transmit FIFO when it is full and this bit set to 1
     * |        |          |0 = No overflow.
     * |        |          |1 = Overflow.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[18]    |TXTHIF    |Transmit FIFO Threshold Flag
     * |        |          |When data word(s) in transmit FIFO is equal or lower than threshold value set in TXTH[3:0] the TXTHIF bit becomes to 1.
     * |        |          |It keeps at 1 till TXCNT[3:0] is higher than TXTH[3:0] after software write I2S_TX registe.
     * |        |          |0 = Data word(s) in FIFO is higher than threshold level.
     * |        |          |1 = Data word(s) in FIFO is equal or lower than threshold level.
     * |        |          |Note: This bit is read only.
     * |[19]    |TXFULL    |Transmit FIFO Full
     * |        |          |This bit reflect data word number in transmit FIFO is 16
     * |        |          |0 = Not full.
     * |        |          |1 = Full.
     * |        |          |Note: This bit is read only.
     * |[20]    |TXEMPTY   |Transmit FIFO Empty
     * |        |          |This bit reflect data word number in transmit FIFO is zero
     * |        |          |0 = Not empty.
     * |        |          |1 = Empty.
     * |        |          |Note: This bit is read only.
     * |[21]    |TXBUSY    |Transmit Busy
     * |        |          |This bit is cleared to 0 when all data in transmit FIFO and shift buffer is shifted out.
     * |        |          |This bit is set to 1 when the first data is loaded to shift buffe.
     * |        |          |0 = Transmit shift buffer is empty.
     * |        |          |1 = Transmit shift buffer is busy.
     * |        |          |Note: This bit is read only.
     * |[22]    |RZCIF     |Right Channel Zero-Cross Flag
     * |        |          |It indicates right channel next sample data sign bit is changed or all data bits are zero.
     * |        |          |0 = No zero-cross.
     * |        |          |1 = Right channel zero-cross is detected.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[23]    |LZCIF     |Left Channel Zero-Cross Flag
     * |        |          |It indicates left channel next sample data sign bit is changed or all data bits are zero.
     * |        |          |0 = No zero-cross.
     * |        |          |1 = Left channel zero-cross is detected.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[27:24] |RXCNT     |Receive FIFO Level
     * |        |          |These bits indicate word number in receive FIFO
     * |        |          |0000 = No data or 16 words (need to check the RX full flag).
     * |        |          |0001 = 1 word in receive FIFO.
     * |        |          |....
     * |        |          |1111 = 15 words in receive FIFO.
     * |        |          |Note: IF RXFULL flag is 1, and RXCNT = 0x0. It means there are 16 words in the FIFO.
     * |[31:28] |TXCNT     |Transmit FIFO Level
     * |        |          |These bits indicate word number in transmit FIFO
     * |        |          |0000 = No data or 16 words ( need to check the TX full flag).
     * |        |          |0001 = 1 word in transmit FIFO.
     * |        |          |....
     * |        |          |1111 = 15 words in transmit FIFO.
     * |        |          |Note: IF TXFULL flag is 1, and TXCNT = 0x0. It means there are 16 words in the FIFO.
    */
    __IO uint32_t STATUS;

    /**
     * TX
     * ===================================================================================================
     * Offset: 0x10  I2S Transmit FIFO Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Transmit FIFO Register
     * |        |          |I2S contains 16 words (16x32 bit) data FIFO for data transmssion.
     * |        |          |Write data to this register to prepare data for transmission.
     * |        |          |The remaining word number is indicated by TXCNT[3:0] in I2S_STATU.
    */
    __O  uint32_t TX;

    /**
     * RX
     * ===================================================================================================
     * Offset: 0x14  I2S Receive FIFO Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Receive FIFO Register
     * |        |          |I2S contains 16 words (16x32 bit) data FIFO for data receiving.
     * |        |          |Read this register to get data in FIFO.
     * |        |          |The remaining data word number is indicated by RXCNT[3:0] in I2S_STATUS registe.
    */
    __I  uint32_t RX;

    /**
     * CODECCTL
     * ===================================================================================================
     * Offset: 0x18  I2S Virtual I2C Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |Data Information
     * |        |          |This parameter is used to read from the internal audio CODEC or write to the internal audio CODEC.
     * |[15:8]  |ADDR      |Address Information
     * |        |          |This parameter is used to read from the internal audio CODEC or write to the internal audio CODEC.
     * |[16]    |RW        |Read Or Write Command
     * |        |          |Control this command to read data from the internal audio CODEC or write data to.
     * |        |          |0 = Read from the internal audio CODEC.
     * |        |          |1 = Write to the internal audio CODEC.
     * |[23:17] |DEVID     |Internal Audio CODEC Device ID
     * |        |          |This parameter should be set to 40H.
     * |[30:24] |I2CCKDIV  |SCK Clock Divider
     * |        |          |Control the SCK Timing Parameter.
     * |        |          |The SCK frequency is (F_I2SCLK / (I2CCKDIV * 16)).
     * |        |          |Note: Cannot be zero.
     * |        |          |Note2: F_SCK must be lower than or equal to F_MCLK / 16.
     * |[31]    |BUSY      |Busy Flag
     * |        |          |If the register 'I2S_CODECCTL' has been written, the HW would change the command to the I2C format because the internal audio codec interface is I2C.
     * |        |          |However, the speed of the I2C is slow.
     * |        |          |Thus, this bit is used to indicate the end of the I2C comman.
     * |        |          |0 = I2C command is finished.
     * |        |          |1 = I2C command is not finished.
    */
    __IO uint32_t CODECCTL;
    uint32_t RESERVE0[1];


    /**
     * TXSTADDR
     * ===================================================================================================
     * Offset: 0x20  I2S TX DMA Start Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |TX DMA Start Address Register
     * |        |          |Note1: The address is word boundary.
     * |        |          |Note2: The address can't be set smaller than 0x2000_0000.
    */
    __IO uint32_t TXSTADDR;

    /**
     * TXTHADDR
     * ===================================================================================================
     * Offset: 0x24  I2S TX DMA Threshold Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |TX DMA Threshold Address Register
     * |        |          |Note1: The address is word boundary.
     * |        |          |Note2: The address can't be set smaller than 0x2000_0000.
    */
    __IO uint32_t TXTHADDR;

    /**
     * TXEADDR
     * ===================================================================================================
     * Offset: 0x28  I2S TX DMA End Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |TX DMA End Address Register
     * |        |          |Note1: The address is word boundary.
     * |        |          |Note2: If WDWIDTH[1:0] is equal to 0x2 or 0x3, user must set the correct end address to avoid the swap between right channel and left channel in stereo mode.
     * |        |          |Note2: The address can't be set smaller than 0x2000_0000.
    */
    __IO uint32_t TXEADDR;

    /**
     * TXCADDR
     * ===================================================================================================
     * Offset: 0x2C  I2S TX DMA Current Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |TX DMA Current Address Register
    */
    __I  uint32_t TXCADDR;

    /**
     * RXSTADDR
     * ===================================================================================================
     * Offset: 0x30  I2S RX DMA Start Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |RX DMA Start Address Register
     * |        |          |Note1: The address is word boundary.
     * |        |          |Note2: The address can't be set smaller than 0x2000_0000.
    */
    __IO uint32_t RXSTADDR;

    /**
     * RXTHADDR
     * ===================================================================================================
     * Offset: 0x34  I2S RX DMA Threshold Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |RX DMA Threshold Address Register
     * |        |          |Note1: The address is word boundary.
     * |        |          |Note2: The address can't be set smaller than 0x2000_0000.
    */
    __IO uint32_t RXTHADDR;

    /**
     * RXEADDR
     * ===================================================================================================
     * Offset: 0x38  I2S RX DMA End Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |RX DMA End Address Register
     * |        |          |Note1: The address is word boundary.
     * |        |          |Note2: If WDWIDTH[1:0] is equal to 0x2 or 0x3, user must set the correct end address to avoid the swap between right channel and left channel in stereo mode.
     * |        |          |Note3: The address can't be set smaller than 0x2000_0000.
    */
    __IO uint32_t RXEADDR;

    /**
     * RXCADDR
     * ===================================================================================================
     * Offset: 0x3C  I2S RX DMA Current Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |RX DMA Current Address Register
    */
    __I  uint32_t RXCADDR;

    /**
     * RXAVGCTL
     * ===================================================================================================
     * Offset: 0x40  I2S RX Data Average Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |WINSEL    |RX Data Average Window Select
     * |        |          |0000 = Average window is 1 (2^0) sample.
     * |        |          |0001 = Average window is 2 (2^1) samples.
     * |        |          |0010 = Average window is 4 (2^2) samples.
     * |        |          |...
     * |        |          |1110 = Average window is 16384 (2^14) samples.
     * |        |          |1111 = Average window is 32768 (2^15) samples.
     * |        |          |Note: Every window size samples will generate one average result.
    */
    __IO uint32_t RXAVGCTL;

    /**
     * RXLCHAVG
     * ===================================================================================================
     * Offset: 0x44  I2S RX Left Channel Data Average
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RESULT    |RX Left Channel Data Average Result
     * |        |          |The average result of left channel received data.
     * |        |          |Note: If MONO (I2S_CTL[6]) = 1, the average result is only in this register whatever RXLCH (I2S_CTL[23]) = 1 or 0
    */
    __I  uint32_t RXLCHAVG;

    /**
     * RXRCHAVG
     * ===================================================================================================
     * Offset: 0x48  I2S RX Right Channel Data Average
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RESULT    |RX Right Channel Data Average Result
     * |        |          |The average result of left channel received data.
     * |        |          |Note: If MONO (I2S_CTL[6]), this register will be useless.
    */
    __I  uint32_t RXRCHAVG;

} I2S_T;

/**
    @addtogroup I2S_CONST I2S Bit Field Definition
    Constant Definitions for I2S Controller
@{ */

#define I2S_CTL_I2SEN_Pos                (0)                                               /*!< I2S_T::CTL: I2SEN Position                */
#define I2S_CTL_I2SEN_Msk                (0x1ul << I2S_CTL_I2SEN_Pos)                      /*!< I2S_T::CTL: I2SEN Mask                    */

#define I2S_CTL_TXEN_Pos                 (1)                                               /*!< I2S_T::CTL: TXEN Position                 */
#define I2S_CTL_TXEN_Msk                 (0x1ul << I2S_CTL_TXEN_Pos)                       /*!< I2S_T::CTL: TXEN Mask                     */

#define I2S_CTL_RXEN_Pos                 (2)                                               /*!< I2S_T::CTL: RXEN Position                 */
#define I2S_CTL_RXEN_Msk                 (0x1ul << I2S_CTL_RXEN_Pos)                       /*!< I2S_T::CTL: RXEN Mask                     */

#define I2S_CTL_MUTE_Pos                 (3)                                               /*!< I2S_T::CTL: MUTE Position                 */
#define I2S_CTL_MUTE_Msk                 (0x1ul << I2S_CTL_MUTE_Pos)                       /*!< I2S_T::CTL: MUTE Mask                     */

#define I2S_CTL_WDWIDTH_Pos              (4)                                               /*!< I2S_T::CTL: WDWIDTH Position              */
#define I2S_CTL_WDWIDTH_Msk              (0x3ul << I2S_CTL_WDWIDTH_Pos)                    /*!< I2S_T::CTL: WDWIDTH Mask                  */

#define I2S_CTL_MONO_Pos                 (6)                                               /*!< I2S_T::CTL: MONO Position                 */
#define I2S_CTL_MONO_Msk                 (0x1ul << I2S_CTL_MONO_Pos)                       /*!< I2S_T::CTL: MONO Mask                     */

#define I2S_CTL_FORMAT_Pos               (7)                                               /*!< I2S_T::CTL: FORMAT Position               */
#define I2S_CTL_FORMAT_Msk               (0x1ul << I2S_CTL_FORMAT_Pos)                     /*!< I2S_T::CTL: FORMAT Mask                   */

#define I2S_CTL_TXTH_Pos                 (8)                                               /*!< I2S_T::CTL: TXTH Position                 */
#define I2S_CTL_TXTH_Msk                 (0xful << I2S_CTL_TXTH_Pos)                       /*!< I2S_T::CTL: TXTH Mask                     */

#define I2S_CTL_RXTH_Pos                 (12)                                              /*!< I2S_T::CTL: RXTH Position                 */
#define I2S_CTL_RXTH_Msk                 (0xful << I2S_CTL_RXTH_Pos)                       /*!< I2S_T::CTL: RXTH Mask                     */

#define I2S_CTL_RZCEN_Pos                (16)                                              /*!< I2S_T::CTL: RZCEN Position                */
#define I2S_CTL_RZCEN_Msk                (0x1ul << I2S_CTL_RZCEN_Pos)                      /*!< I2S_T::CTL: RZCEN Mask                    */

#define I2S_CTL_LZCEN_Pos                (17)                                              /*!< I2S_T::CTL: LZCEN Position                */
#define I2S_CTL_LZCEN_Msk                (0x1ul << I2S_CTL_LZCEN_Pos)                      /*!< I2S_T::CTL: LZCEN Mask                    */

#define I2S_CTL_TXCLR_Pos                (18)                                              /*!< I2S_T::CTL: TXCLR Position                */
#define I2S_CTL_TXCLR_Msk                (0x1ul << I2S_CTL_TXCLR_Pos)                      /*!< I2S_T::CTL: TXCLR Mask                    */

#define I2S_CTL_RXCLR_Pos                (19)                                              /*!< I2S_T::CTL: RXCLR Position                */
#define I2S_CTL_RXCLR_Msk                (0x1ul << I2S_CTL_RXCLR_Pos)                      /*!< I2S_T::CTL: RXCLR Mask                    */

#define I2S_CTL_TXDMAEN_Pos              (20)                                              /*!< I2S_T::CTL: TXDMAEN Position              */
#define I2S_CTL_TXDMAEN_Msk              (0x1ul << I2S_CTL_TXDMAEN_Pos)                    /*!< I2S_T::CTL: TXDMAEN Mask                  */

#define I2S_CTL_RXDMAEN_Pos              (21)                                              /*!< I2S_T::CTL: RXDMAEN Position              */
#define I2S_CTL_RXDMAEN_Msk              (0x1ul << I2S_CTL_RXDMAEN_Pos)                    /*!< I2S_T::CTL: RXDMAEN Mask                  */

#define I2S_CTL_RXLCH_Pos                (23)                                              /*!< I2S_T::CTL: RXLCH Position                */
#define I2S_CTL_RXLCH_Msk                (0x1ul << I2S_CTL_RXLCH_Pos)                      /*!< I2S_T::CTL: RXLCH Mask                    */

#define I2S_CTL_PCMEN_Pos                (24)                                              /*!< I2S_T::CTL: PCMEN Position                */
#define I2S_CTL_PCMEN_Msk                (0x1ul << I2S_CTL_PCMEN_Pos)                      /*!< I2S_T::CTL: PCMEN Mask                    */

#define I2S_CTL_MCLKEN_Pos               (25)                                              /*!< I2S_T::CTL: MCLKEN Position               */
#define I2S_CTL_MCLKEN_Msk               (0x1ul << I2S_CTL_MCLKEN_Pos)                     /*!< I2S_T::CTL: MCLKEN Mask                   */

#define I2S_CTL_SLAVE_Pos                (26)                                              /*!< I2S_T::CTL: SLAVE Position                */
#define I2S_CTL_SLAVE_Msk                (0x1ul << I2S_CTL_SLAVE_Pos)                      /*!< I2S_T::CTL: SLAVE Mask                    */

#define I2S_CTL_CODECSEL_Pos             (28)                                              /*!< I2S_T::CTL: CODECSEL Position             */
#define I2S_CTL_CODECSEL_Msk             (0x1ul << I2S_CTL_CODECSEL_Pos)                   /*!< I2S_T::CTL: CODECSEL Mask                 */

#define I2S_CTL_CODECRST_Pos             (29)                                              /*!< I2S_T::CTL: CODECRST Position             */
#define I2S_CTL_CODECRST_Msk             (0x1ul << I2S_CTL_CODECRST_Pos)                   /*!< I2S_T::CTL: CODECRST Mask                 */

#define I2S_CLKDIV_MCLKDIV_Pos           (0)                                               /*!< I2S_T::CLKDIV: MCLKDIV Position           */
#define I2S_CLKDIV_MCLKDIV_Msk           (0x3ful << I2S_CLKDIV_MCLKDIV_Pos)                /*!< I2S_T::CLKDIV: MCLKDIV Mask               */

#define I2S_CLKDIV_BCLKDIV_Pos           (8)                                               /*!< I2S_T::CLKDIV: BCLKDIV Position           */
#define I2S_CLKDIV_BCLKDIV_Msk           (0x1fful << I2S_CLKDIV_BCLKDIV_Pos)               /*!< I2S_T::CLKDIV: BCLKDIV Mask               */

#define I2S_IEN_RXUDIEN_Pos              (0)                                               /*!< I2S_T::IEN: RXUDIEN Position              */
#define I2S_IEN_RXUDIEN_Msk              (0x1ul << I2S_IEN_RXUDIEN_Pos)                    /*!< I2S_T::IEN: RXUDIEN Mask                  */

#define I2S_IEN_RXOVIEN_Pos              (1)                                               /*!< I2S_T::IEN: RXOVIEN Position              */
#define I2S_IEN_RXOVIEN_Msk              (0x1ul << I2S_IEN_RXOVIEN_Pos)                    /*!< I2S_T::IEN: RXOVIEN Mask                  */

#define I2S_IEN_RXTHIEN_Pos              (2)                                               /*!< I2S_T::IEN: RXTHIEN Position              */
#define I2S_IEN_RXTHIEN_Msk              (0x1ul << I2S_IEN_RXTHIEN_Pos)                    /*!< I2S_T::IEN: RXTHIEN Mask                  */

#define I2S_IEN_RDMAEIEN_Pos             (4)                                               /*!< I2S_T::IEN: RDMAEIEN Position             */
#define I2S_IEN_RDMAEIEN_Msk             (0x1ul << I2S_IEN_RDMAEIEN_Pos)                   /*!< I2S_T::IEN: RDMAEIEN Mask                 */

#define I2S_IEN_RDMATIEN_Pos             (5)                                               /*!< I2S_T::IEN: RDMATIEN Position             */
#define I2S_IEN_RDMATIEN_Msk             (0x1ul << I2S_IEN_RDMATIEN_Pos)                   /*!< I2S_T::IEN: RDMATIEN Mask                 */

#define I2S_IEN_TDMAEIEN_Pos             (6)                                               /*!< I2S_T::IEN: TDMAEIEN Position             */
#define I2S_IEN_TDMAEIEN_Msk             (0x1ul << I2S_IEN_TDMAEIEN_Pos)                   /*!< I2S_T::IEN: TDMAEIEN Mask                 */

#define I2S_IEN_TDMATIEN_Pos             (7)                                               /*!< I2S_T::IEN: TDMATIEN Position             */
#define I2S_IEN_TDMATIEN_Msk             (0x1ul << I2S_IEN_TDMATIEN_Pos)                   /*!< I2S_T::IEN: TDMATIEN Mask                 */

#define I2S_IEN_TXUDIEN_Pos              (8)                                               /*!< I2S_T::IEN: TXUDIEN Position              */
#define I2S_IEN_TXUDIEN_Msk              (0x1ul << I2S_IEN_TXUDIEN_Pos)                    /*!< I2S_T::IEN: TXUDIEN Mask                  */

#define I2S_IEN_TXOVIEN_Pos              (9)                                               /*!< I2S_T::IEN: TXOVIEN Position              */
#define I2S_IEN_TXOVIEN_Msk              (0x1ul << I2S_IEN_TXOVIEN_Pos)                    /*!< I2S_T::IEN: TXOVIEN Mask                  */

#define I2S_IEN_TXTHIEN_Pos              (10)                                              /*!< I2S_T::IEN: TXTHIEN Position              */
#define I2S_IEN_TXTHIEN_Msk              (0x1ul << I2S_IEN_TXTHIEN_Pos)                    /*!< I2S_T::IEN: TXTHIEN Mask                  */

#define I2S_IEN_RZCIEN_Pos               (11)                                              /*!< I2S_T::IEN: RZCIEN Position               */
#define I2S_IEN_RZCIEN_Msk               (0x1ul << I2S_IEN_RZCIEN_Pos)                     /*!< I2S_T::IEN: RZCIEN Mask                   */

#define I2S_IEN_LZCIEN_Pos               (12)                                              /*!< I2S_T::IEN: LZCIEN Position               */
#define I2S_IEN_LZCIEN_Msk               (0x1ul << I2S_IEN_LZCIEN_Pos)                     /*!< I2S_T::IEN: LZCIEN Mask                   */

#define I2S_STATUS_I2SIF_Pos             (0)                                               /*!< I2S_T::STATUS: I2SIF Position             */
#define I2S_STATUS_I2SIF_Msk             (0x1ul << I2S_STATUS_I2SIF_Pos)                   /*!< I2S_T::STATUS: I2SIF Mask                 */

#define I2S_STATUS_RXIF_Pos              (1)                                               /*!< I2S_T::STATUS: RXIF Position              */
#define I2S_STATUS_RXIF_Msk              (0x1ul << I2S_STATUS_RXIF_Pos)                    /*!< I2S_T::STATUS: RXIF Mask                  */

#define I2S_STATUS_TXIF_Pos              (2)                                               /*!< I2S_T::STATUS: TXIF Position              */
#define I2S_STATUS_TXIF_Msk              (0x1ul << I2S_STATUS_TXIF_Pos)                    /*!< I2S_T::STATUS: TXIF Mask                  */

#define I2S_STATUS_RIGHT_Pos             (3)                                               /*!< I2S_T::STATUS: RIGHT Position             */
#define I2S_STATUS_RIGHT_Msk             (0x1ul << I2S_STATUS_RIGHT_Pos)                   /*!< I2S_T::STATUS: RIGHT Mask                 */

#define I2S_STATUS_RDMAEIF_Pos           (4)                                               /*!< I2S_T::STATUS: RDMAEIF Position           */
#define I2S_STATUS_RDMAEIF_Msk           (0x1ul << I2S_STATUS_RDMAEIF_Pos)                 /*!< I2S_T::STATUS: RDMAEIF Mask               */

#define I2S_STATUS_RDMATIF_Pos           (5)                                               /*!< I2S_T::STATUS: RDMATIF Position           */
#define I2S_STATUS_RDMATIF_Msk           (0x1ul << I2S_STATUS_RDMATIF_Pos)                 /*!< I2S_T::STATUS: RDMATIF Mask               */

#define I2S_STATUS_TDMAEIF_Pos           (6)                                               /*!< I2S_T::STATUS: TDMAEIF Position           */
#define I2S_STATUS_TDMAEIF_Msk           (0x1ul << I2S_STATUS_TDMAEIF_Pos)                 /*!< I2S_T::STATUS: TDMAEIF Mask               */

#define I2S_STATUS_TDMATIF_Pos           (7)                                               /*!< I2S_T::STATUS: TDMATIF Position           */
#define I2S_STATUS_TDMATIF_Msk           (0x1ul << I2S_STATUS_TDMATIF_Pos)                 /*!< I2S_T::STATUS: TDMATIF Mask               */

#define I2S_STATUS_RXUDIF_Pos            (8)                                               /*!< I2S_T::STATUS: RXUDIF Position            */
#define I2S_STATUS_RXUDIF_Msk            (0x1ul << I2S_STATUS_RXUDIF_Pos)                  /*!< I2S_T::STATUS: RXUDIF Mask                */

#define I2S_STATUS_RXOVIF_Pos            (9)                                               /*!< I2S_T::STATUS: RXOVIF Position            */
#define I2S_STATUS_RXOVIF_Msk            (0x1ul << I2S_STATUS_RXOVIF_Pos)                  /*!< I2S_T::STATUS: RXOVIF Mask                */

#define I2S_STATUS_RXTHIF_Pos            (10)                                              /*!< I2S_T::STATUS: RXTHIF Position            */
#define I2S_STATUS_RXTHIF_Msk            (0x1ul << I2S_STATUS_RXTHIF_Pos)                  /*!< I2S_T::STATUS: RXTHIF Mask                */

#define I2S_STATUS_RXFULL_Pos            (11)                                              /*!< I2S_T::STATUS: RXFULL Position            */
#define I2S_STATUS_RXFULL_Msk            (0x1ul << I2S_STATUS_RXFULL_Pos)                  /*!< I2S_T::STATUS: RXFULL Mask                */

#define I2S_STATUS_RXEMPTY_Pos           (12)                                              /*!< I2S_T::STATUS: RXEMPTY Position           */
#define I2S_STATUS_RXEMPTY_Msk           (0x1ul << I2S_STATUS_RXEMPTY_Pos)                 /*!< I2S_T::STATUS: RXEMPTY Mask               */

#define I2S_STATUS_TXUDIF_Pos            (16)                                              /*!< I2S_T::STATUS: TXUDIF Position            */
#define I2S_STATUS_TXUDIF_Msk            (0x1ul << I2S_STATUS_TXUDIF_Pos)                  /*!< I2S_T::STATUS: TXUDIF Mask                */

#define I2S_STATUS_TXOVIF_Pos            (17)                                              /*!< I2S_T::STATUS: TXOVIF Position            */
#define I2S_STATUS_TXOVIF_Msk            (0x1ul << I2S_STATUS_TXOVIF_Pos)                  /*!< I2S_T::STATUS: TXOVIF Mask                */

#define I2S_STATUS_TXTHIF_Pos            (18)                                              /*!< I2S_T::STATUS: TXTHIF Position            */
#define I2S_STATUS_TXTHIF_Msk            (0x1ul << I2S_STATUS_TXTHIF_Pos)                  /*!< I2S_T::STATUS: TXTHIF Mask                */

#define I2S_STATUS_TXFULL_Pos            (19)                                              /*!< I2S_T::STATUS: TXFULL Position            */
#define I2S_STATUS_TXFULL_Msk            (0x1ul << I2S_STATUS_TXFULL_Pos)                  /*!< I2S_T::STATUS: TXFULL Mask                */

#define I2S_STATUS_TXEMPTY_Pos           (20)                                              /*!< I2S_T::STATUS: TXEMPTY Position           */
#define I2S_STATUS_TXEMPTY_Msk           (0x1ul << I2S_STATUS_TXEMPTY_Pos)                 /*!< I2S_T::STATUS: TXEMPTY Mask               */

#define I2S_STATUS_TXBUSY_Pos            (21)                                              /*!< I2S_T::STATUS: TXBUSY Position            */
#define I2S_STATUS_TXBUSY_Msk            (0x1ul << I2S_STATUS_TXBUSY_Pos)                  /*!< I2S_T::STATUS: TXBUSY Mask                */

#define I2S_STATUS_RZCIF_Pos             (22)                                              /*!< I2S_T::STATUS: RZCIF Position             */
#define I2S_STATUS_RZCIF_Msk             (0x1ul << I2S_STATUS_RZCIF_Pos)                   /*!< I2S_T::STATUS: RZCIF Mask                 */

#define I2S_STATUS_LZCIF_Pos             (23)                                              /*!< I2S_T::STATUS: LZCIF Position             */
#define I2S_STATUS_LZCIF_Msk             (0x1ul << I2S_STATUS_LZCIF_Pos)                   /*!< I2S_T::STATUS: LZCIF Mask                 */

#define I2S_STATUS_RXCNT_Pos             (24)                                              /*!< I2S_T::STATUS: RXCNT Position             */
#define I2S_STATUS_RXCNT_Msk             (0xful << I2S_STATUS_RXCNT_Pos)                   /*!< I2S_T::STATUS: RXCNT Mask                 */

#define I2S_STATUS_TXCNT_Pos             (28)                                              /*!< I2S_T::STATUS: TXCNT Position             */
#define I2S_STATUS_TXCNT_Msk             (0xful << I2S_STATUS_TXCNT_Pos)                   /*!< I2S_T::STATUS: TXCNT Mask                 */

#define I2S_TX_TX_Pos                    (0)                                               /*!< I2S_T::TX: TX Position                    */
#define I2S_TX_TX_Msk                    (0xfffffffful << I2S_TX_TX_Pos)                   /*!< I2S_T::TX: TX Mask                        */

#define I2S_RX_RX_Pos                    (0)                                               /*!< I2S_T::RX: RX Position                    */
#define I2S_RX_RX_Msk                    (0xfffffffful << I2S_RX_RX_Pos)                   /*!< I2S_T::RX: RX Mask                        */

#define I2S_CODECCTL_DAT_Pos             (0)                                               /*!< I2S_T::CODECCTL: DAT Position             */
#define I2S_CODECCTL_DAT_Msk             (0xfful << I2S_CODECCTL_DAT_Pos)                  /*!< I2S_T::CODECCTL: DAT Mask                 */

#define I2S_CODECCTL_ADDR_Pos            (8)                                               /*!< I2S_T::CODECCTL: ADDR Position            */
#define I2S_CODECCTL_ADDR_Msk            (0xfful << I2S_CODECCTL_ADDR_Pos)                 /*!< I2S_T::CODECCTL: ADDR Mask                */

#define I2S_CODECCTL_RW_Pos              (16)                                              /*!< I2S_T::CODECCTL: RW Position              */
#define I2S_CODECCTL_RW_Msk              (0x1ul << I2S_CODECCTL_RW_Pos)                    /*!< I2S_T::CODECCTL: RW Mask                  */

#define I2S_CODECCTL_DEVID_Pos           (17)                                              /*!< I2S_T::CODECCTL: DEVID Position           */
#define I2S_CODECCTL_DEVID_Msk           (0x7ful << I2S_CODECCTL_DEVID_Pos)                /*!< I2S_T::CODECCTL: DEVID Mask               */

#define I2S_CODECCTL_I2CCKDIV_Pos        (24)                                              /*!< I2S_T::CODECCTL: I2CCKDIV Position        */
#define I2S_CODECCTL_I2CCKDIV_Msk        (0x7ful << I2S_CODECCTL_I2CCKDIV_Pos)             /*!< I2S_T::CODECCTL: I2CCKDIV Mask            */

#define I2S_CODECCTL_BUSY_Pos            (31)                                              /*!< I2S_T::CODECCTL: BUSY Position            */
#define I2S_CODECCTL_BUSY_Msk            (0x1ul << I2S_CODECCTL_BUSY_Pos)                  /*!< I2S_T::CODECCTL: BUSY Mask                */

#define I2S_TXSTADDR_ADDR_Pos            (0)                                               /*!< I2S_T::TXSTADDR: ADDR Position            */
#define I2S_TXSTADDR_ADDR_Msk            (0xfffffffful << I2S_TXSTADDR_ADDR_Pos)           /*!< I2S_T::TXSTADDR: ADDR Mask                */

#define I2S_TXTHADDR_ADDR_Pos            (0)                                               /*!< I2S_T::TXTHADDR: ADDR Position            */
#define I2S_TXTHADDR_ADDR_Msk            (0xfffffffful << I2S_TXTHADDR_ADDR_Pos)           /*!< I2S_T::TXTHADDR: ADDR Mask                */

#define I2S_TXEADDR_ADDR_Pos             (0)                                               /*!< I2S_T::TXEADDR: ADDR Position             */
#define I2S_TXEADDR_ADDR_Msk             (0xfffffffful << I2S_TXEADDR_ADDR_Pos)            /*!< I2S_T::TXEADDR: ADDR Mask                 */

#define I2S_TXCADDR_ADDR_Pos             (0)                                               /*!< I2S_T::TXCADDR: ADDR Position             */
#define I2S_TXCADDR_ADDR_Msk             (0xfffffffful << I2S_TXCADDR_ADDR_Pos)            /*!< I2S_T::TXCADDR: ADDR Mask                 */

#define I2S_RXSTADDR_ADDR_Pos            (0)                                               /*!< I2S_T::RXSTADDR: ADDR Position            */
#define I2S_RXSTADDR_ADDR_Msk            (0xfffffffful << I2S_RXSTADDR_ADDR_Pos)           /*!< I2S_T::RXSTADDR: ADDR Mask                */

#define I2S_RXTHADDR_ADDR_Pos            (0)                                               /*!< I2S_T::RXTHADDR: ADDR Position            */
#define I2S_RXTHADDR_ADDR_Msk            (0xfffffffful << I2S_RXTHADDR_ADDR_Pos)           /*!< I2S_T::RXTHADDR: ADDR Mask                */

#define I2S_RXEADDR_ADDR_Pos             (0)                                               /*!< I2S_T::RXEADDR: ADDR Position             */
#define I2S_RXEADDR_ADDR_Msk             (0xfffffffful << I2S_RXEADDR_ADDR_Pos)            /*!< I2S_T::RXEADDR: ADDR Mask                 */

#define I2S_RXCADDR_ADDR_Pos             (0)                                               /*!< I2S_T::RXCADDR: ADDR Position             */
#define I2S_RXCADDR_ADDR_Msk             (0xfffffffful << I2S_RXCADDR_ADDR_Pos)            /*!< I2S_T::RXCADDR: ADDR Mask                 */

#define I2S_RXAVGCTL_WINSEL_Pos          (0)                                               /*!< I2S_T::RXAVGCTL: WINSEL Position          */
#define I2S_RXAVGCTL_WINSEL_Msk          (0xful << I2S_RXAVGCTL_WINSEL_Pos)                /*!< I2S_T::RXAVGCTL: WINSEL Mask              */

#define I2S_RXLCHAVG_RESULT_Pos          (0)                                               /*!< I2S_T::RXLCHAVG: RESULT Position          */
#define I2S_RXLCHAVG_RESULT_Msk          (0xfffffffful << I2S_RXLCHAVG_RESULT_Pos)         /*!< I2S_T::RXLCHAVG: RESULT Mask              */

#define I2S_RXRCHAVG_RESULT_Pos          (0)                                               /*!< I2S_T::RXRCHAVG: RESULT Position          */
#define I2S_RXRCHAVG_RESULT_Msk          (0xfffffffful << I2S_RXRCHAVG_RESULT_Pos)         /*!< I2S_T::RXRCHAVG: RESULT Mask              */

/**@}*/ /* I2S_CONST */
/**@}*/ /* end of I2S register group */


/*---------------------- Pulse Width Modulation Controller -------------------------*/
/**
    @addtogroup PWM Pulse Width Modulation Controller(PWM)
    Memory Mapped Structure for PWM Controller
@{ */

typedef struct
{


    /**
     * CLKPSC
     * ===================================================================================================
     * Offset: 0x00  PWM Pre-scale Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |CLKPSC01  |Clock Pre-Scale 0 For PWM Counter 0 & 1
     * |        |          |Clock input is divided by (CLKPSC01+1) before it is fed to the counter 0 & 1.
     * |        |          |If CLKPSC01=0, then the pre-scale 0 output clock will be stopped.
     * |[15:8]  |CLKPSC23  |Clock Pre-Scale 1 For PWM Counter 2 & 3
     * |        |          |Clock input is divided by (CLKPSC23+1) before it is fed to the counter 2 & 3.
     * |        |          |If CLKPSC23=0, then the pre-scale 1 output clock will be stopped.
     * |[23:16] |DZCNT01   |Dead-Time Interval Register 0
     * |        |          |These 8-bit determine Dead-time length.
     * |        |          |Dead-time = (DTCNT01[7:0]+1) * PWM_CLK period.
     * |[31:24] |DZCNT23   |Dead-Time Interval Register 1
     * |        |          |These 8-bit determine Dead-time length.
     * |        |          |Dead-time = (DTCNT23[7:0]+1) * PWM_CLK period.
    */
    __IO uint32_t CLKPSC;

    /**
     * CLKDIV
     * ===================================================================================================
     * Offset: 0x04  PWM Clock Select Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |CLKDIV0   |PWM Counter 0 Clock Source Selection
     * |        |          |Select clock input for PWM Counter 0.
     * |        |          |(Table is the same as CLKDIV3)
     * |[6:4]   |CLKDIV1   |PWM Counter 1 Clock Source Selection
     * |        |          |Select clock input for PWM Counter 1.
     * |        |          |(Table is the same as CLKDIV3)
     * |[10:8]  |CLKDIV2   |PWM Counter 2 Clock Source Selection
     * |        |          |Select clock input for PWM Counter 2.
     * |        |          |(Table is the same as CLKDIV3)
     * |[14:12] |CLKDIV3   |PWM Counter 3 Clock Source Selection
     * |        |          |Select clock input for timer 3.
     * |        |          |000 = PWM_CLK/2.
     * |        |          |001 = PWM_CLK/4.
     * |        |          |010 = PWM_CLK/8.
     * |        |          |011 = PWM_CLK/16.
     * |        |          |100 = PWM_CLK/1.
    */
    __IO uint32_t CLKDIV;

    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x08  PWM Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTEN0    |PWM Counter 0 Enable Control
     * |        |          |0 = PWM Counter and clock prescaler stops running.
     * |        |          |1 = PWM Counter and clock prescaler starts running.
     * |[2]     |PINV0     |PWM Counter 0 Inverter ON/OFF
     * |        |          |0 = Inverter OFF.
     * |        |          |1 = Inverter ON.
     * |[3]     |CNTMODE0  |PWM Counter 0 Auto-reload Mode/One-shot Mode
     * |        |          |0 = One-shot mode.
     * |        |          |1 = Auto-reload mode.
     * |        |          |If there is a rising transition at this bit, it will cause PWM_PERIOD0 and PWM_CMPDAT0 be cleared.
     * |[4]     |DTEN01    |Dead-time 0 Generator Enable Control
     * |        |          |0 = Dead-time generator stops running.
     * |        |          |1 = Dead-time generator starts running.
     * |[5]     |DTEN23    |Dead-time 1 Generator Enable Control
     * |        |          |0 = Dead-time generator stops running.
     * |        |          |1 = Dead-time generator starts running.
     * |[8]     |CNTEN1    |PWM Counter 1 Enable Control
     * |        |          |0 = PWM Counter and clock prescaler stops running.
     * |        |          |1 = PWM Counter and clock prescaler starts running.
     * |[10]    |PINV1     |PWM Counter 1 Inverter ON/OFF
     * |        |          |0 = Inverter OFF.
     * |        |          |1 = Inverter ON.
     * |[11]    |CNTMODE1  |PWM Counter 1 Auto-reload Mode/One-shot Mode
     * |        |          |0 = One-shot mode.
     * |        |          |1 = Auto-reload mode.
     * |        |          |Note: If there is a rising transition at this bit, it will cause PWM_PERIOD1 and PWM_CMPDAT1 be cleared.
     * |[16]    |CNTEN2    |PWM Counter 2 Enable Control
     * |        |          |0 = PWM Counter and clock prescaler stops running.
     * |        |          |1 = PWM Counter and clock prescaler starts running.
     * |[18]    |PINV2     |PWM Counter 2 Inverter ON/OFF
     * |        |          |0 = Inverter OFF.
     * |        |          |1 = Inverter ON.
     * |[19]    |CNTMODE2  |PWM Counter 2 Auto-reload Mode/One-shot Mode
     * |        |          |0 = One-shot mode.
     * |        |          |1 = Auto-reload mode.
     * |        |          |Note: If there is a rising transition at this bit, it will cause PWM_PERIOD2 and PWM_CMPDAT2 be cleared.
     * |[24]    |CNTEN3    |PWM Counter 3 Enable Control
     * |        |          |0 = PWM Counter and clock prescaler stops running.
     * |        |          |1 = PWM Counter and clock prescaler starts running.
     * |[26]    |PINV3     |PWM Counter 3 Inverter ON/OFF
     * |        |          |0 = Inverter OFF.
     * |        |          |1 = Inverter ON.
     * |[27]    |CNTMODE3  |PWM Counter 3 Auto-reload Mode/One-shot Mode
     * |        |          |0 = One-Shot mode.
     * |        |          |1 = Auto-Reload mode.
     * |        |          |Note: If there is a rising transition at this bit, it will cause PWM_PERIOD3 and PWM_CMPDAT3 be cleared.
    */
    __IO uint32_t CTL;

    /**
     * PERIOD0
     * ===================================================================================================
     * Offset: 0x0C  PWM Period Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PERIOD    |PWM Period Register
     * |        |          |PERIOD determines the PWM period.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to PERIOD will take effect in the next PWM cycle.
    */
    __IO uint32_t PERIOD0;

    /**
     * CMPDAT0
     * ===================================================================================================
     * Offset: 0x10  PWM Comparator Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMP       |PWM Compare Register
     * |        |          |CMP determines the PWM output duty ratio.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to CMP will take effect in the next PWM cycle.
    */
    __IO uint32_t CMPDAT0;

    /**
     * CNT0
     * ===================================================================================================
     * Offset: 0x14  PWM Data Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CNT       |PWM Data Register
     * |        |          |User can monitor CNT to know current value in 16-bit down counter.
    */
    __I  uint32_t CNT0;

    /**
     * PERIOD1
     * ===================================================================================================
     * Offset: 0x18  PWM Period Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PERIOD    |PWM Period Register
     * |        |          |PERIOD determines the PWM period.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to PERIOD will take effect in the next PWM cycle.
    */
    __IO uint32_t PERIOD1;

    /**
     * CMPDAT1
     * ===================================================================================================
     * Offset: 0x1C  PWM Comparator Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMP       |PWM Compare Register
     * |        |          |CMP determines the PWM output duty ratio.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to CMP will take effect in the next PWM cycle.
    */
    __IO uint32_t CMPDAT1;

    /**
     * CNT1
     * ===================================================================================================
     * Offset: 0x20  PWM Data Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CNT       |PWM Data Register
     * |        |          |User can monitor CNT to know current value in 16-bit down counter.
    */
    __I  uint32_t CNT1;

    /**
     * PERIOD2
     * ===================================================================================================
     * Offset: 0x24  PWM Period Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PERIOD    |PWM Period Register
     * |        |          |PERIOD determines the PWM period.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to PERIOD will take effect in the next PWM cycle.
    */
    __IO uint32_t PERIOD2;

    /**
     * CMPDAT2
     * ===================================================================================================
     * Offset: 0x28  PWM Comparator Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMP       |PWM Compare Register
     * |        |          |CMP determines the PWM output duty ratio.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to CMP will take effect in the next PWM cycle.
    */
    __IO uint32_t CMPDAT2;

    /**
     * CNT2
     * ===================================================================================================
     * Offset: 0x2C  PWM Data Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CNT       |PWM Data Register
     * |        |          |User can monitor CNT to know current value in 16-bit down counter.
    */
    __I  uint32_t CNT2;

    /**
     * PERIOD3
     * ===================================================================================================
     * Offset: 0x30  PWM Period Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |PERIOD    |PWM Period Register
     * |        |          |PERIOD determines the PWM period.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to PERIOD will take effect in the next PWM cycle.
    */
    __IO uint32_t PERIOD3;

    /**
     * CMPDAT3
     * ===================================================================================================
     * Offset: 0x34  PWM Comparator Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CMP       |PWM Compare Register
     * |        |          |CMP determines the PWM output duty ratio.
     * |        |          |PWM frequency = PWM_CLK/[(prescale+1)*(clock divider)*(PERIOD+1)].
     * |        |          |l Duty ratio = (CMP+1)/(PERIOD+1).
     * |        |          |l CMP >= PERIOD: PWM output is always high.
     * |        |          |l CMP < PERIOD: PWM low width = (PERIOD-CMP) unit; PWM high width = (CMP+1) unit.
     * |        |          |l CMP = 0: PWM low width = (PERIOD) unit; PWM high width = 1 unit.
     * |        |          |Note: Any write to CMP will take effect in the next PWM cycle.
    */
    __IO uint32_t CMPDAT3;

    /**
     * CNT3
     * ===================================================================================================
     * Offset: 0x38  PWM Data Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |CNT       |PWM Data Register
     * |        |          |User can monitor CNT to know current value in 16-bit down counter.
    */
    __I  uint32_t CNT3;
    uint32_t RESERVE0[1];


    /**
     * INTEN
     * ===================================================================================================
     * Offset: 0x40  PWM Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PIEN      |PWM Period Interrupt Enable Control
     * |        |          |0 = Period interrupt Disabled.
     * |        |          |1 = Period interrupt Enabled.
     * |        |          |Note: Each bit controls the corresponding PWM channel.
    */
    __IO uint32_t INTEN;

    /**
     * INTSTS
     * ===================================================================================================
     * Offset: 0x44  PWM Interrupt Indication Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |PIF       |PWM Timer Interrupt Flag
     * |        |          |0 = Interrupt Flag OFF.
     * |        |          |1 = Interrupt Flag ON.
     * |        |          |Note1: Each bit controls the corresponding PWM channel.
     * |        |          |Note2: User can clear each interrupt flag by writing a one to corresponding bit
    */
    __IO uint32_t INTSTS;
    uint32_t RESERVE1[2];


    /**
     * CAPCTL01
     * ===================================================================================================
     * Offset: 0x50  Capture Control Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPINV0   |Capture 0 Inverter Enable Control
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO.
     * |[1]     |CRLIEN0   |Channel 0 Rising Interrupt Enable Control
     * |        |          |0 = Channel 0 Rising Interrupt Enable Disabled.
     * |        |          |1 = Channel 0 Rising Interrupt Enable Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 0 has rising transition, Capture issues an Interrupt.
     * |[2]     |CFLIEN0   |Channel 0 Falling Interrupt Enable Control
     * |        |          |0 = Channel 0 Falling Interrupt Disabled.
     * |        |          |1 = Channel 0 Falling Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 0 has falling transition, Capture issues an Interrupt.
     * |[3]     |CAPEN0    |Capture Channel 0 Function Enable Control
     * |        |          |0 = Capture function Disabled.
     * |        |          |1 = Capture function Enabled.
     * |        |          |Note1: When Enabled, Capture latched the PWM-counter value and saved to PWM_RCAPDAT0 (Rising latch) and PWM_FCAPDAT0 (Falling latch).
     * |        |          |Note2: When Disabled, Capture does not update PWM_RCAPDAT0 and PWM_FCAPDAT0, and disable Channel 0 Interrupt.
     * |[4]     |CAPIF0    |Capture 0 Interrupt Indication
     * |        |          |0 = Interrupt Flag OFF.
     * |        |          |1 = Interrupt Flag ON.
     * |        |          |Note: If this bit is "1", PWM-counter 0 will not reload when the next capture interrupt occur.
     * |        |          |Write "1" clea.
     * |[6]     |CRLIF0    |Capture Rising Latch Interrupt Flag
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, and this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
     * |[7]     |CFLIF0    |Capture Falling Latch Interrupt Flag
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, and this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
     * |[16]    |CAPINV1   |Capture 1 Inverter Enable Control
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled.
     * |[17]    |CRLIEN1   |Channel 1 Rising Interrupt Enable Control
     * |        |          |0 = Channel 1 Rising Interrupt Disabled.
     * |        |          |1 = Channel 1 Rising Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 1 has rising transition, Capture issues an Interrupt.
     * |[18]    |CFLIEN1   |Channel1 Falling Interrupt Enable Control
     * |        |          |0 = Channel1 Falling Interrupt Disabled.
     * |        |          |1 = Channel1 Falling Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 1 has falling transition, Capture issues an Interrupt.
     * |[19]    |CAPEN1    |Capture Channel 1 Function Enable Control
     * |        |          |0 = Capture function Disabled.
     * |        |          |1 = Capture function Enabled.
     * |        |          |Note1: When Enabled, Capture latched the PMW-counter 1 and saved to PWM_RCAPDAT1 (Rising latch) and PWM_FCAPDAT1 (Falling latch).
     * |        |          |Note2: When Disabled, Capture does not update PWM_RCAPDAT1 and PWM_FCAPDAT1, and disable Channel 1 Interrupt.
     * |[20]    |CAPIF1    |Capture 1 Interrupt Indication
     * |        |          |0 = Interrupt Flag OFF.
     * |        |          |1 = Interrupt Flag ON.
     * |        |          |Note: If this bit is "1", PWM-counter 1 will not reload when the next capture interrupt occurs.
     * |        |          |Write "1" to clea.
     * |[22]    |CRLIF1    |Capture Rising Latch Interrupt Flag
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, and this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
     * |[23]    |CFLIF1    |Capture Falling Latch Interrupt Flag
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, and this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
    */
    __IO uint32_t CAPCTL01;

    /**
     * CAPCTL23
     * ===================================================================================================
     * Offset: 0x54  Capture Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPINV2   |Capture 2 Inverter Enable Control
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO
     * |[1]     |CRLIEN2   |Channel 2 Rising Interrupt Enable Control
     * |        |          |0 = Channel 2 Rising Interrupt Disabled.
     * |        |          |1 = Channel 2 Rising Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 2 has rising transition, Capture issues an Interrupt.
     * |[2]     |CFLIEN2   |Channel 2 Falling Interrupt Enable Control
     * |        |          |0 = Channel 2 Falling Interrupt Disabled.
     * |        |          |1 = Channel 2 Falling Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 2 has falling transition, Capture issues an Interrupt.
     * |[3]     |CAPEN2    |Capture Channel 2 Function Enable Control
     * |        |          |0 = Capture function Disabled.
     * |        |          |1 = Capture function Enabled.
     * |        |          |Note: When Enabled, Capture latched the PMW-counter value and saved to PWM_RCAPDAT2 (Rising latch) and PWM_FCAPDAT2 (Falling latch).
     * |        |          |When Disabled, Capture does not update PWM_RCAPDAT2 and PWM_FCAPDAT2, and disable Channel 2 Interrup.
     * |[4]     |CAPIF2    |Capture 2 Interrupt Indication
     * |        |          |0 = Interrupt Flag OFF.
     * |        |          |1 = Interrupt Flag ON.
     * |        |          |Note: If this bit is "1", PWM-counter 2 will not reload when next capture interrupt occur.
     * |[6]     |CRLIF2    |Capture Rising Latch Interrupt Flag
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
     * |[7]     |CFLIF2    |Capture Falling Latch Interrupt Flag
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
     * |[16]    |CAPINV3   |Capture 3 Inverter Enable Control
     * |        |          |0 = Capture source inverter Disabled.
     * |        |          |1 = Capture source inverter Enabled. Reverse the input signal from GPIO
     * |[17]    |CRLIEN3   |Channel 3 Rising Interrupt Enable Control
     * |        |          |0 = Channel 3 Rising Interrupt Disabled.
     * |        |          |1 = Channel 3 Rising Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 3 has rising transition, Capture issues an Interrupt.
     * |[18]    |CFLIEN3   |Channel 3 Falling Interrupt Enable Control
     * |        |          |0 = Channel 3 Falling Interrupt Disabled.
     * |        |          |1 = Channel 3 Falling Interrupt Enabled.
     * |        |          |Note: When Enabled, if Capture detects Channel 3 has falling transition, Capture issues an Interrupt.
     * |[19]    |CAPEN3    |Capture Channel 3 Function Enable Control
     * |        |          |0 = Capture function Disabled.
     * |        |          |1 = Capture function Enabled.
     * |        |          |Note: When Enabled, Capture latched the PMW-counter and saved to PWM_RCAPDAT3 (Rising latch) and PWM_FCAPDAT3 (Falling latch).
     * |        |          |When Disabled, Capture does not update PWM_RCAPDAT3 and PWM_FCAPDAT3, and disable Channel 3 Interrup.
     * |[20]    |CAPIF3    |Capture 3 Interrupt Indication
     * |        |          |0 = Interrupt Flag OFF.
     * |        |          |1 = Interrupt Flag ON.
     * |        |          |Note: If this bit is "1", PWM-counter 3 will not reload when next capture interrupt occur.
     * |[22]    |CRLIF3    |Capture Rising Latch Interrupt Flag
     * |        |          |0 = No capture rising latch condition happened.
     * |        |          |1 = Capture rising latch condition happened, this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
     * |[23]    |CFLIF3    |Capture Falling Latch Interrupt Flag
     * |        |          |0 = No capture falling latch condition happened.
     * |        |          |1 = Capture falling latch condition happened, this flag will be set to high.
     * |        |          |Note: This bit must be cleared by writing 1 to it.
    */
    __IO uint32_t CAPCTL23;

    /**
     * RCAPDAT0
     * ===================================================================================================
     * Offset: 0x58  Capture Rising Latch Register (Channel 0)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |Capture Rising Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has rising transition.
    */
    __IO uint32_t RCAPDAT0;

    /**
     * FCAPDAT0
     * ===================================================================================================
     * Offset: 0x5C  Capture Falling Latch Register (Channel 0)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |Capture Falling Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has Falling transition.
    */
    __IO uint32_t FCAPDAT0;

    /**
     * RCAPDAT1
     * ===================================================================================================
     * Offset: 0x60  Capture Rising Latch Register (Channel 1)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |Capture Rising Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has rising transition.
    */
    __IO uint32_t RCAPDAT1;

    /**
     * FCAPDAT1
     * ===================================================================================================
     * Offset: 0x64  Capture Falling Latch Register (Channel 1)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |Capture Falling Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has Falling transition.
    */
    __IO uint32_t FCAPDAT1;

    /**
     * RCAPDAT2
     * ===================================================================================================
     * Offset: 0x68  Capture Rising Latch Register (Channel 2)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |Capture Rising Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has rising transition.
    */
    __IO uint32_t RCAPDAT2;

    /**
     * FCAPDAT2
     * ===================================================================================================
     * Offset: 0x6C  Capture Falling Latch Register (Channel 2)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |Capture Falling Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has Falling transition.
    */
    __IO uint32_t FCAPDAT2;

    /**
     * RCAPDAT3
     * ===================================================================================================
     * Offset: 0x70  Capture Rising Latch Register (Channel 3)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RCAPDAT   |Capture Rising Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has rising transition.
    */
    __IO uint32_t RCAPDAT3;

    /**
     * FCAPDAT3
     * ===================================================================================================
     * Offset: 0x74  Capture Falling Latch Register (Channel 3)
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |FCAPDAT   |Capture Falling Latch Register
     * |        |          |Latch the PWM counter when Channel 0/1/2/3 has Falling transition.
    */
    __IO uint32_t FCAPDAT3;

    /**
     * CAPINEN
     * ===================================================================================================
     * Offset: 0x78  Capture Input Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPINEN   |Capture Input Enable Control
     * |        |          |0 = PWM Channel capture input path Disabled.
     * |        |          |The input of PWM channel capture function is always regarded as 0.
     * |        |          |1 = PWM Channel capture input path Enabled.
     * |        |          |The input of PWM channel capture function comes from correlative multifunction pin.
     * |        |          |n = 0, Capture channel 0 is from PB.10 or PC.9.
     * |        |          |n = 1, Capture channel 1 is from PB.11 or PC.10.
     * |        |          |n = 2, Capture channel 2 is from PB.12 or PC.11.
     * |        |          |n = 3, Capture channel 3 is from PB.13 or PC.12.
    */
    __IO uint32_t CAPINEN;

    /**
     * POEN
     * ===================================================================================================
     * Offset: 0x7C  PWM Output Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |POEN      |PWM Counter Output Enable Control
     * |        |          |0 = PWM Counter Output Disabled.
     * |        |          |1 = PWM Counter Output Enabled.
     * |        |          |Note: Each bit controls the corresponding PWM channel.
    */
    __IO uint32_t POEN;

} PWM_T;

/**
    @addtogroup PWM_CONST PWM Bit Field Definition
    Constant Definitions for PWM Controller
@{ */

#define PWM_CLKPSC_CLKPSC01_Pos          (0)                                               /*!< PWM_T::CLKPSC: CLKPSC01 Position          */
#define PWM_CLKPSC_CLKPSC01_Msk          (0xfful << PWM_CLKPSC_CLKPSC01_Pos)               /*!< PWM_T::CLKPSC: CLKPSC01 Mask              */

#define PWM_CLKPSC_CLKPSC23_Pos          (8)                                               /*!< PWM_T::CLKPSC: CLKPSC23 Position          */
#define PWM_CLKPSC_CLKPSC23_Msk          (0xfful << PWM_CLKPSC_CLKPSC23_Pos)               /*!< PWM_T::CLKPSC: CLKPSC23 Mask              */

#define PWM_CLKPSC_DZCNT01_Pos           (16)                                              /*!< PWM_T::CLKPSC: DZCNT01 Position           */
#define PWM_CLKPSC_DZCNT01_Msk           (0xfful << PWM_CLKPSC_DZCNT01_Pos)                /*!< PWM_T::CLKPSC: DZCNT01 Mask               */

#define PWM_CLKPSC_DZCNT23_Pos           (24)                                              /*!< PWM_T::CLKPSC: DZCNT23 Position           */
#define PWM_CLKPSC_DZCNT23_Msk           (0xfful << PWM_CLKPSC_DZCNT23_Pos)                /*!< PWM_T::CLKPSC: DZCNT23 Mask               */

#define PWM_CLKDIV_CLKDIV0_Pos           (0)                                               /*!< PWM_T::CLKDIV: CLKDIV0 Position           */
#define PWM_CLKDIV_CLKDIV0_Msk           (0x7ul << PWM_CLKDIV_CLKDIV0_Pos)                 /*!< PWM_T::CLKDIV: CLKDIV0 Mask               */

#define PWM_CLKDIV_CLKDIV1_Pos           (4)                                               /*!< PWM_T::CLKDIV: CLKDIV1 Position           */
#define PWM_CLKDIV_CLKDIV1_Msk           (0x7ul << PWM_CLKDIV_CLKDIV1_Pos)                 /*!< PWM_T::CLKDIV: CLKDIV1 Mask               */

#define PWM_CLKDIV_CLKDIV2_Pos           (8)                                               /*!< PWM_T::CLKDIV: CLKDIV2 Position           */
#define PWM_CLKDIV_CLKDIV2_Msk           (0x7ul << PWM_CLKDIV_CLKDIV2_Pos)                 /*!< PWM_T::CLKDIV: CLKDIV2 Mask               */

#define PWM_CLKDIV_CLKDIV3_Pos           (12)                                              /*!< PWM_T::CLKDIV: CLKDIV3 Position           */
#define PWM_CLKDIV_CLKDIV3_Msk           (0x7ul << PWM_CLKDIV_CLKDIV3_Pos)                 /*!< PWM_T::CLKDIV: CLKDIV3 Mask               */

#define PWM_CTL_CNTEN0_Pos               (0)                                               /*!< PWM_T::CTL: CNTEN0 Position               */
#define PWM_CTL_CNTEN0_Msk               (0x1ul << PWM_CTL_CNTEN0_Pos)                     /*!< PWM_T::CTL: CNTEN0 Mask                   */

#define PWM_CTL_PINV0_Pos                (2)                                               /*!< PWM_T::CTL: PINV0 Position                */
#define PWM_CTL_PINV0_Msk                (0x1ul << PWM_CTL_PINV0_Pos)                      /*!< PWM_T::CTL: PINV0 Mask                    */

#define PWM_CTL_CNTMODE0_Pos             (3)                                               /*!< PWM_T::CTL: CNTMODE0 Position             */
#define PWM_CTL_CNTMODE0_Msk             (0x1ul << PWM_CTL_CNTMODE0_Pos)                   /*!< PWM_T::CTL: CNTMODE0 Mask                 */

#define PWM_CTL_DTEN01_Pos               (4)                                               /*!< PWM_T::CTL: DTEN01 Position               */
#define PWM_CTL_DTEN01_Msk               (0x1ul << PWM_CTL_DTEN01_Pos)                     /*!< PWM_T::CTL: DTEN01 Mask                   */

#define PWM_CTL_DTEN23_Pos               (5)                                               /*!< PWM_T::CTL: DTEN23 Position               */
#define PWM_CTL_DTEN23_Msk               (0x1ul << PWM_CTL_DTEN23_Pos)                     /*!< PWM_T::CTL: DTEN23 Mask                   */

#define PWM_CTL_CNTEN1_Pos               (8)                                               /*!< PWM_T::CTL: CNTEN1 Position               */
#define PWM_CTL_CNTEN1_Msk               (0x1ul << PWM_CTL_CNTEN1_Pos)                     /*!< PWM_T::CTL: CNTEN1 Mask                   */

#define PWM_CTL_PINV1_Pos                (10)                                              /*!< PWM_T::CTL: PINV1 Position                */
#define PWM_CTL_PINV1_Msk                (0x1ul << PWM_CTL_PINV1_Pos)                      /*!< PWM_T::CTL: PINV1 Mask                    */

#define PWM_CTL_CNTMODE1_Pos             (11)                                              /*!< PWM_T::CTL: CNTMODE1 Position             */
#define PWM_CTL_CNTMODE1_Msk             (0x1ul << PWM_CTL_CNTMODE1_Pos)                   /*!< PWM_T::CTL: CNTMODE1 Mask                 */

#define PWM_CTL_CNTEN2_Pos               (16)                                              /*!< PWM_T::CTL: CNTEN2 Position               */
#define PWM_CTL_CNTEN2_Msk               (0x1ul << PWM_CTL_CNTEN2_Pos)                     /*!< PWM_T::CTL: CNTEN2 Mask                   */

#define PWM_CTL_PINV2_Pos                (18)                                              /*!< PWM_T::CTL: PINV2 Position                */
#define PWM_CTL_PINV2_Msk                (0x1ul << PWM_CTL_PINV2_Pos)                      /*!< PWM_T::CTL: PINV2 Mask                    */

#define PWM_CTL_CNTMODE2_Pos             (19)                                              /*!< PWM_T::CTL: CNTMODE2 Position             */
#define PWM_CTL_CNTMODE2_Msk             (0x1ul << PWM_CTL_CNTMODE2_Pos)                   /*!< PWM_T::CTL: CNTMODE2 Mask                 */

#define PWM_CTL_CNTEN3_Pos               (24)                                              /*!< PWM_T::CTL: CNTEN3 Position               */
#define PWM_CTL_CNTEN3_Msk               (0x1ul << PWM_CTL_CNTEN3_Pos)                     /*!< PWM_T::CTL: CNTEN3 Mask                   */

#define PWM_CTL_PINV3_Pos                (26)                                              /*!< PWM_T::CTL: PINV3 Position                */
#define PWM_CTL_PINV3_Msk                (0x1ul << PWM_CTL_PINV3_Pos)                      /*!< PWM_T::CTL: PINV3 Mask                    */

#define PWM_CTL_CNTMODE3_Pos             (27)                                              /*!< PWM_T::CTL: CNTMODE3 Position             */
#define PWM_CTL_CNTMODE3_Msk             (0x1ul << PWM_CTL_CNTMODE3_Pos)                   /*!< PWM_T::CTL: CNTMODE3 Mask                 */

#define PWM_PERIOD0_PERIOD_Pos           (0)                                               /*!< PWM_T::PERIOD0: PERIOD Position           */
#define PWM_PERIOD0_PERIOD_Msk           (0xfffful << PWM_PERIOD0_PERIOD_Pos)              /*!< PWM_T::PERIOD0: PERIOD Mask               */

#define PWM_CMPDAT0_CMP_Pos              (0)                                               /*!< PWM_T::CMPDAT0: CMP Position              */
#define PWM_CMPDAT0_CMP_Msk              (0xfffful << PWM_CMPDAT0_CMP_Pos)                 /*!< PWM_T::CMPDAT0: CMP Mask                  */

#define PWM_CNT0_CNT_Pos                 (0)                                               /*!< PWM_T::CNT0: CNT Position                 */
#define PWM_CNT0_CNT_Msk                 (0xfffful << PWM_CNT0_CNT_Pos)                    /*!< PWM_T::CNT0: CNT Mask                     */

#define PWM_PERIOD1_PERIOD_Pos           (0)                                               /*!< PWM_T::PERIOD1: PERIOD Position           */
#define PWM_PERIOD1_PERIOD_Msk           (0xfffful << PWM_PERIOD1_PERIOD_Pos)              /*!< PWM_T::PERIOD1: PERIOD Mask               */

#define PWM_CMPDAT1_CMP_Pos              (0)                                               /*!< PWM_T::CMPDAT1: CMP Position              */
#define PWM_CMPDAT1_CMP_Msk              (0xfffful << PWM_CMPDAT1_CMP_Pos)                 /*!< PWM_T::CMPDAT1: CMP Mask                  */

#define PWM_CNT1_CNT_Pos                 (0)                                               /*!< PWM_T::CNT1: CNT Position                 */
#define PWM_CNT1_CNT_Msk                 (0xfffful << PWM_CNT1_CNT_Pos)                    /*!< PWM_T::CNT1: CNT Mask                     */

#define PWM_PERIOD2_PERIOD_Pos           (0)                                               /*!< PWM_T::PERIOD2: PERIOD Position           */
#define PWM_PERIOD2_PERIOD_Msk           (0xfffful << PWM_PERIOD2_PERIOD_Pos)              /*!< PWM_T::PERIOD2: PERIOD Mask               */

#define PWM_CMPDAT2_CMP_Pos              (0)                                               /*!< PWM_T::CMPDAT2: CMP Position              */
#define PWM_CMPDAT2_CMP_Msk              (0xfffful << PWM_CMPDAT2_CMP_Pos)                 /*!< PWM_T::CMPDAT2: CMP Mask                  */

#define PWM_CNT2_CNT_Pos                 (0)                                               /*!< PWM_T::CNT2: CNT Position                 */
#define PWM_CNT2_CNT_Msk                 (0xfffful << PWM_CNT2_CNT_Pos)                    /*!< PWM_T::CNT2: CNT Mask                     */

#define PWM_PERIOD3_PERIOD_Pos           (0)                                               /*!< PWM_T::PERIOD3: PERIOD Position           */
#define PWM_PERIOD3_PERIOD_Msk           (0xfffful << PWM_PERIOD3_PERIOD_Pos)              /*!< PWM_T::PERIOD3: PERIOD Mask               */

#define PWM_CMPDAT3_CMP_Pos              (0)                                               /*!< PWM_T::CMPDAT3: CMP Position              */
#define PWM_CMPDAT3_CMP_Msk              (0xfffful << PWM_CMPDAT3_CMP_Pos)                 /*!< PWM_T::CMPDAT3: CMP Mask                  */

#define PWM_CNT3_CNT_Pos                 (0)                                               /*!< PWM_T::CNT3: CNT Position                 */
#define PWM_CNT3_CNT_Msk                 (0xfffful << PWM_CNT3_CNT_Pos)                    /*!< PWM_T::CNT3: CNT Mask                     */

#define PWM_INTEN_PIEN_Pos               (0)                                               /*!< PWM_T::INTEN: PIEN Position               */
#define PWM_INTEN_PIEN_Msk               (0xful << PWM_INTEN_PIEN_Pos)                     /*!< PWM_T::INTEN: PIEN Mask                   */

#define PWM_INTSTS_PIF_Pos               (0)                                               /*!< PWM_T::INTSTS: PIF Position               */
#define PWM_INTSTS_PIF_Msk               (0xful << PWM_INTSTS_PIF_Pos)                     /*!< PWM_T::INTSTS: PIF Mask                   */

#define PWM_CAPCTL01_CAPINV0_Pos         (0)                                               /*!< PWM_T::CAPCTL01: CAPINV0 Position         */
#define PWM_CAPCTL01_CAPINV0_Msk         (0x1ul << PWM_CAPCTL01_CAPINV0_Pos)               /*!< PWM_T::CAPCTL01: CAPINV0 Mask             */

#define PWM_CAPCTL01_CRLIEN0_Pos         (1)                                               /*!< PWM_T::CAPCTL01: CRLIEN0 Position         */
#define PWM_CAPCTL01_CRLIEN0_Msk         (0x1ul << PWM_CAPCTL01_CRLIEN0_Pos)               /*!< PWM_T::CAPCTL01: CRLIEN0 Mask             */

#define PWM_CAPCTL01_CFLIEN0_Pos         (2)                                               /*!< PWM_T::CAPCTL01: CFLIEN0 Position         */
#define PWM_CAPCTL01_CFLIEN0_Msk         (0x1ul << PWM_CAPCTL01_CFLIEN0_Pos)               /*!< PWM_T::CAPCTL01: CFLIEN0 Mask             */

#define PWM_CAPCTL01_CAPEN0_Pos          (3)                                               /*!< PWM_T::CAPCTL01: CAPEN0 Position          */
#define PWM_CAPCTL01_CAPEN0_Msk          (0x1ul << PWM_CAPCTL01_CAPEN0_Pos)                /*!< PWM_T::CAPCTL01: CAPEN0 Mask              */

#define PWM_CAPCTL01_CAPIF0_Pos          (4)                                               /*!< PWM_T::CAPCTL01: CAPIF0 Position          */
#define PWM_CAPCTL01_CAPIF0_Msk          (0x1ul << PWM_CAPCTL01_CAPIF0_Pos)                /*!< PWM_T::CAPCTL01: CAPIF0 Mask              */

#define PWM_CAPCTL01_CRLIF0_Pos          (6)                                               /*!< PWM_T::CAPCTL01: CRLIF0 Position          */
#define PWM_CAPCTL01_CRLIF0_Msk          (0x1ul << PWM_CAPCTL01_CRLIF0_Pos)                /*!< PWM_T::CAPCTL01: CRLIF0 Mask              */

#define PWM_CAPCTL01_CFLIF0_Pos          (7)                                               /*!< PWM_T::CAPCTL01: CFLIF0 Position          */
#define PWM_CAPCTL01_CFLIF0_Msk          (0x1ul << PWM_CAPCTL01_CFLIF0_Pos)                /*!< PWM_T::CAPCTL01: CFLIF0 Mask              */

#define PWM_CAPCTL01_CAPINV1_Pos         (16)                                              /*!< PWM_T::CAPCTL01: CAPINV1 Position         */
#define PWM_CAPCTL01_CAPINV1_Msk         (0x1ul << PWM_CAPCTL01_CAPINV1_Pos)               /*!< PWM_T::CAPCTL01: CAPINV1 Mask             */

#define PWM_CAPCTL01_CRLIEN1_Pos         (17)                                              /*!< PWM_T::CAPCTL01: CRLIEN1 Position         */
#define PWM_CAPCTL01_CRLIEN1_Msk         (0x1ul << PWM_CAPCTL01_CRLIEN1_Pos)               /*!< PWM_T::CAPCTL01: CRLIEN1 Mask             */

#define PWM_CAPCTL01_CFLIEN1_Pos         (18)                                              /*!< PWM_T::CAPCTL01: CFLIEN1 Position         */
#define PWM_CAPCTL01_CFLIEN1_Msk         (0x1ul << PWM_CAPCTL01_CFLIEN1_Pos)               /*!< PWM_T::CAPCTL01: CFLIEN1 Mask             */

#define PWM_CAPCTL01_CAPEN1_Pos          (19)                                              /*!< PWM_T::CAPCTL01: CAPEN1 Position          */
#define PWM_CAPCTL01_CAPEN1_Msk          (0x1ul << PWM_CAPCTL01_CAPEN1_Pos)                /*!< PWM_T::CAPCTL01: CAPEN1 Mask              */

#define PWM_CAPCTL01_CAPIF1_Pos          (20)                                              /*!< PWM_T::CAPCTL01: CAPIF1 Position          */
#define PWM_CAPCTL01_CAPIF1_Msk          (0x1ul << PWM_CAPCTL01_CAPIF1_Pos)                /*!< PWM_T::CAPCTL01: CAPIF1 Mask              */

#define PWM_CAPCTL01_CRLIF1_Pos          (22)                                              /*!< PWM_T::CAPCTL01: CRLIF1 Position          */
#define PWM_CAPCTL01_CRLIF1_Msk          (0x1ul << PWM_CAPCTL01_CRLIF1_Pos)                /*!< PWM_T::CAPCTL01: CRLIF1 Mask              */

#define PWM_CAPCTL01_CFLIF1_Pos          (23)                                              /*!< PWM_T::CAPCTL01: CFLIF1 Position          */
#define PWM_CAPCTL01_CFLIF1_Msk          (0x1ul << PWM_CAPCTL01_CFLIF1_Pos)                /*!< PWM_T::CAPCTL01: CFLIF1 Mask              */

#define PWM_CAPCTL23_CAPINV2_Pos         (0)                                               /*!< PWM_T::CAPCTL23: CAPINV2 Position         */
#define PWM_CAPCTL23_CAPINV2_Msk         (0x1ul << PWM_CAPCTL23_CAPINV2_Pos)               /*!< PWM_T::CAPCTL23: CAPINV2 Mask             */

#define PWM_CAPCTL23_CRLIEN2_Pos         (1)                                               /*!< PWM_T::CAPCTL23: CRLIEN2 Position         */
#define PWM_CAPCTL23_CRLIEN2_Msk         (0x1ul << PWM_CAPCTL23_CRLIEN2_Pos)               /*!< PWM_T::CAPCTL23: CRLIEN2 Mask             */

#define PWM_CAPCTL23_CFLIEN2_Pos         (2)                                               /*!< PWM_T::CAPCTL23: CFLIEN2 Position         */
#define PWM_CAPCTL23_CFLIEN2_Msk         (0x1ul << PWM_CAPCTL23_CFLIEN2_Pos)               /*!< PWM_T::CAPCTL23: CFLIEN2 Mask             */

#define PWM_CAPCTL23_CAPEN2_Pos          (3)                                               /*!< PWM_T::CAPCTL23: CAPEN2 Position          */
#define PWM_CAPCTL23_CAPEN2_Msk          (0x1ul << PWM_CAPCTL23_CAPEN2_Pos)                /*!< PWM_T::CAPCTL23: CAPEN2 Mask              */

#define PWM_CAPCTL23_CAPIF2_Pos          (4)                                               /*!< PWM_T::CAPCTL23: CAPIF2 Position          */
#define PWM_CAPCTL23_CAPIF2_Msk          (0x1ul << PWM_CAPCTL23_CAPIF2_Pos)                /*!< PWM_T::CAPCTL23: CAPIF2 Mask              */

#define PWM_CAPCTL23_CRLIF2_Pos          (6)                                               /*!< PWM_T::CAPCTL23: CRLIF2 Position          */
#define PWM_CAPCTL23_CRLIF2_Msk          (0x1ul << PWM_CAPCTL23_CRLIF2_Pos)                /*!< PWM_T::CAPCTL23: CRLIF2 Mask              */

#define PWM_CAPCTL23_CFLIF2_Pos          (7)                                               /*!< PWM_T::CAPCTL23: CFLIF2 Position          */
#define PWM_CAPCTL23_CFLIF2_Msk          (0x1ul << PWM_CAPCTL23_CFLIF2_Pos)                /*!< PWM_T::CAPCTL23: CFLIF2 Mask              */

#define PWM_CAPCTL23_CAPINV3_Pos         (16)                                              /*!< PWM_T::CAPCTL23: CAPINV3 Position         */
#define PWM_CAPCTL23_CAPINV3_Msk         (0x1ul << PWM_CAPCTL23_CAPINV3_Pos)               /*!< PWM_T::CAPCTL23: CAPINV3 Mask             */

#define PWM_CAPCTL23_CRLIEN3_Pos         (17)                                              /*!< PWM_T::CAPCTL23: CRLIEN3 Position         */
#define PWM_CAPCTL23_CRLIEN3_Msk         (0x1ul << PWM_CAPCTL23_CRLIEN3_Pos)               /*!< PWM_T::CAPCTL23: CRLIEN3 Mask             */

#define PWM_CAPCTL23_CFLIEN3_Pos         (18)                                              /*!< PWM_T::CAPCTL23: CFLIEN3 Position         */
#define PWM_CAPCTL23_CFLIEN3_Msk         (0x1ul << PWM_CAPCTL23_CFLIEN3_Pos)               /*!< PWM_T::CAPCTL23: CFLIEN3 Mask             */

#define PWM_CAPCTL23_CAPEN3_Pos          (19)                                              /*!< PWM_T::CAPCTL23: CAPEN3 Position          */
#define PWM_CAPCTL23_CAPEN3_Msk          (0x1ul << PWM_CAPCTL23_CAPEN3_Pos)                /*!< PWM_T::CAPCTL23: CAPEN3 Mask              */

#define PWM_CAPCTL23_CAPIF3_Pos          (20)                                              /*!< PWM_T::CAPCTL23: CAPIF3 Position          */
#define PWM_CAPCTL23_CAPIF3_Msk          (0x1ul << PWM_CAPCTL23_CAPIF3_Pos)                /*!< PWM_T::CAPCTL23: CAPIF3 Mask              */

#define PWM_CAPCTL23_CRLIF3_Pos          (22)                                              /*!< PWM_T::CAPCTL23: CRLIF3 Position          */
#define PWM_CAPCTL23_CRLIF3_Msk          (0x1ul << PWM_CAPCTL23_CRLIF3_Pos)                /*!< PWM_T::CAPCTL23: CRLIF3 Mask              */

#define PWM_CAPCTL23_CFLIF3_Pos          (23)                                              /*!< PWM_T::CAPCTL23: CFLIF3 Position          */
#define PWM_CAPCTL23_CFLIF3_Msk          (0x1ul << PWM_CAPCTL23_CFLIF3_Pos)                /*!< PWM_T::CAPCTL23: CFLIF3 Mask              */

#define PWM_RCAPDAT0_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT0: RCAPDAT Position         */
#define PWM_RCAPDAT0_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT0_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT0: RCAPDAT Mask             */

#define PWM_FCAPDAT0_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT0: FCAPDAT Position         */
#define PWM_FCAPDAT0_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT0_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT0: FCAPDAT Mask             */

#define PWM_RCAPDAT1_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT1: RCAPDAT Position         */
#define PWM_RCAPDAT1_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT1_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT1: RCAPDAT Mask             */

#define PWM_FCAPDAT1_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT1: FCAPDAT Position         */
#define PWM_FCAPDAT1_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT1_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT1: FCAPDAT Mask             */

#define PWM_RCAPDAT2_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT2: RCAPDAT Position         */
#define PWM_RCAPDAT2_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT2_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT2: RCAPDAT Mask             */

#define PWM_FCAPDAT2_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT2: FCAPDAT Position         */
#define PWM_FCAPDAT2_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT2_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT2: FCAPDAT Mask             */

#define PWM_RCAPDAT3_RCAPDAT_Pos         (0)                                               /*!< PWM_T::RCAPDAT3: RCAPDAT Position         */
#define PWM_RCAPDAT3_RCAPDAT_Msk         (0xfffful << PWM_RCAPDAT3_RCAPDAT_Pos)            /*!< PWM_T::RCAPDAT3: RCAPDAT Mask             */

#define PWM_FCAPDAT3_FCAPDAT_Pos         (0)                                               /*!< PWM_T::FCAPDAT3: FCAPDAT Position         */
#define PWM_FCAPDAT3_FCAPDAT_Msk         (0xfffful << PWM_FCAPDAT3_FCAPDAT_Pos)            /*!< PWM_T::FCAPDAT3: FCAPDAT Mask             */

#define PWM_CAPINEN_CAPINEN_Pos          (0)                                               /*!< PWM_T::CAPINEN: CAPINEN Position          */
#define PWM_CAPINEN_CAPINEN_Msk          (0xful << PWM_CAPINEN_CAPINEN_Pos)                /*!< PWM_T::CAPINEN: CAPINEN Mask              */

#define PWM_POEN_POEN_Pos                (0)                                               /*!< PWM_T::POEN: POEN Position                */
#define PWM_POEN_POEN_Msk                (0xful << PWM_POEN_POEN_Pos)                      /*!< PWM_T::POEN: POEN Mask                    */

/**@}*/ /* PWM_CONST */
/**@}*/ /* end of PWM register group */


/*---------------------- Real Time Clock Controller -------------------------*/
/**
    @addtogroup RTC Real Time Clock Controller(RTC)
    Memory Mapped Structure for RTC Controller
@{ */

typedef struct
{


    /**
     * INIT
     * ===================================================================================================
     * Offset: 0x00  RTC Initiation Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ACTIVE    |RTC Active Status (Read Only)
     * |        |          |0 = RTC is at reset state.
     * |        |          |1 = RTC is at normal active state.
     * |[31:1]  |INIT_STS  |RTC Initiation (While Writing)
     * |        |          |When RTC block is powered on, RTC is at reset state.
     * |        |          |User has to write a number (0x a5eb1357) to INIT to make RTC leaving reset state.
     * |        |          |Once the INIT is written as 0xa5eb1357, the RTC will be in un-reset state permanentl.
     * |        |          |The INIT is a write-only field and read value will be always "0".
     * |        |          |RTC Internal Status (While Reading)
     * |        |          |[31:8]: INIT[31:8]
     * |        |          |[7:5]: RTC internal state machine of key detection
     * |        |          |[4]: Status of power key, 0:pressed and 1:released
     * |        |          |[3]: Status of power off request pwr_key_off
     * |        |          |[2]: Level shifter reset
     * |        |          |[1]: Level shifter enable
    */
    __IO  uint32_t INIT;

    /**
     * RWEN
     * ===================================================================================================
     * Offset: 0x04  RTC Access Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |RWEN      |RTC Register Access Enable Password (R/W)
     * |        |          |0xA965 = Access Password.
     * |        |          |Others = Access Disabled.
     * |[16]    |RWENF     |RTC Register Access Enable Flag (Read Only)
     * |        |          |0 = RTC register access Disabled.
     * |        |          |1 = RTC register access Enabled.
     * |        |          |This bit will be set after RTC_RWEN[15:0] register is load a 0xA965, and it will be cleared when RTC_RWEN[15:0] is not 0xA965.
    */
    __IO uint32_t RWEN;

    /**
     * FREQADJ
     * ===================================================================================================
     * Offset: 0x08  RTC Frequency Compensation Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[5:0]   |FRACTION  |Fraction Part
     * |        |          |Formula = (fraction part of detected RTC clock rate ) x 60-1.
     * |        |          |Digit in RTC_FREQADJ must be expressed as hexadecimal number.
     * |[23:8]  |INTEGER   |Integer Part
     * |        |          |Real Oscillator Frequency = (Integer part of detected RTC clock rate -1).
     * |[27:24] |PKEYTIME  |Minimum Duration That Power Key Must Be Pressed To Turn On Core Power
     * |        |          |Minimum power key duration = 0.25*(PKEYTIME+1) sec.
     * |[31]    |ADJTRG    |RTC Clock Calibration Control
     * |        |          |This bit will be kept at "High" while the calibration is ongoing and cleared to "Low" automatically while the calibration is done and the content of RTC 1Hz Counter Register is valid calibration flow is as follows.
     * |        |          |0 = RTC Clock calibration mechanism Disabled.
     * |        |          |1 = RTC Clock calibration mechanism Enabled.
    */
    __IO uint32_t FREQADJ;

    /**
     * TIME
     * ===================================================================================================
     * Offset: 0x0C  RTC Time Loading Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SEC       |1-Sec Time Digit
     * |[6:4]   |TENSEC    |10-Sec Time Digit
     * |[11:8]  |MIN       |1-Min Time Digit
     * |[14:12] |TENMIN    |10-Min Time Digit
     * |[19:16] |HR        |1-Hour Time Digit
     * |[21:20] |TENHR     |10-Hour Time Digit
    */
    __IO uint32_t TIME;

    /**
     * CAL
     * ===================================================================================================
     * Offset: 0x10  RTC Calendar Loading Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DAY       |1-Day Calendar Digit
     * |[5:4]   |TENDAY    |10-Day Calendar Digit
     * |[11:8]  |MON       |1-Month Calendar Digit
     * |[12]    |TENMON    |10-Month Calendar Digit
     * |[19:16] |YEAR      |1-Year Calendar Digit
     * |[23:20] |TENYEAR   |10-Year Calendar Digit
    */
    __IO uint32_t CAL;

    /**
     * CLKFMT
     * ===================================================================================================
     * Offset: 0x14  RTC Time Scale Selection Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |24HEN     |24-Hour / 12-Hour Mode Selection
     * |        |          |Indicate that RTC_TIME and RTC_TALM are in 24-hour mode or 12-hour mode.
     * |        |          |0 = 12-hour time scale with AM and PM indication selected.
     * |        |          |1 = 24-hour time scale selected.
    */
    __IO uint32_t CLKFMT;

    /**
     * WEEKDAY
     * ===================================================================================================
     * Offset: 0x18  RTC Day of the Week Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |WEEKDAY   |Day Of The Week Register
     * |        |          |0 = Sunday.
     * |        |          |1 = Monday.
     * |        |          |2 = Tuesday.
     * |        |          |3 = Wednesday.
     * |        |          |4 = Thursday.
     * |        |          |5 = Friday.
     * |        |          |6 = Saturday.
    */
    __IO uint32_t WEEKDAY;

    /**
     * TALM
     * ===================================================================================================
     * Offset: 0x1C  RTC Time Alarm Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |SEC       |1-Sec Time Digit of Alarm Setting (0-9)
     * |[6:4]   |TENSEC    |10-Sec Time Digit of Alarm Setting (0-5)
     * |[7]     |MSKSEC    |Mask Alarm By Second
     * |        |          |0 =Activate.
     * |        |          |1 = Mask.
     * |[11:8]  |MIN       |1-Min Time Digit of Alarm Setting (0-9)
     * |[14:12] |TENMIN    |10-Min Time Digit of Alarm Setting (0-5)
     * |[15]    |MSKMIN    |Mask Alarm By Minute
     * |        |          |0 = Activate.
     * |        |          |1 = Mask.
     * |[19:16] |HR        |1-Hour Time Digit of Alarm Setting (0-9)
     * |[21:20] |TENHR     |10-Hour Time Digit Of Alarm Setting (0-2)
     * |        |          |When RTC runs as 12-hour time scale mode, RTC_TIME[21] (the high bit of TENHR[1:0]) means AM/PM indication.
     * |        |          |(If RTC_TIME[21] is 1, it indicates PM time message.).
     * |[23]    |MSKHR     |Mask Alarm By Hour
     * |        |          |0 = Activate.
     * |        |          |1 = Mask.
    */
    __IO uint32_t TALM;

    /**
     * CALM
     * ===================================================================================================
     * Offset: 0x20  RTC Calendar Alarm Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DAY       |1-Day Calendar Digit of Alarm Setting (0-9)
     * |[5:4]   |TENDAY    |10-Day Calendar Digit of Alarm Setting (0-3)
     * |[7]     |MSKDAY    |Mask Alarm By Day
     * |        |          |0 = Activate.
     * |        |          |1 = Mask.
     * |[11:8]  |MON       |1-Month Calendar Digit of Alarm Setting (0-9)
     * |[12]    |TENMON    |10-Month Calendar Digit of Alarm Setting (0-1)
     * |[15]    |MSKMON    |Mask Alarm By Month
     * |        |          |0 = Activate.
     * |        |          |1 = Mask.
     * |[19:16] |YEAR      |1-Year Calendar Digit of Alarm Setting (0-9)
     * |[23:20] |TENYEAR   |10-Year Calendar Digit of Alarm Setting (0-9)
     * |[24]    |MSKYEAR   |Mask Alarm By Year
     * |        |          |0 = Activate.
     * |        |          |1 = Mask.
     * |[30:28] |WEEKDAY   |Week Day Alarm Digit
     * |[31]    |MSKWEEKDAY|Mask Alarm By Week Day
     * |        |          |0 =Activate.
     * |        |          |1 =Mask.
    */
    __IO uint32_t CALM;

    /**
     * LEAPYEAR
     * ===================================================================================================
     * Offset: 0x24  RTC Leap Year Indication Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |LEAPYEAR  |Leap Year Indication Register (Read Only)
     * |        |          |0 = This year is not a leap year.
     * |        |          |1 = This year is leap year.
    */
    __I  uint32_t LEAPYEAR;

    /**
     * INTEN
     * ===================================================================================================
     * Offset: 0x28  RTC Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ALMIEN    |Alarm Interrupt Enable Control
     * |        |          |0 = RTC Alarm Interrupt Disabled.
     * |        |          |1 = RTC Alarm Interrupt Enabled.
     * |[1]     |TICKIEN   |Time Tick Interrupt Enable Control
     * |        |          |0 = RTC Time Tick Interrupt and counter Disabled.
     * |        |          |1 = RTC Time Tick Interrupt and counter Enabled.
     * |[2]     |PKEYIEN   |Power Switch Interrupt Enable Control
     * |        |          |0 = Power Switch Be Pressed Interrupt Disabled.
     * |        |          |1 = Power Switch Be Pressed Interrupt Enabled.
     * |[3]     |RALMIEN   |Relative Alarm Interrupt Enable Control
     * |        |          |0 = RTC Relative Alarm Interrupt Disabled.
     * |        |          |1 = RTC Relative Alarm Interrupt Enabled.
    */
    __IO uint32_t INTEN;

    /**
     * INTSTS
     * ===================================================================================================
     * Offset: 0x2C  RTC Interrupt Indication Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ALMIF     |RTC Alarm Interrupt Flag
     * |        |          |When RTC time counters RTC_TIME and RTC_CAL match the alarm setting time registers RTC_TALM and RTC_CALM, this bit will be set to 1 and an interrupt will be generated if RTC Alarm Interrupt enabled ALMIEN (RTC_INTEN[0]) is set to 1
     * |        |          |Chip will be woken up if RTC Alarm Interrupt is enabled when chip is at Power-down mode.
     * |        |          |0 = Alarm condition is not matched.
     * |        |          |1 = Alarm condition is matched.
     * |        |          |Note: Write 1 to clear this bit.
     * |[1]     |TICKIF    |RTC Time Tick Interrupt Flag
     * |        |          |When RTC time tick happened, this bit will be set to 1 and an interrupt will be generated if RTC Tick Interrupt enabled TICKIEN (RTC_INTEN[1]) is set to 1
     * |        |          |Chip will also be woken up if RTC Tick Interrupt is enabled and this bit is set to 1 when chip is running at Power-down mode.
     * |        |          |0 = Tick condition does not occur.
     * |        |          |1 = Tick condition occur.
     * |        |          |Note: Write 1 to clear to clear this bit.
     * |[2]     |POWKEYIF  |Power Switch Interrupt Flag
     * |        |          |When RTC detect power key (RTC_nRWAKE) is pressed , the POWKEYIF (RTC_INTSYS[2]) is set to 1
     * |        |          |0 = The power switch interrupt never occurred.
     * |        |          |1 = The power switch has been activated.
     * |        |          |Note: Software can also clear this bit after RTC interrupt has occurred
     * |[3]     |RELALMIF  |RTC Relative Alarm Interrupt Indication
     * |        |          |0 = Relative alarm interrupt never occurred.
     * |        |          |1 = Relative time counter and calendar counter have counted to a specified time recorded in RTC_TALM and RTC_CALM
     * |        |          |RTC alarm interrupt has been activated.
     * |        |          |Note: Software can also clear this bit after RTC interrupt has occurred
    */
    __IO uint32_t INTSTS;

    /**
     * TICK
     * ===================================================================================================
     * Offset: 0x30  RTC Time Tick Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |TICKSEL   |Time Tick Register
     * |        |          |These bits are used to select RTC time tick period for Periodic Time Tick Interrupt request.
     * |        |          |000 = Time tick is 1 second.
     * |        |          |001 = Time tick is 1/2 second.
     * |        |          |010 = Time tick is 1/4 second.
     * |        |          |011 = Time tick is 1/8 second.
     * |        |          |100 = Time tick is 1/16 second.
     * |        |          |101 = Time tick is 1/32 second.
     * |        |          |110 = Time tick is 1/64 second.
     * |        |          |111 = Time tick is 1/28 second.
    */
    __IO uint32_t TICK;

    /**
     * POWCTL
     * ===================================================================================================
     * Offset: 0x34  RTC Power Time On Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |POWEN     |Power ON
     * |        |          |RTC_RPWR will change to high state when POWEN value change from 0 to 1.
     * |        |          |Note : The following conditions will make RTC_RPWR low
     * |        |          |Set POWEN bit to 0
     * |        |          |POWOFFEN is set to 1 and the power key is pressed over the period of POWOFFT.
     * |        |          |This register can be read back after the RTC enable is active.
     * |[1]     |SWPOWOFF  |Software Core Power Disable Control
     * |        |          |If the power key is pressed, the RTC_RPWR pin can be cleared by setting this bit and this can be cleared to 0 when the pressed power key, RTC_RPWR is released.
     * |        |          |If the power is not pressed, it is not used to set this bi.
     * |        |          |1 = Force the RTC_RPWR to low.
     * |[2]     |POWOFFEN  |Hardware Power Clear Enable Control
     * |        |          |0 = The RTC_RPWR pin won't be influenced by the pressed time of power key.
     * |        |          |1 = The RTC_RPWR pin will clear to low when the power key is pressed over the POWOFFT second.
     * |[3]     |ALMIEN    |Normal Time Alarm
     * |        |          |0 = Normal time alarm control Disabled.
     * |        |          |1 = Normal time alarm control Enabled.
     * |[4]     |RALMIEN   |Relative Time Alarm
     * |        |          |0 = The relative time alarm control Disabled.
     * |        |          |1 = The relative time alarm control Enabled.
     * |[5]     |EDGE_TRIG |Power Key Trigger Mode
     * |        |          |0 = LEVEL TRIGGER, RTC is powered on while power key is pressed longer programmed duration.
     * |        |          |1 = EDGE TRIGE, RTC is powered on while power key is pressed longer than programmed duration and then released.
     * |[7]     |POWKEY    |Power Key Status
     * |        |          |0 = The power key is pressed to low.
     * |        |          |1 = The power key status is high.
     * |[19:16] |POWOFFT   |Power Clear Period
     * |        |          |Indicates that the period of the power core will be cleared after the power key is pressed
     * |        |          |Its time scalar is one second so that the default is 5 second.
     * |[31:20] |RALMTIME  |Relative Time Alarm Period (Second Unit)
     * |        |          |Indicates the period of the relative time alarm. Its maximum value is 1800.
     * |        |          |When RALMIEN (RTC_INTEN[3]) = 0 , it will be cleared to 0.
    */
    __IO uint32_t POWCTL;

    /**
     * SET
     * ===================================================================================================
     * Offset: 0x38  RTC Setting Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |CBEN      |32768 Hz (LXT) Crystal Control
     * |        |          |0 = Crystal Disabled.
     * |        |          |1 = Crystal Enabled.
     * |[2]     |IOMSEL    |X32_IN And X32_OUT PAD Digital Input Mode Control
     * |        |          |0 = Digital input mode.
     * |        |          |1 = Crystal mode (default value).
     * |[3]     |XININDAT  |X32_IN PAD Status
     * |        |          |input signal when IOMSEL = 0 (Read Only).
     * |[4]     |XOUTDAT   |X32_OUT PAD Status
     * |        |          |input signal when IOMSEL = 0 (Read Only).
    */
    __IO uint32_t SET;

    /**
     * CLKSRC
     * ===================================================================================================
     * Offset: 0x3C  RC Oscillator Setting Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CKSRC     |Internal RC Oscillator Control
     * |        |          |0 = Internal RC oscillator Disabled.
     * |        |          |1 = Internal RC oscillator Enabled.
    */
    __IO uint32_t CLKSRC;

    /**
     * CALCNT
     * ===================================================================================================
     * Offset: 0x40  RC Oscillator Calibration Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |CALCNT    |Cycle Number Of PCLK During 1Hz
     * |        |          |That is generated by dividing RTC Clock.
     * |        |          |This number can be used to deduct the real clock rate of RTC clock.
    */
    __I  uint32_t CALCNT;

    /**
     * SYNC
     * ===================================================================================================
     * Offset: 0x44  RTC Register Write Complete
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SYNC      |Polling The Flag To Detect RTC Register Write Complete
     * |        |          |0 = Register cannot be written.
     * |        |          |1 = Register can be written because write complete.
    */
    __I  uint32_t SYNC;
    __I  uint32_t RESERVE0[3];


    /**
     * SPRx
     * ===================================================================================================
     * Offset: 0x54 ~ 0x70  RTC Spare Register 0 ~ 7
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RTC_SPRn  |RTC Spare Register
     * |        |          |n = 0~7.
    */
    __IO uint32_t SPR[8];

} RTC_T;

/**
    @addtogroup RTC_CONST RTC Bit Field Definition
    Constant Definitions for RTC Controller
@{ */

#define RTC_INIT_ACTIVE_Pos              (0)                                               /*!< RTC_T::INIT: ACTIVE Position              */
#define RTC_INIT_ACTIVE_Msk              (0x1ul << RTC_INIT_ACTIVE_Pos)                    /*!< RTC_T::INIT: ACTIVE Mask                  */

#define RTC_INIT_STS_Pos                 (1)                                               /*!< RTC_T::INIT: STS Position            */
#define RTC_INIT_STS_Msk                 (0x7ffffffful << RTC_INIT_STS_Pos)                /*!< RTC_T::INIT: STS Mask                */

#define RTC_INIT_INIT_Pos                (1)                                               /*!< RTC_T::INIT: INIT Position                */
#define RTC_INIT_INIT_Msk                (0x7ffffffful << RTC_INIT_INIT_Pos)               /*!< RTC_T::INIT: INIT Mask                    */

#define RTC_RWEN_RWEN_Pos                (0)                                               /*!< RTC_T::RWEN: RWEN Position                */
#define RTC_RWEN_RWEN_Msk                (0xfffful << RTC_RWEN_RWEN_Pos)                   /*!< RTC_T::RWEN: RWEN Mask                    */

#define RTC_RWEN_RWENF_Pos               (16)                                              /*!< RTC_T::RWEN: RWENF Position               */
#define RTC_RWEN_RWENF_Msk               (0x1ul << RTC_RWEN_RWENF_Pos)                     /*!< RTC_T::RWEN: RWENF Mask                   */

#define RTC_FREQADJ_FRACTION_Pos         (0)                                               /*!< RTC_T::FREQADJ: FRACTION Position         */
#define RTC_FREQADJ_FRACTION_Msk         (0x3ful << RTC_FREQADJ_FRACTION_Pos)              /*!< RTC_T::FREQADJ: FRACTION Mask             */

#define RTC_FREQADJ_INTEGER_Pos          (8)                                               /*!< RTC_T::FREQADJ: INTEGER Position          */
#define RTC_FREQADJ_INTEGER_Msk          (0xfffful << RTC_FREQADJ_INTEGER_Pos)             /*!< RTC_T::FREQADJ: INTEGER Mask              */

#define RTC_FREQADJ_PKEYTIME_Pos         (24)                                              /*!< RTC_T::FREQADJ: PKEYTIME Position         */
#define RTC_FREQADJ_PKEYTIME_Msk         (0xful << RTC_FREQADJ_PKEYTIME_Pos)               /*!< RTC_T::FREQADJ: PKEYTIME Mask             */

#define RTC_FREQADJ_ADJTRG_Pos           (31)                                              /*!< RTC_T::FREQADJ: ADJTRG Position           */
#define RTC_FREQADJ_ADJTRG_Msk           (0x1ul << RTC_FREQADJ_ADJTRG_Pos)                 /*!< RTC_T::FREQADJ: ADJTRG Mask               */

#define RTC_TIME_SEC_Pos                 (0)                                               /*!< RTC_T::TIME: SEC Position                 */
#define RTC_TIME_SEC_Msk                 (0xful << RTC_TIME_SEC_Pos)                       /*!< RTC_T::TIME: SEC Mask                     */

#define RTC_TIME_TENSEC_Pos              (4)                                               /*!< RTC_T::TIME: TENSEC Position              */
#define RTC_TIME_TENSEC_Msk              (0x7ul << RTC_TIME_TENSEC_Pos)                    /*!< RTC_T::TIME: TENSEC Mask                  */

#define RTC_TIME_MIN_Pos                 (8)                                               /*!< RTC_T::TIME: MIN Position                 */
#define RTC_TIME_MIN_Msk                 (0xful << RTC_TIME_MIN_Pos)                       /*!< RTC_T::TIME: MIN Mask                     */

#define RTC_TIME_TENMIN_Pos              (12)                                              /*!< RTC_T::TIME: TENMIN Position              */
#define RTC_TIME_TENMIN_Msk              (0x7ul << RTC_TIME_TENMIN_Pos)                    /*!< RTC_T::TIME: TENMIN Mask                  */

#define RTC_TIME_HR_Pos                  (16)                                              /*!< RTC_T::TIME: HR Position                  */
#define RTC_TIME_HR_Msk                  (0xful << RTC_TIME_HR_Pos)                        /*!< RTC_T::TIME: HR Mask                      */

#define RTC_TIME_TENHR_Pos               (20)                                              /*!< RTC_T::TIME: TENHR Position               */
#define RTC_TIME_TENHR_Msk               (0x3ul << RTC_TIME_TENHR_Pos)                     /*!< RTC_T::TIME: TENHR Mask                   */

#define RTC_CAL_DAY_Pos                  (0)                                               /*!< RTC_T::CAL: DAY Position                  */
#define RTC_CAL_DAY_Msk                  (0xful << RTC_CAL_DAY_Pos)                        /*!< RTC_T::CAL: DAY Mask                      */

#define RTC_CAL_TENDAY_Pos               (4)                                               /*!< RTC_T::CAL: TENDAY Position               */
#define RTC_CAL_TENDAY_Msk               (0x3ul << RTC_CAL_TENDAY_Pos)                     /*!< RTC_T::CAL: TENDAY Mask                   */

#define RTC_CAL_MON_Pos                  (8)                                               /*!< RTC_T::CAL: MON Position                  */
#define RTC_CAL_MON_Msk                  (0xful << RTC_CAL_MON_Pos)                        /*!< RTC_T::CAL: MON Mask                      */

#define RTC_CAL_TENMON_Pos               (12)                                              /*!< RTC_T::CAL: TENMON Position               */
#define RTC_CAL_TENMON_Msk               (0x1ul << RTC_CAL_TENMON_Pos)                     /*!< RTC_T::CAL: TENMON Mask                   */

#define RTC_CAL_YEAR_Pos                 (16)                                              /*!< RTC_T::CAL: YEAR Position                 */
#define RTC_CAL_YEAR_Msk                 (0xful << RTC_CAL_YEAR_Pos)                       /*!< RTC_T::CAL: YEAR Mask                     */

#define RTC_CAL_TENYEAR_Pos              (20)                                              /*!< RTC_T::CAL: TENYEAR Position              */
#define RTC_CAL_TENYEAR_Msk              (0xful << RTC_CAL_TENYEAR_Pos)                    /*!< RTC_T::CAL: TENYEAR Mask                  */

#define RTC_CLKFMT_24HEN_Pos             (0)                                               /*!< RTC_T::CLKFMT: 24HEN Position             */
#define RTC_CLKFMT_24HEN_Msk             (0x1ul << RTC_CLKFMT_24HEN_Pos)                   /*!< RTC_T::CLKFMT: 24HEN Mask                 */

#define RTC_WEEKDAY_WEEKDAY_Pos          (0)                                               /*!< RTC_T::WEEKDAY: WEEKDAY Position          */
#define RTC_WEEKDAY_WEEKDAY_Msk          (0x7ul << RTC_WEEKDAY_WEEKDAY_Pos)                /*!< RTC_T::WEEKDAY: WEEKDAY Mask              */

#define RTC_TALM_SEC_Pos                 (0)                                               /*!< RTC_T::TALM: SEC Position                 */
#define RTC_TALM_SEC_Msk                 (0xful << RTC_TALM_SEC_Pos)                       /*!< RTC_T::TALM: SEC Mask                     */

#define RTC_TALM_TENSEC_Pos              (4)                                               /*!< RTC_T::TALM: TENSEC Position              */
#define RTC_TALM_TENSEC_Msk              (0x7ul << RTC_TALM_TENSEC_Pos)                    /*!< RTC_T::TALM: TENSEC Mask                  */

#define RTC_TALM_MSKSEC_Pos              (7)                                               /*!< RTC_T::TALM: MSKSEC Position              */
#define RTC_TALM_MSKSEC_Msk              (0x1ul << RTC_TALM_MSKSEC_Pos)                    /*!< RTC_T::TALM: MSKSEC Mask                  */

#define RTC_TALM_MIN_Pos                 (8)                                               /*!< RTC_T::TALM: MIN Position                 */
#define RTC_TALM_MIN_Msk                 (0xful << RTC_TALM_MIN_Pos)                       /*!< RTC_T::TALM: MIN Mask                     */

#define RTC_TALM_TENMIN_Pos              (12)                                              /*!< RTC_T::TALM: TENMIN Position              */
#define RTC_TALM_TENMIN_Msk              (0x7ul << RTC_TALM_TENMIN_Pos)                    /*!< RTC_T::TALM: TENMIN Mask                  */

#define RTC_TALM_MSKMIN_Pos              (15)                                              /*!< RTC_T::TALM: MSKMIN Position              */
#define RTC_TALM_MSKMIN_Msk              (0x1ul << RTC_TALM_MSKMIN_Pos)                    /*!< RTC_T::TALM: MSKMIN Mask                  */

#define RTC_TALM_HR_Pos                  (16)                                              /*!< RTC_T::TALM: HR Position                  */
#define RTC_TALM_HR_Msk                  (0xful << RTC_TALM_HR_Pos)                        /*!< RTC_T::TALM: HR Mask                      */

#define RTC_TALM_TENHR_Pos               (20)                                              /*!< RTC_T::TALM: TENHR Position               */
#define RTC_TALM_TENHR_Msk               (0x3ul << RTC_TALM_TENHR_Pos)                     /*!< RTC_T::TALM: TENHR Mask                   */

#define RTC_TALM_MSKHR_Pos               (23)                                              /*!< RTC_T::TALM: MSKHR Position               */
#define RTC_TALM_MSKHR_Msk               (0x1ul << RTC_TALM_MSKHR_Pos)                     /*!< RTC_T::TALM: MSKHR Mask                   */

#define RTC_CALM_DAY_Pos                 (0)                                               /*!< RTC_T::CALM: DAY Position                 */
#define RTC_CALM_DAY_Msk                 (0xful << RTC_CALM_DAY_Pos)                       /*!< RTC_T::CALM: DAY Mask                     */

#define RTC_CALM_TENDAY_Pos              (4)                                               /*!< RTC_T::CALM: TENDAY Position              */
#define RTC_CALM_TENDAY_Msk              (0x3ul << RTC_CALM_TENDAY_Pos)                    /*!< RTC_T::CALM: TENDAY Mask                  */

#define RTC_CALM_MSKDAY_Pos              (7)                                               /*!< RTC_T::CALM: MSKDAY Position              */
#define RTC_CALM_MSKDAY_Msk              (0x1ul << RTC_CALM_MSKDAY_Pos)                    /*!< RTC_T::CALM: MSKDAY Mask                  */

#define RTC_CALM_MON_Pos                 (8)                                               /*!< RTC_T::CALM: MON Position                 */
#define RTC_CALM_MON_Msk                 (0xful << RTC_CALM_MON_Pos)                       /*!< RTC_T::CALM: MON Mask                     */

#define RTC_CALM_TENMON_Pos              (12)                                              /*!< RTC_T::CALM: TENMON Position              */
#define RTC_CALM_TENMON_Msk              (0x1ul << RTC_CALM_TENMON_Pos)                    /*!< RTC_T::CALM: TENMON Mask                  */

#define RTC_CALM_MSKMON_Pos              (15)                                              /*!< RTC_T::CALM: MSKMON Position              */
#define RTC_CALM_MSKMON_Msk              (0x1ul << RTC_CALM_MSKMON_Pos)                    /*!< RTC_T::CALM: MSKMON Mask                  */

#define RTC_CALM_YEAR_Pos                (16)                                              /*!< RTC_T::CALM: YEAR Position                */
#define RTC_CALM_YEAR_Msk                (0xful << RTC_CALM_YEAR_Pos)                      /*!< RTC_T::CALM: YEAR Mask                    */

#define RTC_CALM_TENYEAR_Pos             (20)                                              /*!< RTC_T::CALM: TENYEAR Position             */
#define RTC_CALM_TENYEAR_Msk             (0xful << RTC_CALM_TENYEAR_Pos)                   /*!< RTC_T::CALM: TENYEAR Mask                 */

#define RTC_CALM_MSKYEAR_Pos             (24)                                              /*!< RTC_T::CALM: MSKYEAR Position             */
#define RTC_CALM_MSKYEAR_Msk             (0x1ul << RTC_CALM_MSKYEAR_Pos)                   /*!< RTC_T::CALM: MSKYEAR Mask                 */

#define RTC_CALM_WEEKDAY_Pos             (28)                                              /*!< RTC_T::CALM: WEEKDAY Position             */
#define RTC_CALM_WEEKDAY_Msk             (0x7ul << RTC_CALM_WEEKDAY_Pos)                   /*!< RTC_T::CALM: WEEKDAY Mask                 */

#define RTC_CALM_MSKWEEKDAY_Pos          (31)                                              /*!< RTC_T::CALM: MSKWEEKDAY Position          */
#define RTC_CALM_MSKWEEKDAY_Msk          (0x1ul << RTC_CALM_MSKWEEKDAY_Pos)                /*!< RTC_T::CALM: MSKWEEKDAY Mask              */

#define RTC_LEAPYEAR_LEAPYEAR_Pos        (0)                                               /*!< RTC_T::LEAPYEAR: LEAPYEAR Position        */
#define RTC_LEAPYEAR_LEAPYEAR_Msk        (0x1ul << RTC_LEAPYEAR_LEAPYEAR_Pos)              /*!< RTC_T::LEAPYEAR: LEAPYEAR Mask            */

#define RTC_INTEN_ALMIEN_Pos             (0)                                               /*!< RTC_T::INTEN: ALMIEN Position             */
#define RTC_INTEN_ALMIEN_Msk             (0x1ul << RTC_INTEN_ALMIEN_Pos)                   /*!< RTC_T::INTEN: ALMIEN Mask                 */

#define RTC_INTEN_TICKIEN_Pos            (1)                                               /*!< RTC_T::INTEN: TICKIEN Position            */
#define RTC_INTEN_TICKIEN_Msk            (0x1ul << RTC_INTEN_TICKIEN_Pos)                  /*!< RTC_T::INTEN: TICKIEN Mask                */

#define RTC_INTEN_PKEYIEN_Pos            (2)                                               /*!< RTC_T::INTEN: PKEYIEN Position            */
#define RTC_INTEN_PKEYIEN_Msk            (0x1ul << RTC_INTEN_PKEYIEN_Pos)                  /*!< RTC_T::INTEN: PKEYIEN Mask                */

#define RTC_INTEN_RALMIEN_Pos            (3)                                               /*!< RTC_T::INTEN: RALMIEN Position            */
#define RTC_INTEN_RALMIEN_Msk            (0x1ul << RTC_INTEN_RALMIEN_Pos)                  /*!< RTC_T::INTEN: RALMIEN Mask                */

#define RTC_INTSTS_ALMIF_Pos             (0)                                               /*!< RTC_T::INTSTS: ALMIF Position             */
#define RTC_INTSTS_ALMIF_Msk             (0x1ul << RTC_INTSTS_ALMIF_Pos)                   /*!< RTC_T::INTSTS: ALMIF Mask                 */

#define RTC_INTSTS_TICKIF_Pos            (1)                                               /*!< RTC_T::INTSTS: TICKIF Position            */
#define RTC_INTSTS_TICKIF_Msk            (0x1ul << RTC_INTSTS_TICKIF_Pos)                  /*!< RTC_T::INTSTS: TICKIF Mask                */

#define RTC_INTSTS_POWKEYIF_Pos          (2)                                               /*!< RTC_T::INTSTS: POWKEYIF Position          */
#define RTC_INTSTS_POWKEYIF_Msk          (0x1ul << RTC_INTSTS_POWKEYIF_Pos)                /*!< RTC_T::INTSTS: POWKEYIF Mask              */

#define RTC_INTSTS_RELALMIF_Pos          (3)                                               /*!< RTC_T::INTSTS: RELALMIF Position          */
#define RTC_INTSTS_RELALMIF_Msk          (0x1ul << RTC_INTSTS_RELALMIF_Pos)                /*!< RTC_T::INTSTS: RELALMIF Mask              */

#define RTC_TICK_TICKSEL_Pos             (0)                                               /*!< RTC_T::TICK: TICKSEL Position             */
#define RTC_TICK_TICKSEL_Msk             (0x7ul << RTC_TICK_TICKSEL_Pos)                   /*!< RTC_T::TICK: TICKSEL Mask                 */

#define RTC_POWCTL_POWEN_Pos             (0)                                               /*!< RTC_T::POWCTL: POWEN Position             */
#define RTC_POWCTL_POWEN_Msk             (0x1ul << RTC_POWCTL_POWEN_Pos)                   /*!< RTC_T::POWCTL: POWEN Mask                 */

#define RTC_POWCTL_SWPOWOFF_Pos          (1)                                               /*!< RTC_T::POWCTL: SWPOWOFF Position          */
#define RTC_POWCTL_SWPOWOFF_Msk          (0x1ul << RTC_POWCTL_SWPOWOFF_Pos)                /*!< RTC_T::POWCTL: SWPOWOFF Mask              */

#define RTC_POWCTL_POWOFFEN_Pos          (2)                                               /*!< RTC_T::POWCTL: POWOFFEN Position          */
#define RTC_POWCTL_POWOFFEN_Msk          (0x1ul << RTC_POWCTL_POWOFFEN_Pos)                /*!< RTC_T::POWCTL: POWOFFEN Mask              */

#define RTC_POWCTL_ALMIEN_Pos            (3)                                               /*!< RTC_T::POWCTL: ALMIEN Position            */
#define RTC_POWCTL_ALMIEN_Msk            (0x1ul << RTC_POWCTL_ALMIEN_Pos)                  /*!< RTC_T::POWCTL: ALMIEN Mask                */

#define RTC_POWCTL_RALMIEN_Pos           (4)                                               /*!< RTC_T::POWCTL: RALMIEN Position           */
#define RTC_POWCTL_RALMIEN_Msk           (0x1ul << RTC_POWCTL_RALMIEN_Pos)                 /*!< RTC_T::POWCTL: RALMIEN Mask               */

#define RTC_POWCTL_EDGE_TRIG_Pos         (5)                                               /*!< RTC_T::POWCTL: EDGE_TRIG Position         */
#define RTC_POWCTL_EDGE_TRIG_Msk         (0x1ul << RTC_POWCTL_EDGE_TRIG_Pos)               /*!< RTC_T::POWCTL: EDGE_TRIG Mask             */

#define RTC_POWCTL_POWKEY_Pos            (7)                                               /*!< RTC_T::POWCTL: POWKEY Position            */
#define RTC_POWCTL_POWKEY_Msk            (0x1ul << RTC_POWCTL_POWKEY_Pos)                  /*!< RTC_T::POWCTL: POWKEY Mask                */

#define RTC_POWCTL_POWOFFT_Pos           (16)                                              /*!< RTC_T::POWCTL: POWOFFT Position           */
#define RTC_POWCTL_POWOFFT_Msk           (0xful << RTC_POWCTL_POWOFFT_Pos)                 /*!< RTC_T::POWCTL: POWOFFT Mask               */

#define RTC_POWCTL_RALMTIME_Pos          (20)                                              /*!< RTC_T::POWCTL: RALMTIME Position          */
#define RTC_POWCTL_RALMTIME_Msk          (0xffful << RTC_POWCTL_RALMTIME_Pos)              /*!< RTC_T::POWCTL: RALMTIME Mask              */

#define RTC_SET_CBEN_Pos                 (1)                                               /*!< RTC_T::SET: CBEN Position                 */
#define RTC_SET_CBEN_Msk                 (0x1ul << RTC_SET_CBEN_Pos)                       /*!< RTC_T::SET: CBEN Mask                     */

#define RTC_SET_IOMSEL_Pos               (2)                                               /*!< RTC_T::SET: IOMSEL Position               */
#define RTC_SET_IOMSEL_Msk               (0x1ul << RTC_SET_IOMSEL_Pos)                     /*!< RTC_T::SET: IOMSEL Mask                   */

#define RTC_SET_XININDAT_Pos             (3)                                               /*!< RTC_T::SET: XININDAT Position             */
#define RTC_SET_XININDAT_Msk             (0x1ul << RTC_SET_XININDAT_Pos)                   /*!< RTC_T::SET: XININDAT Mask                 */

#define RTC_SET_XOUTDAT_Pos              (4)                                               /*!< RTC_T::SET: XOUTDAT Position              */
#define RTC_SET_XOUTDAT_Msk              (0x1ul << RTC_SET_XOUTDAT_Pos)                    /*!< RTC_T::SET: XOUTDAT Mask                  */

#define RTC_CLKSRC_CKSRC_Pos             (0)                                               /*!< RTC_T::CLKSRC: CKSRC Position             */
#define RTC_CLKSRC_CKSRC_Msk             (0x1ul << RTC_CLKSRC_CKSRC_Pos)                   /*!< RTC_T::CLKSRC: CKSRC Mask                 */

#define RTC_CALCNT_CALCNT_Pos            (0)                                               /*!< RTC_T::CALCNT: CALCNT Position            */
#define RTC_CALCNT_CALCNT_Msk            (0xfffffffful << RTC_CALCNT_CALCNT_Pos)           /*!< RTC_T::CALCNT: CALCNT Mask                */

#define RTC_SYNC_SYNC_Pos                (0)                                               /*!< RTC_T::SYNC: SYNC Position                */
#define RTC_SYNC_SYNC_Msk                (0x1ul << RTC_SYNC_SYNC_Pos)                      /*!< RTC_T::SYNC: SYNC Mask                    */

#define RTC_SPRn_RTC_SPARn_Pos           (0)                                               /*!< RTC_T::SPRn: RTC_SPARn Position           */
#define RTC_SPRn_RTC_SPARn_Msk           (0xfffffffful << RTC_SPRn_RTC_SPARn_Pos)          /*!< RTC_T::SPRn: RTC_SPARn Mask               */

/**@}*/ /* RTC_CONST */
/**@}*/ /* end of RTC register group */


/*---------------------- SD Card Host Interface -------------------------*/
/**
    @addtogroup SDH SD Card Host Interface(SDH)
    Memory Mapped Structure for SDH Controller
@{ */

typedef struct
{


    /**
     * FBx
     * ===================================================================================================
     * Offset: 0x00 ~ 0x1F Shared Buffer (FIFO) Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DATA      |SDH Receiving/Transmit Flash Buffer
     * |        |          |This buffer is used to receive/transmit data. It can be accessed by CPU or DMAC.
    */
    uint32_t FB[32];
    __I  uint32_t RESERVE0[224];


    /**
     * DMACTL
     * ===================================================================================================
     * Offset: 0x400  DMAC Control and Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DMAEN     |DMAC Engine Enable Control
     * |        |          |0 = DMAC Disabled.
     * |        |          |1 = DMAC Enabled.
     * |        |          |Note1: If this bit is cleared, DMAC will ignore all DMA request and force Bus Master into IDLE state.
     * |        |          |Note2: If a target abort occurs, DMAEN will be cleared.
     * |[1]     |DMARST    |Software Engine Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset internal state machine and pointers.
     * |        |          |The contents of control register will not be cleared.
     * |        |          |This bit will auto be cleared after few clock cycle.
     * |        |          |Note: Software resets DMA region.
     * |[3]     |SGEN      |Scatter-Gather Function Enable Control
     * |        |          |0 = Scatter-gather function Disabled (DMAC will treat the starting address in SDH_DMASA as starting pointer of a single block memory).
     * |        |          |1 = Scatter-gather function Enabled (DMAC will treat the starting address in SDH_DMASA as a starting address of Physical Address Descriptor (PAD) table.
     * |        |          |The format of these PADs' will be described later.
     * |[9]     |DMABUSY   |DMA Transfer In Progress
     * |        |          |This bit indicates that the DMA is transferred or not.
     * |        |          |0 = DMA transfer is not in progress.
     * |        |          |1 = DMA transfer is in progress.
    */
    __IO uint32_t DMACTL;
    uint32_t RESERVE1[1];


    /**
     * DMASA
     * ===================================================================================================
     * Offset: 0x408  DMAC Transfer Starting Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ORDER     |Determined To The PAD Table Fetching In Order Or Out Of Order
     * |        |          |0 = PAD table is fetched in order.
     * |        |          |1 = PAD table is fetched out of order.
     * |        |          |Note: The bit0 is valid in scatter-gather mode when SGEN = 1.
     * |[31:1]  |DMASA     |DMA Transfer Starting Address
     * |        |          |This field pads 0 as least significant bit indicates a 32-bit starting address of system memory (SRAM) for DMAC to retrieve or fill in data.
     * |        |          |If DMAC is not in Normal mode, this field will be interpreted as a starting address of Physical Address Descriptor (PAD) table.
    */
    __IO uint32_t DMASA;

    /**
     * DMABCNT
     * ===================================================================================================
     * Offset: 0x40C  DMAC Transfer Byte Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[25:0]  |BCNT      |DMA Transfer Byte Count (Read Only)
     * |        |          |This field indicates the remained byte count of DMAC transfer.
     * |        |          |The value of this field is valid only when DMAC is busy; otherwise, it is .
    */
    __I  uint32_t DMABCNT;

    /**
     * DMAINTEN
     * ===================================================================================================
     * Offset: 0x410  DMAC Interrupt Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ABORTIEN  |DMA Read/Write Target Abort Interrupt Enable Control
     * |        |          |0 = Target abort interrupt generation Disabled during DMA transfer.
     * |        |          |1 = Target abort interrupt generation Enabled during DMA transfer.
     * |[1]     |WEOTIEN   |Wrong EOT Encountered Interrupt Enable Control
     * |        |          |0 = Interrupt generation Disabled when wrong EOT is encountered.
     * |        |          |1 = Interrupt generation Enabled when wrong EOT is encountered.
    */
    __IO uint32_t DMAINTEN;

    /**
     * DMAINTSTS
     * ===================================================================================================
     * Offset: 0x414  DMAC Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |ABORTIF   |DMA Read/Write Target Abort Interrupt Flag
     * |        |          |0 = No bus ERROR response received.
     * |        |          |1 = Bus ERROR response received.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[1]     |WEOTIF    |Wrong EOT Encountered Interrupt Flag
     * |        |          |When DMA Scatter-Gather function is enabled, and EOT of the descriptor is encountered before DMA transfer finished (that means the total sector count of all PAD is less than the sector count of DMAC), this bit will be set.
     * |        |          |0 = No EOT encountered before DMA transfer finished.
     * |        |          |1 = EOT encountered before DMA transfer finished.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
    */
    __IO uint32_t DMAINTSTS;
    uint32_t RESERVE2[250];


    /**
     * GCTL
     * ===================================================================================================
     * Offset: 0x800  Global Control and Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |GCTLRST   |Software Engine Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset all SDH engines.
     * |        |          |The contents of control registers will not be cleared.
     * |        |          |This bit will auto cleared after few clock cycle.
     * |[1]     |SDEN      |Secure-Digital Functionality Enable Control
     * |        |          |0 = SD functionality Disabled.
     * |        |          |1 = SD functionality Enabled.
    */
    __IO uint32_t GCTL;

    /**
     * GINTEN
     * ===================================================================================================
     * Offset: 0x804  Global Interrupt Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DTAIEN    |DMAC READ/WRITE Target Abort Interrupt Enable Control
     * |        |          |0 = DMAC READ/WRITE target abort interrupt generation Disabled.
     * |        |          |1 = DMAC READ/WRITE target abort interrupt generation Enabled.
    */
    __IO uint32_t GINTEN;

    /**
     * GINTSTS
     * ===================================================================================================
     * Offset: 0x808  Global Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DTAIF     |DMAC READ/WRITE Target Abort Interrupt Flag (Read Only)
     * |        |          |This bit indicates DMAC received an ERROR response from internal AHB bus during DMA read/write operation.
     * |        |          |When Target Abort is occurred, please reset all engine.
     * |        |          |0 = No bus ERROR response received.
     * |        |          |1 = Bus ERROR response received.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
    */
    __IO  uint32_t GINTSTS;
    uint32_t RESERVE3[5];


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x820  SD Control and Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |COEN      |Command Output Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, SD host will output a command to SD card.
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[1]     |RIEN      |Response Input Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, SD host will wait to receive a response from SD card.
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[2]     |DIEN      |Data Input Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, SD host will wait to receive block data and the CRC-16 value from SD card.
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[3]     |DOEN      |Data Output Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, SD host will transfer block data and the CRC-16 value to SD card.
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[4]     |R2EN      |Response R2 Input Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, SD host will wait to receive a response R2 from SD card and store the response data into DMAC's flash buffer (exclude CRC-7).
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[5]     |CLK74OEN  |Generating 74 Clock Cycles Output Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, SD host will output 74 clock cycles to SD card.
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[6]     |CLK8OEN   |Generating 8 Clock Cycles Output Enable Control
     * |        |          |0 = No effect. (Please use SDH_CTL[CTLRST] to clear this bit.)
     * |        |          |1 = Enabled, and SD host will output 8 clock cycles.
     * |        |          |Note: When operation is finished, this bit will be cleared automatically.
     * |        |          |Thus, don't write 0 to this bit (the controller will be abnormal.
     * |[7]     |CLKKEEP0  |SD Clock Enable Control For Port 0
     * |        |          |0 = SD host decides when to output clock and when to disable clock output automatically.
     * |        |          |1 = SD clock always keeps free running.
     * |[13:8]  |CMDCODE   |SD Command Code
     * |        |          |This register contains the SD command code (0x00 - 0x3F).
     * |[14]    |CTLRST    |Software Engine Reset
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the internal state machine and counters.
     * |        |          |The contents of control register will not be cleared (but RIEN, DIEN, DOEN and R2EN will be cleared).
     * |        |          |This bit will be auto cleared after few clock cycle.
     * |[15]    |DBW       |SD Data Bus Width (For 1-Bit / 4-Bit Selection)
     * |        |          |0 = Data bus width is 1-bit.
     * |        |          |1 = Data bus width is 4-bit.
     * |[23:16] |BLKCNT    |Block Counts To Be Transferred Or Received
     * |        |          |This field contains the block counts for data-in and data-out transfer.
     * |        |          |For READ_MULTIPLE_BLOCK and WRITE_MULTIPLE_BLOCK command, software can use this function to accelerate data transfer and improve performance.
     * |        |          |Don't fill 0x0 to this fiel.
     * |        |          |For READ_MULTIPLE_BLOCK and WRITE_MULTIPLE_BLOCK command, the actual total length is BLKCNT * (BLKLEN +1).
     * |[27:24] |SDNWR     |NWR Parameter For Block Write Operation
     * |        |          |This value indicates the NWR parameter for data block write operation in SD clock counts.
     * |        |          |The actual clock cycle will be SDNWR+.
    */
    __IO uint32_t CTL;

    /**
     * CMDARG
     * ===================================================================================================
     * Offset: 0x824  SD Command Argument Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ARGUMENT  |SD Command Argument
     * |        |          |This register contains a 32-bit value specifies the argument of SD command from host controller to SD card.
     * |        |          |Before triggering SDH_CTL [COEN], software should fill argument in this fiel.
    */
    __IO uint32_t CMDARG;

    /**
     * INTEN
     * ===================================================================================================
     * Offset: 0x828  SD Interrupt Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BLKDIEN   |Block Transfer Done Interrupt Enable Control
     * |        |          |0 = SD host will not generate interrupt when data-in (out) transfer done.
     * |        |          |1 = SD host will generate interrupt when data-in (out) transfer done.
     * |[1]     |CRCIEN    |CRC7, CRC16 And CRC Status Error Interrupt Enable Control
     * |        |          |0 = SD host will not generate interrupt when CRC7, CRC16 and CRC status is error.
     * |        |          |1 = SD host will generate interrupt when CRC7, CRC16 and CRC status is error.
     * |[8]     |CDIEN0    |SD0 Card Detection Interrupt Enable Control
     * |        |          |Enable/Disable interrupt generation of SD controller when card 0 is inserted or removed.
     * |        |          |0 = SD0 Card Detection Interrupt Disabled.
     * |        |          |1 = SD0 Card Detection Interrupt Enabled.
     * |[10]    |SDHIEN0   |SDH Interrupt Enable Control For Port 0
     * |        |          |Enable/Disable interrupt generation of SD host when SDH card 0 issues an interrupt via DAT [1] to host.
     * |        |          |0 = SDH Port 0 Interrupt Disabled.
     * |        |          |1 = SDH Port 0 Interrupt Enabled.
     * |[12]    |RTOIEN    |Response Time-out Interrupt Enable Control
     * |        |          |Enable/Disable interrupt generation of SD controller when receiving response or R2 time-out.
     * |        |          |Time-out value is specified at TOU.
     * |        |          |0 = Response Time-out Interrupt Disabled.
     * |        |          |1 = Response Time-out Interrupt Enabled.
     * |[13]    |DITOIEN   |Data Input Time-out Interrupt Enable Control
     * |        |          |Enable/Disable interrupt generation of SD controller when data input time-out.
     * |        |          |Time-out value is specified at TOU.
     * |        |          |0 = Data Input Time-out Interrupt Disabled.
     * |        |          |1 = Data Input Time-out Interrupt Enabled.
     * |[30]    |CDSRC0    |SD0 Card Detect Source Selection
     * |        |          |0 = From SD0 card's DAT3 pin.
     * |        |          |Host need clock to get data on pin DAT3.
     * |        |          |Please make sure SDH_CTL[CLKKEEP0] is 1 in order to generate free running clock for DAT3 pi.
     * |        |          |1 = From GPIO pin.
    */
    __IO uint32_t INTEN;

    /**
     * INTSTS
     * ===================================================================================================
     * Offset: 0x82C  SD Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BLKDIF    |Block Transfer Done Interrupt Flag (Read Only)
     * |        |          |This bit indicates that SD host has finished all data-in or data-out block transfer.
     * |        |          |If there is a CRC16 error or incorrect CRC status during multiple block data transfer, the transfer will be broken and this bit will also be se.
     * |        |          |0 = Not finished yet.
     * |        |          |1 = Done.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[1]     |CRCIF     |CRC7, CRC16 And CRC Status Error Interrupt Flag (Read Only)
     * |        |          |This bit indicates that SD host has occurred CRC error during response in, data-in or data-out (CRC status error) transfer.
     * |        |          |When CRC error occurred, software should reset SD engine.
     * |        |          |Some responses (e.g.
     * |        |          |R3) do not have CRC-7 information with it; the SD host will still calculate CRC-7, get CRC error and set this flag.
     * |        |          |In this condition, software should ignore CRC error and clear this bit manuall.
     * |        |          |0 = No CRC error is occurred.
     * |        |          |1 = CRC error is occurred.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[2]     |CRC7      |CRC-7 Check Status (Read Only)
     * |        |          |SD host will check CRC-7 correctness during each response in.
     * |        |          |If that response does not contain CRC-7 information (e.g.
     * |        |          |R3), then software should turn off CRCIEN (SDH_INTEN[1]) and ignore this bi.
     * |        |          |0 = CRC-7 Transfer incorrectness.
     * |        |          |1 = CRC-7 Transfer correctness.
     * |[3]     |CRC16     |CRC-16 Check Status Of Data-In Transfer (Read Only)
     * |        |          |SD host will check CRC-16 correctness after data-in transfer.
     * |        |          |0 = CRC-16 Transfer incorrectness.
     * |        |          |1 = CRC-16 Transfer correctness.
     * |[6:4]   |CRCSTS    |CRC Status Value Of Data-Out Transfer (Read Only)
     * |        |          |SD host will record CRC status of data-out transfer.
     * |        |          |Software could use this value to identify what type of error is during data-out transfe.
     * |        |          |010 = Positive CRC status.
     * |        |          |101 = Negative CRC status.
     * |        |          |111 = SD card programming error occurs.
     * |        |          |Others = Reserved.
     * |[7]     |DAT0STS   |DAT0 Pin Status Of Current Selected SD Port (Read Only)
     * |        |          |This bit is the DAT0 pin status of current selected SD port.
     * |[8]     |CDIF0     |SD0 Card Detection Interrupt Flag (Read Only)
     * |        |          |This bit indicates that SD card 0 is inserted or removed.
     * |        |          |Only when CDIEN0 (SDH_INTEN[8]) is set to 1, this bit is activ.
     * |        |          |0 = No card is inserted or removed.
     * |        |          |1 = There is a card inserted in or removed from SD0.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[10]    |SDHIF0    |SDH 0 Interrupt Flag (Read Only)
     * |        |          |This bit indicates that SDH card 0 issues an interrupt to host.
     * |        |          |This interrupt is designed to level sensitive.
     * |        |          |Before clearing it, turn off SDHIEN0 (SDH_INTEN[10]) firs.
     * |        |          |0 = No interrupt is issued by SDH card 0.
     * |        |          |1 = an interrupt is issued by SDH card 0.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[12]    |RTOIF     |Response Time-out Interrupt Flag (Read Only)
     * |        |          |This bit indicates that SD host counts to time-out value when receiving response or R2 (waiting start bit).
     * |        |          |0 = No time-out.
     * |        |          |1 = Response time-out.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[13]    |DITOIF    |Data Input Time-out Interrupt Flag (Read Only)
     * |        |          |This bit indicates that SD host counts to time-out value when receiving data (waiting start bit).
     * |        |          |0 = No time-out.
     * |        |          |1 = Data input time-out.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[16]    |CDSTS0    |Card Detect Status Of SD0 (Read Only)
     * |        |          |This bit indicates the card detect pin status of SD0, and is used for card detection.
     * |        |          |When there is a card inserted in or removed from SD0, software should check this bit to confirm if there is really a card insertion or remova.
     * |        |          |If CDSRC0 (SDH_INTEN[30]) = 0, to select DAT3 for card detection:
     * |        |          |0 = Card removed.
     * |        |          |1 = Card inserted.
     * |        |          |If CDSRC0 (SDH_INTEN[30]) = 1, to select GPIO for card detection:
     * |        |          |0 = Card inserted.
     * |        |          |1 = Card removed.
     * |[18]    |DAT1STS   |DAT1 Pin Status Of SD Port (Read Only)
     * |        |          |This bit indicates the DAT1 pin status of SD port.
    */
    __IO  uint32_t INTSTS;

    /**
     * RESP0
     * ===================================================================================================
     * Offset: 0x830  SD Receiving Response Token Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RESPTK0   |SD Receiving Response Token 0
     * |        |          |SD host controller will receive a response token for getting a reply from SD card when RIEN (SDH_CTL[1]) is set.
     * |        |          |This field contains response bit 47-16 of the response toke.
    */
    __I  uint32_t RESP0;

    /**
     * RESP1
     * ===================================================================================================
     * Offset: 0x834  SD Receiving Response Token Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |RESPTK1   |SD Receiving Response Token 1
     * |        |          |SD host controller will receive a response token for getting a reply from SD card when RIEN (SDH_CTL[1]) is set.
     * |        |          |This register contains the bit 15-8 of the response toke.
    */
    __I  uint32_t RESP1;

    /**
     * BLEN
     * ===================================================================================================
     * Offset: 0x838  SD Block Length Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[10:0]  |BLKLEN    |SD BLOCK LENGTH In Byte Unit
     * |        |          |An 11-bit value specifies the SD transfer byte count of a block.
     * |        |          |The actual byte count is equal to BLKLEN +.
     * |        |          |Note: The default SD block length is 512 bytes.
    */
    __IO uint32_t BLEN;

    /**
     * TOUT
     * ===================================================================================================
     * Offset: 0x83C  SD Response/Data-in Time-out Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |TOUT      |SD Response/Data-In Time-Out Value
     * |        |          |A 24-bit value specifies the time-out count of response and data input.
     * |        |          |SD host controller will wait the start bit of response or data-in until this value reached.
     * |        |          |The time period depends on SD engine clock frequency.
     * |        |          |Do not write a small number into this field, or you may never get response or data due to time-ou.
     * |        |          |Note: Filling 0x0 into this field will disable hardware time-out function.
    */
    __IO uint32_t TOUT;

} SDH_T;

/**
    @addtogroup SDH_CONST SDH Bit Field Definition
    Constant Definitions for SDH Controller
@{ */

#define SDH_DMACTL_DMAEN_Pos             (0)                                               /*!< SDH_T::DMACTL: DMAEN Position             */
#define SDH_DMACTL_DMAEN_Msk             (0x1ul << SDH_DMACTL_DMAEN_Pos)                   /*!< SDH_T::DMACTL: DMAEN Mask                 */

#define SDH_DMACTL_DMARST_Pos            (1)                                               /*!< SDH_T::DMACTL: DMARST Position            */
#define SDH_DMACTL_DMARST_Msk            (0x1ul << SDH_DMACTL_DMARST_Pos)                  /*!< SDH_T::DMACTL: DMARST Mask                */

#define SDH_DMACTL_SGEN_Pos              (3)                                               /*!< SDH_T::DMACTL: SGEN Position              */
#define SDH_DMACTL_SGEN_Msk              (0x1ul << SDH_DMACTL_SGEN_Pos)                    /*!< SDH_T::DMACTL: SGEN Mask                  */

#define SDH_DMACTL_DMABUSY_Pos           (9)                                               /*!< SDH_T::DMACTL: DMABUSY Position           */
#define SDH_DMACTL_DMABUSY_Msk           (0x1ul << SDH_DMACTL_DMABUSY_Pos)                 /*!< SDH_T::DMACTL: DMABUSY Mask               */

#define SDH_DMASA_ORDER_Pos              (0)                                               /*!< SDH_T::DMASA: ORDER Position              */
#define SDH_DMASA_ORDER_Msk              (0x1ul << SDH_DMASA_ORDER_Pos)                    /*!< SDH_T::DMASA: ORDER Mask                  */

#define SDH_DMASA_DMASA_Pos              (1)                                               /*!< SDH_T::DMASA: DMASA Position              */
#define SDH_DMASA_DMASA_Msk              (0x7ffffffful << SDH_DMASA_DMASA_Pos)             /*!< SDH_T::DMASA: DMASA Mask                  */

#define SDH_DMABCNT_BCNT_Pos             (0)                                               /*!< SDH_T::DMABCNT: BCNT Position             */
#define SDH_DMABCNT_BCNT_Msk             (0x3fffffful << SDH_DMABCNT_BCNT_Pos)             /*!< SDH_T::DMABCNT: BCNT Mask                 */

#define SDH_DMAINTEN_ABORTIEN_Pos        (0)                                               /*!< SDH_T::DMAINTEN: ABORTIEN Position        */
#define SDH_DMAINTEN_ABORTIEN_Msk        (0x1ul << SDH_DMAINTEN_ABORTIEN_Pos)              /*!< SDH_T::DMAINTEN: ABORTIEN Mask            */

#define SDH_DMAINTEN_WEOTIEN_Pos         (1)                                               /*!< SDH_T::DMAINTEN: WEOTIEN Position         */
#define SDH_DMAINTEN_WEOTIEN_Msk         (0x1ul << SDH_DMAINTEN_WEOTIEN_Pos)               /*!< SDH_T::DMAINTEN: WEOTIEN Mask             */

#define SDH_DMAINTSTS_ABORTIF_Pos        (0)                                               /*!< SDH_T::DMAINTSTS: ABORTIF Position        */
#define SDH_DMAINTSTS_ABORTIF_Msk        (0x1ul << SDH_DMAINTSTS_ABORTIF_Pos)              /*!< SDH_T::DMAINTSTS: ABORTIF Mask            */

#define SDH_DMAINTSTS_WEOTIF_Pos         (1)                                               /*!< SDH_T::DMAINTSTS: WEOTIF Position         */
#define SDH_DMAINTSTS_WEOTIF_Msk         (0x1ul << SDH_DMAINTSTS_WEOTIF_Pos)               /*!< SDH_T::DMAINTSTS: WEOTIF Mask             */

#define SDH_GCTL_GCTLRST_Pos             (0)                                               /*!< SDH_T::GCTL: GCTLRST Position             */
#define SDH_GCTL_GCTLRST_Msk             (0x1ul << SDH_GCTL_GCTLRST_Pos)                   /*!< SDH_T::GCTL: GCTLRST Mask                 */

#define SDH_GCTL_SDEN_Pos                (1)                                               /*!< SDH_T::GCTL: SDEN Position                */
#define SDH_GCTL_SDEN_Msk                (0x1ul << SDH_GCTL_SDEN_Pos)                      /*!< SDH_T::GCTL: SDEN Mask                    */

#define SDH_GINTEN_DTAIEN_Pos            (0)                                               /*!< SDH_T::GINTEN: DTAIEN Position            */
#define SDH_GINTEN_DTAIEN_Msk            (0x1ul << SDH_GINTEN_DTAIEN_Pos)                  /*!< SDH_T::GINTEN: DTAIEN Mask                */

#define SDH_GINTSTS_DTAIF_Pos            (0)                                               /*!< SDH_T::GINTSTS: DTAIF Position            */
#define SDH_GINTSTS_DTAIF_Msk            (0x1ul << SDH_GINTSTS_DTAIF_Pos)                  /*!< SDH_T::GINTSTS: DTAIF Mask                */

#define SDH_CTL_COEN_Pos                 (0)                                               /*!< SDH_T::CTL: COEN Position                 */
#define SDH_CTL_COEN_Msk                 (0x1ul << SDH_CTL_COEN_Pos)                       /*!< SDH_T::CTL: COEN Mask                     */

#define SDH_CTL_RIEN_Pos                 (1)                                               /*!< SDH_T::CTL: RIEN Position                 */
#define SDH_CTL_RIEN_Msk                 (0x1ul << SDH_CTL_RIEN_Pos)                       /*!< SDH_T::CTL: RIEN Mask                     */

#define SDH_CTL_DIEN_Pos                 (2)                                               /*!< SDH_T::CTL: DIEN Position                 */
#define SDH_CTL_DIEN_Msk                 (0x1ul << SDH_CTL_DIEN_Pos)                       /*!< SDH_T::CTL: DIEN Mask                     */

#define SDH_CTL_DOEN_Pos                 (3)                                               /*!< SDH_T::CTL: DOEN Position                 */
#define SDH_CTL_DOEN_Msk                 (0x1ul << SDH_CTL_DOEN_Pos)                       /*!< SDH_T::CTL: DOEN Mask                     */

#define SDH_CTL_R2EN_Pos                 (4)                                               /*!< SDH_T::CTL: R2EN Position                 */
#define SDH_CTL_R2EN_Msk                 (0x1ul << SDH_CTL_R2EN_Pos)                       /*!< SDH_T::CTL: R2EN Mask                     */

#define SDH_CTL_CLK74OEN_Pos             (5)                                               /*!< SDH_T::CTL: CLK74OEN Position             */
#define SDH_CTL_CLK74OEN_Msk             (0x1ul << SDH_CTL_CLK74OEN_Pos)                   /*!< SDH_T::CTL: CLK74OEN Mask                 */

#define SDH_CTL_CLK8OEN_Pos              (6)                                               /*!< SDH_T::CTL: CLK8OEN Position              */
#define SDH_CTL_CLK8OEN_Msk              (0x1ul << SDH_CTL_CLK8OEN_Pos)                    /*!< SDH_T::CTL: CLK8OEN Mask                  */

#define SDH_CTL_CLKKEEP0_Pos             (7)                                               /*!< SDH_T::CTL: CLKKEEP0 Position             */
#define SDH_CTL_CLKKEEP0_Msk             (0x1ul << SDH_CTL_CLKKEEP0_Pos)                   /*!< SDH_T::CTL: CLKKEEP0 Mask                 */

#define SDH_CTL_CMDCODE_Pos              (8)                                               /*!< SDH_T::CTL: CMDCODE Position              */
#define SDH_CTL_CMDCODE_Msk              (0x3ful << SDH_CTL_CMDCODE_Pos)                   /*!< SDH_T::CTL: CMDCODE Mask                  */

#define SDH_CTL_CTLRST_Pos               (14)                                              /*!< SDH_T::CTL: CTLRST Position               */
#define SDH_CTL_CTLRST_Msk               (0x1ul << SDH_CTL_CTLRST_Pos)                     /*!< SDH_T::CTL: CTLRST Mask                   */

#define SDH_CTL_DBW_Pos                  (15)                                              /*!< SDH_T::CTL: DBW Position                  */
#define SDH_CTL_DBW_Msk                  (0x1ul << SDH_CTL_DBW_Pos)                        /*!< SDH_T::CTL: DBW Mask                      */

#define SDH_CTL_BLKCNT_Pos               (16)                                              /*!< SDH_T::CTL: BLKCNT Position               */
#define SDH_CTL_BLKCNT_Msk               (0xfful << SDH_CTL_BLKCNT_Pos)                    /*!< SDH_T::CTL: BLKCNT Mask                   */

#define SDH_CTL_SDNWR_Pos                (24)                                              /*!< SDH_T::CTL: SDNWR Position                */
#define SDH_CTL_SDNWR_Msk                (0xful << SDH_CTL_SDNWR_Pos)                      /*!< SDH_T::CTL: SDNWR Mask                    */

#define SDH_CMDARG_ARGUMENT_Pos          (0)                                               /*!< SDH_T::CMDARG: ARGUMENT Position          */
#define SDH_CMDARG_ARGUMENT_Msk          (0xfffffffful << SDH_CMDARG_ARGUMENT_Pos)         /*!< SDH_T::CMDARG: ARGUMENT Mask              */

#define SDH_INTEN_BLKDIEN_Pos            (0)                                               /*!< SDH_T::INTEN: BLKDIEN Position            */
#define SDH_INTEN_BLKDIEN_Msk            (0x1ul << SDH_INTEN_BLKDIEN_Pos)                  /*!< SDH_T::INTEN: BLKDIEN Mask                */

#define SDH_INTEN_CRCIEN_Pos             (1)                                               /*!< SDH_T::INTEN: CRCIEN Position             */
#define SDH_INTEN_CRCIEN_Msk             (0x1ul << SDH_INTEN_CRCIEN_Pos)                   /*!< SDH_T::INTEN: CRCIEN Mask                 */

#define SDH_INTEN_CDIEN0_Pos             (8)                                               /*!< SDH_T::INTEN: CDIEN0 Position             */
#define SDH_INTEN_CDIEN0_Msk             (0x1ul << SDH_INTEN_CDIEN0_Pos)                   /*!< SDH_T::INTEN: CDIEN0 Mask                 */

#define SDH_INTEN_SDHIEN0_Pos            (10)                                              /*!< SDH_T::INTEN: SDHIEN0 Position            */
#define SDH_INTEN_SDHIEN0_Msk            (0x1ul << SDH_INTEN_SDHIEN0_Pos)                  /*!< SDH_T::INTEN: SDHIEN0 Mask                */

#define SDH_INTEN_RTOIEN_Pos             (12)                                              /*!< SDH_T::INTEN: RTOIEN Position             */
#define SDH_INTEN_RTOIEN_Msk             (0x1ul << SDH_INTEN_RTOIEN_Pos)                   /*!< SDH_T::INTEN: RTOIEN Mask                 */

#define SDH_INTEN_DITOIEN_Pos            (13)                                              /*!< SDH_T::INTEN: DITOIEN Position            */
#define SDH_INTEN_DITOIEN_Msk            (0x1ul << SDH_INTEN_DITOIEN_Pos)                  /*!< SDH_T::INTEN: DITOIEN Mask                */

#define SDH_INTEN_CDSRC0_Pos             (30)                                              /*!< SDH_T::INTEN: CDSRC0 Position             */
#define SDH_INTEN_CDSRC0_Msk             (0x1ul << SDH_INTEN_CDSRC0_Pos)                   /*!< SDH_T::INTEN: CDSRC0 Mask                 */

#define SDH_INTSTS_BLKDIF_Pos            (0)                                               /*!< SDH_T::INTSTS: BLKDIF Position            */
#define SDH_INTSTS_BLKDIF_Msk            (0x1ul << SDH_INTSTS_BLKDIF_Pos)                  /*!< SDH_T::INTSTS: BLKDIF Mask                */

#define SDH_INTSTS_CRCIF_Pos             (1)                                               /*!< SDH_T::INTSTS: CRCIF Position             */
#define SDH_INTSTS_CRCIF_Msk             (0x1ul << SDH_INTSTS_CRCIF_Pos)                   /*!< SDH_T::INTSTS: CRCIF Mask                 */

#define SDH_INTSTS_CRC7_Pos              (2)                                               /*!< SDH_T::INTSTS: CRC7 Position              */
#define SDH_INTSTS_CRC7_Msk              (0x1ul << SDH_INTSTS_CRC7_Pos)                    /*!< SDH_T::INTSTS: CRC7 Mask                  */

#define SDH_INTSTS_CRC16_Pos             (3)                                               /*!< SDH_T::INTSTS: CRC16 Position             */
#define SDH_INTSTS_CRC16_Msk             (0x1ul << SDH_INTSTS_CRC16_Pos)                   /*!< SDH_T::INTSTS: CRC16 Mask                 */

#define SDH_INTSTS_CRCSTS_Pos            (4)                                               /*!< SDH_T::INTSTS: CRCSTS Position            */
#define SDH_INTSTS_CRCSTS_Msk            (0x7ul << SDH_INTSTS_CRCSTS_Pos)                  /*!< SDH_T::INTSTS: CRCSTS Mask                */

#define SDH_INTSTS_DAT0STS_Pos           (7)                                               /*!< SDH_T::INTSTS: DAT0STS Position           */
#define SDH_INTSTS_DAT0STS_Msk           (0x1ul << SDH_INTSTS_DAT0STS_Pos)                 /*!< SDH_T::INTSTS: DAT0STS Mask               */

#define SDH_INTSTS_CDIF0_Pos             (8)                                               /*!< SDH_T::INTSTS: CDIF0 Position             */
#define SDH_INTSTS_CDIF0_Msk             (0x1ul << SDH_INTSTS_CDIF0_Pos)                   /*!< SDH_T::INTSTS: CDIF0 Mask                 */

#define SDH_INTSTS_SDHIF0_Pos            (10)                                              /*!< SDH_T::INTSTS: SDHIF0 Position            */
#define SDH_INTSTS_SDHIF0_Msk            (0x1ul << SDH_INTSTS_SDHIF0_Pos)                  /*!< SDH_T::INTSTS: SDHIF0 Mask                */

#define SDH_INTSTS_RTOIF_Pos             (12)                                              /*!< SDH_T::INTSTS: RTOIF Position             */
#define SDH_INTSTS_RTOIF_Msk             (0x1ul << SDH_INTSTS_RTOIF_Pos)                   /*!< SDH_T::INTSTS: RTOIF Mask                 */

#define SDH_INTSTS_DITOIF_Pos            (13)                                              /*!< SDH_T::INTSTS: DITOIF Position            */
#define SDH_INTSTS_DITOIF_Msk            (0x1ul << SDH_INTSTS_DITOIF_Pos)                  /*!< SDH_T::INTSTS: DITOIF Mask                */

#define SDH_INTSTS_CDSTS0_Pos            (16)                                              /*!< SDH_T::INTSTS: CDSTS0 Position            */
#define SDH_INTSTS_CDSTS0_Msk            (0x1ul << SDH_INTSTS_CDSTS0_Pos)                  /*!< SDH_T::INTSTS: CDSTS0 Mask                */

#define SDH_INTSTS_DAT1STS_Pos           (18)                                              /*!< SDH_T::INTSTS: DAT1STS Position           */
#define SDH_INTSTS_DAT1STS_Msk           (0x1ul << SDH_INTSTS_DAT1STS_Pos)                 /*!< SDH_T::INTSTS: DAT1STS Mask               */

#define SDH_RESP0_RESPTK0_Pos            (0)                                               /*!< SDH_T::RESP0: RESPTK0 Position            */
#define SDH_RESP0_RESPTK0_Msk            (0xfffffffful << SDH_RESP0_RESPTK0_Pos)           /*!< SDH_T::RESP0: RESPTK0 Mask                */

#define SDH_RESP1_RESPTK1_Pos            (0)                                               /*!< SDH_T::RESP1: RESPTK1 Position            */
#define SDH_RESP1_RESPTK1_Msk            (0xfful << SDH_RESP1_RESPTK1_Pos)                 /*!< SDH_T::RESP1: RESPTK1 Mask                */

#define SDH_BLEN_BLKLEN_Pos              (0)                                               /*!< SDH_T::BLEN: BLKLEN Position              */
#define SDH_BLEN_BLKLEN_Msk              (0x7fful << SDH_BLEN_BLKLEN_Pos)                  /*!< SDH_T::BLEN: BLKLEN Mask                  */

#define SDH_TOUT_TOUT_Pos                (0)                                               /*!< SDH_T::TOUT: TOUT Position                */
#define SDH_TOUT_TOUT_Msk                (0xfffffful << SDH_TOUT_TOUT_Pos)                 /*!< SDH_T::TOUT: TOUT Mask                    */

/**@}*/ /* SDH_CONST */
/**@}*/ /* end of SDH register group */


/*---------------------- SPI Master -------------------------*/
/**
    @addtogroup SPIM SPI Master(SPIM)
    Memory Mapped Structure for SPIM Controller
@{ */

typedef struct
{


    /**
     * CTL0
     * ===================================================================================================
     * Offset: 0x00  Control and Status Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CIPHOFF   |Cipher Disable Control
     * |        |          |0 = Cipher function Enabled.
     * |        |          |1 = Cipher function Disabled.
     * |        |          |Note1: Cipher function only can be disabled.
     * |        |          |If user wants to enable cipher function after disabled processing, the chip must be reset agai.
     * |        |          |Note2: If there is not any key in the PMOC module, the cipher will be disabled automatically.
     * |        |          |Note3: In ICE mode, the cipher will be disabled automatically.
     * |[2]     |BALEN     |Balance The AHB Control Time Between Cipher Enable And Disable Control
     * |        |          |When cipher is enabled, the AHB control signal will delay some time caused by the encoding or decoding calculation.
     * |        |          |Therefore, if set BALEN to 1, it will make the AHB signal processing time with cipher disabled be equal to that with cipher enable.
     * |        |          |Note: Only useful when cipher is disabled.
     * |[5]     |B4ADDREN  |4-Byte Address Mode Enable Control
     * |        |          |0 = 4-byte address mode Disabled.
     * |        |          |1 = 4-byte address mode Enabled.
     * |        |          |Note: Used for DMA Write/DMA Read/DMM mode.
     * |[6]     |IEN       |Interrupt Enable Control
     * |        |          |0 = SPIM Interrupt Disabled.
     * |        |          |1 = SPIM Interrupt Enabled.
     * |[7]     |IF        |Interrupt Flag
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Write 1 to clear.
     * |        |          |Read Operation:
     * |        |          |0 = The transfer has not finished yet.
     * |        |          |1 = The transfer has done.
     * |[12:8]  |DWIDTH    |Transmit/Receive Bit Length
     * |        |          |This field specifies how many bits are transmitted/received in one transmit/receive transaction.
     * |        |          |0x7 = 8 bits.
     * |        |          |0xF = 16 bits.
     * |        |          |0x17 = 24 bits.
     * |        |          |0x1F = 32 bits.
     * |        |          |Others = Incorrect transfer result.
     * |        |          |Note1: Only used for I/O mode.
     * |        |          |Note2: Only 8-, 16-, 24-, and 32-bit are allowed. Other bit length will result in incorrect transfer.
     * |[14:13] |BURSTNUM  |Transmit/Receive Burst Number
     * |        |          |This field specifies how many transmit/receive transactions should be executed continuously in one transfer.
     * |        |          |00 = Only one transmit/receive transaction will be executed in one transfer.
     * |        |          |01 = Two successive transmit/receive transactions will be executed in one transfer.
     * |        |          |10 = Three successive transmit/receive transactions will be executed in one transfer.
     * |        |          |11 = Four successive transmit/receive transactions will be executed in one transfer.
     * |        |          |Note: Only used for I/O Mode.
     * |[15]    |QDIODIR   |SPI Interface Direction Select For Quad/Dual Mode
     * |        |          |0 = Interface signals are input.
     * |        |          |1 = Interface signals are output.
     * |        |          |Note: Only used for I/O mode.
     * |[19:16] |SUSPITV   |Suspend Interval
     * |        |          |These four bits provide the configuration of suspend interval between two successive transmit/receive transactions in a transfer.
     * |        |          |The default value is 0x00.
     * |        |          |When BURSTNUM = 00, setting this field has no effect on transfer.
     * |        |          |The desired interval is obtained according to the following equation (from the last falling edge of current sclk to the first rising edge of next sclk):
     * |        |          |(SUSPITV+2)*period of SCLK
     * |        |          |0x0 = 2 SCLK clock cycles.
     * |        |          |0x1 = 3 SCLK clock cycles.
     * |        |          |......
     * |        |          |0xE = 16 SCLK clock cycles.
     * |        |          |0xF = 17 SCLK clock cycles.
     * |        |          |Note: Only used for I/O mode.
     * |[21:20] |BITMODE   |SPI Interface Bit Mode
     * |        |          |00 = Standard mode.
     * |        |          |01 = Dual mode.
     * |        |          |10 = Quad mode.
     * |        |          |11 = Reserved.
     * |        |          |Note: Only used for I/O mode.
     * |[23:22] |OPMODE    |SPI Function Operation Mode
     * |        |          |00 = I/O mode.
     * |        |          |01 = DMA Write mode.
     * |        |          |10 = DMA Read mode.
     * |        |          |11 = Direct Memory Map (DMM) mode (Default).
     * |        |          |Note: In DMA Write mode, hardware will send just one page program command per operation.
     * |        |          |Users must take care of cross-page case.
     * |[31:24] |CMDCODE   |Page Program Command Code
     * |        |          |0x02 = Page program (Used for DMA Write mode).
     * |        |          |0x32 = Quad page program with TYPE_1 program flow (Used for DMA Write mode).
     * |        |          |0x38 = Quad page program with TYPE_2 program flow (Used for DMA Write mode).
     * |        |          |0x40 = Quad page program with TYPE_3 program flow (Used for DMA Write mode).
     * |        |          |The Others = Reserved.
     * |        |          |Read Command Code
     * |        |          |0x03 = Standard read (Used for DMA Read/DMM mode).
     * |        |          |0x0B = Fast read (Used for DMA Read/DMM mode).
     * |        |          |0x3B = Fast dual read (Used for DMA Read/DMM mode).
     * |        |          |0xEB = Quad read (Used for DMA Read/DMM mode).
     * |        |          |The Others = Reserved.
     * |        |          |Note1: Quad mode of SPI Flash must be enabled first by I/O mode before using quad page program/quad read commands.
     * |        |          |Note2: See support list for SPI Flash which support these command codes.
     * |        |          |Note3: For TYPE_1, TYPE_2, and TYPE_3 program flows, refer to Figure 6.14-3, 6.14-4, and 6.14-5.
    */
    __IO uint32_t CTL0;

    /**
     * CTL1
     * ===================================================================================================
     * Offset: 0x04  Control Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SPIMEN    |Go And Busy Status
     * |        |          |Write Operation:
     * |        |          |0 = No effect.
     * |        |          |1 = Start the transfer.
     * |        |          |This bit remains set during the transfer and is automatically cleared after transfer finishe.
     * |        |          |Read Operation:
     * |        |          |0 = The transfer has done.
     * |        |          |1 = The transfer has not finished yet.
     * |        |          |Note: All registers should be set before writing 1 to the SPIMEN bit.
     * |        |          |When a transfer is in progress, you should not write to any register of this periphera.
     * |[4]     |SS        |Slave Select Active Enable Control
     * |        |          |0 = SPIM_SS is in active level.
     * |        |          |1 = SPIM_SS is in inactive level.
     * |        |          |Note: This interface can only drive one device/slave at a given time.
     * |        |          |Therefore, the slave selects of the selected device must be set to its active level before starting any read or write transfe.
     * |[5]     |SSACTPOL  |Slave Select Active Level
     * |        |          |It defines the active level of device/slave select signal (SPIM_SS).
     * |        |          |0 = The SPIM_SS slave select signal is Active Low.
     * |        |          |1 = The SPIM_SS slave select signal is Active High.
     * |[7:6]   |IFSEL     |Device/Slave Interface Select
     * |        |          |00 = SPI Interface from GPIO.
     * |        |          |01 = SPI Interface from MCP.
     * |        |          |10 = SPI Interface from MCP64.
     * |        |          |11 = Reserved.
     * |        |          |Note: MCP and MCP64 only can be referenced by MCP SPI Flash pad location.
     * |[11:8]  |IDLECNT   |Idle Interval
     * |        |          |In DMM mode, IDLECNT is set to control the minimum idle time between two SPI Flash accesses.
     * |        |          |Note: Only used for DMM mode.
     * |[31:16] |DIVIDER   |Clock Divider Register
     * |        |          |The value in this field is the frequency divider of the system clock to generate the serial clock on the output SPIM_CLK pin.
     * |        |          |The desired frequency is obtained according to the following equation:
     * |        |          |Note: When set DIVIDER to zero, the frequency of SPIM_CLK will be equal to the frequency of SYS_CLK.
    */
    __IO uint32_t CTL1;

    /**
     * VALIDCTL
     * ===================================================================================================
     * Offset: 0x08  Validation Check Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |VALIDSTS  |Validation Status Bit
     * |        |          |This bit will be updated when the VALIDEN bit changes.
     * |        |          |0 = Code in SPI Flash is not valid.
     * |        |          |1 = Code in SPI Flash is valid.
     * |        |          |Note: Write 0 to clear it to 0.
     * |[16]    |VALIDEN   |Validation Enable Bit
     * |        |          |Setting this bit to enable the validation function.
     * |        |          |The function can check whether the code in SPI Flash is valid or no.
     * |        |          |0=>1 (Rising Edge) = Enable the validation and clear the VALIDSTS bit.
     * |        |          |1=>0 (Falling Edge) = Disable the validation and update the VALIDSTS bit.
    */
    __IO uint32_t VALIDCTL;

    /**
     * RXCLKDLY
     * ===================================================================================================
     * Offset: 0x0C  RX Clock Delay Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DWDELSEL  |SPI Flash Deselect Time Interval of DMA Write Mode
     * |        |          |For DMA write mode only.
     * |        |          |This register sets deselect time interval of SPI Flash (i.e. time interval of inactive level of SPIM_SS) when SPI Flash controller operates in DMA write mode (Note1).
     * |        |          |Deselect time interval of DMA write mode = (DWDELSEL + 1) * AHB clock cycle time (Note2).
     * |        |          |Note1: User needs to check minimum deselect time of used SPI Flash component to set the register value.
     * |        |          |Note2: AHB clock cycle time = 1/AHB clock frequency.
     * |[15:8]  |PHDELSEL  |SPI Flash Phase Delay Time
     * |        |          |For DMA write mode, DMA read mode, and direct memory mapping mode (DMM mode).
     * |        |          |This register sets phase delay time between command data phase, address data phase, and dummy cycle phase, where SPI Flash controller will send those phase data to SPI Flash.
     * |        |          |Phase delay time of DMA write mode and DMA read mode = (PHDELSEL + 1) * AHB clock cycle time (refer to the following Note).
     * |        |          |Note: AHB clock cycle time = 1/AHB clock frequency.
     * |[31:29] |DLYSEL    |RX Sample Clock Source Delay Chain Selection
     * |        |          |000 = Not Delay.
     * |        |          |001 = Select sample clock through 2 Delay Cells.
     * |        |          |010 = Select sample clock through 4 Delay Cells.
     * |        |          |011 = Select sample clock through 6 Delay Cells.
     * |        |          |...
     * |        |          |111 = Select sample clock through 14 Delay Cells.
    */
    __IO  uint32_t RXCLKDLY;

    /**
     * RX0
     * ===================================================================================================
     * Offset: 0x10  Data Receive Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Data Receive Register
     * |        |          |The Data Receive Registers hold the received data of the last executed transfer.
     * |        |          |Number of valid RX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, received data are held in the most significant RX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, received data are held in the least significant byte of RX register firs.
     * |        |          |In a byte, received data are held in the most significant bit of RX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, received data will be held in the order SPIM_RX3[23:0], SPIM_RX2[23:0], SPIM_RX1[23:0], SPIM_RX0[23:0].
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, received data will be held in the order SPIM_RX0[7:0], SPIM_RX0[15:8], SPIM_RX0[23:16].
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, received data will be held in the order SPIM_RX0[7], SPIM_RX0[6], ..., SPIM_RX0[0].
    */
    __I  uint32_t RX0;

    /**
     * RX1
     * ===================================================================================================
     * Offset: 0x14  Data Receive Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Data Receive Register
     * |        |          |The Data Receive Registers hold the received data of the last executed transfer.
     * |        |          |Number of valid RX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, received data are held in the most significant RX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, received data are held in the least significant byte of RX register firs.
     * |        |          |In a byte, received data are held in the most significant bit of RX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, received data will be held in the order SPIM_RX3[23:0], SPIM_RX2[23:0], SPIM_RX1[23:0], SPIM_RX0[23:0].
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, received data will be held in the order SPIM_RX0[7:0], SPIM_RX0[15:8], SPIM_RX0[23:16].
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, received data will be held in the order SPIM_RX0[7], SPIM_RX0[6], ..., SPIM_RX0[0].
    */
    __I  uint32_t RX1;

    /**
     * RX2
     * ===================================================================================================
     * Offset: 0x18  Data Receive Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Data Receive Register
     * |        |          |The Data Receive Registers hold the received data of the last executed transfer.
     * |        |          |Number of valid RX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, received data are held in the most significant RX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, received data are held in the least significant byte of RX register firs.
     * |        |          |In a byte, received data are held in the most significant bit of RX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, received data will be held in the order SPIM_RX3[23:0], SPIM_RX2[23:0], SPIM_RX1[23:0], SPIM_RX0[23:0].
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, received data will be held in the order SPIM_RX0[7:0], SPIM_RX0[15:8], SPIM_RX0[23:16].
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, received data will be held in the order SPIM_RX0[7], SPIM_RX0[6], ..., SPIM_RX0[0].
    */
    __I  uint32_t RX2;

    /**
     * RX3
     * ===================================================================================================
     * Offset: 0x1C  Data Receive Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Data Receive Register
     * |        |          |The Data Receive Registers hold the received data of the last executed transfer.
     * |        |          |Number of valid RX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, received data are held in the most significant RX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, received data are held in the least significant byte of RX register firs.
     * |        |          |In a byte, received data are held in the most significant bit of RX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, received data will be held in the order SPIM_RX3[23:0], SPIM_RX2[23:0], SPIM_RX1[23:0], SPIM_RX0[23:0].
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, received data will be held in the order SPIM_RX0[7:0], SPIM_RX0[15:8], SPIM_RX0[23:16].
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, received data will be held in the order SPIM_RX0[7], SPIM_RX0[6], ..., SPIM_RX0[0].
    */
    __I  uint32_t RX3;

    /**
     * TX0
     * ===================================================================================================
     * Offset: 0x20  Data Transmit Register 0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Data Transmit Register
     * |        |          |The Data Transmit Registers hold the data to be transmitted in next transfer.
     * |        |          |Number of valid TX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, data are transmitted in the most significant TX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, data are transmitted in the least significant byte of TX register firs.
     * |        |          |In a byte, data are transmitted in the most significant bit of TX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX3[23:0], SPIM_TX2[23:0], SPIM_TX1[23:0], SPIM_TX0[23:0] in next transfer.
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX0[7:0], SPIM_TX0[15:8], SPIM_TX0[23:16] in next transfer.
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, data will be transmitted in the order SPIM_TX0[7], SPIM_TX0[6], ..., SPIM_TX0[0] in next transfer.
    */
    __IO uint32_t TX0;

    /**
     * TX1
     * ===================================================================================================
     * Offset: 0x24  Data Transmit Register 1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Data Transmit Register
     * |        |          |The Data Transmit Registers hold the data to be transmitted in next transfer.
     * |        |          |Number of valid TX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, data are transmitted in the most significant TX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, data are transmitted in the least significant byte of TX register firs.
     * |        |          |In a byte, data are transmitted in the most significant bit of TX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX3[23:0], SPIM_TX2[23:0], SPIM_TX1[23:0], SPIM_TX0[23:0] in next transfer.
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX0[7:0], SPIM_TX0[15:8], SPIM_TX0[23:16] in next transfer.
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, data will be transmitted in the order SPIM_TX0[7], SPIM_TX0[6], ..., SPIM_TX0[0] in next transfer.
    */
    __IO uint32_t TX1;

    /**
     * TX2
     * ===================================================================================================
     * Offset: 0x28  Data Transmit Register 2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Data Transmit Register
     * |        |          |The Data Transmit Registers hold the data to be transmitted in next transfer.
     * |        |          |Number of valid TX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, data are transmitted in the most significant TX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, data are transmitted in the least significant byte of TX register firs.
     * |        |          |In a byte, data are transmitted in the most significant bit of TX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX3[23:0], SPIM_TX2[23:0], SPIM_TX1[23:0], SPIM_TX0[23:0] in next transfer.
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX0[7:0], SPIM_TX0[15:8], SPIM_TX0[23:16] in next transfer.
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, data will be transmitted in the order SPIM_TX0[7], SPIM_TX0[6], ..., SPIM_TX0[0] in next transfer.
    */
    __IO uint32_t TX2;

    /**
     * TX3
     * ===================================================================================================
     * Offset: 0x2C  Data Transmit Register 3
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Data Transmit Register
     * |        |          |The Data Transmit Registers hold the data to be transmitted in next transfer.
     * |        |          |Number of valid TX registers is specified in SPIM_CTL0[BURSTNUM].
     * |        |          |If BURSTNUM > 0, data are transmitted in the most significant TX register firs.
     * |        |          |Number of valid-bit is specified in SPIM_CTL0[DWIDTH].
     * |        |          |If DWIDTH is 16, 24, or 32, data are transmitted in the least significant byte of TX register firs.
     * |        |          |In a byte, data are transmitted in the most significant bit of TX register first.
     * |        |          |Example 1: If SPIM_CTL0[BURSTNUM] = 0x3 and SPIM_CTL1[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX3[23:0], SPIM_TX2[23:0], SPIM_TX1[23:0], SPIM_TX0[23:0] in next transfer.
     * |        |          |Example 2: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x17, data will be transmitted in the order SPIM_TX0[7:0], SPIM_TX0[15:8], SPIM_TX0[23:16] in next transfer.
     * |        |          |Example 3: If SPIM_CTL0[BURSTNUM] = 0x0 and SPIM_CTL0[DWIDTH] = 0x07, data will be transmitted in the order SPIM_TX0[7], SPIM_TX0[6], ..., SPIM_TX0[0] in next transfer.
    */
    __IO uint32_t TX3;

    /**
     * SRAMADDR
     * ===================================================================================================
     * Offset: 0x30  SRAM Memory Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |SRAM Memory Address
     * |        |          |For DMA Read mode, this is the destination address for DMA transfer.
     * |        |          |For DMA Write mode, this is the source address for DMA transfer.
     * |        |          |Note: This address must be word-aligned.
    */
    __IO uint32_t SRAMADDR;

    /**
     * DMATBCNT
     * ===================================================================================================
     * Offset: 0x34  DMA Transfer Byte Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CNT       |DMA Transfer Byte Count Register
     * |        |          |It indicates the transfer length for DMA process. The unit for counting is byte.
    */
    __IO uint32_t DMATBCNT;

    /**
     * FADDR
     * ===================================================================================================
     * Offset: 0x38  SPI Flash Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |ADDR      |SPI Flash Address Register
     * |        |          |For DMA Read mode, this is the source address for DMA transfer.
     * |        |          |For DMA Write mode, this is the destination address for DMA transfer.
     * |        |          |Note: This address must be word-aligned.
    */
    __IO uint32_t FADDR;

} SPIM_T;

/**
    @addtogroup SPIM_CONST SPIM Bit Field Definition
    Constant Definitions for SPIM Controller
@{ */

#define SPIM_CTL0_CIPHOFF_Pos            (0)                                               /*!< SPIM_T::CTL0: CIPHOFF Position            */
#define SPIM_CTL0_CIPHOFF_Msk            (0x1ul << SPIM_CTL0_CIPHOFF_Pos)                  /*!< SPIM_T::CTL0: CIPHOFF Mask                */

#define SPIM_CTL0_BALEN_Pos              (2)                                               /*!< SPIM_T::CTL0: BALEN Position              */
#define SPIM_CTL0_BALEN_Msk              (0x1ul << SPIM_CTL0_BALEN_Pos)                    /*!< SPIM_T::CTL0: BALEN Mask                  */

#define SPIM_CTL0_B4ADDREN_Pos           (5)                                               /*!< SPIM_T::CTL0: B4ADDREN Position           */
#define SPIM_CTL0_B4ADDREN_Msk           (0x1ul << SPIM_CTL0_B4ADDREN_Pos)                 /*!< SPIM_T::CTL0: B4ADDREN Mask               */

#define SPIM_CTL0_IEN_Pos                (6)                                               /*!< SPIM_T::CTL0: IEN Position                */
#define SPIM_CTL0_IEN_Msk                (0x1ul << SPIM_CTL0_IEN_Pos)                      /*!< SPIM_T::CTL0: IEN Mask                    */

#define SPIM_CTL0_IF_Pos                 (7)                                               /*!< SPIM_T::CTL0: IF Position                 */
#define SPIM_CTL0_IF_Msk                 (0x1ul << SPIM_CTL0_IF_Pos)                       /*!< SPIM_T::CTL0: IF Mask                     */

#define SPIM_CTL0_DWIDTH_Pos             (8)                                               /*!< SPIM_T::CTL0: DWIDTH Position             */
#define SPIM_CTL0_DWIDTH_Msk             (0x1ful << SPIM_CTL0_DWIDTH_Pos)                  /*!< SPIM_T::CTL0: DWIDTH Mask                 */

#define SPIM_CTL0_BURSTNUM_Pos           (13)                                              /*!< SPIM_T::CTL0: BURSTNUM Position           */
#define SPIM_CTL0_BURSTNUM_Msk           (0x3ul << SPIM_CTL0_BURSTNUM_Pos)                 /*!< SPIM_T::CTL0: BURSTNUM Mask               */

#define SPIM_CTL0_QDIODIR_Pos            (15)                                              /*!< SPIM_T::CTL0: QDIODIR Position            */
#define SPIM_CTL0_QDIODIR_Msk            (0x1ul << SPIM_CTL0_QDIODIR_Pos)                  /*!< SPIM_T::CTL0: QDIODIR Mask                */

#define SPIM_CTL0_SUSPITV_Pos            (16)                                              /*!< SPIM_T::CTL0: SUSPITV Position            */
#define SPIM_CTL0_SUSPITV_Msk            (0xful << SPIM_CTL0_SUSPITV_Pos)                  /*!< SPIM_T::CTL0: SUSPITV Mask                */

#define SPIM_CTL0_BITMODE_Pos            (20)                                              /*!< SPIM_T::CTL0: BITMODE Position            */
#define SPIM_CTL0_BITMODE_Msk            (0x3ul << SPIM_CTL0_BITMODE_Pos)                  /*!< SPIM_T::CTL0: BITMODE Mask                */

#define SPIM_CTL0_OPMODE_Pos             (22)                                              /*!< SPIM_T::CTL0: OPMODE Position             */
#define SPIM_CTL0_OPMODE_Msk             (0x3ul << SPIM_CTL0_OPMODE_Pos)                   /*!< SPIM_T::CTL0: OPMODE Mask                 */

#define SPIM_CTL0_CMDCODE_Pos            (24)                                              /*!< SPIM_T::CTL0: CMDCODE Position            */
#define SPIM_CTL0_CMDCODE_Msk            (0xfful << SPIM_CTL0_CMDCODE_Pos)                 /*!< SPIM_T::CTL0: CMDCODE Mask                */

#define SPIM_CTL1_SPIMEN_Pos             (0)                                               /*!< SPIM_T::CTL1: SPIMEN Position             */
#define SPIM_CTL1_SPIMEN_Msk             (0x1ul << SPIM_CTL1_SPIMEN_Pos)                   /*!< SPIM_T::CTL1: SPIMEN Mask                 */

#define SPIM_CTL1_SS_Pos                 (4)                                               /*!< SPIM_T::CTL1: SS Position                 */
#define SPIM_CTL1_SS_Msk                 (0x1ul << SPIM_CTL1_SS_Pos)                       /*!< SPIM_T::CTL1: SS Mask                     */

#define SPIM_CTL1_SSACTPOL_Pos           (5)                                               /*!< SPIM_T::CTL1: SSACTPOL Position           */
#define SPIM_CTL1_SSACTPOL_Msk           (0x1ul << SPIM_CTL1_SSACTPOL_Pos)                 /*!< SPIM_T::CTL1: SSACTPOL Mask               */

#define SPIM_CTL1_IFSEL_Pos              (6)                                               /*!< SPIM_T::CTL1: IFSEL Position              */
#define SPIM_CTL1_IFSEL_Msk              (0x3ul << SPIM_CTL1_IFSEL_Pos)                    /*!< SPIM_T::CTL1: IFSEL Mask                  */

#define SPIM_CTL1_IDLECNT_Pos            (8)                                               /*!< SPIM_T::CTL1: IDLECNT Position            */
#define SPIM_CTL1_IDLECNT_Msk            (0xful << SPIM_CTL1_IDLECNT_Pos)                  /*!< SPIM_T::CTL1: IDLECNT Mask                */

#define SPIM_CTL1_DIVIDER_Pos            (16)                                              /*!< SPIM_T::CTL1: DIVIDER Position            */
#define SPIM_CTL1_DIVIDER_Msk            (0xfffful << SPIM_CTL1_DIVIDER_Pos)               /*!< SPIM_T::CTL1: DIVIDER Mask                */

#define SPIM_VALIDCTL_VALIDSTS_Pos       (0)                                               /*!< SPIM_T::VALIDCTL: VALIDSTS Position       */
#define SPIM_VALIDCTL_VALIDSTS_Msk       (0x1ul << SPIM_VALIDCTL_VALIDSTS_Pos)             /*!< SPIM_T::VALIDCTL: VALIDSTS Mask           */

#define SPIM_VALIDCTL_VALIDEN_Pos        (16)                                              /*!< SPIM_T::VALIDCTL: VALIDEN Position        */
#define SPIM_VALIDCTL_VALIDEN_Msk        (0x1ul << SPIM_VALIDCTL_VALIDEN_Pos)              /*!< SPIM_T::VALIDCTL: VALIDEN Mask            */

#define SPIM_RXCLKDLY_DWDELSEL_Pos       (0)                                               /*!< SPIM_T::RXCLKDLY: DWDELSEL Position       */
#define SPIM_RXCLKDLY_DWDELSEL_Msk       (0xFFul << SPIM_RXCLKDLY_DWDELSEL_Pos)            /*!< SPIM_T::RXCLKDLY: DWDELSEL Mask           */

#define SPIM_RXCLKDLY_PHDELSEL_Pos       (8)                                               /*!< SPIM_T::RXCLKDLY: PHDELSEL Position       */
#define SPIM_RXCLKDLY_PHDELSEL_Msk       (0xFFul << SPIM_RXCLKDLY_PHDELSEL_Pos)            /*!< SPIM_T::RXCLKDLY: PHDELSEL Mask           */

#define SPIM_RXCLKDLY_DLYSEL_Pos         (29)                                              /*!< SPIM_T::RXCLKDLY: DLYSEL Position         */
#define SPIM_RXCLKDLY_DLYSEL_Msk         (0x7ul << SPIM_RXCLKDLY_DLYSEL_Pos)               /*!< SPIM_T::RXCLKDLY: DLYSEL Mask             */

#define SPIM_RX0_RX_Pos                  (0)                                               /*!< SPIM_T::RX0: RX Position                  */
#define SPIM_RX0_RX_Msk                  (0xfffffffful << SPIM_RX0_RX_Pos)                 /*!< SPIM_T::RX0: RX Mask                      */

#define SPIM_RX1_RX_Pos                  (0)                                               /*!< SPIM_T::RX1: RX Position                  */
#define SPIM_RX1_RX_Msk                  (0xfffffffful << SPIM_RX1_RX_Pos)                 /*!< SPIM_T::RX1: RX Mask                      */

#define SPIM_RX2_RX_Pos                  (0)                                               /*!< SPIM_T::RX2: RX Position                  */
#define SPIM_RX2_RX_Msk                  (0xfffffffful << SPIM_RX2_RX_Pos)                 /*!< SPIM_T::RX2: RX Mask                      */

#define SPIM_RX3_RX_Pos                  (0)                                               /*!< SPIM_T::RX3: RX Position                  */
#define SPIM_RX3_RX_Msk                  (0xfffffffful << SPIM_RX3_RX_Pos)                 /*!< SPIM_T::RX3: RX Mask                      */

#define SPIM_TX0_TX_Pos                  (0)                                               /*!< SPIM_T::TX0: TX Position                  */
#define SPIM_TX0_TX_Msk                  (0xfffffffful << SPIM_TX0_TX_Pos)                 /*!< SPIM_T::TX0: TX Mask                      */

#define SPIM_TX1_TX_Pos                  (0)                                               /*!< SPIM_T::TX1: TX Position                  */
#define SPIM_TX1_TX_Msk                  (0xfffffffful << SPIM_TX1_TX_Pos)                 /*!< SPIM_T::TX1: TX Mask                      */

#define SPIM_TX2_TX_Pos                  (0)                                               /*!< SPIM_T::TX2: TX Position                  */
#define SPIM_TX2_TX_Msk                  (0xfffffffful << SPIM_TX2_TX_Pos)                 /*!< SPIM_T::TX2: TX Mask                      */

#define SPIM_TX3_TX_Pos                  (0)                                               /*!< SPIM_T::TX3: TX Position                  */
#define SPIM_TX3_TX_Msk                  (0xfffffffful << SPIM_TX3_TX_Pos)                 /*!< SPIM_T::TX3: TX Mask                      */

#define SPIM_SRAMADDR_ADDR_Pos           (0)                                               /*!< SPIM_T::SRAMADDR: ADDR Position           */
#define SPIM_SRAMADDR_ADDR_Msk           (0xfffffffful << SPIM_SRAMADDR_ADDR_Pos)          /*!< SPIM_T::SRAMADDR: ADDR Mask               */

#define SPIM_DMATBCNT_CNT_Pos            (0)                                               /*!< SPIM_T::DMATBCNT: CNT Position            */
#define SPIM_DMATBCNT_CNT_Msk            (0xfffffful << SPIM_DMATBCNT_CNT_Pos)             /*!< SPIM_T::DMATBCNT: CNT Mask                */

#define SPIM_FADDR_ADDR_Pos              (0)                                               /*!< SPIM_T::FADDR: ADDR Position              */
#define SPIM_FADDR_ADDR_Msk              (0xfffffffful << SPIM_FADDR_ADDR_Pos)             /*!< SPIM_T::FADDR: ADDR Mask                  */

/**@}*/ /* SPIM_CONST */
/**@}*/ /* end of SPIM register group */


/*---------------------- Serial Peripheral Interface Controller -------------------------*/
/**
    @addtogroup SPI Serial Peripheral Interface Controller(SPI)
    Memory Mapped Structure for SPI Controller
@{ */

typedef struct
{


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x00  SPI Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SPIEN     |SPI Transfer Control Enable Bit
     * |        |          |0 = Transfer control bit Disabled.
     * |        |          |1 = Transfer control bit Enabled.
     * |        |          |Note1: In Master mode, the transfer will start when there is data in the FIFO buffer after this is set to 1.
     * |        |          |In Slave mode, this device is ready to receive data when this bit is set to .
     * |        |          |Note2: The byte reorder function is not supported when the Quad or Dual I/O mode is enabled.
     * |[1]     |RXNEG     |Receive On Negative Edge
     * |        |          |0 = Received data input signal is latched on the rising edge of SPICLK.
     * |        |          |1 = Received data input signal is latched on the falling edge of SPICLK.
     * |[2]     |TXNEG     |Transmit On Negative Edge
     * |        |          |0 = Transmitted data output signal is changed on the rising edge of SPICLK.
     * |        |          |1 = Transmitted data output signal is changed on the falling edge of SPICLK.
     * |[3]     |CLKPOL    |Clock Polarity
     * |        |          |0 = SPICLK is idle low.
     * |        |          |1 = SPICLK is idle high.
     * |[7:4]   |SUSPITV   |Suspend Interval (Master Only)
     * |        |          |The four bits provide configurable suspend interval between two successive transmit/receive transaction in a transfer.
     * |        |          |The definition of the suspend interval is the interval between the last clock edge of the preceding transaction word and the first clock edge of the following transaction word.
     * |        |          |The default value is 0x3.
     * |        |          |The period of the suspend interval is obtained according to the following equatio.
     * |        |          |(SUSPITV[3:0]+0.5) * period of SPICLK clock cycle
     * |        |          |Example:
     * |        |          |SUSPITV = 0x0 ... 0.5 SPICLK clock cycle.
     * |        |          |SUSPITV = 0x1 ... 1.5 SPICLK clock cycle.
     * |        |          |......
     * |        |          |SUSPITV = 0xE ... 14.5 SPICLK clock cycle.
     * |        |          |SUSPITV = 0xF ... 15.5 SPICLK clock cycle.
     * |[12:8]  |DWIDTH    |Data Transmit Bit Width
     * |        |          |This field specifies how many bits can be transmitted / received in one transaction.
     * |        |          |The minimum bit length is 8 bits and can up to 32 bit.
     * |        |          |DWIDTH = 0x08 ... 8 bits.
     * |        |          |DWIDTH = 0x09 ... 9 bits.
     * |        |          |......
     * |        |          |DWIDTH = 0x1F ... 31 bits.
     * |        |          |DWIDTH = 0x00 ... 32 bits.
     * |[13]    |LSB       |Send LSB First
     * |        |          |0 = The MSB, which bit of transmit/receive register depends on the setting of DWIDTH, is transmitted/received first.
     * |        |          |1 = The LSB, bit 0 of the SPI TX register, is sent first to the SPI data output pin, and the first bit received from the SPI data input pin will be put in the LSB position of the RX register (bit 0 of SPI_RX).
     * |[17]    |UNITIEN   |Unit Transfer Interrupt Enable Control
     * |        |          |0 = SPI unit transfer interrupt Disabled.
     * |        |          |1 = SPI unit transfer interrupt Enabled.
     * |[18]    |SLAVE     |Slave Mode Enable Control
     * |        |          |0 = Master mode.
     * |        |          |1 = Slave mode.
     * |[19]    |REORDER   |Byte Reorder Function Enable Control
     * |        |          |0 = Byte reorder function Disabled.
     * |        |          |1 = Byte reorder function Enabled.
     * |        |          |A byte suspend interval will be inserted among each byte.
     * |        |          |The period of the byte suspend interval depends on the setting of SUSPITV (SPI_CTL[7:4].
     * |        |          |Note1: Byte reorder function is only available if DWIDTH is defined as 16, 24, and 32 bits.
    */
    __IO uint32_t CTL;

    /**
     * CLKDIV
     * ===================================================================================================
     * Offset: 0x04  SPI Clock Divider Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DIVIDER   |Clock Divider 1 Register
     * |        |          |The value in this field is the frequency divider for generating the SPI engine clock, fspi_eclk, and the SPI serial clock of SPI master.
     * |        |          |The frequency is obtained according to the following equatio.
     * |        |          |where
     * |        |          |is the SPI engine clock source, which is defined in the clock control, clock control register.
    */
    __IO uint32_t CLKDIV;

    /**
     * SSCTL
     * ===================================================================================================
     * Offset: 0x08  SPI Slave Select Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SS        |Slave Selection Control (Master Only)
     * |        |          |If AUTOSS bit is cleared to 0,
     * |        |          |0 = set the SPIn_SS line to inactive state.
     * |        |          |1 = set the SPIn_SS line to active state.
     * |        |          |If the AUTOSS bit is set to 1,
     * |        |          |0 = Keep the SPIn_SS line at inactive state.
     * |        |          |1 = SPIn_SS line will be automatically driven to active state for the duration of data transfer, and will be driven to inactive state for the rest of the time
     * |        |          |The active state of SPIn_SS is specified in SSACTPOL (SPI_SSCTL[2]).
     * |[2]     |SSACTPOL  |Slave Select Active Level
     * |        |          |This bit defines the active status of slave select signal (SS).
     * |        |          |0 = The slave select signal SS is active on low-level.
     * |        |          |1 = The slave select signal SS is active on high-level.
     * |[3]     |AUTOSS    |Automatic Slave Select Function Enable (Master Only)
     * |        |          |0 = If this bit is cleared, slave select signals will be asserted/de-asserted by setting /clearing the corresponding bit of SPI_SSCTL[0].
     * |        |          |1 = If this bit is set, SS signal will be generated automatically.
     * |        |          |It means that device/slave select signal, which is set in SPI_SSCTL[0], will be asserted by the SPI controller when transmit/receive is started, and will be de-asserted after each transmit/receive is finishe.
     * |[4]     |SLV3WIRE  |Slave 3-Wire Mode Enable Control
     * |        |          |This is used to ignore the slave select signal in Slave mode.
     * |        |          |The SPI controller can work with 3-wire interface including SPI_CLK, SPI_MISO, and SPI_MOS.
     * |        |          |0 = 2-wire bi-direction interface.
     * |        |          |1 = 3-wire bi-direction interface.
     * |[5]     |SLVTOIEN  |Slave Mode Time-out Interrupt Enable Control
     * |        |          |0 = Slave mode time-out interrupt Disabled.
     * |        |          |1 = Slave mode time-out interrupt Enabled.
     * |[6]     |SLVTORST  |Slave Mode Time-out FIFO Clear
     * |        |          |0 = Slave mode Time-out FIFO Clear Disable.
     * |        |          |1 = Slave mode Time-out FIFO Clear Enable.
     * |        |          |Note: Both the FIFO clear function, TX_CLK and RXRST, active automatically when there is a slave mode time-out event
     * |[8]     |SLVBEIEN  |Slave Mode Error 0 Interrupt Enable Control
     * |        |          |0 = Slave mode error 0 interrupt Disable.
     * |        |          |1 = Slave mode error 0 interrupt Enable.
     * |[9]     |SLVURIEN  |Slave Mode Error 1 Interrupt Enable Control
     * |        |          |0 = Slave mode error 1 interrupt Disable.
     * |        |          |1 = Slave mode error 1 interrupt Enable.
     * |[12]    |SSACTIEN  |Slave Select Active Interrupt Enable Control
     * |        |          |0 = Slave select active interrupt Disable.
     * |        |          |1 = Slave select active interrupt Enable.
     * |[13]    |SSINAIEN  |Slave Select Inactive Interrupt Enable Control
     * |        |          |0 = Slave select inactive interrupt Disable.
     * |        |          |1 = Slave select inactive interrupt Enable.
     * |[31:16] |SLVTOCNT  |Slave Mode Time-out Period
     * |        |          |In Slave mode, these bits indicate the time out period when there is serial clock input during slave select active.
     * |        |          |The clock source of the time out counter is Slave engine clock.
     * |        |          |If the value is 0, it indicates the slave mode time-out function is disable.
    */
    __IO uint32_t SSCTL;
    __I  uint32_t RESERVE0[1];


    /**
     * FIFOCTL
     * ===================================================================================================
     * Offset: 0x10  SPI FIFO Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXRST     |Clear Receive FIFO Control
     * |        |          |0 = No effect.
     * |        |          |1 = Clear receive FIFO control
     * |        |          |The RXFULL (SPI_STATUS[9]) will be cleared to 0 and the RXEMPTY (SPI_STATUS[8]) will be set to 1
     * |        |          |This bit will be cleared to 0 by hardware about 3 system clocks+3 SPI engine clock after it is set to 1.
     * |        |          |Note: If there is slave receive time-out event, the RXRST will be set 1 when the SLVTORST (SPI_SSCTL[6]) is enabled.
     * |[1]     |TXRST     |Clear Transmit FIFO Control
     * |        |          |0 = No effect.
     * |        |          |1 = Clear transmit FIFO control
     * |        |          |The TXFULL (SPI_STATUS[17]) will be cleared to 0 and the TXEMPTY (SPI_STATUS[16]) will be set to 1
     * |        |          |This bit will be cleared to 0 by hardware about 3 system clocks+3 SPI engine clock after it is set to 1.
     * |        |          |Note: If there is slave receive time-out event, the TXRST will be set 1 when the SLVTORST (SPI_SSCTL[6]) is enabled.
     * |[2]     |RXTHIEN   |Receive FIFO Threshold Interrupt Enable Control
     * |        |          |0 = RX FIFO threshold interrupt Disabled.
     * |        |          |1 = RX FIFO threshold interrupt Enabled.
     * |[3]     |TXTHIEN   |Transmit FIFO Threshold Interrupt Enable Control
     * |        |          |0 = TX FIFO threshold interrupt Disabled.
     * |        |          |1 = TX FIFO threshold interrupt Enabled.
     * |[4]     |RXTOIEN   |Slave Receive Time-out Interrupt Enable Control
     * |        |          |0 = Receive time-out interrupt Disabled.
     * |        |          |1 = Receive time-out interrupt Enabled.
     * |[5]     |RXOVIEN   |Receive FIFO Overrun Interrupt Enable Control
     * |        |          |0 = Receive FIFO overrun interrupt Disabled.
     * |        |          |1 = Receive FIFO overrun interrupt Enabled.
     * |[6]     |TXUFPOL   |Transmit Under-Run Data Out
     * |        |          |0 = The SPI data out is keep 0 if there is transmit under-run event in Slave mode.
     * |        |          |1 = The SPI data out is keep 1 if there is transmit under-run event in Slave mode.
     * |        |          |Note1: The under run event is active after the serial clock input and the hardware synchronous, so that the first 1~3 bit (depending on the relation between system clock and the engine clock) data out will be the last transaction data.
     * |        |          |Note2: If the frequency of system clock approach to engine clock, they may need 3-bit time to report the transmit under-run data out.
     * |[7]     |TXUFIEN   |Slave Transmit Under Run Interrupt Enable Control
     * |        |          |0 = Slave Transmit FIFO under-run interrupt Disabled.
     * |        |          |1 = Slave Transmit FIFO under-run interrupt Enabled.
     * |[8]     |RXFBCLR   |Clear Receive FIFO
     * |        |          |0 = No effect.
     * |        |          |1 = Clear receive FIFO only.
     * |        |          |Note: Auto cleared by Hardware.
     * |[9]     |TXFBCLR   |Clear Transmit FIFO
     * |        |          |0 = No effect.
     * |        |          |1 = Clear transmit FIFO only.
     * |        |          |Note: Auto cleared by Hardware.
     * |[26:24] |RXTH      |Receive FIFO Threshold
     * |        |          |If the valid data count of the receive FIFO buffer is larger than the RXTH setting, the RXTHIF bit will be set to 1, else the RXTHIF bit will be cleared to 0.
     * |[30:28] |TXTH      |Transmit FIFO Threshold
     * |        |          |If the valid data count of the transmit FIFO buffer is less than or equal to the TXTH setting, the TXTHIF bit will be set to 1, else the TXTHIF bit will be cleared to 0.
    */
    __IO uint32_t FIFOCTL;

    /**
     * STATUS
     * ===================================================================================================
     * Offset: 0x14  SPI Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |BUSY      |SPI Unit Bus Status (Read Only)
     * |        |          |0 = No transaction in the SPI bus.
     * |        |          |1 = SPI controller unit in busy state.
     * |        |          |The following lists the bus busy conditions:
     * |        |          |a.SPIEN (SPI_CTL[0]) = 1 and the TXEMPTY (SPI_STATUS[16]) = 0.
     * |        |          |b.For SPI Master, the TXEMPTY (SPI_STATUS[16]) = 1 but the current transaction is not finished yet.
     * |        |          |c.For SPI Slave receive mode, the SPIEN (SPI_CTL[0]) = 1 and there is serial clock input into the SPI core logic when slave select is active.
     * |        |          |d.For SPI Slave transmit mode, the SPIEN (SPI_CTL[0]) = 1 and the transmit buffer is not empty in SPI core logic event if the slave select is inactive.
     * |[1]     |UNITIF    |Unit Transfer Interrupt Status
     * |        |          |0 = No transaction has been finished since this bit was cleared to 0.
     * |        |          |1 = SPI controller has finished one unit transfer.
     * |        |          |Note: This bit will be cleared by writing 1 to itself.
     * |[2]     |SSACTIF   |Slave Select Active Interrupt Status
     * |        |          |0 = Slave select active interrupt is cleared or did not occur.
     * |        |          |1 = Slave select active interrupt event has occurred.
     * |        |          |Note: This bit will be cleared by writing 1 to itself.
     * |[3]     |SSINAIF   |Slave Select Inactive Interrupt Status
     * |        |          |0 = Slave select inactive interrupt is cleared or did not occur.
     * |        |          |1 = Slave select inactive interrupt event has occurred.
     * |        |          |Note: This bit will be cleared by writing 1 to itself.
     * |[4]     |SSLINE    |Slave Select Line Bus Status (Read Only)
     * |        |          |0 = Indicates the slave select line bus status is 0.
     * |        |          |1 = Indicates the slave select line bus status is 1.
     * |        |          |Note: If SSACTPOL (SPI_SSCTL[2]) is set 0, and the SSLINE is 1, the SPI slave select is in inactive status.
     * |[5]     |SLVTOIF   |Slave Time-out Interrupt Status (Read Only)
     * |        |          |When the Slave Select is active and the value of SLVTOCNT (SPI_SSCTL[31:16]) is not 0 and the serial clock input, the slave time-out counter in SPI controller logic will be start
     * |        |          |When the value of time-out counter greater or equal than the value of SLVTOCNT (SPI_SSCTL[31:16]) during before one transaction done, the slave time-out interrupt event will active.
     * |        |          |0 = Slave time-out is not active.
     * |        |          |1 = Slave time-out is active.
     * |        |          |Note: If the DWIDTH (SPI_CTL[12:8]) is set 0x10, one transaction is equal 16 bits serial clock period.
     * |[6]     |SLVBEIF   |Slave Mode Error 0 Interrupt Status (Read Only)
     * |        |          |In Slave mode, there is bit counter mismatch with DWIDTH (SPI_CTL[12:8]) when the slave select line goes to inactive state.
     * |        |          |0 = No Slave mode error 0 event.
     * |        |          |1 = Slave mode error 0 occurs.
     * |        |          |Note: If the slave select active but there is no any serial clock input, the SLVBEIF (SPI_STATUS[6]) also active when the slave select goes to inactive state.
     * |[7]     |SLVURIF   |Slave Mode Error 1 Interrupt Status (Read Only)
     * |        |          |In Slave mode, transmit under-run occurs when the slave select line goes to inactive state.
     * |        |          |0 = No Slave mode error 1 event.
     * |        |          |1 = Slave mode error 1 occurs.
     * |[8]     |RXEMPTY   |Receive FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Receive FIFO buffer is not empty.
     * |        |          |1 = Receive FIFO buffer is empty.
     * |[9]     |RXFULL    |Receive FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Receive FIFO buffer is not empty.
     * |        |          |1 = Receive FIFO buffer is empty.
     * |[10]    |RXTHIF    |Receive FIFO Threshold Interrupt Status (Read Only)
     * |        |          |0 = The valid data count within the RX FIFO buffer is smaller than or equal to the setting value of RXTH (SPI_FIFOCTL[26:24]).
     * |        |          |1 = The valid data count within the receive FIFO buffer is larger than the setting value of RXTH (SPI_FIFOCTL[26:24]).
     * |        |          |Note: If RXTHIEN (SPI_FIFOCTL[2]) = 1 and RXTHIF = 1, the SPI controller will generate a SPI interrupt request.
     * |[11]    |RXOVIF    |Receive FIFO Overrun Status
     * |        |          |0= No receiver FIFO overrun status.
     * |        |          |1= Receive FIFO buffer is full, the follow-up data will be dropped and this bit will be set to 1.
     * |        |          |Note: This bit will be cleared by writing 1 to itself.
     * |[12]    |RXTOIF    |Receive Time-out Interrupt Status
     * |        |          |0 = No receive FIFO time-out event.
     * |        |          |1 = Receive FIFO buffer is not empty and no read operation on receive FIFO buffer over 64 SPI clock period in Master mode or over 576 SPI engine clock period in Slave mode
     * |        |          |When the received FIFO buffer is read by software, the time-out status will be cleared automatically.
     * |        |          |Note: This bit will be cleared by writing 1 to itself.
     * |[15]    |SPIENSTS  |SPI Enable Bit Status (Read Only)
     * |        |          |0 = Indicate the transmit control bit is disabled.
     * |        |          |1 = Indicate the transfer control bit is active.
     * |        |          |The clock source of SPI controller logic is engine clock, it is asynchronous with the system clock
     * |        |          |In order to make sure the function is disabled in SPI controller logic, this bit indicates the real status of SPIEN (SPI_CTL[0]) in SPI controller logic for user.
     * |[16]    |TXEMPTY   |Transmit FIFO Buffer Empty Indicator (Read Only)
     * |        |          |0 = Transmit FIFO buffer is not empty.
     * |        |          |1 = Transmit FIFO buffer is empty.
     * |[17]    |TXFULL    |Transmit FIFO Buffer Full Indicator (Read Only)
     * |        |          |0 = Transmit FIFO buffer is not full.
     * |        |          |1 = Transmit FIFO buffer is full.
     * |[18]    |TXTHIF    |Transmit FIFO Threshold Interrupt Status (Read Only)
     * |        |          |0 = The valid data count within the transmit FIFO buffer is larger than the setting value of TXTH (SPI_FIFOCTL[30:28]).
     * |        |          |1 = The valid data count within the transmit FIFO buffer is less than or equal to the setting value of TXTH (SPI_FIFOCTL[30:28]).
     * |        |          |Note: If TXTHIEN (SPI_FIFOCTL[3]) = 1 and TXTHIF = 1, the SPI controller will generate a SPI interrupt request.
     * |[19]    |TXUFIF    |TX Underflow Interrupt Flag
     * |        |          |When the TX underflow event occurs, this bit will be set to 1, the state of data output pin depends on the setting of TXUFPOL.
     * |        |          |0 = No effect.
     * |        |          |1 = No data in Transmit FIFO and TX shift register when the slave selection signal is active.
     * |        |          |Note 1: This bit will be cleared by writing 1 to it.
     * |        |          |Note 2: If reset slave's transmission circuit when slave selection signal is active, this flag will be set to 1 after 3 system clock cycles + 2 peripheral clock cycles since the reset operation is done.
     * |[23]    |TXRXRST   |FIFO CLR Status (Read Only)
     * |        |          |0 = Done the FIFO buffer clear function of TXRST (SPI_FIFOCTL[1]) or RXRST (SPI_FIFOCTL[0]).
     * |        |          |1 = Doing the FIFO buffer clear function of TXRST (SPI_FIFOCTL[1])or RXRST (SPI_FIFOCTL[0]).
     * |        |          |Note: Both the TXRST (SPI_FIFOCTL[1]), RXRST (SPI_FIFOCTL[0]), need 3 system clock+3 engine clock , the status of this bit support the user to monitor the clear function is doing or done.
     * |[27:24] |RXCNT     |Receive FIFO Data Count (Read Only)
     * |        |          |This bit field indicates the valid data count of receive FIFO buffer.
     * |[31:28] |TXCNT     |Transmit FIFO Data Count (Read Only)
     * |        |          |This bit field indicates the valid data count of transmit FIFO buffer.
    */
    __IO  uint32_t STATUS;
    uint32_t RESERVE1[2];


    /**
     * TX
     * ===================================================================================================
     * Offset: 0x20  SPI Data Transmit Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |TX        |Data Transmit Register
     * |        |          |The data transmit registers pass through the transmitted data into the 8-level transmit FIFO buffer
     * |        |          |The number of valid bits depends on the setting of transmit bit width field of the SPI_CTL register.
     * |        |          |For example, if DWIDTH (SPI_CTL[12:8]) is set to 0x08, the bits TX[7:0] will be transmitted
     * |        |          |If DWIDTH (SPI_CTL[12:8]) is set to 0x00, the SPI controller will perform a 32-bit transfer.
    */
    __O  uint32_t TX;
    uint32_t RESERVE2[3];


    /**
     * RX
     * ===================================================================================================
     * Offset: 0x30  SPI Data Receive Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |RX        |Data Receive Register
     * |        |          |There is 8-level FIFO buffer in this controller
     * |        |          |The data receive register holds the earliest datum received from SPI data input pin
     * |        |          |If the RXEMPTY (SPI_STATUS[8]) is not set to 1, the receive FIFO buffer can be accessed through software by reading this register
     * |        |          |This is a read-only register
    */
    __I  uint32_t RX;

} SPI_T;

/**
    @addtogroup SPI_CONST SPI Bit Field Definition
    Constant Definitions for SPI Controller
@{ */

#define SPI_CTL_SPIEN_Pos                (0)                                               /*!< SPI_T::CTL: SPIEN Position                */
#define SPI_CTL_SPIEN_Msk                (0x1ul << SPI_CTL_SPIEN_Pos)                      /*!< SPI_T::CTL: SPIEN Mask                    */

#define SPI_CTL_RXNEG_Pos                (1)                                               /*!< SPI_T::CTL: RXNEG Position                */
#define SPI_CTL_RXNEG_Msk                (0x1ul << SPI_CTL_RXNEG_Pos)                      /*!< SPI_T::CTL: RXNEG Mask                    */

#define SPI_CTL_TXNEG_Pos                (2)                                               /*!< SPI_T::CTL: TXNEG Position                */
#define SPI_CTL_TXNEG_Msk                (0x1ul << SPI_CTL_TXNEG_Pos)                      /*!< SPI_T::CTL: TXNEG Mask                    */

#define SPI_CTL_CLKPOL_Pos               (3)                                               /*!< SPI_T::CTL: CLKPOL Position               */
#define SPI_CTL_CLKPOL_Msk               (0x1ul << SPI_CTL_CLKPOL_Pos)                     /*!< SPI_T::CTL: CLKPOL Mask                   */

#define SPI_CTL_SUSPITV_Pos              (4)                                               /*!< SPI_T::CTL: SUSPITV Position              */
#define SPI_CTL_SUSPITV_Msk              (0xful << SPI_CTL_SUSPITV_Pos)                    /*!< SPI_T::CTL: SUSPITV Mask                  */

#define SPI_CTL_DWIDTH_Pos               (8)                                               /*!< SPI_T::CTL: DWIDTH Position               */
#define SPI_CTL_DWIDTH_Msk               (0x1ful << SPI_CTL_DWIDTH_Pos)                    /*!< SPI_T::CTL: DWIDTH Mask                   */

#define SPI_CTL_LSB_Pos                  (13)                                              /*!< SPI_T::CTL: LSB Position                  */
#define SPI_CTL_LSB_Msk                  (0x1ul << SPI_CTL_LSB_Pos)                        /*!< SPI_T::CTL: LSB Mask                      */

#define SPI_CTL_UNITIEN_Pos              (17)                                              /*!< SPI_T::CTL: UNITIEN Position              */
#define SPI_CTL_UNITIEN_Msk              (0x1ul << SPI_CTL_UNITIEN_Pos)                    /*!< SPI_T::CTL: UNITIEN Mask                  */

#define SPI_CTL_SLAVE_Pos                (18)                                              /*!< SPI_T::CTL: SLAVE Position                */
#define SPI_CTL_SLAVE_Msk                (0x1ul << SPI_CTL_SLAVE_Pos)                      /*!< SPI_T::CTL: SLAVE Mask                    */

#define SPI_CTL_REORDER_Pos              (19)                                              /*!< SPI_T::CTL: REORDER Position              */
#define SPI_CTL_REORDER_Msk              (0x1ul << SPI_CTL_REORDER_Pos)                    /*!< SPI_T::CTL: REORDER Mask                  */

#define SPI_CLKDIV_DIVIDER_Pos           (0)                                               /*!< SPI_T::CLKDIV: DIVIDER Position           */
#define SPI_CLKDIV_DIVIDER_Msk           (0xfful << SPI_CLKDIV_DIVIDER_Pos)                /*!< SPI_T::CLKDIV: DIVIDER Mask               */

#define SPI_SSCTL_SS_Pos                 (0)                                               /*!< SPI_T::SSCTL: SS Position                 */
#define SPI_SSCTL_SS_Msk                 (0x1ul << SPI_SSCTL_SS_Pos)                       /*!< SPI_T::SSCTL: SS Mask                     */

#define SPI_SSCTL_SSACTPOL_Pos           (2)                                               /*!< SPI_T::SSCTL: SSACTPOL Position           */
#define SPI_SSCTL_SSACTPOL_Msk           (0x1ul << SPI_SSCTL_SSACTPOL_Pos)                 /*!< SPI_T::SSCTL: SSACTPOL Mask               */

#define SPI_SSCTL_AUTOSS_Pos             (3)                                               /*!< SPI_T::SSCTL: AUTOSS Position             */
#define SPI_SSCTL_AUTOSS_Msk             (0x1ul << SPI_SSCTL_AUTOSS_Pos)                   /*!< SPI_T::SSCTL: AUTOSS Mask                 */

#define SPI_SSCTL_SLV3WIRE_Pos           (4)                                               /*!< SPI_T::SSCTL: SLV3WIRE Position           */
#define SPI_SSCTL_SLV3WIRE_Msk           (0x1ul << SPI_SSCTL_SLV3WIRE_Pos)                 /*!< SPI_T::SSCTL: SLV3WIRE Mask               */

#define SPI_SSCTL_SLVTOIEN_Pos           (5)                                               /*!< SPI_T::SSCTL: SLVTOIEN Position           */
#define SPI_SSCTL_SLVTOIEN_Msk           (0x1ul << SPI_SSCTL_SLVTOIEN_Pos)                 /*!< SPI_T::SSCTL: SLVTOIEN Mask               */

#define SPI_SSCTL_SLVTORST_Pos           (6)                                               /*!< SPI_T::SSCTL: SLVTORST Position           */
#define SPI_SSCTL_SLVTORST_Msk           (0x1ul << SPI_SSCTL_SLVTORST_Pos)                 /*!< SPI_T::SSCTL: SLVTORST Mask               */

#define SPI_SSCTL_SLVBEIEN_Pos           (8)                                               /*!< SPI_T::SSCTL: SLVBEIEN Position           */
#define SPI_SSCTL_SLVBEIEN_Msk           (0x1ul << SPI_SSCTL_SLVBEIEN_Pos)                 /*!< SPI_T::SSCTL: SLVBEIEN Mask               */

#define SPI_SSCTL_SLVURIEN_Pos           (9)                                               /*!< SPI_T::SSCTL: SLVURIEN Position           */
#define SPI_SSCTL_SLVURIEN_Msk           (0x1ul << SPI_SSCTL_SLVURIEN_Pos)                 /*!< SPI_T::SSCTL: SLVURIEN Mask               */

#define SPI_SSCTL_SSACTIEN_Pos           (12)                                              /*!< SPI_T::SSCTL: SSACTIEN Position           */
#define SPI_SSCTL_SSACTIEN_Msk           (0x1ul << SPI_SSCTL_SSACTIEN_Pos)                 /*!< SPI_T::SSCTL: SSACTIEN Mask               */

#define SPI_SSCTL_SSINAIEN_Pos           (13)                                              /*!< SPI_T::SSCTL: SSINAIEN Position           */
#define SPI_SSCTL_SSINAIEN_Msk           (0x1ul << SPI_SSCTL_SSINAIEN_Pos)                 /*!< SPI_T::SSCTL: SSINAIEN Mask               */

#define SPI_SSCTL_SLVTOCNT_Pos           (16)                                              /*!< SPI_T::SSCTL: SLVTOCNT Position           */
#define SPI_SSCTL_SLVTOCNT_Msk           (0xfffful << SPI_SSCTL_SLVTOCNT_Pos)              /*!< SPI_T::SSCTL: SLVTOCNT Mask               */

#define SPI_FIFOCTL_RXRST_Pos            (0)                                               /*!< SPI_T::FIFOCTL: RXRST Position            */
#define SPI_FIFOCTL_RXRST_Msk            (0x1ul << SPI_FIFOCTL_RXRST_Pos)                  /*!< SPI_T::FIFOCTL: RXRST Mask                */

#define SPI_FIFOCTL_TXRST_Pos            (1)                                               /*!< SPI_T::FIFOCTL: TXRST Position            */
#define SPI_FIFOCTL_TXRST_Msk            (0x1ul << SPI_FIFOCTL_TXRST_Pos)                  /*!< SPI_T::FIFOCTL: TXRST Mask                */

#define SPI_FIFOCTL_RXTHIEN_Pos          (2)                                               /*!< SPI_T::FIFOCTL: RXTHIEN Position          */
#define SPI_FIFOCTL_RXTHIEN_Msk          (0x1ul << SPI_FIFOCTL_RXTHIEN_Pos)                /*!< SPI_T::FIFOCTL: RXTHIEN Mask              */

#define SPI_FIFOCTL_TXTHIEN_Pos          (3)                                               /*!< SPI_T::FIFOCTL: TXTHIEN Position          */
#define SPI_FIFOCTL_TXTHIEN_Msk          (0x1ul << SPI_FIFOCTL_TXTHIEN_Pos)                /*!< SPI_T::FIFOCTL: TXTHIEN Mask              */

#define SPI_FIFOCTL_RXTOIEN_Pos          (4)                                               /*!< SPI_T::FIFOCTL: RXTOIEN Position          */
#define SPI_FIFOCTL_RXTOIEN_Msk          (0x1ul << SPI_FIFOCTL_RXTOIEN_Pos)                /*!< SPI_T::FIFOCTL: RXTOIEN Mask              */

#define SPI_FIFOCTL_RXOVIEN_Pos          (5)                                               /*!< SPI_T::FIFOCTL: RXOVIEN Position          */
#define SPI_FIFOCTL_RXOVIEN_Msk          (0x1ul << SPI_FIFOCTL_RXOVIEN_Pos)                /*!< SPI_T::FIFOCTL: RXOVIEN Mask              */

#define SPI_FIFOCTL_TXUFPOL_Pos          (6)                                               /*!< SPI_T::FIFOCTL: TXUFPOL Position          */
#define SPI_FIFOCTL_TXUFPOL_Msk          (0x1ul << SPI_FIFOCTL_TXUFPOL_Pos)                /*!< SPI_T::FIFOCTL: TXUFPOL Mask              */

#define SPI_FIFOCTL_TXUFIEN_Pos          (7)                                               /*!< SPI_T::FIFOCTL: TXUFIEN Position          */
#define SPI_FIFOCTL_TXUFIEN_Msk          (0x1ul << SPI_FIFOCTL_TXUFIEN_Pos)                /*!< SPI_T::FIFOCTL: TXUFIEN Mask              */

#define SPI_FIFOCTL_RXFBCLR_Pos          (8)                                               /*!< SPI_T::FIFOCTL: RXFBCLR Position          */
#define SPI_FIFOCTL_RXFBCLR_Msk          (0x1ul << SPI_FIFOCTL_RXFBCLR_Pos)                /*!< SPI_T::FIFOCTL: RXFBCLR Mask              */

#define SPI_FIFOCTL_TXFBCLR_Pos          (9)                                               /*!< SPI_T::FIFOCTL: TXFBCLR Position          */
#define SPI_FIFOCTL_TXFBCLR_Msk          (0x1ul << SPI_FIFOCTL_TXFBCLR_Pos)                /*!< SPI_T::FIFOCTL: TXFBCLR Mask              */

#define SPI_FIFOCTL_RXTH_Pos             (24)                                              /*!< SPI_T::FIFOCTL: RXTH Position             */
#define SPI_FIFOCTL_RXTH_Msk             (0x7ul << SPI_FIFOCTL_RXTH_Pos)                   /*!< SPI_T::FIFOCTL: RXTH Mask                 */

#define SPI_FIFOCTL_TXTH_Pos             (28)                                              /*!< SPI_T::FIFOCTL: TXTH Position             */
#define SPI_FIFOCTL_TXTH_Msk             (0x7ul << SPI_FIFOCTL_TXTH_Pos)                   /*!< SPI_T::FIFOCTL: TXTH Mask                 */

#define SPI_STATUS_BUSY_Pos              (0)                                               /*!< SPI_T::STATUS: BUSY Position              */
#define SPI_STATUS_BUSY_Msk              (0x1ul << SPI_STATUS_BUSY_Pos)                    /*!< SPI_T::STATUS: BUSY Mask                  */

#define SPI_STATUS_UNITIF_Pos            (1)                                               /*!< SPI_T::STATUS: UNITIF Position            */
#define SPI_STATUS_UNITIF_Msk            (0x1ul << SPI_STATUS_UNITIF_Pos)                  /*!< SPI_T::STATUS: UNITIF Mask                */

#define SPI_STATUS_SSACTIF_Pos           (2)                                               /*!< SPI_T::STATUS: SSACTIF Position           */
#define SPI_STATUS_SSACTIF_Msk           (0x1ul << SPI_STATUS_SSACTIF_Pos)                 /*!< SPI_T::STATUS: SSACTIF Mask               */

#define SPI_STATUS_SSINAIF_Pos           (3)                                               /*!< SPI_T::STATUS: SSINAIF Position           */
#define SPI_STATUS_SSINAIF_Msk           (0x1ul << SPI_STATUS_SSINAIF_Pos)                 /*!< SPI_T::STATUS: SSINAIF Mask               */

#define SPI_STATUS_SSLINE_Pos            (4)                                               /*!< SPI_T::STATUS: SSLINE Position            */
#define SPI_STATUS_SSLINE_Msk            (0x1ul << SPI_STATUS_SSLINE_Pos)                  /*!< SPI_T::STATUS: SSLINE Mask                */

#define SPI_STATUS_SLVTOIF_Pos           (5)                                               /*!< SPI_T::STATUS: SLVTOIF Position           */
#define SPI_STATUS_SLVTOIF_Msk           (0x1ul << SPI_STATUS_SLVTOIF_Pos)                 /*!< SPI_T::STATUS: SLVTOIF Mask               */

#define SPI_STATUS_SLVBEIF_Pos           (6)                                               /*!< SPI_T::STATUS: SLVBEIF Position           */
#define SPI_STATUS_SLVBEIF_Msk           (0x1ul << SPI_STATUS_SLVBEIF_Pos)                 /*!< SPI_T::STATUS: SLVBEIF Mask               */

#define SPI_STATUS_SLVURIF_Pos           (7)                                               /*!< SPI_T::STATUS: SLVURIF Position           */
#define SPI_STATUS_SLVURIF_Msk           (0x1ul << SPI_STATUS_SLVURIF_Pos)                 /*!< SPI_T::STATUS: SLVURIF Mask               */

#define SPI_STATUS_RXEMPTY_Pos           (8)                                               /*!< SPI_T::STATUS: RXEMPTY Position           */
#define SPI_STATUS_RXEMPTY_Msk           (0x1ul << SPI_STATUS_RXEMPTY_Pos)                 /*!< SPI_T::STATUS: RXEMPTY Mask               */

#define SPI_STATUS_RXFULL_Pos            (9)                                               /*!< SPI_T::STATUS: RXFULL Position            */
#define SPI_STATUS_RXFULL_Msk            (0x1ul << SPI_STATUS_RXFULL_Pos)                  /*!< SPI_T::STATUS: RXFULL Mask                */

#define SPI_STATUS_RXTHIF_Pos            (10)                                              /*!< SPI_T::STATUS: RXTHIF Position            */
#define SPI_STATUS_RXTHIF_Msk            (0x1ul << SPI_STATUS_RXTHIF_Pos)                  /*!< SPI_T::STATUS: RXTHIF Mask                */

#define SPI_STATUS_RXOVIF_Pos            (11)                                              /*!< SPI_T::STATUS: RXOVIF Position            */
#define SPI_STATUS_RXOVIF_Msk            (0x1ul << SPI_STATUS_RXOVIF_Pos)                  /*!< SPI_T::STATUS: RXOVIF Mask                */

#define SPI_STATUS_RXTOIF_Pos            (12)                                              /*!< SPI_T::STATUS: RXTOIF Position            */
#define SPI_STATUS_RXTOIF_Msk            (0x1ul << SPI_STATUS_RXTOIF_Pos)                  /*!< SPI_T::STATUS: RXTOIF Mask                */

#define SPI_STATUS_SPIENSTS_Pos          (15)                                              /*!< SPI_T::STATUS: SPIENSTS Position          */
#define SPI_STATUS_SPIENSTS_Msk          (0x1ul << SPI_STATUS_SPIENSTS_Pos)                /*!< SPI_T::STATUS: SPIENSTS Mask              */

#define SPI_STATUS_TXEMPTY_Pos           (16)                                              /*!< SPI_T::STATUS: TXEMPTY Position           */
#define SPI_STATUS_TXEMPTY_Msk           (0x1ul << SPI_STATUS_TXEMPTY_Pos)                 /*!< SPI_T::STATUS: TXEMPTY Mask               */

#define SPI_STATUS_TXFULL_Pos            (17)                                              /*!< SPI_T::STATUS: TXFULL Position            */
#define SPI_STATUS_TXFULL_Msk            (0x1ul << SPI_STATUS_TXFULL_Pos)                  /*!< SPI_T::STATUS: TXFULL Mask                */

#define SPI_STATUS_TXTHIF_Pos            (18)                                              /*!< SPI_T::STATUS: TXTHIF Position            */
#define SPI_STATUS_TXTHIF_Msk            (0x1ul << SPI_STATUS_TXTHIF_Pos)                  /*!< SPI_T::STATUS: TXTHIF Mask                */

#define SPI_STATUS_TXUFIF_Pos            (19)                                              /*!< SPI_T::STATUS: TXUFIF Position            */
#define SPI_STATUS_TXUFIF_Msk            (0x1ul << SPI_STATUS_TXUFIF_Pos)                  /*!< SPI_T::STATUS: TXUFIF Mask                */

#define SPI_STATUS_TXRXRST_Pos           (23)                                              /*!< SPI_T::STATUS: TXRXRST Position           */
#define SPI_STATUS_TXRXRST_Msk           (0x1ul << SPI_STATUS_TXRXRST_Pos)                 /*!< SPI_T::STATUS: TXRXRST Mask               */

#define SPI_STATUS_RXCNT_Pos             (24)                                              /*!< SPI_T::STATUS: RXCNT Position             */
#define SPI_STATUS_RXCNT_Msk             (0xful << SPI_STATUS_RXCNT_Pos)                   /*!< SPI_T::STATUS: RXCNT Mask                 */

#define SPI_STATUS_TXCNT_Pos             (28)                                              /*!< SPI_T::STATUS: TXCNT Position             */
#define SPI_STATUS_TXCNT_Msk             (0xful << SPI_STATUS_TXCNT_Pos)                   /*!< SPI_T::STATUS: TXCNT Mask                 */

#define SPI_TX_TX_Pos                    (0)                                               /*!< SPI_T::TX: TX Position                    */
#define SPI_TX_TX_Msk                    (0xfffffffful << SPI_TX_TX_Pos)                   /*!< SPI_T::TX: TX Mask                        */

#define SPI_RX_RX_Pos                    (0)                                               /*!< SPI_T::RX: RX Position                    */
#define SPI_RX_RX_Msk                    (0xfffffffful << SPI_RX_RX_Pos)                   /*!< SPI_T::RX: RX Mask                        */

/**@}*/ /* SPI_CONST */
/**@}*/ /* end of SPI register group */


/*---------------------- Timer Controller -------------------------*/
/**
    @addtogroup TIMER Timer Controller(TIMER)
    Memory Mapped Structure for TIMER Controller
@{ */

typedef struct
{


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x00  Timer Control and Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |PSC       |PSC Counter
     * |        |          |Timer input clock source is divided by (PSC+1) before it is fed to the timer up counter.
     * |        |          |If this field is 0 (PSC = 0), then there is no scalin.
     * |[16]    |CNTDATEN  |Data Load Enable Control
     * |        |          |When this bit is set, timer counter value (TIMERx_CNT) will be updated continuously to monitor internal 24-bit up counter value as the counter is counting.
     * |        |          |0 = Timer Data Register update Disabled.
     * |        |          |1 = Timer Data Register update Enabled while timer counter is active.
     * |[23]    |WKEN      |Wake-Up Enable Control
     * |        |          |If this bit is set to 1, while timer interrupt flag TIF (TIMERx_INTSTS[0]) is 1 and INTEN (TIMERx_CTL[29]) is enabled, the timer interrupt signal will generate a wake-up trigger event to CPU.
     * |        |          |0 = Wake-up trigger event Disabled if timer interrupt signal generated.
     * |        |          |1 = Wake-up trigger event Enabled if timer interrupt signal generated.
     * |[24]    |EXTCNTEN  |Counter Mode Enable Control
     * |        |          |This bit is for external counting pin function to be enabled.
     * |        |          |When timer is used as an event counter, this bit should be set to 1 and HXT selected as timer clock sourc.
     * |        |          |0 = External counter mode Disabled.
     * |        |          |1 = External counter mode Enabled.
     * |[25]    |ACTSTS    |Timer Active Status Control (Read Only)
     * |        |          |This bit indicates the 24-bit up counter status.
     * |        |          |0 = 24-bit up counter is not active.
     * |        |          |1 = 24-bit up counter is active.
     * |[26]    |RSTCNT    |Timer Reset Control
     * |        |          |Setting this bit will reset the 24-bit up counter value (TIMERx_CNT) and also force CNTEN (TIMERx_CTL[30]) to 0 if ACTSTS (TIMERx_CTL[25]) is 1.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset 8-bit PSC counter, 24-bit up counter value and CNTEN bit.
     * |[28:27] |OPMODE    |Timer Operation Mode
     * |        |          |00 = Timer controller is operated in One-shot mode.
     * |        |          |01 = Timer controller is operated in Periodic mode.
     * |        |          |10 = Timer controller is operated in Toggle-output mode.
     * |        |          |11 = Timer controller is operated in Continuous Counting mode.
     * |[29]    |INTEN     |Interrupt Enable Control
     * |        |          |0 = Timer Interrupt Disabled.
     * |        |          |1 = Timer Interrupt Enabled.
     * |        |          |Note: If this bit is enabled, when the timer interrupt flag TIF is set to 1, the timer interrupt signal is generated and inform to CPU.
     * |[30]    |CNTEN     |Timer Enable Control
     * |        |          |0 = Stops/Suspends counting.
     * |        |          |1 = Starts counting.
     * |        |          |Note1: In stop status, and then set CNTEN to 1 will enable the 24-bit up counter to keep counting from the last stop counting value.
     * |        |          |Note2: This bit is auto-cleared by hardware in One-shot mode (TIMERx_CTL [28:27] = 00) when the timer interrupt flag TIF (TIMERx_INTSTS[0]) is generated.
     * |[31]    |ICEDEBUG  |ICE Debug Mode Acknowledge Disable Control
     * |        |          |0 = ICE debug mode acknowledgement effects TIMER counting.
     * |        |          |Timer counter will be held while CPU is held by ICE.
     * |        |          |1 = ICE debug mode acknowledgement Disabled.
     * |        |          |Timer counter will keep going no matter CPU is held by ICE or not.
    */
    __IO uint32_t CTL;

    /**
     * CMP
     * ===================================================================================================
     * Offset: 0x04  Timer Compare Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CMPDAT    |Timer Compared Value
     * |        |          |CMPDAT is a 24-bit compared value register.
     * |        |          |When the internal 24-bit up counter value is equal to CMPDAT value, the TIF (TIMERx_INTSTS[0] timer interrupt flag) will set to .
     * |        |          |Time-out period = (Period of timer clock input) * (8-bit PSC +1) * (24-bit CMPDAT).
     * |        |          |Note1: Never write 0x0 or 0x1 in the CMPDAT field, or the timer will run into unknown state.
     * |        |          |Note2: When timer is operating in Continuous Counting mode, the 24-bit up counter will keep counting continuously even if software writes a new value into CMPDAT field.
     * |        |          |But if timer is operating at other modes, the 24-bit up counter will restart counting and using the newest CMPDAT value to be the timer compared value if software writes a new value to the CMPDAT fiel.
    */
    __IO uint32_t CMP;

    /**
     * INTSTS
     * ===================================================================================================
     * Offset: 0x08  Timer Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TIF       |Timer Interrupt Flag
     * |        |          |This bit indicates the interrupt flag status of Timer while TIMERx_CNT value reaches to CMPDAT value.
     * |        |          |0 = No effect.
     * |        |          |1 = TIMERx_CNT value matches the CMPDAT value.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[1]     |TWKF      |Timer Wake-Up Flag
     * |        |          |This bit indicates the interrupt wake-up flag status of timer.
     * |        |          |0 = Timer does not cause CPU wake-up.
     * |        |          |1 = CPU wake-up from Power-down mode and Deep Power-down mode, if timer time-out interrupt signal generated.
     * |        |          |Note: This bit is cleared by writing 1 to it.
    */
    __IO uint32_t INTSTS;

    /**
     * CNT
     * ===================================================================================================
     * Offset: 0x0C  Timer Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CNT       |Timer Data Register
     * |        |          |1. EXTCNTEN (TIMERx_CTL[24] ) = 0 : CNT is 24- bit counter value.
     * |        |          |User can read TIMERx_CNT for getting current 24- bit counter value if TIMERx_CTL[24] is set to 0
     * |        |          |2. EXTCNTEN (TIMERx_CTL[24] ) = 1 : CNT is 24- bit event counter value.
     * |        |          |Note: User can read CNT for getting the current 24- bit event counter value if TIMERx_CTL[24] is 1
    */
    __I  uint32_t CNT;

    /**
     * CAP
     * ===================================================================================================
     * Offset: 0x10  Timer Capture Data Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[23:0]  |CAPDAT    |Timer Capture Data Register
     * |        |          |When the CAPEN (TIMERx_EXTCTL[3]) bit is set, CAPFUNCS (TIMERx_EXTCTL[4]) bit is 0, and a transition on TMx_EXT pin matched the CAPEDGE (TIMERx_EXTCTL[2:1]) setting, CAPIF (TIMERx_EINTSTS[0]) will be set to 1 and the current timer counter value (TIMERx_CNT value) will be auto-loaded into the CAPDAT field.
    */
    __I  uint32_t CAP;

    /**
     * EXTCTL
     * ===================================================================================================
     * Offset: 0x14  Timer External Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CNTPHASE  |Timer External Count Phase
     * |        |          |This bit indicates the detection phase of external counting pin.
     * |        |          |0 = A falling edge of external counting pin will be counted.
     * |        |          |1 = A rising edge of external counting pin will be counted.
     * |[2:1]   |CAPEDGE   |Timer External Pin Edge Detect
     * |        |          |00 = A 1 to 0 transition on TMx_EXT (x= 0~3) pin will be detected.
     * |        |          |01 = A 0 to 1 transition on TMx_EXT (x= 0~3) pin will be detected.
     * |        |          |10 = Either 1 to 0 or 0 to 1 transition on TMx_EXT (x= 0~3) pin will be detected.
     * |        |          |11 = Reserved.
     * |[3]     |CAPEN     |Timer External Pin Enable Control
     * |        |          |This bit enables the CAPFUNCS (TIMERx_EXTCTL[4]) function on the TMx_EXT pin.
     * |        |          |0 = CAPFUNCS function of TMx_EXT (x= 0~3) pin will be ignored.
     * |        |          |1 = CAPFUNCS function of TMx_EXT (x= 0~3) pin is active.
     * |[4]     |CAPFUNCS  |Timer External Reset Counter / Capture Mode Select
     * |        |          |0 = Transition on TMx_EXT (x= 0~3) pin is using to save the 24-bit timer counter value.
     * |        |          |(TIMERx_CNT value) to timer capture value (TIMERx_CAP value) if CAPIF (TIMERx_EINTSTS[0]) is set to 1
     * |        |          |1 = Transition on TMx_EXT (x= 0~3) pin is using to reset the 24-bit timer counter value.
     * |[5]     |CAPIEN    |Timer External Interrupt Enable Control
     * |        |          |0 = TMx_EXT (x= 0~3) pin detection Interrupt Disabled.
     * |        |          |1 = TMx_EXT (x= 0~3) pin detection Interrupt Enabled.
     * |        |          |Note: CAPIEN is used to enable timer external interrupt.
     * |        |          |If CAPIEN enabled, timer will rise an interrupt when CAPIF = .
     * |        |          |For example, while CAPIEN = 1, CAPEN = 1, and CAPEDGE = 00, a 1 to 0 transition on the TMx_EXT pin will cause the CAPIF(TIMERx_EINTSTS[0]) interrupt flag to be set then the interrupt signal is generated and sent to NVIC to inform CPU.
     * |[6]     |CAPDBEN   |Timer External Capture Pin De-Bounce Enable Control
     * |        |          |0 = TMx_EXT (x= 0~3) pin de-bounce Disabled.
     * |        |          |1 = TMx_EXT (x= 0~3) pin de-bounce Enabled.
     * |        |          |Note: If this bit is enabled, the edge detection of TMx_EXT pin is detected with de-bounce circuit.
     * |[7]     |CNTDBEN   |Timer Counter Pin De-Bounce Enable Control
     * |        |          |0 = TMx_CNT_OUT (x= 0~3) pin de-bounce Disabled.
     * |        |          |1 = TMx_CNT_OUT (x= 0~3) pin de-bounce Enabled.
     * |        |          |Note: If this bit is enabled, the edge detection of TMx_CNT_OUT pin is detected with de-bounce circuit.
    */
    __IO uint32_t EXTCTL;

    /**
     * EINTSTS
     * ===================================================================================================
     * Offset: 0x18  Timer External Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CAPIF     |Timer External Interrupt Flag
     * |        |          |This bit indicates the timer external interrupt flag status.
     * |        |          |When CAPEN (TIMERx_EXTCTL[3]) bit is set, CAPFUNCS (TIMERx_EXTCTL[4]) bit is 0, and a transition on TMx_EXT (x= 0~3) pin matched the CAPEDGE (TIMERx_EXTCTL[2:1]) setting, this bit will set to 1 by hardware.
     * |        |          |0 = TMx_EXT (x= 0~3) pin interrupt did not occur.
     * |        |          |1 = TMx_EXT (x= 0~3) pin interrupt occurred.
     * |        |          |Note: This bit is cleared by writing 1 to it.
    */
    __IO uint32_t EINTSTS;
    __I  uint32_t RESERVE0[1];

} TIMER_T;

/**
    @addtogroup TIMER_CONST TIMER Bit Field Definition
    Constant Definitions for TIMER Controller
@{ */

#define TIMER_CTL_PSC_Pos                (0)                                               /*!< TIMER_T::CTL: PSC Position                */
#define TIMER_CTL_PSC_Msk                (0xfful << TIMER_CTL_PSC_Pos)                     /*!< TIMER_T::CTL: PSC Mask                    */

#define TIMER_CTL_CNTDATEN_Pos           (16)                                              /*!< TIMER_T::CTL: CNTDATEN Position           */
#define TIMER_CTL_CNTDATEN_Msk           (0x1ul << TIMER_CTL_CNTDATEN_Pos)                 /*!< TIMER_T::CTL: CNTDATEN Mask               */

#define TIMER_CTL_WKEN_Pos               (23)                                              /*!< TIMER_T::CTL: WKEN Position               */
#define TIMER_CTL_WKEN_Msk               (0x1ul << TIMER_CTL_WKEN_Pos)                     /*!< TIMER_T::CTL: WKEN Mask                   */

#define TIMER_CTL_EXTCNTEN_Pos           (24)                                              /*!< TIMER_T::CTL: EXTCNTEN Position           */
#define TIMER_CTL_EXTCNTEN_Msk           (0x1ul << TIMER_CTL_EXTCNTEN_Pos)                 /*!< TIMER_T::CTL: EXTCNTEN Mask               */

#define TIMER_CTL_ACTSTS_Pos             (25)                                              /*!< TIMER_T::CTL: ACTSTS Position             */
#define TIMER_CTL_ACTSTS_Msk             (0x1ul << TIMER_CTL_ACTSTS_Pos)                   /*!< TIMER_T::CTL: ACTSTS Mask                 */

#define TIMER_CTL_RSTCNT_Pos             (26)                                              /*!< TIMER_T::CTL: RSTCNT Position             */
#define TIMER_CTL_RSTCNT_Msk             (0x1ul << TIMER_CTL_RSTCNT_Pos)                   /*!< TIMER_T::CTL: RSTCNT Mask                 */

#define TIMER_CTL_OPMODE_Pos             (27)                                              /*!< TIMER_T::CTL: OPMODE Position             */
#define TIMER_CTL_OPMODE_Msk             (0x3ul << TIMER_CTL_OPMODE_Pos)                   /*!< TIMER_T::CTL: OPMODE Mask                 */

#define TIMER_CTL_INTEN_Pos              (29)                                              /*!< TIMER_T::CTL: INTEN Position              */
#define TIMER_CTL_INTEN_Msk              (0x1ul << TIMER_CTL_INTEN_Pos)                    /*!< TIMER_T::CTL: INTEN Mask                  */

#define TIMER_CTL_CNTEN_Pos              (30)                                              /*!< TIMER_T::CTL: CNTEN Position              */
#define TIMER_CTL_CNTEN_Msk              (0x1ul << TIMER_CTL_CNTEN_Pos)                    /*!< TIMER_T::CTL: CNTEN Mask                  */

#define TIMER_CTL_ICEDEBUG_Pos           (31)                                              /*!< TIMER_T::CTL: ICEDEBUG Position           */
#define TIMER_CTL_ICEDEBUG_Msk           (0x1ul << TIMER_CTL_ICEDEBUG_Pos)                 /*!< TIMER_T::CTL: ICEDEBUG Mask               */

#define TIMER_CMP_CMPDAT_Pos             (0)                                               /*!< TIMER_T::CMP: CMPDAT Position             */
#define TIMER_CMP_CMPDAT_Msk             (0xfffffful << TIMER_CMP_CMPDAT_Pos)              /*!< TIMER_T::CMP: CMPDAT Mask                 */

#define TIMER_INTSTS_TIF_Pos             (0)                                               /*!< TIMER_T::INTSTS: TIF Position             */
#define TIMER_INTSTS_TIF_Msk             (0x1ul << TIMER_INTSTS_TIF_Pos)                   /*!< TIMER_T::INTSTS: TIF Mask                 */

#define TIMER_INTSTS_TWKF_Pos            (1)                                               /*!< TIMER_T::INTSTS: TWKF Position            */
#define TIMER_INTSTS_TWKF_Msk            (0x1ul << TIMER_INTSTS_TWKF_Pos)                  /*!< TIMER_T::INTSTS: TWKF Mask                */

#define TIMER_CNT_CNT_Pos                (0)                                               /*!< TIMER_T::CNT: CNT Position                */
#define TIMER_CNT_CNT_Msk                (0xfffffful << TIMER_CNT_CNT_Pos)                 /*!< TIMER_T::CNT: CNT Mask                    */

#define TIMER_CAP_CAPDAT_Pos             (0)                                               /*!< TIMER_T::CAP: CAPDAT Position             */
#define TIMER_CAP_CAPDAT_Msk             (0xfffffful << TIMER_CAP_CAPDAT_Pos)              /*!< TIMER_T::CAP: CAPDAT Mask                 */

#define TIMER_EXTCTL_CNTPHASE_Pos        (0)                                               /*!< TIMER_T::EXTCTL: CNTPHASE Position        */
#define TIMER_EXTCTL_CNTPHASE_Msk        (0x1ul << TIMER_EXTCTL_CNTPHASE_Pos)              /*!< TIMER_T::EXTCTL: CNTPHASE Mask            */

#define TIMER_EXTCTL_CAPEDGE_Pos         (1)                                               /*!< TIMER_T::EXTCTL: CAPEDGE Position         */
#define TIMER_EXTCTL_CAPEDGE_Msk         (0x3ul << TIMER_EXTCTL_CAPEDGE_Pos)               /*!< TIMER_T::EXTCTL: CAPEDGE Mask             */

#define TIMER_EXTCTL_CAPEN_Pos           (3)                                               /*!< TIMER_T::EXTCTL: CAPEN Position           */
#define TIMER_EXTCTL_CAPEN_Msk           (0x1ul << TIMER_EXTCTL_CAPEN_Pos)                 /*!< TIMER_T::EXTCTL: CAPEN Mask               */

#define TIMER_EXTCTL_CAPFUNCS_Pos        (4)                                               /*!< TIMER_T::EXTCTL: CAPFUNCS Position        */
#define TIMER_EXTCTL_CAPFUNCS_Msk        (0x1ul << TIMER_EXTCTL_CAPFUNCS_Pos)              /*!< TIMER_T::EXTCTL: CAPFUNCS Mask            */

#define TIMER_EXTCTL_CAPIEN_Pos          (5)                                               /*!< TIMER_T::EXTCTL: CAPIEN Position          */
#define TIMER_EXTCTL_CAPIEN_Msk          (0x1ul << TIMER_EXTCTL_CAPIEN_Pos)                /*!< TIMER_T::EXTCTL: CAPIEN Mask              */

#define TIMER_EXTCTL_CAPDBEN_Pos         (6)                                               /*!< TIMER_T::EXTCTL: CAPDBEN Position         */
#define TIMER_EXTCTL_CAPDBEN_Msk         (0x1ul << TIMER_EXTCTL_CAPDBEN_Pos)               /*!< TIMER_T::EXTCTL: CAPDBEN Mask             */

#define TIMER_EXTCTL_CNTDBEN_Pos         (7)                                               /*!< TIMER_T::EXTCTL: CNTDBEN Position         */
#define TIMER_EXTCTL_CNTDBEN_Msk         (0x1ul << TIMER_EXTCTL_CNTDBEN_Pos)               /*!< TIMER_T::EXTCTL: CNTDBEN Mask             */

#define TIMER_EINTSTS_CAPIF_Pos          (0)                                               /*!< TIMER_T::EINTSTS: CAPIF Position          */
#define TIMER_EINTSTS_CAPIF_Msk          (0x1ul << TIMER_EINTSTS_CAPIF_Pos)                /*!< TIMER_T::EINTSTS: CAPIF Mask              */

/**@}*/ /* TIMER_CONST */
/**@}*/ /* end of TIMER register group */


/*---------------------- Universal Asynchronous Receiver/Transmitter Controller -------------------------*/
/**
    @addtogroup UART Universal Asynchronous Receiver/Transmitter Controller(UART)
    Memory Mapped Structure for UART Controller
@{ */

typedef struct
{


    /**
     * DAT
     * ===================================================================================================
     * Offset: 0x00  UART Receive/Transmit Buffer Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |DAT       |Receiving/Transmit Buffer
     * |        |          |Write Operation:
     * |        |          |By writing one byte to this register, the data byte will be stored in transmitter FIFO.
     * |        |          |The UART Controller will send out the data stored in transmitter FIFO top location through the UART_TX.
     * |        |          |Read Operation:
     * |        |          |By reading this register, the UART will return an 8-bit data received from receiving FIFO.
    */
    __IO uint32_t DAT;

    /**
     * INTEN
     * ===================================================================================================
     * Offset: 0x04  UART Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIEN    |Receive Data Available Interrupt Enable Control
     * |        |          |0 = Receive data available interrupt Disabled.
     * |        |          |1 = Receive data available interrupt Enabled.
     * |[1]     |THREIEN   |Transmit Holding Register Empty Interrupt Enable Control
     * |        |          |0 = Transmit holding register empty interrupt Disabled.
     * |        |          |1 = Transmit holding register empty interrupt Enabled.
     * |[2]     |RLSIEN    |Receive Line Status Interrupt Enable Control
     * |        |          |0 = Receive Line Status interrupt Disabled.
     * |        |          |1 = Receive Line Status interrupt Enabled.
     * |[3]     |MODEMIEN  |Modem Status Interrupt Enable Control
     * |        |          |0 = Modem status interrupt Disabled.
     * |        |          |1 = Modem status interrupt Enabled.
     * |[4]     |RXTOIEN   |RX Time-out Interrupt Enable Control
     * |        |          |0 = RX time-out interrupt Disabled.
     * |        |          |1 = RX time-out interrupt Enabled.
     * |[5]     |BUFERRIEN |Buffer Error Interrupt Enable Control
     * |        |          |0 = Buffer error interrupt Disabled.
     * |        |          |1 = Buffer error interrupt Enabled.
     * |[8]     |LINIEN    |LIN Bus Interrupt Enable Control (Not Available In UART1/UART2)
     * |        |          |0 = LIN bus interrupt Disabled.
     * |        |          |1 = LIN bus interrupt Enabled.
     * |        |          |This bit is used for LIN function mode.
     * |[9]     |WKCTSIEN  |NCTS Wake-Up Interrupt Enable Control
     * |        |          |0 = nCTS wake-up system function Disabled.
     * |        |          |1 = Wake-up system function Enabled.
     * |        |          |When the system is in Power-down mode, an external nCTS change will wake up system from Power-down mod.
     * |[10]    |WKDATIEN  |Incoming Data Wake-Up Interrupt Enable Control
     * |        |          |0 = Incoming data wake-up system function Disabled.
     * |        |          |1 = Incoming data wake-up system function Enabled.
     * |        |          |When the system is in Power-down mode, incoming data will wake up system from Power-down mode.
     * |        |          |Hardware will clear this bit when the incoming data wake-up operation finishes and "system clock" work stable
     * |[11]    |TOCNTEN   |Time-out Counter Enable Control
     * |        |          |0 = Time-out counter Disabled.
     * |        |          |1 = Time-out counter Enabled.
     * |[12]    |ATORTSEN  |NRTS Auto-Flow Control Enable Control
     * |        |          |0 = nRTS auto-flow control Disabled.
     * |        |          |1 = nRTS auto-flow control Enabled.
     * |        |          |When nRTS auto-flow is enabled, if the number of bytes in the RX FIFO equals the RTSTRGLV (UART_FIFO[19:16]), the UART will de-assert nRTS signal.
     * |[13]    |ATOCTSEN  |NCTS Auto-Flow Control Enable Control
     * |        |          |0 = nCTS auto-flow control Disabled.
     * |        |          |1 = nCTS auto-flow control Enabled.
     * |        |          |When nCTS auto-flow is enabled, the UART will send data to external device if nCTS input assert (UART will not send data to device until nCTS is asserted).
     * |[18]    |ABRIEN    |Auto-Baud Rate Interrupt Enable Control
     * |        |          |0 = Auto-baud rate interrupt Disabled.
     * |        |          |1 = Auto-baud rate interrupt Enabled.
    */
    __IO uint32_t INTEN;

    /**
     * FIFO
     * ===================================================================================================
     * Offset: 0x08  UART FIFO Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |RXRST     |RX Field Software Reset Control
     * |        |          |When RXRST (UART_FIFO[1]) is set, all the byte in the receiver FIFO and RX internal state machine are cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the RX internal state machine and pointers.
     * |        |          |Note: This bit will automatically clear at least 3 UART peripheral clock cycles.
     * |[2]     |TXRST     |TX Field Software Reset Control
     * |        |          |When TXRST (UART_FIFO[2]) is set, all the byte in the transmit FIFO and TX internal state machine are cleared.
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the TX internal state machine and pointers.
     * |        |          |Note: This bit will automatically clear at least 3 UART peripheral clock cycles.
     * |[7:4]   |RFITL     |RX FIFO Interrupt Trigger Level
     * |        |          |When the number of bytes in the receive FIFO equals the RFITL, the RDAIF will be set (if RDAIEN (UART_INTEN [0]) enabled, and an interrupt will be generated).
     * |        |          |0000 = RX FIFO Interrupt Trigger Level is 1 byte.
     * |        |          |0001 = RX FIFO Interrupt Trigger Level is 4 bytes.
     * |        |          |0010 = RX FIFO Interrupt Trigger Level is 8 bytes.
     * |        |          |0011 = RX FIFO Interrupt Trigger Level is 14 bytes.
     * |        |          |0100 = RX FIFO Interrupt Trigger Level is 30/14 (64 FIFO/16 FIFO).
     * |        |          |0101 = RX FIFO Interrupt Trigger Level is 46/14 (64 FIFO/16 FIFO).
     * |        |          |0110 = RX FIFO Interrupt Trigger Level is 62/14 (64 FIFO/16 FIFO).
     * |        |          |Others = Reserved.
     * |[8]     |RXOFF     |Receiver Disable Control
     * |        |          |The receiver is disabled or not (set 1 to disable receiver)
     * |        |          |0 = Receiver Enabled.
     * |        |          |1 = Receiver Disabled.
     * |        |          |Note: This bit is used for RS-485 Normal Multi-drop mode.
     * |        |          |It should be programmed before RS485NMM (UART_ALTCTL [8]) is programme.
     * |[19:16] |RTSTRGLV  |NRTS Trigger Level For Auto-Flow Control Use
     * |        |          |0000 = nRTS Trigger Level is 1 bytes.
     * |        |          |0001 = nRTS Trigger Level is 4bytes.
     * |        |          |0010 = nRTS Trigger Level is 8 bytes.
     * |        |          |0011 = nRTS Trigger Level is 14 bytes.
     * |        |          |0100 = nRTS Trigger Level is 30/14 (64 FIFO/16 FIFO).
     * |        |          |0101 = nRTS Trigger Level is 46/14 (64 FIFO/16 FIFO).
     * |        |          |0110 = nRTS Trigger Level is 62/14 (64 FIFO/16 FIFO).
     * |        |          |Others = Reserved.
     * |        |          |Note: This field is used for automatic nRTS flow control.
    */
    __IO uint32_t FIFO;

    /**
     * LINE
     * ===================================================================================================
     * Offset: 0x0C  UART Line Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |WLS       |Word Length Selection
     * |        |          |This field sets UART word length.
     * |        |          |00 = 5 bits.
     * |        |          |01 = 6 bits.
     * |        |          |10 = 7 bits.
     * |        |          |11 = 8 bits.
     * |[2]     |NSB       |Number Of "STOP Bit"
     * |        |          |0 = One "STOP bit" is generated in the transmitted data.
     * |        |          |1 = When select 5-bit word length, 1.5 "STOP bit" is generated in the transmitted data.
     * |        |          |When select 6-, 7- and 8-bit word length, 2 "STOP bit" is generated in the transmitted dat.
     * |[3]     |PBE       |Parity Bit Enable Control
     * |        |          |0 = No parity bit generated Disabled.
     * |        |          |1 = Parity bit generated Enabled.
     * |        |          |Note: Parity bit is generated on each outgoing character and is checked on each incoming data.
     * |[4]     |EPE       |Even Parity Enable Control
     * |        |          |0 = Odd number of logic 1's is transmitted and checked in each word.
     * |        |          |1 = Even number of logic 1's is transmitted and checked in each word.
     * |        |          |Note: This bit is effective only when PBE (UART_LINE[3]) is set.
     * |[5]     |SPE       |Stick Parity Enable Control
     * |        |          |0 = Stick parity Disabled.
     * |        |          |1 = Stick parity Enabled.
     * |        |          |Note: If PBE (UART_LINE[3]) and EPE (UART_LINE[4]) are logic 1, the parity bit is transmitted and checked as logic 0.
     * |        |          |If PBE (UART_LINE[3]) is 1 and EPE (UART_LINE[4]) is 0 then the parity bit is transmitted and checked as .
     * |[6]     |BCB       |Break Control
     * |        |          |0 = Break Control Disabled.
     * |        |          |1 = Break Control Enabled.
     * |        |          |Note: When this bit is set to logic 1, the serial data output (TX) is forced to the Spacing State (logic 0).
     * |        |          |This bit acts only on TX line and has no effect on the transmitter logi.
    */
    __IO uint32_t LINE;

    /**
     * MODEM
     * ===================================================================================================
     * Offset: 0x10  UART Modem Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |RTS       |NRTS (Request-To-Send) Signal Control
     * |        |          |This bit is direct control internal nRTS signal active or not, and then drive the nRTS pin output with RTSACTLV bit configuration.
     * |        |          |0 = nRTS signal is active.
     * |        |          |1 = nRTS signal is inactive.
     * |        |          |Note1: This nRTS signal control bit is not effective when nRTS auto-flow control is enabled in UART function mode.
     * |        |          |Note2: This nRTS signal control bit is not effective when RS-485 auto direction mode (AUD) is enabled in RS-485 function mode.
     * |[9]     |RTSACTLV  |NRTS Pin Active Level
     * |        |          |This bit defines the active level state of nRTS pin output.
     * |        |          |0 = nRTS pin output is high level active.
     * |        |          |1 = nRTS pin output is low level active. (Default)
     * |        |          |Note1: Refer to Figure 6.10-10 and Figure 6.10-11 for UART function mode.
     * |        |          |Note2: Refer to Figure 6.10-21 and Figure 6.10-22 for RS-485 function mode.
     * |[13]    |RTSSTS    |NRTS Pin Status (Read Only)
     * |        |          |This bit mirror from nRTS pin output of voltage logic status.
     * |        |          |0 = nRTS pin output is low level voltage logic state.
     * |        |          |1 = nRTS pin output is high level voltage logic state.
    */
    __IO uint32_t MODEM;

    /**
     * MODEMSTS
     * ===================================================================================================
     * Offset: 0x14  UART Modem Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |CTSDETF   |Detect NCTS State Change Flag (Read Only)
     * |        |          |This bit is set whenever nCTS input has change state, and it will generate Modem interrupt to CPU when MODEMIEN (UART_INTEN [3]) is set to 1.
     * |        |          |0 = nCTS input has not change state.
     * |        |          |1 = nCTS input has change state.
     * |        |          |Note: This bit is read only, but can be cleared by writing "1" to it.
     * |[4]     |CTSSTS    |NCTS Pin Status (Read Only)
     * |        |          |This bit mirror from nCTS pin input of voltage logic status.
     * |        |          |0 = nCTS pin input is low level voltage logic state.
     * |        |          |1 = nCTS pin input is high level voltage logic state.
     * |        |          |Note: This bit echoes when UART Controller peripheral clock is enabled, and nCTS multi-function port is selected.
     * |[8]     |CTSACTLV  |NCTS Pin Active Level
     * |        |          |This bit defines the active level state of nCTS pin input.
     * |        |          |0 = nCTS pin input is high level active.
     * |        |          |1 = nCTS pin input is low level active. (Default)
    */
    __IO  uint32_t MODEMSTS;

    /**
     * FIFOSTS
     * ===================================================================================================
     * Offset: 0x18  UART FIFO Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RXOVIF    |RX Overflow Error Interrupt Flag (Read Only)
     * |        |          |This bit is set when RX FIFO overflow.
     * |        |          |If the number of bytes of received data is greater than RX_FIFO (UART_DAT) size, 16/64 bytes this bit will be set.
     * |        |          |0 = RX FIFO is not overflow.
     * |        |          |1 = RX FIFO is overflow.
     * |        |          |Note: This bit is read only, but can be cleared by writing "1" to it.
     * |[1]     |ABRDIF    |Auto-baud Rate Detect Interrupt (Read Only)
     * |        |          |0 = Auto-baud rate detect function is not finished.
     * |        |          |1 = Auto-baud rate detect function is finished.
     * |        |          |Note1: This bit is set to logic "1" when auto-baud rate detect function is finished.
     * |        |          |Note2: This bit is read only, but can be cleared by writing "1" to it.
     * |[2]     |ABRDTOIF  |Auto-baud Rate Time-out Interrupt (Read Only)
     * |        |          |0 = Auto-baud rate counter is underflow.
     * |        |          |1 = Auto-baud rate counter is overflow.
     * |        |          |Note1: This bit is set to logic "1" in Auto-baud Rate Detect mode and the baud rate counter is overflow.
     * |        |          |Note2: This bit is read only, but can be cleared by writing "1" to it.
     * |[3]     |ADDRDETF  |RS-485 Address Byte Detect Flag (Read Only)
     * |        |          |0 = Receiver detects a data that is not an address bit (bit 9 ='0').
     * |        |          |1 = Receiver detects a data that is an address bit (bit 9 ='1').
     * |        |          |Note1: This field is used for RS-485 function mode and ADDRDEN (UART_ALTCTL[15]) is set to 1 to enable Address detection mode .
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     * |[4]     |PEF       |Parity Error Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid "parity bit".
     * |        |          |0 = No parity error is generated.
     * |        |          |1 = Parity error is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[5]     |FEF       |Framing Error Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received character does not have a valid "stop bit" (that is, the stop bit following the last data bit or parity bit is detected as logic 0).
     * |        |          |0 = No framing error is generated.
     * |        |          |1 = Framing error is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[6]     |BIF       |Break Interrupt Flag (Read Only)
     * |        |          |This bit is set to logic 1 whenever the received data input (RX) is held in the "spacing state" (logic 0) for longer than a full word transmission time (that is, the total time of "start bit" + data bits + parity + stop bits).
     * |        |          |0 = No Break interrupt is generated.
     * |        |          |1 = Break interrupt is generated.
     * |        |          |Note: This bit is read only, but can be cleared by writing '1' to it.
     * |[13:8]  |RXPTR     |RX FIFO Pointer (Read Only)
     * |        |          |This field indicates the RX FIFO Buffer Pointer.
     * |        |          |When UART receives one byte from external device, RXPTR increases one.
     * |        |          |When one byte of RX FIFO is read by CPU, RXPTR decreases on.
     * |        |          |The Maximum value shown in RXPTR is 15/63.
     * |        |          |When the using level of RX FIFO Buffer equal to 16/64, the RXFULL bit is set to 1 and RXPTR will show 0.
     * |        |          |As one byte of RX FIFO is read by CPU, the RXFULL bit is cleared to 0 and RXPTR will show 15/63.
     * |[14]    |RXEMPTY   |Receiver FIFO Empty (Read Only)
     * |        |          |This bit initiate RX FIFO empty or not.
     * |        |          |0 = RX FIFO is not empty.
     * |        |          |1 = RX FIFO is empty.
     * |        |          |Note: When the last byte of RX FIFO has been read by CPU, hardware sets this bit high.
     * |        |          |It will be cleared when UART receives any new dat.
     * |[15]    |RXFULL    |Receiver FIFO Full (Read Only)
     * |        |          |This bit initiates RX FIFO full or not.
     * |        |          |0 = RX FIFO is not full.
     * |        |          |1 = RX FIFO is full.
     * |        |          |Note: This bit is set when the number of usage in RX FIFO Buffer is equal to 16/64, otherwise is cleared by hardware.
     * |[21:16] |TXPTR     |TX FIFO Pointer (Read Only)
     * |        |          |This field indicates the TX FIFO Buffer Pointer.
     * |        |          |When CPU writes one byte into UART_DAT, TXPTR increases one.
     * |        |          |When one byte of TX FIFO is transferred to Transmitter Shift Register, TXPTR decreases on.
     * |        |          |The Maximum value shown in TXPTR is 15/63.
     * |        |          |When the using level of TX FIFO Buffer equal to 16/64, the TXFULL bit is set to 1 and TXPTR will show 0.
     * |        |          |As one byte of TX FIFO is transferred to Transmitter Shift Register, the TXFULL bit is cleared to 0 and TXPTR will show 15/63.
     * |[22]    |TXEMPTY   |Transmitter FIFO Empty (Read Only)
     * |        |          |This bit indicates TX FIFO empty or not.
     * |        |          |0 = TX FIFO is not empty.
     * |        |          |1 = TX FIFO is empty.
     * |        |          |Note: When the last byte of TX FIFO has been transferred to Transmitter Shift Register, hardware sets this bit high.
     * |        |          |It will be cleared when writing data into DAT (TX FIFO not empty.
     * |[23]    |TXFULL    |Transmitter FIFO Full (Read Only)
     * |        |          |This bit indicates TX FIFO full or not.
     * |        |          |0 = TX FIFO is not full.
     * |        |          |1 = TX FIFO is full.
     * |        |          |Note: This bit is set when the number of usage in TX FIFO Buffer is equal to 16/64, otherwise is cleared by hardware.
     * |[24]    |TXOVIF    |TX Overflow Error Interrupt Flag (Read Only)
     * |        |          |If TX FIFO (UART_DAT) is full, an additional write to UART_DAT will cause this bit to logic 1.
     * |        |          |0 = TX FIFO is not overflow.
     * |        |          |1 = TX FIFO is overflow.
     * |        |          |Note: This bit is read only, but can be cleared by writing "1" to it.
     * |[28]    |TXEMPTYF  |Transmitter Empty Flag (Read Only)
     * |        |          |This bit is set by hardware when TX FIFO (UART_DAT) is empty and the STOP bit of the last byte has been transmitted.
     * |        |          |0 = TX FIFO is not empty.
     * |        |          |1 = TX FIFO is empty.
     * |        |          |Note: This bit is cleared automatically when TX FIFO is not empty or the last byte transmission has not completed.
    */
    __IO  uint32_t FIFOSTS;

    /**
     * INTSTS
     * ===================================================================================================
     * Offset: 0x1C  UART Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RDAIF     |Receive Data Available Interrupt Flag (Read Only)
     * |        |          |When the number of bytes in the RX FIFO equals the RFITL then the RDAIF(UART_INTSTS[0]) will be set.
     * |        |          |If RDAIEN (UART_INTEN [0]) is enabled, the RDA interrupt will be generate.
     * |        |          |0 = No RDA interrupt flag is generated.
     * |        |          |1 = RDA interrupt flag is generated.
     * |        |          |Note: This bit is read only and it will be cleared when the number of unread bytes of RX FIFO drops below the threshold level (RFITL(UART_FIFO[7:4]).
     * |[1]     |THREIF    |Transmit Holding Register Empty Interrupt Flag (Read Only)
     * |        |          |This bit is set when the last data of TX FIFO is transferred to Transmitter Shift Register.
     * |        |          |If THREIEN (UART_INTEN[1]) is enabled, the THRE interrupt will be generate.
     * |        |          |0 = No THRE interrupt flag is generated.
     * |        |          |1 = THRE interrupt flag is generated.
     * |        |          |Note: This bit is read only and it will be cleared when writing data into UART_DAT (TX FIFO not empty).
     * |[2]     |RLSIF     |Receive Line Interrupt Flag (Read Only)
     * |        |          |This bit is set when the RX receive data have parity error, frame error or break error (at least one of 3 bits, BIF(UART_FIFOSTS[6]), FEF(UART_FIFOSTS[5]) and PEF(UART_FIFOSTS[4]), is set).
     * |        |          |If RLSIEN (UART_INTEN [2]) is enabled, the RLS interrupt will be generate.
     * |        |          |0 = No RLS interrupt flag is generated.
     * |        |          |1 = RLS interrupt flag is generated.
     * |        |          |Note1: In RS-485 function mode, this field is set include "receiver detect and received address byte character (bit9 = '1') bit".
     * |        |          |At the same time, the bit of ADDRDETF (UART_FIFOSTS[3]) is also se.
     * |        |          |Note2: This bit is read only and reset to 0 when all bits of BIF (UART_FIFOSTS[6]), FEF(UART_FIFOSTS[5]) and PEF(UART_FIFOSTS[4]) are cleared.
     * |        |          |Note3: In RS-485 function mode, this bit is read only and reset to 0 when all bits of BIF (UART_FIFOSTS[6]) , FEF(UART_FIFOSTS[5]) and PEF(UART_FIFOSTS[4]) and ADDRDETF (UART_FIFOSTS[3]) are cleared.
     * |[3]     |MODEMIF   |MODEM Interrupt Flag (Read Only) Channel
     * |        |          |This bit is set when the nCTS pin has state change (CTSDETF (UART_MODEMSTS[0]) = 1).
     * |        |          |If MODEMIEN (UART_INTEN [3]) is enabled, the Modem interrupt will be generate.
     * |        |          |0 = No Modem interrupt flag is generated.
     * |        |          |1 = Modem interrupt flag is generated.
     * |        |          |Note: This bit is read only and reset to 0 when bit CTSDETF is cleared by a write 1 on CTSDETF(UART_MODEMSTS[0]).
     * |[4]     |RXTOIF    |Time-out Interrupt Flag (Read Only)
     * |        |          |This bit is set when the RX FIFO is not empty and no activities occurred in the RX FIFO and the time-out counter equal to TOIC.
     * |        |          |If TOUTIEN (UART_INTEN [4]) is enabled, the Tout interrupt will be generate.
     * |        |          |0 = No Time-out interrupt flag is generated.
     * |        |          |1 = Time-out interrupt flag is generated.
     * |        |          |Note: This bit is read only and user can read UART_DAT (RX is in active) to clear it.
     * |[5]     |BUFERRIF  |Buffer Error Interrupt Flag (Read Only)
     * |        |          |This bit is set when the TX FIFO or RX FIFO overflows (TXOVIF (UART_FIFOSTS[24]) or RXOVIF (UART_FIFOSTS[0]) is set).
     * |        |          |When BUFERRIF (UART_INTSTS[5])is set, the transfer is not correct.
     * |        |          |If BFERRIEN (UART_INTEN [8]) is enabled, the buffer error interrupt will be generate.
     * |        |          |0 = No buffer error interrupt flag is generated.
     * |        |          |1 = Buffer error interrupt flag is generated.
     * |        |          |Note: This bit is read only.
     * |        |          |This bit is cleared if both of RXOVIF(UART_FIFOSTS[0]) and TXOVIF(UART_FIFOSTS[24]) are cleared to 0 by writing 1 to RXOVIF(UART_FIFOSTS[0]) and TXOVIF(UART_FIFOSTS[24].
     * |[6]     |WKIF      |UART Wake-up Interrupt Flag (Read Only)
     * |        |          |This bit is set when DATWKIF (UART_INTSTS[17]) or CTSWKIF(UART_INTSTS[16]) is set to 1.
     * |        |          |0 = No DATWKIF and CTSWKIF are generated.
     * |        |          |1 = DATWKIF or CTSWKIF.
     * |        |          |Note: This bit is read only
     * |        |          |This bit is cleared if both of DATWKIF (UART_INTSTS[17]) and CTSWKIF(UART_INTSTS[16]) are cleared to 0 by writing 1 to DATWKIF (UART_INTSTS[17]) and CTSWKIF (UART_INTSTS[17].
     * |[7]     |LINIF     |LIN Bus Interrupt Flag (Read Only) (Not Available In UART0 Channel)
     * |        |          |This bit is set when LIN slave header detect (SLVHDETF (UART_LINSTS[0] =1)), LIN break detect (BRKDETF(UART_LINSTS[9]=1)), bit error detect (BITEF(UART_LINSTS[9]=1), LIN slave ID parity error (SLVIDPEF(UART_LINSTS[2] = 1) or LIN slave header error detect (SLVHEF (UART_LINSTS[1])).
     * |        |          |If LIN_ IEN (UART_INTEN [8]) is enabled the LIN interrupt will be generate.
     * |        |          |0 = None of SLVHDETF, BRKDETF, BITEF, SLVIDPEF and SLVHEF is generated.
     * |        |          |1 = At least one of SLVHDETF, BRKDETF, BITEF, SLVIDPEF and SLVHEF is generated.
     * |        |          |Note: This bit is read only.
     * |        |          |This bit is cleared when SLVHDETF(UART_LINSTS[0]), BRKDETF(UART_LINSTS[8]), BITEF(UART_LINSTS[9]), SLVIDPEF (UART_LINSTS[2]), SLVHEF(UART_LINSTS[1]) and SLVSYNCF(UART_LINSTS[3]) all are cleare.
     * |[8]     |RDAINT    |Receive Data Available Interrupt Indicator (Read Only)
     * |        |          |This bit is set if RDAIEN (UART_INTEN[0]) and RDAIF (UART_INTSTS[0]) are both set to 1.
     * |        |          |0 = No RDA interrupt is generated.
     * |        |          |1 = RDA interrupt is generated.
     * |[9]     |THREINT   |Transmit Holding Register Empty Interrupt Indicator (Read Only)
     * |        |          |This bit is set if THREIEN (UART_INTEN[1])and THREIF(UART_INTSTS[1]) are both set to 1.
     * |        |          |0 = No DATE interrupt is generated.
     * |        |          |1 = DATE interrupt is generated.
     * |[10]    |RLSINT    |Receive Line Status Interrupt Indicator (Read Only)
     * |        |          |This bit is set if RLSIEN (UART_INTEN[2]) and RLSIF(UART_INTSTS[2]) are both set to 1.
     * |        |          |0 = No RLS interrupt is generated.
     * |        |          |1 = RLS interrupt is generated.
     * |[11]    |MODEMINT  |MODEM Status Interrupt Indicator (Read Only)
     * |        |          |This bit is set if MODEMIEN(UART_INTEN[3] and MODEMIF(UART_INTSTS[4]) are both set to 1
     * |        |          |0 = No Modem interrupt is generated.
     * |        |          |1 = Modem interrupt is generate.
     * |[12]    |RXTOINT   |Time-out Interrupt Indicator (Read Only)
     * |        |          |This bit is set if TOUTIEN(UART_INTEN[4]) and RXTOIF(UART_INTSTS[4]) are both set to 1.
     * |        |          |0 = No Tout interrupt is generated.
     * |        |          |1 = Tout interrupt is generated.
     * |[13]    |BUFERRINT |Buffer Error Interrupt Indicator (Read Only)
     * |        |          |This bit is set if BFERRIEN(UART_INTEN[5] and BUFERRIF(UART_INTSTS[5]) are both set to 1.
     * |        |          |0 = No buffer error interrupt is generated.
     * |        |          |1 = Buffer error interrupt is generated.
     * |[14]    |WKINT     |UART Wake-up Interrupt Indicator (Read Only)
     * |        |          |This bit is set when DATWKIF or CTSWKIF is set to 1.
     * |        |          |0 = NO data or nCTS wake-up interrupt are generated.
     * |        |          |1 = Data or nCTS wake-up interrupt are generated.
     * |[15]    |LININT    |LIN Bus Interrupt Indicator (Read Only)(Not Available in UART0 Channel)
     * |        |          |This bit is set if LINIEN (UART_INTEN[8]) and LIN IF(UART_INTSTS[7]) are both set to 1.
     * |        |          |0 = No LIN Bus interrupt is generated.
     * |        |          |1 = The LIN Bus interrupt is generated.
     * |[16]    |CTSWKIF   |NCTS Wake-Up Interrupt Flag (Read Only)
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by nCTS wake-up.
     * |        |          |Note1: If WKCTSIEN (UART_INTEN[9])is enabled, the wake-up interrupt is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
     * |[17]    |DATWKIF   |Data Wake-Up Interrupt Flag (Read Only)
     * |        |          |This bit is set if chip wake-up from power-down state by data wake-up.
     * |        |          |0 = Chip stays in power-down state.
     * |        |          |1 = Chip wake-up from power-down state by data wake-up.
     * |        |          |Note1: If WKDATIEN (UART_INTEN[10]) is enabled, the wake-up interrupt is generated.
     * |        |          |Note2: This bit is read only, but can be cleared by writing '1' to it.
    */
    __IO  uint32_t INTSTS;

    /**
     * TOUT
     * ===================================================================================================
     * Offset: 0x20  UART Time-out Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |TOIC      |Time-out Interrupt Comparator
     * |        |          |The time-out counter resets and starts counting (the counting clock = baud rate) whenever the RX FIFO receives a new data word.
     * |        |          |Once the content of time-out counter is equal to that of time-out interrupt comparator (TOIC (UART_TOUT[7:0])), a receiver time-out interrupt (RXTOINT(UART_INTSTS[12])) is generated if RXTOIEN (UART_INTEN [4]) enabled.
     * |        |          |A new incoming data word or RX FIFO empty will clear RXTOINT(UART_INTSTS[12]).
     * |        |          |In order to avoid receiver time-out interrupt generation immediately during one character is being received, TOIC value should be set between 40 and 255.
     * |        |          |Thus, for example, if TOIC is set with 40, the time-out interrupt is generated after four characters are not received when 1 stop bit and no parity check is set for UART transfe.
     * |[15:8]  |DLY       |TX Delay Time Value
     * |        |          |This field is used to programming the transfer delay time between the last stop bit and next start bit.
     * |        |          |The unit is bit tim.
    */
    __IO uint32_t TOUT;

    /**
     * BAUD
     * ===================================================================================================
     * Offset: 0x24  UART Baud Rate Divisor Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |BRD       |Baud Rate Divider
     * |        |          |The field indicates the baud rate divider.
     * |        |          |This filed is used in baud rate calculation.
     * |        |          |The detail description is shown in Table 6.10-.
     * |[27:24] |EDIVM1    |Extra Divider For BAUD Rate Mode 1
     * |        |          |This field is used for baud rate calculation in mode 1 and has no effect for baud rate calculation in mode 0 and mode 2.
     * |        |          |The detail description is shown in Table 6.10-2.
     * |[28]    |BAUDM0    |BAUD Rate Mode Selection Bit 0
     * |        |          |This bit is baud rate mode selection bit 0.
     * |        |          |UART provides three baud rate calculation modes.
     * |        |          |This bit combines with BAUDM1 (UART_BAUD[29]) to select baud rate calculation mode.
     * |        |          |The detail description is shown in Table 6.10-.
     * |[29]    |BAUDM1    |BAUD Rate Mode Selection Bit 1
     * |        |          |This bit is baud rate mode selection bit 1.
     * |        |          |UART provides three baud rate calculation modes.
     * |        |          |This bit combines with BAUDM0 (UART_BAUD[28]) to select baud rate calculation mode.
     * |        |          |The detail description is shown in Table 6.10-.
     * |        |          |In IrDA mode must be operated in mode 0.
    */
    __IO uint32_t BAUD;

    /**
     * IRDA
     * ===================================================================================================
     * Offset: 0x28  UART IrDA Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1]     |TXEN      |IrDA Receiver/Transmitter Selection Enable Control
     * |        |          |0 = IrDA Transmitter Disabled and Receiver Enabled. (Default)
     * |        |          |1 = IrDA Transmitter Enabled and Receiver Disabled.
     * |        |          |In IrDA function mode (FUNCSEL(UART_FUNCSEL[1:0])=10), the first received data is unreliable and it should be skipped if IrDA receiver is enabled (TXEN(UART_IRDA[1])=0) at the first time.
     * |[5]     |TXINV     |IrDA Inverse Transmitting Output Signal
     * |        |          |0 = None inverse transmitting signal. (Default)
     * |        |          |1 = Inverse transmitting output signal.
     * |[6]     |RXINV     |IrDA Inverse Receive Input Signal
     * |        |          |0 = None inverse receiving input signal.
     * |        |          |1 = Inverse receiving input signal. (Default)
    */
    __IO uint32_t IRDA;

    /**
     * ALTCTL
     * ===================================================================================================
     * Offset: 0x2C  UART Alternate Control/Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |BRKFL     |UART LIN Break Field Length (Only Available In UART1/UART2 Channel)
     * |        |          |This field indicates a 4-bit LIN TX break field count.
     * |        |          |Note1: This break field length is BRKFL + 1
     * |        |          |Note2: According to LIN spec, the reset value is 0xC (break field length = 13).
     * |[6]     |LINRXEN   |LIN RX Enable Control (Only Available In UART1/UART2 Channel)
     * |        |          |0 = LIN RX mode Disabled.
     * |        |          |1 = LIN RX mode Enabled.
     * |[7]     |LINTXEN   |LIN TX Break Mode Enable Control (Only Available In UART1/UART2 Channel)
     * |        |          |0 = LIN TX Break mode Disabled.
     * |        |          |1 = LIN TX Break mode Enabled.
     * |        |          |Note: When TX break field transfer operation finished, this bit will be cleared automatically.
     * |[8]     |RS485NMM  |RS-485 Normal Multi-Drop Operation Mode (NMM)
     * |        |          |0 = RS-485 Normal Multi-drop Operation mode (NMM) Disabled.
     * |        |          |1 = RS-485 Normal Multi-drop Operation mode (NMM) Enabled.
     * |        |          |Note: It cannot be active with RS-485_AAD operation mode.
     * |[9]     |RS485AAD  |RS-485 Auto Address Detection Operation Mode (AAD)
     * |        |          |0 = RS-485 Auto Address Detection Operation mode (AAD) Disabled.
     * |        |          |1 = RS-485 Auto Address Detection Operation mode (AAD) Enabled.
     * |        |          |It cannot be active with RS-485_NMM operation mode.
     * |[10]    |RS485AUD  |RS-485 Auto Direction Function (AUD)
     * |        |          |0 = RS-485 Auto Direction Operation function (AUD) Disabled.
     * |        |          |1 = RS-485 Auto Direction Operation function (AUD) Enabled.
     * |        |          |Note: It can be active with RS-485_AAD or RS-485_NMM operation mode.
     * |[15]    |ADDRDEN   |RS-485 Address Detection Enable Control
     * |        |          |This bit is used to enable RS-485 Address Detection mode.
     * |        |          |0 = Address detection mode Disabled.
     * |        |          |1 = Address detection mode Enabled.
     * |        |          |Note: This bit is used for RS-485 any operation mode.
     * |[17]    |ABRIF     |Auto-Baud Rate Interrupt Flag (Read Only)
     * |        |          |This bit is set when auto-baud rate detection function finished or the auto-baud rate counter was overflow and if ABRIEN(UART_INTEN [18]) is set then the auto-baud rate interrupt will be generated.
     * |        |          |Note: This bit is read only, but it can be cleared by writing "1" to ABRDTOIF (UART_FIFOSTS[2]) and ABRDIF(UART_FIFOSTS[1])
     * |[18]    |ABRDEN    |Auto-Baud Rate Detect Enable Control
     * |        |          |0 = Auto-baud rate detect function Disabled.
     * |        |          |1 = Auto-baud rate detect function Enabled.
     * |        |          |Note: This bit is cleared automatically after auto-baud detection is finished.
     * |[20:19] |ABRDBITS  |Auto-Baud Rate Detect Bit Length
     * |        |          |00 = 1-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x01.
     * |        |          |01 = 2-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x02.
     * |        |          |10 = 4-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x08.
     * |        |          |11 = 8-bit time from Start bit to the 1st rising edge. The input pattern shall be 0x80.
     * |        |          |Note: The calculation of bit number includes the START bit.
     * |[31:24] |ADDRMV    |Address Match Value
     * |        |          |This field contains the RS-485 address match values.
     * |        |          |This field is used for RS-485 auto address detection mode.
    */
    __IO uint32_t ALTCTL;

    /**
     * FUNCSEL
     * ===================================================================================================
     * Offset: 0x30  UART Function Select Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |FUNCSEL   |Function Select
     * |        |          |00 = UART function.
     * |        |          |01 = LIN function (Only Available in UART1/UART2 Channel).
     * |        |          |10 = IrDA function.
     * |        |          |11 = RS-485 function.
     * |        |          |Note: In IrDA function mode (FUNCSEL(UART_FUNCSEL[1:0])=10), the first received data is unreliable and it should be skipped if IrDA receiver is enabled (TXEN(UART_IRDA[1])=0) at the first time.
    */
    __IO uint32_t FUNCSEL;

    /**
     * LINCTL
     * ===================================================================================================
     * Offset: 0x34  UART LIN Control Register *
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SLVEN     |LIN Slave Mode Enable Control
     * |        |          |0 = LIN Slave mode Disabled.
     * |        |          |1 = LIN Slave mode Enabled.
     * |[1]     |SLVHDEN   |LIN Slave Header Detection Enable Control
     * |        |          |0 = LIN slave header detection Disabled.
     * |        |          |1 = LIN slave header detection Enabled.
     * |        |          |Note1: This bit is only valid in LIN Slave mode (SLVEN (UART_LINCTL[0]) = 1).
     * |        |          |Note2: In LIN function mode, when detect header field (break + sync + frame ID), SLVHDETF (UART_LINSTS [0]) flag will be asserted.
     * |        |          |If the LINIEN (UART_INTEN[8]) = 1, an interrupt will be generate.
     * |[2]     |SLVAREN   |LIN Slave Automatic Resynchronization Mode Enable Control
     * |        |          |0 = LIN automatic resynchronization Disabled.
     * |        |          |1 = LIN automatic resynchronization Enabled.
     * |        |          |Note1: This bit is only valid in LIN Slave mode (SLVEN (UART_LINCTL[0]) = 1).
     * |        |          |Note2: When operating in Automatic Resynchronization mode, the baud rate setting must be mode2 (BAUDM1 (UART_BAUD [29]) and BAUDM0 (UART_BAUD [28]) must be 1).
     * |        |          |Note3: The control and interactions of this field are explained in 6.10.5.9(Slave mode with automatic resynchronization).
     * |[3]     |SLVDUEN   |LIN Slave Divider Update Method Enable Control
     * |        |          |0 = UART_BAUD updated is written by software (if no automatic resynchronization update occurs at the same time).
     * |        |          |1 = UART_BAUD is updated at the next received character
     * |        |          |User must set the bit before checksum reception.
     * |        |          |Note1: This bit is only valid in LIN Slave mode (SLVEN (UART_LINCTL[0]) = 1).
     * |        |          |Note2: This bit is used for LIN Slave Automatic Resynchronization mode
     * |        |          |(for Non-Automatic Resynchronization mode, this bit should be kept cleared)
     * |        |          |Note3: The control and interactions of this field are explained in 6.10.5.9 (Slave mode with automatic resynchronization).
     * |[4]     |MUTE      |LIN Mute Mode Enable Control
     * |        |          |0 = LIN mute mode Disabled.
     * |        |          |1 = LIN mute mode Enabled.
     * |        |          |Note: The exit from mute mode condition and each control and interactions of this field are explained in 6.10.5.9 (LIN Slave mode).
     * |[8]     |SENDH     |LIN TX Send Header Enable Control
     * |        |          |The LIN TX header can be "break field" or "break and sync field" or "break, sync and frame ID field", it is depend on setting HSEL (UART_LINCTL[23:22]).
     * |        |          |0 = Send LIN TX header Disabled.
     * |        |          |1 = Send LIN TX header Enabled.
     * |        |          |Note1: These registers are shadow registers of LINTXEN (UART_ALTCTL [7]); user can read/write it by setting LINTXEN (UART_ALTCTL [7]) or SENDH (UART_LINCTL [8]).
     * |        |          |Note2: When transmitter header field (it may be "break" or "break + sync" or "break + sync + frame ID" selected by HSEL (UART_LINCTL[23:22]) field) transfer operation finished, this bit will be cleared automatically.
     * |[9]     |IDPEN     |LIN ID Parity Enable Control
     * |        |          |0 = LIN frame ID parity Disabled.
     * |        |          |1 = LIN frame ID parity Enabled.
     * |        |          |Note1: This bit can be used for LIN master to sending header field (SENDH (UART_LINCTL[8])) = 1 and HSEL (UART_LINCTL[23:22]) = 10) or be used for enable LIN slave received frame ID parity checked.
     * |        |          |Note2: This bit is only used when the operation header transmitter is in HSEL (UART_LINCTL[23:22]) = 10
     * |[10]    |BRKDETEN  |LIN Break Detection Enable Control
     * |        |          |When detect consecutive dominant greater than 11 bits, and are followed by a delimiter character, the BRKDETF (UART_LINSTS[8]) flag is set in UART_LINSTS register at the end of break field
     * |        |          |If the LINIEN (UART_INTEN [8])=1, an interrupt will be generated.
     * |        |          |0 = LIN break detection Disabled.
     * |        |          |1 = LIN break detection Enabled.
     * |[11]    |RXOFF     |LIN Receiver Disable Control
     * |        |          |If the receiver is enabled (RXOFF (UART_LINCTL[11]) = 0), all received byte data will be accepted and stored in the RX-FIFO, and if the receiver is disabled (RXOFF (UART_LINCTL[11] = 1), all received byte data will be ignore.
     * |        |          |0 = LIN receiver Enabled.
     * |        |          |1 = LIN receiver Disabled.
     * |        |          |Note: This bit is only valid when operating in LIN function mode (FUNCSEL (UART_FUNCSEL[1:0]) = 01).
     * |[12]    |BITERREN  |Bit Error Detect Enable Control
     * |        |          |0 = Bit error detection function Disabled.
     * |        |          |1 = Bit error detection Enabled.
     * |        |          |Note: In LIN function mode, when a bit error occurs, the BITEF (UART_LINSTS[9]) flag will be asserted
     * |        |          |If the LINIEN (UART_INTEN[8]) = 1, an interrupt will be generated.
     * |[19:16] |BRKFL     |LIN Break Field Length
     * |        |          |This field indicates a 4-bit LIN TX break field count.
     * |        |          |Note1: These registers are shadow registers of BRKFL, User can read/write it by setting BRKFL (UART_ALTCTL[3:0]) or BRKFL (UART_LINCTL[19:16]).
     * |        |          |Note2: This break field length is BRKFL + 1.
     * |        |          |Note3: According to LIN spec, the reset value is 12 (break field length = 13).
     * |[21:20] |BSL       |LIN Break/Sync Delimiter Length
     * |        |          |00 = The LIN break/sync delimiter length is 1-bit time.
     * |        |          |01 = The LIN break/sync delimiter length is 2-bit time.
     * |        |          |10 = The LIN break/sync delimiter length is 3-bit time.
     * |        |          |11 = The LIN break/sync delimiter length is 4-bit time.
     * |        |          |Note: This bit is used for LIN master to send header field.
     * |[23:22] |HSEL      |LIN Header Select
     * |        |          |00 = The LIN header includes "break field".
     * |        |          |01 = The LIN header includes "break field" and "sync field".
     * |        |          |10 = The LIN header includes "break field", "sync field" and "frame ID field".
     * |        |          |11 = Reserved.
     * |        |          |Note: This bit is used to master mode for LIN to send header field (SENDH (UART_LINCTL [8]) = 1) or used to slave to indicates exit from mute mode condition (MUTE (UART_LINCTL[4] = 1).
     * |[31:24] |PID       |LIN PID Bits
     * |        |          |This field contains the LIN frame ID value in LIN function mode, the frame ID parity can be generated by software or hardware depending on whether IDPEN (UART_LINCTL[9]) = 1.
     * |        |          |If the parity generated by hardware, and user fills in ID0~ID5, (PID [29:24]), hardware will calculate P0 (PID[30]) and P1 (PID[31]), otherwise user must fill a frame ID and parity in this field.
     * |        |          |Note1: User can fill in any 8-bit value to this field and the bit 24 indicates ID0 (LSB first).
     * |        |          |Note2: This field can be used for LIN Master mode or Slave mode.
    */
    __IO uint32_t LINCTL;

    /**
     * LINSTS
     * ===================================================================================================
     * Offset: 0x38  UART LIN Status Register *
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SLVHDETF  |LIN Slave Header Detection Flag (Read Only)
     * |        |          |This bit is set by hardware when a LIN header is detected in LIN Slave mode and be cleared by writing 1 to it.
     * |        |          |0 = LIN header not detected.
     * |        |          |1 = LIN header detected (break + sync + frame ID).
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is only valid in LIN Slave mode (SLVEN (UART_LINCTL [0]) = 1) and enable LIN slave header detection function (SLVHDEN (UART_LINCTL [1])).
     * |        |          |Note3: When enable ID parity check IDPEN (UART_LINCTL [9]), if hardware detect complete header ("break + sync + frame ID"), the SLVHDETF will be set whether the frame ID correct or not.
     * |[1]     |SLVHEF    |LIN Slave Header Error Flag (Read Only)
     * |        |          |This bit is set by hardware when a LIN header error is detected in LIN Slave mode and be cleared by writing 1 to it.
     * |        |          |The header errors include "break delimiter is too short (less than 0.5 bit time)", "frame error in sync field or Identifier field", "sync field data is not 0x55 in Non-Automatic Resynchronization mode", "sync field deviation error with Automatic Resynchronization mode", "sync field measure time-out with Automatic Resynchronization mode" and "LIN header reception time-out".
     * |        |          |0 = LIN header error not detected.
     * |        |          |1 = LIN header error detected.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is only valid when UART is operated in LIN Slave mode (SLVEN (UART_LINCTL [0]) = 1) and enables LIN slave header detection function (SLVHDEN (UART_LINCTL [1])).
     * |[2]     |SLVIDPEF  |LIN Slave ID Parity Error Flag
     * |        |          |This bit is set by hardware when receipted frame ID parity is not correct.
     * |        |          |0 = No active.
     * |        |          |1 = Receipted frame ID parity is not correct.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is only valid in LIN Slave mode (SLVEN (UART_LINCTL [0])= 1) and enable LIN frame ID parity check function IDPEN (UART_LINCTL [9]).
     * |[3]     |SLVSYNCF  |LIN Slave Sync Field (Read Only)
     * |        |          |This bit indicates that the LIN sync field is being analyzed in Automatic Resynchronization mode.
     * |        |          |When the receiver header have some error been detect, user must reset the internal circuit to re-search new frame header by writing 1 to this bi.
     * |        |          |0 = The current character is not at LIN sync state.
     * |        |          |1 = The current character is at LIN sync state.
     * |        |          |Note1: This bit is only valid in LIN Slave mode (SLVEN(UART_LINCTL[0]) = 1).
     * |        |          |Note2: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note3: When writing 1 to it, hardware will reload the initial baud rate and re-search a new frame header.
     * |[8]     |BRKDETF   |LIN Break Detection Flag (Read Only)
     * |        |          |This bit is set by hardware when a break is detected and be cleared by writing 1 to it through software.
     * |        |          |0 = LIN break not detected.
     * |        |          |1 = LIN break detected.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is only valid when LIN break detection function is enabled (BRKDETEN (UART_LINCTL[10]) =1).
     * |[9]     |BITEF     |Bit Error Detect Status Flag (Read Only)
     * |        |          |At TX transfer state, hardware will monitoring the bus state, if the input pin (SIN) state not equals to the output pin (SOUT) state, BITEF (UART_LINSTS[9]) will be set.
     * |        |          |When occur bit error, if the LINIEN (UART_INTEN[8]) = 1, an interrupt will be generated.
     * |        |          |Note1: This bit is read only, but it can be cleared by writing 1 to it.
     * |        |          |Note2: This bit is only valid when enable bit error detection function (BITERREN (UART_LINCTL [12]) = 1).
    */
    __IO  uint32_t LINSTS;

} UART_T;

/**
    @addtogroup UART_CONST UART Bit Field Definition
    Constant Definitions for UART Controller
@{ */

#define UART_DAT_DAT_Pos                 (0)                                               /*!< UART_T::DAT: DAT Position                 */
#define UART_DAT_DAT_Msk                 (0xfful << UART_DAT_DAT_Pos)                      /*!< UART_T::DAT: DAT Mask                     */

#define UART_INTEN_RDAIEN_Pos            (0)                                               /*!< UART_T::INTEN: RDAIEN Position            */
#define UART_INTEN_RDAIEN_Msk            (0x1ul << UART_INTEN_RDAIEN_Pos)                  /*!< UART_T::INTEN: RDAIEN Mask                */

#define UART_INTEN_THREIEN_Pos           (1)                                               /*!< UART_T::INTEN: THREIEN Position           */
#define UART_INTEN_THREIEN_Msk           (0x1ul << UART_INTEN_THREIEN_Pos)                 /*!< UART_T::INTEN: THREIEN Mask               */

#define UART_INTEN_RLSIEN_Pos            (2)                                               /*!< UART_T::INTEN: RLSIEN Position            */
#define UART_INTEN_RLSIEN_Msk            (0x1ul << UART_INTEN_RLSIEN_Pos)                  /*!< UART_T::INTEN: RLSIEN Mask                */

#define UART_INTEN_MODEMIEN_Pos          (3)                                               /*!< UART_T::INTEN: MODEMIEN Position          */
#define UART_INTEN_MODEMIEN_Msk          (0x1ul << UART_INTEN_MODEMIEN_Pos)                /*!< UART_T::INTEN: MODEMIEN Mask              */

#define UART_INTEN_RXTOIEN_Pos           (4)                                               /*!< UART_T::INTEN: RXTOIEN Position           */
#define UART_INTEN_RXTOIEN_Msk           (0x1ul << UART_INTEN_RXTOIEN_Pos)                 /*!< UART_T::INTEN: RXTOIEN Mask               */

#define UART_INTEN_BUFERRIEN_Pos         (5)                                               /*!< UART_T::INTEN: BUFERRIEN Position         */
#define UART_INTEN_BUFERRIEN_Msk         (0x1ul << UART_INTEN_BUFERRIEN_Pos)               /*!< UART_T::INTEN: BUFERRIEN Mask             */

#define UART_INTEN_LINIEN_Pos            (8)                                               /*!< UART_T::INTEN: LINIEN Position            */
#define UART_INTEN_LINIEN_Msk            (0x1ul << UART_INTEN_LINIEN_Pos)                  /*!< UART_T::INTEN: LINIEN Mask                */

#define UART_INTEN_WKCTSIEN_Pos          (9)                                               /*!< UART_T::INTEN: WKCTSIEN Position          */
#define UART_INTEN_WKCTSIEN_Msk          (0x1ul << UART_INTEN_WKCTSIEN_Pos)                /*!< UART_T::INTEN: WKCTSIEN Mask              */

#define UART_INTEN_WKDATIEN_Pos          (10)                                              /*!< UART_T::INTEN: WKDATIEN Position          */
#define UART_INTEN_WKDATIEN_Msk          (0x1ul << UART_INTEN_WKDATIEN_Pos)                /*!< UART_T::INTEN: WKDATIEN Mask              */

#define UART_INTEN_TOCNTEN_Pos           (11)                                              /*!< UART_T::INTEN: TOCNTEN Position           */
#define UART_INTEN_TOCNTEN_Msk           (0x1ul << UART_INTEN_TOCNTEN_Pos)                 /*!< UART_T::INTEN: TOCNTEN Mask               */

#define UART_INTEN_ATORTSEN_Pos          (12)                                              /*!< UART_T::INTEN: ATORTSEN Position          */
#define UART_INTEN_ATORTSEN_Msk          (0x1ul << UART_INTEN_ATORTSEN_Pos)                /*!< UART_T::INTEN: ATORTSEN Mask              */

#define UART_INTEN_ATOCTSEN_Pos          (13)                                              /*!< UART_T::INTEN: ATOCTSEN Position          */
#define UART_INTEN_ATOCTSEN_Msk          (0x1ul << UART_INTEN_ATOCTSEN_Pos)                /*!< UART_T::INTEN: ATOCTSEN Mask              */

#define UART_INTEN_ABRIEN_Pos            (18)                                              /*!< UART_T::INTEN: ABRIEN Position            */
#define UART_INTEN_ABRIEN_Msk            (0x1ul << UART_INTEN_ABRIEN_Pos)                  /*!< UART_T::INTEN: ABRIEN Mask                */

#define UART_FIFO_RXRST_Pos              (1)                                               /*!< UART_T::FIFO: RXRST Position              */
#define UART_FIFO_RXRST_Msk              (0x1ul << UART_FIFO_RXRST_Pos)                    /*!< UART_T::FIFO: RXRST Mask                  */

#define UART_FIFO_TXRST_Pos              (2)                                               /*!< UART_T::FIFO: TXRST Position              */
#define UART_FIFO_TXRST_Msk              (0x1ul << UART_FIFO_TXRST_Pos)                    /*!< UART_T::FIFO: TXRST Mask                  */

#define UART_FIFO_RFITL_Pos              (4)                                               /*!< UART_T::FIFO: RFITL Position              */
#define UART_FIFO_RFITL_Msk              (0xful << UART_FIFO_RFITL_Pos)                    /*!< UART_T::FIFO: RFITL Mask                  */

#define UART_FIFO_RXOFF_Pos              (8)                                               /*!< UART_T::FIFO: RXOFF Position              */
#define UART_FIFO_RXOFF_Msk              (0x1ul << UART_FIFO_RXOFF_Pos)                    /*!< UART_T::FIFO: RXOFF Mask                  */

#define UART_FIFO_RTSTRGLV_Pos           (16)                                              /*!< UART_T::FIFO: RTSTRGLV Position           */
#define UART_FIFO_RTSTRGLV_Msk           (0xful << UART_FIFO_RTSTRGLV_Pos)                 /*!< UART_T::FIFO: RTSTRGLV Mask               */

#define UART_LINE_WLS_Pos                (0)                                               /*!< UART_T::LINE: WLS Position                */
#define UART_LINE_WLS_Msk                (0x3ul << UART_LINE_WLS_Pos)                      /*!< UART_T::LINE: WLS Mask                    */

#define UART_LINE_NSB_Pos                (2)                                               /*!< UART_T::LINE: NSB Position                */
#define UART_LINE_NSB_Msk                (0x1ul << UART_LINE_NSB_Pos)                      /*!< UART_T::LINE: NSB Mask                    */

#define UART_LINE_PBE_Pos                (3)                                               /*!< UART_T::LINE: PBE Position                */
#define UART_LINE_PBE_Msk                (0x1ul << UART_LINE_PBE_Pos)                      /*!< UART_T::LINE: PBE Mask                    */

#define UART_LINE_EPE_Pos                (4)                                               /*!< UART_T::LINE: EPE Position                */
#define UART_LINE_EPE_Msk                (0x1ul << UART_LINE_EPE_Pos)                      /*!< UART_T::LINE: EPE Mask                    */

#define UART_LINE_SPE_Pos                (5)                                               /*!< UART_T::LINE: SPE Position                */
#define UART_LINE_SPE_Msk                (0x1ul << UART_LINE_SPE_Pos)                      /*!< UART_T::LINE: SPE Mask                    */

#define UART_LINE_BCB_Pos                (6)                                               /*!< UART_T::LINE: BCB Position                */
#define UART_LINE_BCB_Msk                (0x1ul << UART_LINE_BCB_Pos)                      /*!< UART_T::LINE: BCB Mask                    */

#define UART_MODEM_RTS_Pos               (1)                                               /*!< UART_T::MODEM: RTS Position               */
#define UART_MODEM_RTS_Msk               (0x1ul << UART_MODEM_RTS_Pos)                     /*!< UART_T::MODEM: RTS Mask                   */

#define UART_MODEM_RTSACTLV_Pos          (9)                                               /*!< UART_T::MODEM: RTSACTLV Position          */
#define UART_MODEM_RTSACTLV_Msk          (0x1ul << UART_MODEM_RTSACTLV_Pos)                /*!< UART_T::MODEM: RTSACTLV Mask              */

#define UART_MODEM_RTSSTS_Pos            (13)                                              /*!< UART_T::MODEM: RTSSTS Position            */
#define UART_MODEM_RTSSTS_Msk            (0x1ul << UART_MODEM_RTSSTS_Pos)                  /*!< UART_T::MODEM: RTSSTS Mask                */

#define UART_MODEMSTS_CTSDETF_Pos        (0)                                               /*!< UART_T::MODEMSTS: CTSDETF Position        */
#define UART_MODEMSTS_CTSDETF_Msk        (0x1ul << UART_MODEMSTS_CTSDETF_Pos)              /*!< UART_T::MODEMSTS: CTSDETF Mask            */

#define UART_MODEMSTS_CTSSTS_Pos         (4)                                               /*!< UART_T::MODEMSTS: CTSSTS Position         */
#define UART_MODEMSTS_CTSSTS_Msk         (0x1ul << UART_MODEMSTS_CTSSTS_Pos)               /*!< UART_T::MODEMSTS: CTSSTS Mask             */

#define UART_MODEMSTS_CTSACTLV_Pos       (8)                                               /*!< UART_T::MODEMSTS: CTSACTLV Position       */
#define UART_MODEMSTS_CTSACTLV_Msk       (0x1ul << UART_MODEMSTS_CTSACTLV_Pos)             /*!< UART_T::MODEMSTS: CTSACTLV Mask           */

#define UART_FIFOSTS_RXOVIF_Pos          (0)                                               /*!< UART_T::FIFOSTS: RXOVIF Position          */
#define UART_FIFOSTS_RXOVIF_Msk          (0x1ul << UART_FIFOSTS_RXOVIF_Pos)                /*!< UART_T::FIFOSTS: RXOVIF Mask              */

#define UART_FIFOSTS_ABRDIF_Pos          (1)                                               /*!< UART_T::FIFOSTS: ABRDIF Position          */
#define UART_FIFOSTS_ABRDIF_Msk          (0x1ul << UART_FIFOSTS_ABRDIF_Pos)                /*!< UART_T::FIFOSTS: ABRDIF Mask              */

#define UART_FIFOSTS_ABRDTOIF_Pos        (2)                                               /*!< UART_T::FIFOSTS: ABRDTOIF Position        */
#define UART_FIFOSTS_ABRDTOIF_Msk        (0x1ul << UART_FIFOSTS_ABRDTOIF_Pos)              /*!< UART_T::FIFOSTS: ABRDTOIF Mask            */

#define UART_FIFOSTS_ADDRDETF_Pos        (3)                                               /*!< UART_T::FIFOSTS: ADDRDETF Position        */
#define UART_FIFOSTS_ADDRDETF_Msk        (0x1ul << UART_FIFOSTS_ADDRDETF_Pos)              /*!< UART_T::FIFOSTS: ADDRDETF Mask            */

#define UART_FIFOSTS_PEF_Pos             (4)                                               /*!< UART_T::FIFOSTS: PEF Position             */
#define UART_FIFOSTS_PEF_Msk             (0x1ul << UART_FIFOSTS_PEF_Pos)                   /*!< UART_T::FIFOSTS: PEF Mask                 */

#define UART_FIFOSTS_FEF_Pos             (5)                                               /*!< UART_T::FIFOSTS: FEF Position             */
#define UART_FIFOSTS_FEF_Msk             (0x1ul << UART_FIFOSTS_FEF_Pos)                   /*!< UART_T::FIFOSTS: FEF Mask                 */

#define UART_FIFOSTS_BIF_Pos             (6)                                               /*!< UART_T::FIFOSTS: BIF Position             */
#define UART_FIFOSTS_BIF_Msk             (0x1ul << UART_FIFOSTS_BIF_Pos)                   /*!< UART_T::FIFOSTS: BIF Mask                 */

#define UART_FIFOSTS_RXPTR_Pos           (8)                                               /*!< UART_T::FIFOSTS: RXPTR Position           */
#define UART_FIFOSTS_RXPTR_Msk           (0x3ful << UART_FIFOSTS_RXPTR_Pos)                /*!< UART_T::FIFOSTS: RXPTR Mask               */

#define UART_FIFOSTS_RXEMPTY_Pos         (14)                                              /*!< UART_T::FIFOSTS: RXEMPTY Position         */
#define UART_FIFOSTS_RXEMPTY_Msk         (0x1ul << UART_FIFOSTS_RXEMPTY_Pos)               /*!< UART_T::FIFOSTS: RXEMPTY Mask             */

#define UART_FIFOSTS_RXFULL_Pos          (15)                                              /*!< UART_T::FIFOSTS: RXFULL Position          */
#define UART_FIFOSTS_RXFULL_Msk          (0x1ul << UART_FIFOSTS_RXFULL_Pos)                /*!< UART_T::FIFOSTS: RXFULL Mask              */

#define UART_FIFOSTS_TXPTR_Pos           (16)                                              /*!< UART_T::FIFOSTS: TXPTR Position           */
#define UART_FIFOSTS_TXPTR_Msk           (0x3ful << UART_FIFOSTS_TXPTR_Pos)                /*!< UART_T::FIFOSTS: TXPTR Mask               */

#define UART_FIFOSTS_TXEMPTY_Pos         (22)                                              /*!< UART_T::FIFOSTS: TXEMPTY Position         */
#define UART_FIFOSTS_TXEMPTY_Msk         (0x1ul << UART_FIFOSTS_TXEMPTY_Pos)               /*!< UART_T::FIFOSTS: TXEMPTY Mask             */

#define UART_FIFOSTS_TXFULL_Pos          (23)                                              /*!< UART_T::FIFOSTS: TXFULL Position          */
#define UART_FIFOSTS_TXFULL_Msk          (0x1ul << UART_FIFOSTS_TXFULL_Pos)                /*!< UART_T::FIFOSTS: TXFULL Mask              */

#define UART_FIFOSTS_TXOVIF_Pos          (24)                                              /*!< UART_T::FIFOSTS: TXOVIF Position          */
#define UART_FIFOSTS_TXOVIF_Msk          (0x1ul << UART_FIFOSTS_TXOVIF_Pos)                /*!< UART_T::FIFOSTS: TXOVIF Mask              */

#define UART_FIFOSTS_TXEMPTYF_Pos        (28)                                              /*!< UART_T::FIFOSTS: TXEMPTYF Position        */
#define UART_FIFOSTS_TXEMPTYF_Msk        (0x1ul << UART_FIFOSTS_TXEMPTYF_Pos)              /*!< UART_T::FIFOSTS: TXEMPTYF Mask            */

#define UART_INTSTS_RDAIF_Pos            (0)                                               /*!< UART_T::INTSTS: RDAIF Position            */
#define UART_INTSTS_RDAIF_Msk            (0x1ul << UART_INTSTS_RDAIF_Pos)                  /*!< UART_T::INTSTS: RDAIF Mask                */

#define UART_INTSTS_THREIF_Pos           (1)                                               /*!< UART_T::INTSTS: THREIF Position           */
#define UART_INTSTS_THREIF_Msk           (0x1ul << UART_INTSTS_THREIF_Pos)                 /*!< UART_T::INTSTS: THREIF Mask               */

#define UART_INTSTS_RLSIF_Pos            (2)                                               /*!< UART_T::INTSTS: RLSIF Position            */
#define UART_INTSTS_RLSIF_Msk            (0x1ul << UART_INTSTS_RLSIF_Pos)                  /*!< UART_T::INTSTS: RLSIF Mask                */

#define UART_INTSTS_MODEMIF_Pos          (3)                                               /*!< UART_T::INTSTS: MODEMIF Position          */
#define UART_INTSTS_MODEMIF_Msk          (0x1ul << UART_INTSTS_MODEMIF_Pos)                /*!< UART_T::INTSTS: MODEMIF Mask              */

#define UART_INTSTS_RXTOIF_Pos           (4)                                               /*!< UART_T::INTSTS: RXTOIF Position           */
#define UART_INTSTS_RXTOIF_Msk           (0x1ul << UART_INTSTS_RXTOIF_Pos)                 /*!< UART_T::INTSTS: RXTOIF Mask               */

#define UART_INTSTS_BUFERRIF_Pos         (5)                                               /*!< UART_T::INTSTS: BUFERRIF Position         */
#define UART_INTSTS_BUFERRIF_Msk         (0x1ul << UART_INTSTS_BUFERRIF_Pos)               /*!< UART_T::INTSTS: BUFERRIF Mask             */

#define UART_INTSTS_WKIF_Pos             (6)                                               /*!< UART_T::INTSTS: WKIF Position             */
#define UART_INTSTS_WKIF_Msk             (0x1ul << UART_INTSTS_WKIF_Pos)                   /*!< UART_T::INTSTS: WKIF Mask                 */

#define UART_INTSTS_LINIF_Pos            (7)                                               /*!< UART_T::INTSTS: LINIF Position            */
#define UART_INTSTS_LINIF_Msk            (0x1ul << UART_INTSTS_LINIF_Pos)                  /*!< UART_T::INTSTS: LINIF Mask                */

#define UART_INTSTS_RDAINT_Pos           (8)                                               /*!< UART_T::INTSTS: RDAINT Position           */
#define UART_INTSTS_RDAINT_Msk           (0x1ul << UART_INTSTS_RDAINT_Pos)                 /*!< UART_T::INTSTS: RDAINT Mask               */

#define UART_INTSTS_THREINT_Pos          (9)                                               /*!< UART_T::INTSTS: THREINT Position          */
#define UART_INTSTS_THREINT_Msk          (0x1ul << UART_INTSTS_THREINT_Pos)                /*!< UART_T::INTSTS: THREINT Mask              */

#define UART_INTSTS_RLSINT_Pos           (10)                                              /*!< UART_T::INTSTS: RLSINT Position           */
#define UART_INTSTS_RLSINT_Msk           (0x1ul << UART_INTSTS_RLSINT_Pos)                 /*!< UART_T::INTSTS: RLSINT Mask               */

#define UART_INTSTS_MODEMINT_Pos         (11)                                              /*!< UART_T::INTSTS: MODEMINT Position         */
#define UART_INTSTS_MODEMINT_Msk         (0x1ul << UART_INTSTS_MODEMINT_Pos)               /*!< UART_T::INTSTS: MODEMINT Mask             */

#define UART_INTSTS_RXTOINT_Pos          (12)                                              /*!< UART_T::INTSTS: RXTOINT Position          */
#define UART_INTSTS_RXTOINT_Msk          (0x1ul << UART_INTSTS_RXTOINT_Pos)                /*!< UART_T::INTSTS: RXTOINT Mask              */

#define UART_INTSTS_BUFERRINT_Pos        (13)                                              /*!< UART_T::INTSTS: BUFERRINT Position        */
#define UART_INTSTS_BUFERRINT_Msk        (0x1ul << UART_INTSTS_BUFERRINT_Pos)              /*!< UART_T::INTSTS: BUFERRINT Mask            */

#define UART_INTSTS_WKINT_Pos            (14)                                              /*!< UART_T::INTSTS: WKINT Position            */
#define UART_INTSTS_WKINT_Msk            (0x1ul << UART_INTSTS_WKINT_Pos)                  /*!< UART_T::INTSTS: WKINT Mask                */

#define UART_INTSTS_LININT_Pos           (15)                                              /*!< UART_T::INTSTS: LININT Position           */
#define UART_INTSTS_LININT_Msk           (0x1ul << UART_INTSTS_LININT_Pos)                 /*!< UART_T::INTSTS: LININT Mask               */

#define UART_INTSTS_CTSWKIF_Pos          (16)                                              /*!< UART_T::INTSTS: CTSWKIF Position          */
#define UART_INTSTS_CTSWKIF_Msk          (0x1ul << UART_INTSTS_CTSWKIF_Pos)                /*!< UART_T::INTSTS: CTSWKIF Mask              */

#define UART_INTSTS_DATWKIF_Pos          (17)                                              /*!< UART_T::INTSTS: DATWKIF Position          */
#define UART_INTSTS_DATWKIF_Msk          (0x1ul << UART_INTSTS_DATWKIF_Pos)                /*!< UART_T::INTSTS: DATWKIF Mask              */

#define UART_TOUT_TOIC_Pos               (0)                                               /*!< UART_T::TOUT: TOIC Position               */
#define UART_TOUT_TOIC_Msk               (0xfful << UART_TOUT_TOIC_Pos)                    /*!< UART_T::TOUT: TOIC Mask                   */

#define UART_TOUT_DLY_Pos                (8)                                               /*!< UART_T::TOUT: DLY Position                */
#define UART_TOUT_DLY_Msk                (0xfful << UART_TOUT_DLY_Pos)                     /*!< UART_T::TOUT: DLY Mask                    */

#define UART_BAUD_BRD_Pos                (0)                                               /*!< UART_T::BAUD: BRD Position                */
#define UART_BAUD_BRD_Msk                (0xfffful << UART_BAUD_BRD_Pos)                   /*!< UART_T::BAUD: BRD Mask                    */

#define UART_BAUD_EDIVM1_Pos             (24)                                              /*!< UART_T::BAUD: EDIVM1 Position             */
#define UART_BAUD_EDIVM1_Msk             (0xful << UART_BAUD_EDIVM1_Pos)                   /*!< UART_T::BAUD: EDIVM1 Mask                 */

#define UART_BAUD_BAUDM0_Pos             (28)                                              /*!< UART_T::BAUD: BAUDM0 Position             */
#define UART_BAUD_BAUDM0_Msk             (0x1ul << UART_BAUD_BAUDM0_Pos)                   /*!< UART_T::BAUD: BAUDM0 Mask                 */

#define UART_BAUD_BAUDM1_Pos             (29)                                              /*!< UART_T::BAUD: BAUDM1 Position             */
#define UART_BAUD_BAUDM1_Msk             (0x1ul << UART_BAUD_BAUDM1_Pos)                   /*!< UART_T::BAUD: BAUDM1 Mask                 */

#define UART_IRDA_TXEN_Pos               (1)                                               /*!< UART_T::IRDA: TXEN Position               */
#define UART_IRDA_TXEN_Msk               (0x1ul << UART_IRDA_TXEN_Pos)                     /*!< UART_T::IRDA: TXEN Mask                   */

#define UART_IRDA_TXINV_Pos              (5)                                               /*!< UART_T::IRDA: TXINV Position              */
#define UART_IRDA_TXINV_Msk              (0x1ul << UART_IRDA_TXINV_Pos)                    /*!< UART_T::IRDA: TXINV Mask                  */

#define UART_IRDA_RXINV_Pos              (6)                                               /*!< UART_T::IRDA: RXINV Position              */
#define UART_IRDA_RXINV_Msk              (0x1ul << UART_IRDA_RXINV_Pos)                    /*!< UART_T::IRDA: RXINV Mask                  */

#define UART_ALTCTL_BRKFL_Pos            (0)                                               /*!< UART_T::ALTCTL: BRKFL Position            */
#define UART_ALTCTL_BRKFL_Msk            (0xful << UART_ALTCTL_BRKFL_Pos)                  /*!< UART_T::ALTCTL: BRKFL Mask                */

#define UART_ALTCTL_LINRXEN_Pos          (6)                                               /*!< UART_T::ALTCTL: LINRXEN Position          */
#define UART_ALTCTL_LINRXEN_Msk          (0x1ul << UART_ALTCTL_LINRXEN_Pos)                /*!< UART_T::ALTCTL: LINRXEN Mask              */

#define UART_ALTCTL_LINTXEN_Pos          (7)                                               /*!< UART_T::ALTCTL: LINTXEN Position          */
#define UART_ALTCTL_LINTXEN_Msk          (0x1ul << UART_ALTCTL_LINTXEN_Pos)                /*!< UART_T::ALTCTL: LINTXEN Mask              */

#define UART_ALTCTL_RS485NMM_Pos         (8)                                               /*!< UART_T::ALTCTL: RS485NMM Position         */
#define UART_ALTCTL_RS485NMM_Msk         (0x1ul << UART_ALTCTL_RS485NMM_Pos)               /*!< UART_T::ALTCTL: RS485NMM Mask             */

#define UART_ALTCTL_RS485AAD_Pos         (9)                                               /*!< UART_T::ALTCTL: RS485AAD Position         */
#define UART_ALTCTL_RS485AAD_Msk         (0x1ul << UART_ALTCTL_RS485AAD_Pos)               /*!< UART_T::ALTCTL: RS485AAD Mask             */

#define UART_ALTCTL_RS485AUD_Pos         (10)                                              /*!< UART_T::ALTCTL: RS485AUD Position         */
#define UART_ALTCTL_RS485AUD_Msk         (0x1ul << UART_ALTCTL_RS485AUD_Pos)               /*!< UART_T::ALTCTL: RS485AUD Mask             */

#define UART_ALTCTL_ADDRDEN_Pos          (15)                                              /*!< UART_T::ALTCTL: ADDRDEN Position          */
#define UART_ALTCTL_ADDRDEN_Msk          (0x1ul << UART_ALTCTL_ADDRDEN_Pos)                /*!< UART_T::ALTCTL: ADDRDEN Mask              */

#define UART_ALTCTL_ABRIF_Pos            (17)                                              /*!< UART_T::ALTCTL: ABRIF Position            */
#define UART_ALTCTL_ABRIF_Msk            (0x1ul << UART_ALTCTL_ABRIF_Pos)                  /*!< UART_T::ALTCTL: ABRIF Mask                */

#define UART_ALTCTL_ABRDEN_Pos           (18)                                              /*!< UART_T::ALTCTL: ABRDEN Position           */
#define UART_ALTCTL_ABRDEN_Msk           (0x1ul << UART_ALTCTL_ABRDEN_Pos)                 /*!< UART_T::ALTCTL: ABRDEN Mask               */

#define UART_ALTCTL_ABRDBITS_Pos         (19)                                              /*!< UART_T::ALTCTL: ABRDBITS Position         */
#define UART_ALTCTL_ABRDBITS_Msk         (0x3ul << UART_ALTCTL_ABRDBITS_Pos)               /*!< UART_T::ALTCTL: ABRDBITS Mask             */

#define UART_ALTCTL_ADDRMV_Pos           (24)                                              /*!< UART_T::ALTCTL: ADDRMV Position           */
#define UART_ALTCTL_ADDRMV_Msk           (0xfful << UART_ALTCTL_ADDRMV_Pos)                /*!< UART_T::ALTCTL: ADDRMV Mask               */

#define UART_FUNCSEL_FUNCSEL_Pos         (0)                                               /*!< UART_T::FUNCSEL: FUNCSEL Position         */
#define UART_FUNCSEL_FUNCSEL_Msk         (0x3ul << UART_FUNCSEL_FUNCSEL_Pos)               /*!< UART_T::FUNCSEL: FUNCSEL Mask             */

#define UART_LINCTL_SLVEN_Pos            (0)                                               /*!< UART_T::LINCTL: SLVEN Position            */
#define UART_LINCTL_SLVEN_Msk            (0x1ul << UART_LINCTL_SLVEN_Pos)                  /*!< UART_T::LINCTL: SLVEN Mask                */

#define UART_LINCTL_SLVHDEN_Pos          (1)                                               /*!< UART_T::LINCTL: SLVHDEN Position          */
#define UART_LINCTL_SLVHDEN_Msk          (0x1ul << UART_LINCTL_SLVHDEN_Pos)                /*!< UART_T::LINCTL: SLVHDEN Mask              */

#define UART_LINCTL_SLVAREN_Pos          (2)                                               /*!< UART_T::LINCTL: SLVAREN Position          */
#define UART_LINCTL_SLVAREN_Msk          (0x1ul << UART_LINCTL_SLVAREN_Pos)                /*!< UART_T::LINCTL: SLVAREN Mask              */

#define UART_LINCTL_SLVDUEN_Pos          (3)                                               /*!< UART_T::LINCTL: SLVDUEN Position          */
#define UART_LINCTL_SLVDUEN_Msk          (0x1ul << UART_LINCTL_SLVDUEN_Pos)                /*!< UART_T::LINCTL: SLVDUEN Mask              */

#define UART_LINCTL_MUTE_Pos             (4)                                               /*!< UART_T::LINCTL: MUTE Position             */
#define UART_LINCTL_MUTE_Msk             (0x1ul << UART_LINCTL_MUTE_Pos)                   /*!< UART_T::LINCTL: MUTE Mask                 */

#define UART_LINCTL_SENDH_Pos            (8)                                               /*!< UART_T::LINCTL: SENDH Position            */
#define UART_LINCTL_SENDH_Msk            (0x1ul << UART_LINCTL_SENDH_Pos)                  /*!< UART_T::LINCTL: SENDH Mask                */

#define UART_LINCTL_IDPEN_Pos            (9)                                               /*!< UART_T::LINCTL: IDPEN Position            */
#define UART_LINCTL_IDPEN_Msk            (0x1ul << UART_LINCTL_IDPEN_Pos)                  /*!< UART_T::LINCTL: IDPEN Mask                */

#define UART_LINCTL_BRKDETEN_Pos         (10)                                              /*!< UART_T::LINCTL: BRKDETEN Position         */
#define UART_LINCTL_BRKDETEN_Msk         (0x1ul << UART_LINCTL_BRKDETEN_Pos)               /*!< UART_T::LINCTL: BRKDETEN Mask             */

#define UART_LINCTL_RXOFF_Pos            (11)                                              /*!< UART_T::LINCTL: RXOFF Position            */
#define UART_LINCTL_RXOFF_Msk            (0x1ul << UART_LINCTL_RXOFF_Pos)                  /*!< UART_T::LINCTL: RXOFF Mask                */

#define UART_LINCTL_BITERREN_Pos         (12)                                              /*!< UART_T::LINCTL: BITERREN Position         */
#define UART_LINCTL_BITERREN_Msk         (0x1ul << UART_LINCTL_BITERREN_Pos)               /*!< UART_T::LINCTL: BITERREN Mask             */

#define UART_LINCTL_BRKFL_Pos            (16)                                              /*!< UART_T::LINCTL: BRKFL Position            */
#define UART_LINCTL_BRKFL_Msk            (0xful << UART_LINCTL_BRKFL_Pos)                  /*!< UART_T::LINCTL: BRKFL Mask                */

#define UART_LINCTL_BSL_Pos              (20)                                              /*!< UART_T::LINCTL: BSL Position              */
#define UART_LINCTL_BSL_Msk              (0x3ul << UART_LINCTL_BSL_Pos)                    /*!< UART_T::LINCTL: BSL Mask                  */

#define UART_LINCTL_HSEL_Pos             (22)                                              /*!< UART_T::LINCTL: HSEL Position             */
#define UART_LINCTL_HSEL_Msk             (0x3ul << UART_LINCTL_HSEL_Pos)                   /*!< UART_T::LINCTL: HSEL Mask                 */

#define UART_LINCTL_PID_Pos              (24)                                              /*!< UART_T::LINCTL: PID Position              */
#define UART_LINCTL_PID_Msk              (0xfful << UART_LINCTL_PID_Pos)                   /*!< UART_T::LINCTL: PID Mask                  */

#define UART_LINSTS_SLVHDETF_Pos         (0)                                               /*!< UART_T::LINSTS: SLVHDETF Position         */
#define UART_LINSTS_SLVHDETF_Msk         (0x1ul << UART_LINSTS_SLVHDETF_Pos)               /*!< UART_T::LINSTS: SLVHDETF Mask             */

#define UART_LINSTS_SLVHEF_Pos           (1)                                               /*!< UART_T::LINSTS: SLVHEF Position           */
#define UART_LINSTS_SLVHEF_Msk           (0x1ul << UART_LINSTS_SLVHEF_Pos)                 /*!< UART_T::LINSTS: SLVHEF Mask               */

#define UART_LINSTS_SLVIDPEF_Pos         (2)                                               /*!< UART_T::LINSTS: SLVIDPEF Position         */
#define UART_LINSTS_SLVIDPEF_Msk         (0x1ul << UART_LINSTS_SLVIDPEF_Pos)               /*!< UART_T::LINSTS: SLVIDPEF Mask             */

#define UART_LINSTS_SLVSYNCF_Pos         (3)                                               /*!< UART_T::LINSTS: SLVSYNCF Position         */
#define UART_LINSTS_SLVSYNCF_Msk         (0x1ul << UART_LINSTS_SLVSYNCF_Pos)               /*!< UART_T::LINSTS: SLVSYNCF Mask             */

#define UART_LINSTS_BRKDETF_Pos          (8)                                               /*!< UART_T::LINSTS: BRKDETF Position          */
#define UART_LINSTS_BRKDETF_Msk          (0x1ul << UART_LINSTS_BRKDETF_Pos)                /*!< UART_T::LINSTS: BRKDETF Mask              */

#define UART_LINSTS_BITEF_Pos            (9)                                               /*!< UART_T::LINSTS: BITEF Position            */
#define UART_LINSTS_BITEF_Msk            (0x1ul << UART_LINSTS_BITEF_Pos)                  /*!< UART_T::LINSTS: BITEF Mask                */

/**@}*/ /* UART_CONST */
/**@}*/ /* end of UART register group */


/*---------------------- USB Device Controller -------------------------*/
/**
    @addtogroup USBD USB Device Controller(USBD)
    Memory Mapped Structure for USBD Controller
@{ */

typedef struct
{

    union
    {

        /**
         * EPxDAT
         * ===================================================================================================
         * Offset: 0x64+x*0x28   Endpoint x Data Register
         * ---------------------------------------------------------------------------------------------------
         * |Bits    |Field     |Descriptions
         * | :----: | :----:   | :---- |
         * |[31:0]  |EPDAT     |Endpoint A~L Data Register
         * |        |          |Endpoint A~L data buffer for the buffer transaction (read or write).
         * |        |          |Only word or byte access are supported.
        */
        __IO uint32_t EPDAT;
        /**
           * EPxDAT_BYTE
           * ===================================================================================================
           * Offset: 0x64+x*0x28   Endpoint x Data Register for Byte Access
           * ---------------------------------------------------------------------------------------------------
           * |Bits    |Field     |Descriptions
           * | :----: | :----:   | :---- |
           * |[31:0]  |EPDAT     |Endpoint A~L Data Register
           * |        |          |Endpoint A~L data buffer for the buffer transaction (read or write).
           * |        |          |Only word or byte access are supported.
        */
        __IO uint8_t  EPDAT_BYTE;

    }; ///< Define EPx Data Register 32-bit or 8-bit access
    /**
        * EPxINTSTS
        * ===================================================================================================
        * Offset: 0x68+x*0x28  Endpoint x Interrupt Status Register
        * ---------------------------------------------------------------------------------------------------
        * |Bits    |Field     |Descriptions
        * | :----: | :----:   | :---- |
        * |[0]     |BUFFULLIF |Buffer Full
        * |        |          |For an IN endpoint, the currently selected buffer is full, or no buffer is available to the local side for writing (no space to write).
        * |        |          |For an OUT endpoint, there is a buffer available on the local side, and there are FIFO full of bytes available to be read (entire packet is available for reading.
        * |        |          |0 = The endpoint packet buffer is not full.
        * |        |          |1 = The endpoint packet buffer is full.
        * |        |          |Note: This bit is read-only.
        * |[1]     |BUFEMPTYIF|Buffer Empty
        * |        |          |For an IN endpoint, a buffer is available to the local side for writing up to FIFO full of bytes.
        * |        |          |0 = The endpoint buffer is not empty.
        * |        |          |1 = The endpoint buffer is empty.
        * |        |          |For an OUT endpoint:
        * |        |          |0 = The currently selected buffer does not have a count of 0.
        * |        |          |1 = The currently selected buffer has a count of 0, or no buffer is available on the local side (nothing to read).
        * |        |          |Note: This bit is read-only.
        * |[2]     |SHORTTXIF |Short Packet Transferred Interrupt
        * |        |          |0 = The length of the last packet was not less than the Maximum Packet Size (EPMPS).
        * |        |          |1 = The length of the last packet was less than the Maximum Packet Size (EPMPS).
        * |        |          |Note: Write 1 to clear this bit to 0.
        * |[3]     |TXPKIF    |Data Packet Transmitted Interrupt
        * |        |          |0 = No data packet is transmitted from the endpoint to the host.
        * |        |          |1 = A data packet is transmitted from the endpoint to the host.
        * |        |          |Note: Write 1 to clear this bit to 0.
        * |[4]     |RXPKIF    |Data Packet Received Interrupt
        * |        |          |0 = No data packet is received from the host by the endpoint.
        * |        |          |1 = A data packet is received from the host by the endpoint.
        * |        |          |Note: Write 1 to clear this bit to 0.
        * |[5]     |OUTTKIF   |Data OUT Token Interrupt
        * |        |          |0 = A Data OUT token has not been received from the host.
        * |        |          |1 = A Data OUT token has been received from the host.
        * |        |          |This bit also set by PING token (in high-speed only.
        * |        |          |Note: Write 1 to clear this bit to 0.
        * |[6]     |INTKIF    |Data IN Token Interrupt
        * |        |          |0 = No Data IN token has been received from the host.
        * |        |          |1 = A Data IN token has been received from the host.
        * |        |          |Write 1 to clear this bit to 0.
        * |[7]     |PINGIF    |PING Token Interrupt
        * |        |          |0 = A Data PING token has not been received from the host.
        * |        |          |1 = A Data PING token has been received from the host.
        * |        |          |Write 1 to clear this bit to 0.
        * |[8]     |NAKIF     |USB NAK Sent
        * |        |          |0 = The last USB IN packet could be provided, and was acknowledged with an ACK.
        * |        |          |1 = The last USB IN packet could not be provided, and was acknowledged with a NAK.
        * |        |          |Write 1 to clear this bit to 0.
        * |[9]     |STALLIF   |USB STALL Sent
        * |        |          |0 = The last USB packet could be accepted or provided because the endpoint was stalled, and was acknowledged with a STALL.
        * |        |          |1 = The last USB packet could not be accepted or provided because the endpoint was stalled, and was acknowledged with a STALL.
        * |        |          |Write 1 to clear this bit to 0.
        * |[10]    |NYETIF    |NYET Sent
        * |        |          |0 = The space available in the RAM is sufficient to accommodate the next on coming data packet.
        * |        |          |1 = The space available in the RAM is not sufficient to accommodate the next on coming data packet.
        * |        |          |Write 1 to clear this bit to 0.
        * |[11]    |ERRIF     |ERR Sent
        * |        |          |0 = No any error in the transaction.
        * |        |          |1 = There occurs any error in the transaction.
        * |        |          |Write 1 to clear this bit to 0.
        * |[12]    |SHORTRXIF |Bulk Out Short Packet Received
        * |        |          |0 = No bulk out short packet is received.
        * |        |          |1 = Received bulk out short packet (including zero length packet).
        * |        |          |Write 1 to clear this bit to 0.
    */
    __IO uint32_t EPINTSTS;
    /**
      * EPxINTEN
      * ===================================================================================================
      * Offset: 0x6C+x*0x28   Endpoint x Interrupt Enable Register
      * ---------------------------------------------------------------------------------------------------
      * |Bits    |Field     |Descriptions
      * | :----: | :----:   | :---- |
      * |[0]     |BUFFULLIEN|Buffer Full Interrupt
      * |        |          |When set, this bit enables a local interrupt to be set when a buffer full condition is detected on the bus.
      * |        |          |0 = Buffer full interrupt Disabled.
      * |        |          |1 = Buffer full interrupt Enabled.
      * |[1]     |BUFEMPTYIEN|Buffer Empty Interrupt
      * |        |          |When set, this bit enables a local interrupt to be set when a buffer empty condition is detected on the bus.
      * |        |          |0 = Buffer empty interrupt Disabled.
      * |        |          |1 = Buffer empty interrupt Enabled.
      * |[2]     |SHORTTXIEN|Short Packet Transferred Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a short data packet has been transferred to/from the host.
      * |        |          |0 = Short data packet interrupt Disabled.
      * |        |          |1 = Short data packet interrupt Enabled.
      * |[3]     |TXPKIEN   |Data Packet Transmitted Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a data packet has been received from the host.
      * |        |          |0 = Data packet has been received from the host interrupt Disabled.
      * |        |          |1 = Data packet has been received from the host interrupt Enabled.
      * |[4]     |RXPKIEN   |Data Packet Received Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a data packet has been transmitted to the host.
      * |        |          |0 = Data packet has been transmitted to the host interrupt Disabled.
      * |        |          |1 = Data packet has been transmitted to the host interrupt Enabled.
      * |[5]     |OUTTKIEN  |Data OUT Token Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a Data OUT token has been received from the host.
      * |        |          |0 = Data OUT token interrupt Disabled.
      * |        |          |1 = Data OUT token interrupt Enabled.
      * |[6]     |INTKIEN   |Data IN Token Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a Data IN token has been received from the host.
      * |        |          |0 = Data IN token interrupt Disabled.
      * |        |          |1 = Data IN token interrupt Enabled.
      * |[7]     |PINGIEN   |PING Token Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a PING token has been received from the host.
      * |        |          |0 = PING token interrupt Disabled.
      * |        |          |1 = PING token interrupt Enabled.
      * |[8]     |NAKIEN    |USB NAK Sent Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a NAK token is sent to the host.
      * |        |          |0 = NAK token interrupt Disabled.
      * |        |          |1 = NAK token interrupt Enabled.
      * |[9]     |STALLIEN  |USB STALL Sent Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set when a stall token is sent to the host.
      * |        |          |0 = STALL token interrupt Disabled.
      * |        |          |1 = STALL token interrupt Enabled.
      * |[10]    |NYETIEN   |NYET Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set whenever NYET condition occurs on the bus for this endpoint.
      * |        |          |0 = NYET condition interrupt Disabled.
      * |        |          |1 = NYET condition interrupt Enabled.
      * |[11]    |ERRIEN    |ERR Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set whenever ERR condition occurs on the bus for this endpoint.
      * |        |          |0 = Error event interrupt Disabled.
      * |        |          |1 = Error event interrupt Enabled.
      * |[12]    |SHORTRXIEN|Bulk Out Short Packet Interrupt Enable Bit
      * |        |          |When set, this bit enables a local interrupt to be set whenever bulk out short packet occurs on the bus for this endpoint.
      * |        |          |0 = Bulk out interrupt Disabled.
      * |        |          |1 = Bulk out interrupt Enabled.
    */
    __IO uint32_t EPINTEN;
    /**
     * EPxDATCNT
     * ===================================================================================================
     * Offset: 0x70+x*0x28   Endpoint x Data Available Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |DATCNT    |Data Count
     * |        |          |For an IN endpoint (EPDIR(USBD_EPxCFG[3] is high.), this register returns the number of valid bytes in the IN endpoint packet buffer.
     * |        |          |For an OUT endpoint (EPDIR(USBD_EPxCFG[3] is low.), this register returns the number of received valid bytes in the Host OUT transfer.
     * |[30:16] |DMALOOP   |DMA Loop
     * |        |          |This register is the remaining DMA loop to complete. Each loop means 32-byte transfer.
    */
    __I  uint32_t EPDATCNT;

    /**
     * EPxRSPCTL
     * ===================================================================================================
     * Offset: 0x74+x*0x28  Endpoint x Response Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FLUSH     |Buffer Flush
     * |        |          |Writing 1 to this bit causes the packet buffer to be flushed and the corresponding EP_AVAIL register to be cleared.
     * |        |          |This bit is self-cleared.
     * |        |          |This bit should always be written after a configuration even.
     * |        |          |0 = The packet buffer is not flushed.
     * |        |          |1 = The packet buffer is flushed by user.
     * |[2:1]   |MODE      |Mode Control
     * |        |          |The two bits determine the operation mode of the in-endpoint.
     * |        |          |00 = Auto-Validate Mode.
     * |        |          |01 = Manual-Validate Mode.
     * |        |          |10 = Fly Mode.
     * |        |          |11 = Reserved.
     * |        |          |Note: These bits are not valid for an out-endpoint.
     * |        |          |The auto validate mode will be activated when the reserved mode is selected.
     * |[3]     |TOGGLE    |Endpoint Toggle
     * |        |          |This bit is used to clear the endpoint data toggle bit.
     * |        |          |Reading this bit returns the current state of the endpoint data toggle bi.
     * |        |          |The local CPU may use this bit to initialize the end-point's toggle in case of reception of a Set Interface request or a Clear Feature (ep_halt) request from the host.
     * |        |          |Only when toggle bit is "1", this bit can be written into the inversed write data bit[3.
     * |        |          |0 = Do not clear the endpoint data toggle bit.
     * |        |          |1 = Clear the endpoint data toggle bit.
     * |[4]     |HALT      |Endpoint Halt
     * |        |          |This bit is used to send a STALL handshake as response to the token from the host.
     * |        |          |When an Endpoint Set Feature (ep_halt) is detected by the local CPU, it must write a '1' to this bi.
     * |        |          |0 = Do not send a STALL handshake as response to the token from the host.
     * |        |          |1 = Send a STALL handshake as response to the token from the host.
     * |[5]     |ZEROLEN   |Zero Length
     * |        |          |This bit is used to send a zero-length packet response to an IN-token.
     * |        |          |When this bit is set, a zero packet is sent to the host on reception of an IN-token.
     * |        |          |This bit gets cleared once the zero length data packet is sen.
     * |        |          |0 = A zero packet is not sent to the host on reception of an IN-token.
     * |        |          |1 = A zero packet is sent to the host on reception of an IN-token.
     * |[6]     |SHORTTXEN |Short Packet Transfer Enable Bit
     * |        |          |This bit is applicable only in case of Auto-Validate Method.
     * |        |          |This bit is set to validate any remaining data in the buffer which is not equal to the MPS of the endpoint, and happens to be the last transfer.
     * |        |          |This bit gets cleared once the data packet is sen.
     * |        |          |0 = Not validate any remaining data in the buffer which is not equal to the MPS of the endpoint.
     * |        |          |1 = Validate any remaining data in the buffer which is not equal to the MPS of the endpoint.
     * |[7]     |DISBUF    |Buffer Disable Bit
     * |        |          |This bit is used to receive unknown size OUT short packet.
     * |        |          |The received packet size is reference USBD_EPxDATCNT registe.
     * |        |          |0 = Buffer Not Disabled when Bulk-OUT short packet is received.
     * |        |          |1 = Buffer Disabled when Bulk-OUT short packet is received.
    */
    __IO uint32_t EPRSPCTL;
    /**
     * EPxMPS
     * ===================================================================================================
     * Offset: 0x78+x*0x28  Endpoint x Maximum Packet Size Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[10:0]  |EPMPS     |Endpoint Maximum Packet Size
     * |        |          |This field determines the Maximum Packet Size of the Endpoint.
    */
    __IO uint32_t EPMPS;
    /**
     * EPxTXCNT
     * ===================================================================================================
     * Offset: 0x7C+x*0x28  Endpoint x Transfer Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[10:0]  |TXCNT     |Endpoint Transfer Count
     * |        |          |For IN endpoints, this field determines the total number of bytes to be sent to the host in case of manual validation method.
     * |        |          |For OUT endpoints, this field has no effect.
    */
    __IO uint32_t EPTXCNT;

    /**
     * EPxCFG
     * ===================================================================================================
     * Offset: 0x80+x*0x28   Endpoint x Configuration Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |EPEN      |Endpoint Valid
     * |        |          |When set, this bit enables this endpoint.
     * |        |          |This bit has no effect on Endpoint 0, which is always enabled.
     * |        |          |0 = The endpoint Disabled.
     * |        |          |1 = The endpoint Enabled.
     * |[2:1]   |EPTYPE    |Endpoint Type
     * |        |          |This field selects the type of this endpoint. Endpoint 0 is forced to a Control type.
     * |        |          |00 = Reserved.
     * |        |          |01 = Bulk.
     * |        |          |10 = Interrupt.
     * |        |          |11 = Isochronous.
     * |[3]     |EPDIR     |Endpoint Direction
     * |        |          |0 = out-endpoint (Host OUT to Device).
     * |        |          |1 = in-endpoint (Host IN to Device).
     * |        |          |A maximum of one OUT and IN endpoint is allowed for each endpoint number.
     * |[7:4]   |EPNUM     |Endpoint Number
     * |        |          |This field selects the number of the endpoint. Valid numbers 1 to 15.
     * |        |          |Do not support two endpoints have same endpoint number.
    */
    __IO uint32_t EPCFG;
    /**
     * EPxBUFSTART
     * ===================================================================================================
     * Offset: 0x84+x*0x28  Endpoint x RAM Start Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |SADDR     |Endpoint Start Address
     * |        |          |This is the start-address of the RAM space allocated for the endpoint A~L.
    */
    __IO uint32_t EPBUFSTART;
    /**
     * EPxBUFEND
     * ===================================================================================================
     * Offset: 0x88+x*0x28  Endpoint x RAM End Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |EADDR     |Endpoint End Address
     * |        |          |This is the end-address of the RAM space allocated for the endpoint A~L.
    */
    __IO uint32_t EPBUFEND;


} USBD_EP_T;

typedef struct
{


    /**
     * GINTSTS
     * ===================================================================================================
     * Offset: 0x00  Global Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |USBIF     |USB Interrupt
     * |        |          |This bit conveys the interrupt status for USB specific events endpoint.
     * |        |          |When set, USB interrupt status register should be read to determine the cause of the interrup.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[1]     |CEPIF     |Control Endpoint Interrupt
     * |        |          |This bit conveys the interrupt status for control endpoint.
     * |        |          |When set, Control-ep's interrupt status register should be read to determine the cause of the interrup.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[2]     |EPAIF     |Endpoints A Interrupt
     * |        |          |When set, the corresponding Endpoint A's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[3]     |EPBIF     |Endpoints B Interrupt
     * |        |          |When set, the corresponding Endpoint B's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[4]     |EPCIF     |Endpoints C Interrupt
     * |        |          |When set, the corresponding Endpoint C's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[5]     |EPDIF     |Endpoints D Interrupt
     * |        |          |When set, the corresponding Endpoint D's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[6]     |EPEIF     |Endpoints E Interrupt
     * |        |          |When set, the corresponding Endpoint E's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[7]     |EPFIF     |Endpoints F Interrupt
     * |        |          |When set, the corresponding Endpoint F's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[8]     |EPGIF     |Endpoints G Interrupt
     * |        |          |When set, the corresponding Endpoint G's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[9]     |EPHIF     |Endpoints H Interrupt
     * |        |          |When set, the corresponding Endpoint H's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[10]    |EPIIF     |Endpoints I Interrupt
     * |        |          |When set, the corresponding Endpoint I's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[11]    |EPJIF     |Endpoints J Interrupt
     * |        |          |When set, the corresponding Endpoint J's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[12]    |EPKIF     |Endpoints K Interrupt
     * |        |          |When set, the corresponding Endpoint K's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
     * |[13]    |EPLIF     |Endpoints L Interrupt
     * |        |          |When set, the corresponding Endpoint L's interrupt status register should be read to determine the cause of the interrupt.
     * |        |          |0 = No interrupt event occurred.
     * |        |          |1 = The related interrupt event is occurred.
    */
    __IO  uint32_t GINTSTS;
    uint32_t RESERVE0[1];


    /**
     * GINTEN
     * ===================================================================================================
     * Offset: 0x08  Global Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |USBIEN    |USB Interrupt Enable Bit
     * |        |          |When set, this bit enables a local interrupt to be generated when a USB event occurs on the bus.
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[1]     |CEPIEN    |Control Endpoint Interrupt Enable Bit
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the control endpoint.
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[2]     |EPAIEN    |Interrupt Enable Control For Endpoint A
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint A.
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[3]     |EPBIEN    |Interrupt Enable Control For Endpoint B
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint B
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[4]     |EPCIEN    |Interrupt Enable Control For Endpoint C
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint C
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[5]     |EPDIEN    |Interrupt Enable Control For Endpoint D
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint D
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[6]     |EPEIEN    |Interrupt Enable Control For Endpoint E
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint E
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[7]     |EPFIEN    |Interrupt Enable Control For Endpoint F
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint F
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[8]     |EPGIEN    |Interrupt Enable Control For Endpoint G
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint G
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[9]     |EPHIEN    |Interrupt Enable Control For Endpoint H
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint H
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[10]    |EPIIEN    |Interrupt Enable Control For Endpoint I
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint I
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[11]    |EPJIEN    |Interrupt Enable Control For Endpoint J
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint J
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[12]    |EPKIEN    |Interrupt Enable Control For Endpoint K
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint K
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
     * |[13]    |EPLIEN    |Interrupt Enable Control For Endpoint L
     * |        |          |When set, this bit enables a local interrupt to be generated when an interrupt is pending for the endpoint L
     * |        |          |0 = The related interrupt Disabled.
     * |        |          |1 = The related interrupt Enabled.
    */
    __IO uint32_t GINTEN;
    uint32_t RESERVE1[1];


    /**
     * BUSINTSTS
     * ===================================================================================================
     * Offset: 0x10  USB Bus Interrupt Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SOFIF     |SOF Receive Control
     * |        |          |This bit indicates when a start-of-frame packet has been received.
     * |        |          |0 = No start-of-frame packet has been received.
     * |        |          |1 = Start-of-frame packet has been received.
     * |        |          |Write 1 to clear this bit to 0.
     * |[1]     |RSTIF     |Reset Status
     * |        |          |When set, this bit indicates that either the USB root port reset is end.
     * |        |          |0 = No USB root port reset is end.
     * |        |          |1 = USB root port reset is end.
     * |        |          |Write 1 to clear this bit to 0.
     * |[2]     |RESUMEIF  |Resume
     * |        |          |When set, this bit indicates that a device resume has occurred.
     * |        |          |0 = No device resume has occurred.
     * |        |          |1 = Device resume has occurred.
     * |        |          |Write 1 to clear this bit to 0.
     * |[3]     |SUSPENDIF |Suspend Request
     * |        |          |This bit is set as default and it has to be cleared by writing '1' before the USB reset.
     * |        |          |This bit is also set when a USB Suspend request is detected from the hos.
     * |        |          |0 = No USB Suspend request is detected from the host.
     * |        |          |1= USB Suspend request is detected from the host.
     * |        |          |Write 1 to clear this bit to 0.
     * |[4]     |HISPDIF   |High-Speed Settle
     * |        |          |0 = No valid high-speed reset protocol is detected.
     * |        |          |1 = Valid high-speed reset protocol is over and the device has settled in high-speed.
     * |        |          |Write 1 to clear this bit to 0.
     * |[5]     |DMADONEIF |DMA Completion Interrupt
     * |        |          |0 = No DMA transfer over.
     * |        |          |1 = DMA transfer is over.
     * |        |          |Write 1 to clear this bit to 0.
     * |[6]     |PHYCLKVLDIF|Usable Clock Interrupt
     * |        |          |0 = Usable clock is not available.
     * |        |          |1 = Usable clock is available from the transceiver.
     * |        |          |Write 1 to clear this bit to 0.
     * |[8]     |VBUSDETIF |VBUS Detection Interrupt Status
     * |        |          |0 = No VBUS is plug-in.
     * |        |          |1 = VBUS is plug-in.
     * |        |          |Write 1 to clear this bit to 0.
    */
    __IO uint32_t BUSINTSTS;

    /**
     * BUSINTEN
     * ===================================================================================================
     * Offset: 0x14  USB Bus Interrupt Enable Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SOFIEN    |SOF Interrupt
     * |        |          |This bit enables the SOF interrupt.
     * |        |          |0 = SOF interrupt Disabled.
     * |        |          |1 = SOF interrupt Enabled.
     * |[1]     |RSTIEN    |Reset Status
     * |        |          |This bit enables the USB-Reset interrupt.
     * |        |          |0 = USB-Reset interrupt Disabled.
     * |        |          |1 = USB-Reset interrupt Enabled.
     * |[2]     |RESUMEIEN |Resume
     * |        |          |This bit enables the Resume interrupt.
     * |        |          |0 = Resume interrupt Disabled.
     * |        |          |1 = Resume interrupt Enabled.
     * |[3]     |SUSPENDIEN|Suspend Request
     * |        |          |This bit enables the Suspend interrupt.
     * |        |          |0 = Suspend interrupt Disabled.
     * |        |          |1 = Suspend interrupt Enabled.
     * |[4]     |HISPDIEN  |High-Speed Settle
     * |        |          |This bit enables the high-speed settle interrupt.
     * |        |          |0 = High-speed settle interrupt Disabled.
     * |        |          |1 = High-speed settle interrupt Enabled.
     * |[5]     |DMADONEIEN|DMA Completion Interrupt
     * |        |          |This bit enables the DMA completion interrupt
     * |        |          |0 = DMA completion interrupt Disabled.
     * |        |          |1 = DMA completion interrupt Enabled.
     * |[6]     |PHYCLKVLDIEN|Usable Clock Interrupt
     * |        |          |This bit enables the usable clock interrupt.
     * |        |          |0 = Usable clock interrupt Disabled.
     * |        |          |1 = Usable clock interrupt Enabled.
     * |[8]     |VBUSDETIEN|VBUS Detection Interrupt Enable Control
     * |        |          |This bit enables the VBUS floating detection interrupt.
     * |        |          |0 = VBUS floating detection interrupt Disabled.
     * |        |          |1 = VBUS floating detection interrupt Enabled.
    */
    __IO uint32_t BUSINTEN;

    /**
     * OPER
     * ===================================================================================================
     * Offset: 0x18  USB Operational Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RESUMEEN  |Generate Resume
     * |        |          |0 = No resume sequence to be initiated to the host.
     * |        |          |1 = A resume sequence to be initiated to the host if device remote wake-up is enabled.
     * |        |          |Note: This bit is self-cleared.
     * |[1]     |HISPDEN   |USB High-Speed
     * |        |          |0 = The USB device controller suppresses the chirp-sequence during reset protocol, thereby allowing the USB device controller to settle in full-speed, even though it is connected to a USB2.0 Host.
     * |        |          |1 = The USB device controller initiates a chirp-sequence during reset protocol.
     * |[2]     |CURSPD    |Current USB Speed
     * |        |          |0 = The device has settled in Full Speed.
     * |        |          |1 = The USB device controller has settled in High-speed.
    */
    __IO uint32_t OPER;

    /**
     * FRAMECNT
     * ===================================================================================================
     * Offset: 0x1C  USB Frame Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |MFRAMECNT |Micro-Frame Counter
     * |        |          |This field contains the micro-frame number for the frame number in the frame counter field.
     * |[13:3]  |FRAMECNT  |Frame Counter
     * |        |          |This field contains the frame count from the most recent start-of-frame packet.
    */
    __I  uint32_t FRAMECNT;

    /**
     * FADDR
     * ===================================================================================================
     * Offset: 0x20  USB Function Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[6:0]   |FADDR     |USB Function Address
     * |        |          |This field contains the current USB address of the device.
     * |        |          |This field is cleared when a root port reset is detected.
    */
    __IO uint32_t FADDR;

    /**
     * TEST
     * ===================================================================================================
     * Offset: 0x24  USB Test Mode Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[2:0]   |TESTMODE  |Test Mode Selection
     * |        |          |000 = Normal Operation.
     * |        |          |001 = Test_J.
     * |        |          |010 = Test_K.
     * |        |          |011 = Test_SE0_NAK.
     * |        |          |100 = Test_Packet.
     * |        |          |101 = Test_Force_Enable.
     * |        |          |110 = Reserved.
     * |        |          |111 = Reserved.
     * |        |          |Note: This field is cleared when root port reset is detected.
    */
    __IO uint32_t TEST;

    union
    {
        /**
         * CEPDAT
         * ===================================================================================================
         * Offset: 0x28  Control-Endpoint Data Buffer
         * ---------------------------------------------------------------------------------------------------
         * |Bits    |Field     |Descriptions
         * | :----: | :----:   | :---- |
         * |[31:0]  |DAT       |Control-Endpoint Data Buffer
         * |        |          |Control endpoint data buffer for the buffer transaction (read or write).
         * |        |          |Only word or byte access is supported.
        */
        __IO uint32_t CEPDAT;
        /**
         * CEPDAT_BYTE
         * ===================================================================================================
         * Offset: 0x28  Control-Endpoint Data Buffer for Byte Access
         * ---------------------------------------------------------------------------------------------------
         * |Bits    |Field     |Descriptions
         * | :----: | :----:   | :---- |
         * |[0:7]   |DAT       |Control-Endpoint Data Buffer
         * |        |          |Control endpoint data buffer for the buffer transaction (read or write).
         * |        |          |Only word or byte access is supported.
        */
        __IO uint8_t  CEPDAT_BYTE;
    };///< Define Control-Endpoint Data Register 32-bit or 8-bit access

    /**
     * CEPCTL
     * ===================================================================================================
     * Offset: 0x2C  Control-Endpoint Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |NAKCLR    |No Acknowledge Control
     * |        |          |This bit plays a crucial role in any control transfer.
     * |        |          |0 = The bit is being cleared by the local CPU by writing zero, the USB device controller will be responding with NAKs for the subsequent status phase.
     * |        |          |This mechanism holds the host from moving to the next request, until the local CPU is also ready to process the next reques.
     * |        |          |1 = This bit is set to one by the USB device controller, whenever a setup token is received.
     * |        |          |The local CPU can take its own time to finish off any house-keeping work based on the request and then clear this bi.
     * |        |          |Note: This bit can be updated only when CPU writes data[1:0] is 0x2 or 0x0.
     * |[1]     |STALLEN   |Stall Enable Control
     * |        |          |When this stall bit is set, the control endpoint sends a stall handshake in response to any in or out token thereafter.
     * |        |          |This is typically used for response to invalid/unsupported requests.
     * |        |          |When this bit is being set the NAK clear bit has to be cleared at the same time since the NAK clear bit has highest priority than STALL.
     * |        |          |It is automatically cleared on receipt of a next setup-token.
     * |        |          |Thus, the local CPU does not need to write again to clear this bi.
     * |        |          |0 = No sends a stall handshake in response to any in or out token thereafter.
     * |        |          |1 = The control endpoint sends a stall handshake in response to any in or out token thereafter.
     * |        |          |Note: This bit can be updated only when CPU writes data[1:0] 0x2 or 0x0.
     * |[2]     |ZEROLEN   |Zero Packet Length
     * |        |          |This bit is valid for Auto Validation mode only.
     * |        |          |0 = No zero length packet to the host during Data stage to an IN token.
     * |        |          |1 = USB device controller can send a zero length packet to the host during Data stage to an IN token.
     * |        |          |This bit gets cleared once the zero length data packet is sent.
     * |        |          |Thus, the local CPU does not need to write again to clear this bi.
     * |[3]     |FLUSH     |CEP-FLUSH Bit
     * |        |          |0 = No packet buffer and its corresponding USBD_CEPDATCNT register to be cleared.
     * |        |          |1 = The packet buffer and its corresponding USBD_CEPDATCNT register to be cleared.
     * |        |          |This bit is self-cleared.
    */
    __IO uint32_t CEPCTL;

    /**
     * CEPINTEN
     * ===================================================================================================
     * Offset: 0x30  Control-Endpoint Interrupt Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SETUPTKIEN|Setup Token Interrupt Enable Bit
     * |        |          |0 = The SETUP token interrupt in Control Endpoint Disabled.
     * |        |          |1 = The SETUP token interrupt in Control Endpoint Enabled.
     * |[1]     |SETUPPKIEN|Setup Packet Interrupt
     * |        |          |0 = The SETUP packet interrupt in Control Endpoint Disabled.
     * |        |          |1 = The SETUP packet interrupt in Control Endpoint Enabled.
     * |[2]     |OUTTKIEN  |Out Token Interrupt
     * |        |          |0 = The OUT token interrupt in Control Endpoint Disabled.
     * |        |          |1 = The OUT token interrupt in Control Endpoint Enabled.
     * |[3]     |INTKIEN   |In Token Interrupt
     * |        |          |0 = The IN token interrupt in Control Endpoint Disabled.
     * |        |          |1 = The IN token interrupt in Control Endpoint Enabled.
     * |[4]     |PINGIEN   |Ping Token Interrupt
     * |        |          |0 = The ping token interrupt in Control Endpoint Disabled.
     * |        |          |1 = The ping token interrupt Control Endpoint Enabled.
     * |[5]     |TXPKIEN   |Data Packet Transmitted Interrupt
     * |        |          |0 = The data packet transmitted interrupt in Control Endpoint Disabled.
     * |        |          |1 = The data packet transmitted interrupt in Control Endpoint Enabled.
     * |[6]     |RXPKIEN   |Data Packet Received Interrupt
     * |        |          |0 = The data received interrupt in Control Endpoint Disabled.
     * |        |          |1 = The data received interrupt in Control Endpoint Enabled.
     * |[7]     |NAKIEN    |NAK Sent Interrupt
     * |        |          |0 = The NAK sent interrupt in Control Endpoint Disabled.
     * |        |          |1 = The NAK sent interrupt in Control Endpoint Enabled.
     * |[8]     |STALLIEN  |STALL Sent Interrupt
     * |        |          |0 = The STALL sent interrupt in Control Endpoint Disabled.
     * |        |          |1 = The STALL sent interrupt in Control Endpoint Enabled.
     * |[9]     |ERRIEN    |USB Error Interrupt
     * |        |          |0 = The USB Error interrupt in Control Endpoint Disabled.
     * |        |          |1 = The USB Error interrupt in Control Endpoint Enabled.
     * |[10]    |STSDONEIEN|Status Completion Interrupt
     * |        |          |0 = The Status Completion interrupt in Control Endpoint Disabled.
     * |        |          |1 = The Status Completion interrupt in Control Endpoint Enabled.
     * |[11]    |BUFFULLIEN|Buffer Full Interrupt
     * |        |          |0 = The buffer full interrupt in Control Endpoint Disabled.
     * |        |          |1 = The buffer full interrupt in Control Endpoint Enabled.
     * |[12]    |BUFEMPTYIEN|Buffer Empty Interrupt
     * |        |          |0 = The buffer empty interrupt in Control Endpoint Disabled.
     * |        |          |1= The buffer empty interrupt in Control Endpoint Enabled.
    */
    __IO uint32_t CEPINTEN;

    /**
     * CEPINTSTS
     * ===================================================================================================
     * Offset: 0x34  Control-Endpoint Interrupt Status
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |SETUPTKIF |Setup Token Interrupt
     * |        |          |0 = No setup token is received.
     * |        |          |1 = A setup token is received. Writing 1 clears this status bit
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[1]     |SETUPPKIF |Setup Packet Interrupt
     * |        |          |This bit must be cleared (by writing 1) before the next setup packet can be received.
     * |        |          |If the bit is not cleared, then the successive setup packets will be overwritten in the setup packet buffe.
     * |        |          |0 = No setup packet has been received from the host.
     * |        |          |1 = A setup packet has been received from the host.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[2]     |OUTTKIF   |Out Token Interrupt
     * |        |          |0 = The control-endpoint does not receive an OUT token from the host.
     * |        |          |1 = The control-endpoint receives an OUT token from the host.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[3]     |INTKIF    |In Token Interrupt
     * |        |          |0 = The control-endpoint does not receive an IN token from the host.
     * |        |          |1 = The control-endpoint receives an IN token from the host.
     * |        |          |Write 1 to clear this bit to 0.
     * |[4]     |PINGIF    |Ping Token Interrupt
     * |        |          |0 = The control-endpoint does not receive a ping token from the host.
     * |        |          |1 = The control-endpoint receives a ping token from the host.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[5]     |TXPKIF    |Data Packet Transmitted Interrupt
     * |        |          |0 = No data packet is successfully transmitted to the host in response to an IN-token and an ACK-token is received for the same.
     * |        |          |1 = A data packet is successfully transmitted to the host in response to an IN-token and an ACK-token is received for the same.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[6]     |RXPKIF    |Data Packet Received Interrupt
     * |        |          |0 = No data packet is successfully received from the host for an OUT-token and an ACK is sent to the host.
     * |        |          |1 = A data packet is successfully received from the host for an OUT-token and an ACK is sent to the host.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[7]     |NAKIF     |NAK Sent Interrupt
     * |        |          |0 = No NAK-token is sent in response to an IN/OUT token.
     * |        |          |1 = A NAK-token is sent in response to an IN/OUT token.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[8]     |STALLIF   |STALL Sent Interrupt
     * |        |          |0 = No stall-token is sent in response to an IN/OUT token.
     * |        |          |1 = A stall-token is sent in response to an IN/OUT token.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[9]     |ERRIF     |USB Error Interrupt
     * |        |          |0 = No error had occurred during the transaction.
     * |        |          |1 = An error had occurred during the transaction.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[10]    |STSDONEIF |Status Completion Interrupt
     * |        |          |0 = No USB transaction has completed successfully.
     * |        |          |1 = The status stage of a USB transaction has completed successfully.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[11]    |BUFFULLIF |Buffer Full Interrupt
     * |        |          |0 = The control-endpoint buffer is not full.
     * |        |          |1 = The control-endpoint buffer is full.
     * |        |          |Note: Write 1 to clear this bit to 0.
     * |[12]    |BUFEMPTYIF|Buffer Empty Interrupt
     * |        |          |0 = The control-endpoint buffer is not empty.
     * |        |          |1 = The control-endpoint buffer is empty.
     * |        |          |Note: Write 1 to clear this bit to 0.
    */
    __IO uint32_t CEPINTSTS;

    /**
     * CEPTXCNT
     * ===================================================================================================
     * Offset: 0x38  Control-Endpoint In-transfer Data Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |TXCNT     |In-Transfer Data Count
     * |        |          |There is no mode selection for the control endpoint (but it operates like manual mode).The local-CPU has to fill the control-endpoint buffer with the data to be sent for an in-token and to write the count of bytes in this register.
     * |        |          |When zero is written into this field, a zero length packet is sent to the host.
     * |        |          |When the count written in the register is more than the MPS, the data sent will be of only MP.
    */
    __IO uint32_t CEPTXCNT;

    /**
     * CEPRXCNT
     * ===================================================================================================
     * Offset: 0x3C  Control-Endpoint Out-transfer Data Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |RXCNT     |Out-Transfer Data Count
     * |        |          |The USB device controller maintains the count of the data received in case of an out transfer, during the control transfer.
    */
    __I  uint32_t CEPRXCNT;

    /**
     * CEPDATCNT
     * ===================================================================================================
     * Offset: 0x40  Control-Endpoint Data Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[15:0]  |DATCNT    |Control-Endpoint Data Count
     * |        |          |The USB device controller maintains the count of the data of control-endpoint.
    */
    __I  uint32_t CEPDATCNT;

    /**
     * SETUP1_0
     * ===================================================================================================
     * Offset: 0x44  Setup1 & Setup0 bytes
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |SETUP0    |Setup Byte 0[7:0]
     * |        |          |This register provides byte 0 of the last setup packet received.
     * |        |          |For a Standard Device Request, the following bmRequestType information is returne.
     * |        |          |Bit 7(Direction):
     * |        |          |0 = Host to device.
     * |        |          |1 = Device to host.
     * |        |          |Bit 6-5 (Type):
     * |        |          |00 = Standard.
     * |        |          |01 = Class.
     * |        |          |10 = Vendor.
     * |        |          |11 = Reserved.
     * |        |          |Bit 4-0 (Recipient)
     * |        |          |00000 = Device.
     * |        |          |00001 = nterface.
     * |        |          |00010 = Endpoint.
     * |        |          |00011 =:Other.
     * |        |          |Others =:Reserved.
     * |[15:8]  |SETUP1    |Setup Byte 1[15:8]
     * |        |          |This register provides byte 1 of the last setup packet received.
     * |        |          |For a Standard Device Request, the following bRequest Code information is returne.
     * |        |          |00000000 = Get Status.
     * |        |          |00000001 = Clear Feature.
     * |        |          |00000010 = Reserved.
     * |        |          |00000011 = Set Feature.
     * |        |          |00000100 = Reserved.
     * |        |          |00000101 = Set Address.
     * |        |          |00000110 = Get Descriptor.
     * |        |          |00000111 = Set Descriptor.
     * |        |          |00001000 = Get Configuration.
     * |        |          |00001001 = Set Configuration.
     * |        |          |00001010 = Get Interface.
     * |        |          |00001011 = Set Interface.
     * |        |          |00001100 = Synch Frame.
    */
    __I  uint32_t SETUP1_0;

    /**
     * SETUP3_2
     * ===================================================================================================
     * Offset: 0x48  Setup3 & Setup2 Bytes
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |SETUP2    |Setup Byte 2 [7:0]
     * |        |          |This register provides byte 2 of the last setup packet received.
     * |        |          |For a Standard Device Request, the least significant byte of the wValue field is returned.
     * |[15:8]  |SETUP3    |Setup Byte 3 [15:8]
     * |        |          |This register provides byte 3 of the last setup packet received.
     * |        |          |For a Standard Device Request, the most significant byte of the wValue field is returne.
    */
    __I  uint32_t SETUP3_2;

    /**
     * SETUP5_4
     * ===================================================================================================
     * Offset: 0x4C  Setup5 & Setup4 Bytes
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |SETUP4    |Setup Byte 4[7:0]
     * |        |          |This register provides byte 4 of the last setup packet received.
     * |        |          |For a Standard Device Request, the least significant byte of the wIndex is returne.
     * |[15:8]  |SETUP5    |Setup Byte 5[15:8]
     * |        |          |This register provides byte 5 of the last setup packet received.
     * |        |          |For a Standard Device Request, the most significant byte of the wIndex field is returne.
    */
    __I  uint32_t SETUP5_4;

    /**
     * SETUP7_6
     * ===================================================================================================
     * Offset: 0x50  Setup7 & Setup6 Bytes
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |SETUP6    |Setup Byte 6[7:0]
     * |        |          |This register provides byte 6 of the last setup packet received.
     * |        |          |For a Standard Device Request, the least significant byte of the wLength field is returne.
     * |[15:8]  |SETUP7    |Setup Byte 7[15:8]
     * |        |          |This register provides byte 7 of the last setup packet received.
     * |        |          |For a Standard Device Request, the most significant byte of the wLength field is returne.
    */
    __I  uint32_t SETUP7_6;

    /**
     * CEPBUFSTART
     * ===================================================================================================
     * Offset: 0x54  Control Endpoint RAM Start Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |SADDR     |Control-Endpoint Start Address
     * |        |          |This is the start-address of the RAM space allocated for the control-endpoint.
    */
    __IO uint32_t CEPBUFSTART;

    /**
     * CEPBUFEND
     * ===================================================================================================
     * Offset: 0x58  Control Endpoint RAM End Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[11:0]  |EADDR     |Control-Endpoint End Address
     * |        |          |This is the end-address of the RAM space allocated for the control-endpoint.
    */
    __IO uint32_t CEPBUFEND;

    /**
     * DMACTL
     * ===================================================================================================
     * Offset: 0x5C  DMA Control Status Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |EPNUM     |DMA Endpoint Address Bits
     * |        |          |Used to define the Endpoint Address
     * |[4]     |DMARD     |DMA Operation
     * |        |          |0 = The operation is a DMA write (read from USB buffer).
     * |        |          |DMA will check endpoint data available count (USBD_EPxDATCNT) according to EPNM setting before to perform DMA write operatio.
     * |        |          |1 = The operation is a DMA read (write to USB buffer).
     * |[5]     |DMAEN     |DMA Enable Bit
     * |        |          |0 = DMA function Disabled.
     * |        |          |1 = DMA function Enabled.
     * |[6]     |SGEN      |Scatter Gather Function Enable Bit
     * |        |          |0 = Scatter gather function Disabled.
     * |        |          |1 = Scatter gather function Enabled.
     * |[7]     |DMARST    |Reset DMA State Machine
     * |        |          |0 = No reset the DMA state machine.
     * |        |          |1 = Reset the DMA state machine.
    */
    __IO uint32_t DMACTL;

    /**
     * DMACNT
     * ===================================================================================================
     * Offset: 0x60  DMA Count Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[19:0]  |DMACNT    |DMA Transfer Count
     * |        |          |The transfer count of the DMA operation to be performed is written to this register.
    */
    __IO uint32_t DMACNT;

    USBD_EP_T EP[12];
    uint32_t RESERVE2[1];

    /**
     * UVCHDAT0
     * ===================================================================================================
     * Offset: 0x248  USB Header Word0
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DAT       |The first head data(byte 0 was sent first)
    */
    __IO uint32_t UVCHDAT0;

    /**
     * UVCHDAT1
     * ===================================================================================================
     * Offset: 0x24C  USB Header Word1
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DAT       |The second head data(byte 0 was sent first)
    */
    __IO uint32_t UVCHDAT1;

    /**
     * UVCHDAT2
     * ===================================================================================================
     * Offset: 0x250  USB Header Word2
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DAT       |The third head data(byte 0 was sent first)
    */
    __IO uint32_t UVCHDAT2;

    /**
     * UVCEPAHCNT
     * ===================================================================================================
     * Offset: 0x254  Endpoint A Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPAHCNT;

    /**
     * UVCEPBHCNT
     * ===================================================================================================
     * Offset: 0x258  Endpoint B Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPBHCNT;

    /**
     * UVCEPCHCNT
     * ===================================================================================================
     * Offset: 0x25C  Endpoint C Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPCHCNT;

    /**
     * UVCEPDHCNT
     * ===================================================================================================
     * Offset: 0x260  Endpoint D Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPDHCNT;

    /**
     * UVCEPEHCNT
     * ===================================================================================================
     * Offset: 0x264  Endpoint E Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPEHCNT;

    /**
     * UVCEPFHCNT
     * ===================================================================================================
     * Offset: 0x268  Endpoint F Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPFHCNT;

    /**
     * UVCEPGHCNT
     * ===================================================================================================
     * Offset: 0x26C  Endpoint G Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPGHCNT;

    /**
     * UVCEPHHCNT
     * ===================================================================================================
     * Offset: 0x270  Endpoint H Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPHHCNT;

    /**
     * UVCEPIHCNT
     * ===================================================================================================
     * Offset: 0x274  Endpoint I Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPIHCNT;

    /**
     * UVCEPJHCNT
     * ===================================================================================================
     * Offset: 0x278  Endpoint J Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPJHCNT;

    /**
     * UVCEPKHCNT
     * ===================================================================================================
     * Offset: 0x27C  Endpoint K Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPKHCNT;

    /**
     * UVCEPLHCNT
     * ===================================================================================================
     * Offset: 0x280  Endpoint L Header Count
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |CNT       |This is the header count for the endpoint A~L The header count must be EVEN.
    */
    __IO uint32_t UVCEPLHCNT;
    uint32_t RESERVE3[287];


    /**
     * DMAADDR
     * ===================================================================================================
     * Offset: 0x700  AHB DMA Address Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |DMAADDR   |DMAADDR
     * |        |          |The register specifies the address from which the DMA has to read / write.
     * |        |          |The address must WORD (32-bit) aligne.
    */
    __IO uint32_t DMAADDR;

    /**
     * PHYCTL
     * ===================================================================================================
     * Offset: 0x704  USB PHY Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3]     |LOWPWREN  |PHY Low Power Mode Enable Bit (Low Active)
     * |        |          |USB PHY HS low power mode configuration control bit.
     * |        |          |When this bit is 0, USB PHY is in Low Power Mode and USB PHY TX driver is enable only when USB Device Controller (USBD) transmit the data out to USB bus
     * |        |          |In this mode, HS signal quality is not guaranteed.
     * |        |          |0 = USB PHY Low Power Mode Enabled.
     * |        |          |1 = USB PHY Low Power Mode Disabled.
     * |[8]     |DPPUEN    |DP Pull-Up
     * |        |          |0 = Pull-up resistor on D+ Disabled.
     * |        |          |1 = Pull-up resistor on D+ Enabled.
     * |[9]     |PHYEN     |PHY Suspend Enable Bit
     * |        |          |0 = The USB PHY is suspend.
     * |        |          |1 = The USB PHY is not suspend.
     * |[24]    |WKEN      |Wake-Up Enable Bit
     * |        |          |0 = The wake-up function Disabled.
     * |        |          |1 = The wake-up function Enabled.
     * |[31]    |VBUSDET   |VBUS Status
     * |        |          |0 = The VBUS is not detected yet.
     * |        |          |1 = The VBUS is detected.
    */
    __IO uint32_t PHYCTL;

} USBD_T;

/**
    @addtogroup USBD_CONST USBD Bit Field Definition
    Constant Definitions for USBD Controller
@{ */

#define USBD_GINTSTS_USBIF_Pos           (0)                                               /*!< USBD_T::GINTSTS: USBIF Position           */
#define USBD_GINTSTS_USBIF_Msk           (0x1ul << USBD_GINTSTS_USBIF_Pos)                 /*!< USBD_T::GINTSTS: USBIF Mask               */

#define USBD_GINTSTS_CEPIF_Pos           (1)                                               /*!< USBD_T::GINTSTS: CEPIF Position           */
#define USBD_GINTSTS_CEPIF_Msk           (0x1ul << USBD_GINTSTS_CEPIF_Pos)                 /*!< USBD_T::GINTSTS: CEPIF Mask               */

#define USBD_GINTSTS_EPAIF_Pos           (2)                                               /*!< USBD_T::GINTSTS: EPAIF Position           */
#define USBD_GINTSTS_EPAIF_Msk           (0x1ul << USBD_GINTSTS_EPAIF_Pos)                 /*!< USBD_T::GINTSTS: EPAIF Mask               */

#define USBD_GINTSTS_EPBIF_Pos           (3)                                               /*!< USBD_T::GINTSTS: EPBIF Position           */
#define USBD_GINTSTS_EPBIF_Msk           (0x1ul << USBD_GINTSTS_EPBIF_Pos)                 /*!< USBD_T::GINTSTS: EPBIF Mask               */

#define USBD_GINTSTS_EPCIF_Pos           (4)                                               /*!< USBD_T::GINTSTS: EPCIF Position           */
#define USBD_GINTSTS_EPCIF_Msk           (0x1ul << USBD_GINTSTS_EPCIF_Pos)                 /*!< USBD_T::GINTSTS: EPCIF Mask               */

#define USBD_GINTSTS_EPDIF_Pos           (5)                                               /*!< USBD_T::GINTSTS: EPDIF Position           */
#define USBD_GINTSTS_EPDIF_Msk           (0x1ul << USBD_GINTSTS_EPDIF_Pos)                 /*!< USBD_T::GINTSTS: EPDIF Mask               */

#define USBD_GINTSTS_EPEIF_Pos           (6)                                               /*!< USBD_T::GINTSTS: EPEIF Position           */
#define USBD_GINTSTS_EPEIF_Msk           (0x1ul << USBD_GINTSTS_EPEIF_Pos)                 /*!< USBD_T::GINTSTS: EPEIF Mask               */

#define USBD_GINTSTS_EPFIF_Pos           (7)                                               /*!< USBD_T::GINTSTS: EPFIF Position           */
#define USBD_GINTSTS_EPFIF_Msk           (0x1ul << USBD_GINTSTS_EPFIF_Pos)                 /*!< USBD_T::GINTSTS: EPFIF Mask               */

#define USBD_GINTSTS_EPGIF_Pos           (8)                                               /*!< USBD_T::GINTSTS: EPGIF Position           */
#define USBD_GINTSTS_EPGIF_Msk           (0x1ul << USBD_GINTSTS_EPGIF_Pos)                 /*!< USBD_T::GINTSTS: EPGIF Mask               */

#define USBD_GINTSTS_EPHIF_Pos           (9)                                               /*!< USBD_T::GINTSTS: EPHIF Position           */
#define USBD_GINTSTS_EPHIF_Msk           (0x1ul << USBD_GINTSTS_EPHIF_Pos)                 /*!< USBD_T::GINTSTS: EPHIF Mask               */

#define USBD_GINTSTS_EPIIF_Pos           (10)                                              /*!< USBD_T::GINTSTS: EPIIF Position           */
#define USBD_GINTSTS_EPIIF_Msk           (0x1ul << USBD_GINTSTS_EPIIF_Pos)                 /*!< USBD_T::GINTSTS: EPIIF Mask               */

#define USBD_GINTSTS_EPJIF_Pos           (11)                                              /*!< USBD_T::GINTSTS: EPJIF Position           */
#define USBD_GINTSTS_EPJIF_Msk           (0x1ul << USBD_GINTSTS_EPJIF_Pos)                 /*!< USBD_T::GINTSTS: EPJIF Mask               */

#define USBD_GINTSTS_EPKIF_Pos           (12)                                              /*!< USBD_T::GINTSTS: EPKIF Position           */
#define USBD_GINTSTS_EPKIF_Msk           (0x1ul << USBD_GINTSTS_EPKIF_Pos)                 /*!< USBD_T::GINTSTS: EPKIF Mask               */

#define USBD_GINTSTS_EPLIF_Pos           (13)                                              /*!< USBD_T::GINTSTS: EPLIF Position           */
#define USBD_GINTSTS_EPLIF_Msk           (0x1ul << USBD_GINTSTS_EPLIF_Pos)                 /*!< USBD_T::GINTSTS: EPLIF Mask               */

#define USBD_GINTEN_USBIEN_Pos           (0)                                               /*!< USBD_T::GINTEN: USBIEN Position           */
#define USBD_GINTEN_USBIEN_Msk           (0x1ul << USBD_GINTEN_USBIEN_Pos)                 /*!< USBD_T::GINTEN: USBIEN Mask               */

#define USBD_GINTEN_CEPIEN_Pos           (1)                                               /*!< USBD_T::GINTEN: CEPIEN Position           */
#define USBD_GINTEN_CEPIEN_Msk           (0x1ul << USBD_GINTEN_CEPIEN_Pos)                 /*!< USBD_T::GINTEN: CEPIEN Mask               */

#define USBD_GINTEN_EPAIEN_Pos           (2)                                               /*!< USBD_T::GINTEN: EPAIEN Position           */
#define USBD_GINTEN_EPAIEN_Msk           (0x1ul << USBD_GINTEN_EPAIEN_Pos)                 /*!< USBD_T::GINTEN: EPAIEN Mask               */

#define USBD_GINTEN_EPBIEN_Pos           (3)                                               /*!< USBD_T::GINTEN: EPBIEN Position           */
#define USBD_GINTEN_EPBIEN_Msk           (0x1ul << USBD_GINTEN_EPBIEN_Pos)                 /*!< USBD_T::GINTEN: EPBIEN Mask               */

#define USBD_GINTEN_EPCIEN_Pos           (4)                                               /*!< USBD_T::GINTEN: EPCIEN Position           */
#define USBD_GINTEN_EPCIEN_Msk           (0x1ul << USBD_GINTEN_EPCIEN_Pos)                 /*!< USBD_T::GINTEN: EPCIEN Mask               */

#define USBD_GINTEN_EPDIEN_Pos           (5)                                               /*!< USBD_T::GINTEN: EPDIEN Position           */
#define USBD_GINTEN_EPDIEN_Msk           (0x1ul << USBD_GINTEN_EPDIEN_Pos)                 /*!< USBD_T::GINTEN: EPDIEN Mask               */

#define USBD_GINTEN_EPEIEN_Pos           (6)                                               /*!< USBD_T::GINTEN: EPEIEN Position           */
#define USBD_GINTEN_EPEIEN_Msk           (0x1ul << USBD_GINTEN_EPEIEN_Pos)                 /*!< USBD_T::GINTEN: EPEIEN Mask               */

#define USBD_GINTEN_EPFIEN_Pos           (7)                                               /*!< USBD_T::GINTEN: EPFIEN Position           */
#define USBD_GINTEN_EPFIEN_Msk           (0x1ul << USBD_GINTEN_EPFIEN_Pos)                 /*!< USBD_T::GINTEN: EPFIEN Mask               */

#define USBD_GINTEN_EPGIEN_Pos           (8)                                               /*!< USBD_T::GINTEN: EPGIEN Position           */
#define USBD_GINTEN_EPGIEN_Msk           (0x1ul << USBD_GINTEN_EPGIEN_Pos)                 /*!< USBD_T::GINTEN: EPGIEN Mask               */

#define USBD_GINTEN_EPHIEN_Pos           (9)                                               /*!< USBD_T::GINTEN: EPHIEN Position           */
#define USBD_GINTEN_EPHIEN_Msk           (0x1ul << USBD_GINTEN_EPHIEN_Pos)                 /*!< USBD_T::GINTEN: EPHIEN Mask               */

#define USBD_GINTEN_EPIIEN_Pos           (10)                                              /*!< USBD_T::GINTEN: EPIIEN Position           */
#define USBD_GINTEN_EPIIEN_Msk           (0x1ul << USBD_GINTEN_EPIIEN_Pos)                 /*!< USBD_T::GINTEN: EPIIEN Mask               */

#define USBD_GINTEN_EPJIEN_Pos           (11)                                              /*!< USBD_T::GINTEN: EPJIEN Position           */
#define USBD_GINTEN_EPJIEN_Msk           (0x1ul << USBD_GINTEN_EPJIEN_Pos)                 /*!< USBD_T::GINTEN: EPJIEN Mask               */

#define USBD_GINTEN_EPKIEN_Pos           (12)                                              /*!< USBD_T::GINTEN: EPKIEN Position           */
#define USBD_GINTEN_EPKIEN_Msk           (0x1ul << USBD_GINTEN_EPKIEN_Pos)                 /*!< USBD_T::GINTEN: EPKIEN Mask               */

#define USBD_GINTEN_EPLIEN_Pos           (13)                                              /*!< USBD_T::GINTEN: EPLIEN Position           */
#define USBD_GINTEN_EPLIEN_Msk           (0x1ul << USBD_GINTEN_EPLIEN_Pos)                 /*!< USBD_T::GINTEN: EPLIEN Mask               */

#define USBD_BUSINTSTS_SOFIF_Pos         (0)                                               /*!< USBD_T::BUSINTSTS: SOFIF Position         */
#define USBD_BUSINTSTS_SOFIF_Msk         (0x1ul << USBD_BUSINTSTS_SOFIF_Pos)               /*!< USBD_T::BUSINTSTS: SOFIF Mask             */

#define USBD_BUSINTSTS_RSTIF_Pos         (1)                                               /*!< USBD_T::BUSINTSTS: RSTIF Position         */
#define USBD_BUSINTSTS_RSTIF_Msk         (0x1ul << USBD_BUSINTSTS_RSTIF_Pos)               /*!< USBD_T::BUSINTSTS: RSTIF Mask             */

#define USBD_BUSINTSTS_RESUMEIF_Pos      (2)                                               /*!< USBD_T::BUSINTSTS: RESUMEIF Position      */
#define USBD_BUSINTSTS_RESUMEIF_Msk      (0x1ul << USBD_BUSINTSTS_RESUMEIF_Pos)            /*!< USBD_T::BUSINTSTS: RESUMEIF Mask          */

#define USBD_BUSINTSTS_SUSPENDIF_Pos     (3)                                               /*!< USBD_T::BUSINTSTS: SUSPENDIF Position     */
#define USBD_BUSINTSTS_SUSPENDIF_Msk     (0x1ul << USBD_BUSINTSTS_SUSPENDIF_Pos)           /*!< USBD_T::BUSINTSTS: SUSPENDIF Mask         */

#define USBD_BUSINTSTS_HISPDIF_Pos       (4)                                               /*!< USBD_T::BUSINTSTS: HISPDIF Position       */
#define USBD_BUSINTSTS_HISPDIF_Msk       (0x1ul << USBD_BUSINTSTS_HISPDIF_Pos)             /*!< USBD_T::BUSINTSTS: HISPDIF Mask           */

#define USBD_BUSINTSTS_DMADONEIF_Pos     (5)                                               /*!< USBD_T::BUSINTSTS: DMADONEIF Position     */
#define USBD_BUSINTSTS_DMADONEIF_Msk     (0x1ul << USBD_BUSINTSTS_DMADONEIF_Pos)           /*!< USBD_T::BUSINTSTS: DMADONEIF Mask         */

#define USBD_BUSINTSTS_PHYCLKVLDIF_Pos   (6)                                               /*!< USBD_T::BUSINTSTS: PHYCLKVLDIF Position   */
#define USBD_BUSINTSTS_PHYCLKVLDIF_Msk   (0x1ul << USBD_BUSINTSTS_PHYCLKVLDIF_Pos)         /*!< USBD_T::BUSINTSTS: PHYCLKVLDIF Mask       */

#define USBD_BUSINTSTS_VBUSDETIF_Pos     (8)                                               /*!< USBD_T::BUSINTSTS: VBUSDETIF Position     */
#define USBD_BUSINTSTS_VBUSDETIF_Msk     (0x1ul << USBD_BUSINTSTS_VBUSDETIF_Pos)           /*!< USBD_T::BUSINTSTS: VBUSDETIF Mask         */

#define USBD_BUSINTEN_SOFIEN_Pos         (0)                                               /*!< USBD_T::BUSINTEN: SOFIEN Position         */
#define USBD_BUSINTEN_SOFIEN_Msk         (0x1ul << USBD_BUSINTEN_SOFIEN_Pos)               /*!< USBD_T::BUSINTEN: SOFIEN Mask             */

#define USBD_BUSINTEN_RSTIEN_Pos         (1)                                               /*!< USBD_T::BUSINTEN: RSTIEN Position         */
#define USBD_BUSINTEN_RSTIEN_Msk         (0x1ul << USBD_BUSINTEN_RSTIEN_Pos)               /*!< USBD_T::BUSINTEN: RSTIEN Mask             */

#define USBD_BUSINTEN_RESUMEIEN_Pos      (2)                                               /*!< USBD_T::BUSINTEN: RESUMEIEN Position      */
#define USBD_BUSINTEN_RESUMEIEN_Msk      (0x1ul << USBD_BUSINTEN_RESUMEIEN_Pos)            /*!< USBD_T::BUSINTEN: RESUMEIEN Mask          */

#define USBD_BUSINTEN_SUSPENDIEN_Pos     (3)                                               /*!< USBD_T::BUSINTEN: SUSPENDIEN Position     */
#define USBD_BUSINTEN_SUSPENDIEN_Msk     (0x1ul << USBD_BUSINTEN_SUSPENDIEN_Pos)           /*!< USBD_T::BUSINTEN: SUSPENDIEN Mask         */

#define USBD_BUSINTEN_HISPDIEN_Pos       (4)                                               /*!< USBD_T::BUSINTEN: HISPDIEN Position       */
#define USBD_BUSINTEN_HISPDIEN_Msk       (0x1ul << USBD_BUSINTEN_HISPDIEN_Pos)             /*!< USBD_T::BUSINTEN: HISPDIEN Mask           */

#define USBD_BUSINTEN_DMADONEIEN_Pos     (5)                                               /*!< USBD_T::BUSINTEN: DMADONEIEN Position     */
#define USBD_BUSINTEN_DMADONEIEN_Msk     (0x1ul << USBD_BUSINTEN_DMADONEIEN_Pos)           /*!< USBD_T::BUSINTEN: DMADONEIEN Mask         */

#define USBD_BUSINTEN_PHYCLKVLDIEN_Pos   (6)                                               /*!< USBD_T::BUSINTEN: PHYCLKVLDIEN Position   */
#define USBD_BUSINTEN_PHYCLKVLDIEN_Msk   (0x1ul << USBD_BUSINTEN_PHYCLKVLDIEN_Pos)         /*!< USBD_T::BUSINTEN: PHYCLKVLDIEN Mask       */

#define USBD_BUSINTEN_VBUSDETIEN_Pos     (8)                                               /*!< USBD_T::BUSINTEN: VBUSDETIEN Position     */
#define USBD_BUSINTEN_VBUSDETIEN_Msk     (0x1ul << USBD_BUSINTEN_VBUSDETIEN_Pos)           /*!< USBD_T::BUSINTEN: VBUSDETIEN Mask         */

#define USBD_OPER_RESUMEEN_Pos           (0)                                               /*!< USBD_T::OPER: RESUMEEN Position           */
#define USBD_OPER_RESUMEEN_Msk           (0x1ul << USBD_OPER_RESUMEEN_Pos)                 /*!< USBD_T::OPER: RESUMEEN Mask               */

#define USBD_OPER_HISPDEN_Pos            (1)                                               /*!< USBD_T::OPER: HISPDEN Position            */
#define USBD_OPER_HISPDEN_Msk            (0x1ul << USBD_OPER_HISPDEN_Pos)                  /*!< USBD_T::OPER: HISPDEN Mask                */

#define USBD_OPER_CURSPD_Pos             (2)                                               /*!< USBD_T::OPER: CURSPD Position             */
#define USBD_OPER_CURSPD_Msk             (0x1ul << USBD_OPER_CURSPD_Pos)                   /*!< USBD_T::OPER: CURSPD Mask                 */

#define USBD_FRAMECNT_MFRAMECNT_Pos      (0)                                               /*!< USBD_T::FRAMECNT: MFRAMECNT Position      */
#define USBD_FRAMECNT_MFRAMECNT_Msk      (0x7ul << USBD_FRAMECNT_MFRAMECNT_Pos)            /*!< USBD_T::FRAMECNT: MFRAMECNT Mask          */

#define USBD_FRAMECNT_FRAMECNT_Pos       (3)                                               /*!< USBD_T::FRAMECNT: FRAMECNT Position       */
#define USBD_FRAMECNT_FRAMECNT_Msk       (0x7fful << USBD_FRAMECNT_FRAMECNT_Pos)           /*!< USBD_T::FRAMECNT: FRAMECNT Mask           */

#define USBD_FADDR_FADDR_Pos             (0)                                               /*!< USBD_T::FADDR: FADDR Position             */
#define USBD_FADDR_FADDR_Msk             (0x7ful << USBD_FADDR_FADDR_Pos)                  /*!< USBD_T::FADDR: FADDR Mask                 */

#define USBD_TEST_TESTMODE_Pos           (0)                                               /*!< USBD_T::TEST: TESTMODE Position           */
#define USBD_TEST_TESTMODE_Msk           (0x7ul << USBD_TEST_TESTMODE_Pos)                 /*!< USBD_T::TEST: TESTMODE Mask               */

#define USBD_CEPDAT_DAT_Pos              (0)                                               /*!< USBD_T::CEPDAT: DAT Position              */
#define USBD_CEPDAT_DAT_Msk              (0xfffffffful << USBD_CEPDAT_DAT_Pos)             /*!< USBD_T::CEPDAT: DAT Mask                  */

#define USBD_CEPCTL_NAKCLR_Pos           (0)                                               /*!< USBD_T::CEPCTL: NAKCLR Position           */
#define USBD_CEPCTL_NAKCLR_Msk           (0x1ul << USBD_CEPCTL_NAKCLR_Pos)                 /*!< USBD_T::CEPCTL: NAKCLR Mask               */

#define USBD_CEPCTL_STALLEN_Pos          (1)                                               /*!< USBD_T::CEPCTL: STALLEN Position          */
#define USBD_CEPCTL_STALLEN_Msk          (0x1ul << USBD_CEPCTL_STALLEN_Pos)                /*!< USBD_T::CEPCTL: STALLEN Mask              */

#define USBD_CEPCTL_ZEROLEN_Pos          (2)                                               /*!< USBD_T::CEPCTL: ZEROLEN Position          */
#define USBD_CEPCTL_ZEROLEN_Msk          (0x1ul << USBD_CEPCTL_ZEROLEN_Pos)                /*!< USBD_T::CEPCTL: ZEROLEN Mask              */

#define USBD_CEPCTL_FLUSH_Pos            (3)                                               /*!< USBD_T::CEPCTL: FLUSH Position            */
#define USBD_CEPCTL_FLUSH_Msk            (0x1ul << USBD_CEPCTL_FLUSH_Pos)                  /*!< USBD_T::CEPCTL: FLUSH Mask                */

#define USBD_CEPINTEN_SETUPTKIEN_Pos     (0)                                               /*!< USBD_T::CEPINTEN: SETUPTKIEN Position     */
#define USBD_CEPINTEN_SETUPTKIEN_Msk     (0x1ul << USBD_CEPINTEN_SETUPTKIEN_Pos)           /*!< USBD_T::CEPINTEN: SETUPTKIEN Mask         */

#define USBD_CEPINTEN_SETUPPKIEN_Pos     (1)                                               /*!< USBD_T::CEPINTEN: SETUPPKIEN Position     */
#define USBD_CEPINTEN_SETUPPKIEN_Msk     (0x1ul << USBD_CEPINTEN_SETUPPKIEN_Pos)           /*!< USBD_T::CEPINTEN: SETUPPKIEN Mask         */

#define USBD_CEPINTEN_OUTTKIEN_Pos       (2)                                               /*!< USBD_T::CEPINTEN: OUTTKIEN Position       */
#define USBD_CEPINTEN_OUTTKIEN_Msk       (0x1ul << USBD_CEPINTEN_OUTTKIEN_Pos)             /*!< USBD_T::CEPINTEN: OUTTKIEN Mask           */

#define USBD_CEPINTEN_INTKIEN_Pos        (3)                                               /*!< USBD_T::CEPINTEN: INTKIEN Position        */
#define USBD_CEPINTEN_INTKIEN_Msk        (0x1ul << USBD_CEPINTEN_INTKIEN_Pos)              /*!< USBD_T::CEPINTEN: INTKIEN Mask            */

#define USBD_CEPINTEN_PINGIEN_Pos        (4)                                               /*!< USBD_T::CEPINTEN: PINGIEN Position        */
#define USBD_CEPINTEN_PINGIEN_Msk        (0x1ul << USBD_CEPINTEN_PINGIEN_Pos)              /*!< USBD_T::CEPINTEN: PINGIEN Mask            */

#define USBD_CEPINTEN_TXPKIEN_Pos        (5)                                               /*!< USBD_T::CEPINTEN: TXPKIEN Position        */
#define USBD_CEPINTEN_TXPKIEN_Msk        (0x1ul << USBD_CEPINTEN_TXPKIEN_Pos)              /*!< USBD_T::CEPINTEN: TXPKIEN Mask            */

#define USBD_CEPINTEN_RXPKIEN_Pos        (6)                                               /*!< USBD_T::CEPINTEN: RXPKIEN Position        */
#define USBD_CEPINTEN_RXPKIEN_Msk        (0x1ul << USBD_CEPINTEN_RXPKIEN_Pos)              /*!< USBD_T::CEPINTEN: RXPKIEN Mask            */

#define USBD_CEPINTEN_NAKIEN_Pos         (7)                                               /*!< USBD_T::CEPINTEN: NAKIEN Position         */
#define USBD_CEPINTEN_NAKIEN_Msk         (0x1ul << USBD_CEPINTEN_NAKIEN_Pos)               /*!< USBD_T::CEPINTEN: NAKIEN Mask             */

#define USBD_CEPINTEN_STALLIEN_Pos       (8)                                               /*!< USBD_T::CEPINTEN: STALLIEN Position       */
#define USBD_CEPINTEN_STALLIEN_Msk       (0x1ul << USBD_CEPINTEN_STALLIEN_Pos)             /*!< USBD_T::CEPINTEN: STALLIEN Mask           */

#define USBD_CEPINTEN_ERRIEN_Pos         (9)                                               /*!< USBD_T::CEPINTEN: ERRIEN Position         */
#define USBD_CEPINTEN_ERRIEN_Msk         (0x1ul << USBD_CEPINTEN_ERRIEN_Pos)               /*!< USBD_T::CEPINTEN: ERRIEN Mask             */

#define USBD_CEPINTEN_STSDONEIEN_Pos     (10)                                              /*!< USBD_T::CEPINTEN: STSDONEIEN Position     */
#define USBD_CEPINTEN_STSDONEIEN_Msk     (0x1ul << USBD_CEPINTEN_STSDONEIEN_Pos)           /*!< USBD_T::CEPINTEN: STSDONEIEN Mask         */

#define USBD_CEPINTEN_BUFFULLIEN_Pos     (11)                                              /*!< USBD_T::CEPINTEN: BUFFULLIEN Position     */
#define USBD_CEPINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_CEPINTEN_BUFFULLIEN_Pos)           /*!< USBD_T::CEPINTEN: BUFFULLIEN Mask         */

#define USBD_CEPINTEN_BUFEMPTYIEN_Pos    (12)                                              /*!< USBD_T::CEPINTEN: BUFEMPTYIEN Position    */
#define USBD_CEPINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_CEPINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_T::CEPINTEN: BUFEMPTYIEN Mask        */

#define USBD_CEPINTSTS_SETUPTKIF_Pos     (0)                                               /*!< USBD_T::CEPINTSTS: SETUPTKIF Position     */
#define USBD_CEPINTSTS_SETUPTKIF_Msk     (0x1ul << USBD_CEPINTSTS_SETUPTKIF_Pos)           /*!< USBD_T::CEPINTSTS: SETUPTKIF Mask         */

#define USBD_CEPINTSTS_SETUPPKIF_Pos     (1)                                               /*!< USBD_T::CEPINTSTS: SETUPPKIF Position     */
#define USBD_CEPINTSTS_SETUPPKIF_Msk     (0x1ul << USBD_CEPINTSTS_SETUPPKIF_Pos)           /*!< USBD_T::CEPINTSTS: SETUPPKIF Mask         */

#define USBD_CEPINTSTS_OUTTKIF_Pos       (2)                                               /*!< USBD_T::CEPINTSTS: OUTTKIF Position       */
#define USBD_CEPINTSTS_OUTTKIF_Msk       (0x1ul << USBD_CEPINTSTS_OUTTKIF_Pos)             /*!< USBD_T::CEPINTSTS: OUTTKIF Mask           */

#define USBD_CEPINTSTS_INTKIF_Pos        (3)                                               /*!< USBD_T::CEPINTSTS: INTKIF Position        */
#define USBD_CEPINTSTS_INTKIF_Msk        (0x1ul << USBD_CEPINTSTS_INTKIF_Pos)              /*!< USBD_T::CEPINTSTS: INTKIF Mask            */

#define USBD_CEPINTSTS_PINGIF_Pos        (4)                                               /*!< USBD_T::CEPINTSTS: PINGIF Position        */
#define USBD_CEPINTSTS_PINGIF_Msk        (0x1ul << USBD_CEPINTSTS_PINGIF_Pos)              /*!< USBD_T::CEPINTSTS: PINGIF Mask            */

#define USBD_CEPINTSTS_TXPKIF_Pos        (5)                                               /*!< USBD_T::CEPINTSTS: TXPKIF Position        */
#define USBD_CEPINTSTS_TXPKIF_Msk        (0x1ul << USBD_CEPINTSTS_TXPKIF_Pos)              /*!< USBD_T::CEPINTSTS: TXPKIF Mask            */

#define USBD_CEPINTSTS_RXPKIF_Pos        (6)                                               /*!< USBD_T::CEPINTSTS: RXPKIF Position        */
#define USBD_CEPINTSTS_RXPKIF_Msk        (0x1ul << USBD_CEPINTSTS_RXPKIF_Pos)              /*!< USBD_T::CEPINTSTS: RXPKIF Mask            */

#define USBD_CEPINTSTS_NAKIF_Pos         (7)                                               /*!< USBD_T::CEPINTSTS: NAKIF Position         */
#define USBD_CEPINTSTS_NAKIF_Msk         (0x1ul << USBD_CEPINTSTS_NAKIF_Pos)               /*!< USBD_T::CEPINTSTS: NAKIF Mask             */

#define USBD_CEPINTSTS_STALLIF_Pos       (8)                                               /*!< USBD_T::CEPINTSTS: STALLIF Position       */
#define USBD_CEPINTSTS_STALLIF_Msk       (0x1ul << USBD_CEPINTSTS_STALLIF_Pos)             /*!< USBD_T::CEPINTSTS: STALLIF Mask           */

#define USBD_CEPINTSTS_ERRIF_Pos         (9)                                               /*!< USBD_T::CEPINTSTS: ERRIF Position         */
#define USBD_CEPINTSTS_ERRIF_Msk         (0x1ul << USBD_CEPINTSTS_ERRIF_Pos)               /*!< USBD_T::CEPINTSTS: ERRIF Mask             */

#define USBD_CEPINTSTS_STSDONEIF_Pos     (10)                                              /*!< USBD_T::CEPINTSTS: STSDONEIF Position     */
#define USBD_CEPINTSTS_STSDONEIF_Msk     (0x1ul << USBD_CEPINTSTS_STSDONEIF_Pos)           /*!< USBD_T::CEPINTSTS: STSDONEIF Mask         */

#define USBD_CEPINTSTS_BUFFULLIF_Pos     (11)                                              /*!< USBD_T::CEPINTSTS: BUFFULLIF Position     */
#define USBD_CEPINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_CEPINTSTS_BUFFULLIF_Pos)           /*!< USBD_T::CEPINTSTS: BUFFULLIF Mask         */

#define USBD_CEPINTSTS_BUFEMPTYIF_Pos    (12)                                              /*!< USBD_T::CEPINTSTS: BUFEMPTYIF Position    */
#define USBD_CEPINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_CEPINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_T::CEPINTSTS: BUFEMPTYIF Mask        */

#define USBD_CEPTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_T::CEPTXCNT: TXCNT Position          */
#define USBD_CEPTXCNT_TXCNT_Msk          (0xfful << USBD_CEPTXCNT_TXCNT_Pos)               /*!< USBD_T::CEPTXCNT: TXCNT Mask              */

#define USBD_CEPRXCNT_RXCNT_Pos          (0)                                               /*!< USBD_T::CEPRXCNT: RXCNT Position          */
#define USBD_CEPRXCNT_RXCNT_Msk          (0xfful << USBD_CEPRXCNT_RXCNT_Pos)               /*!< USBD_T::CEPRXCNT: RXCNT Mask              */

#define USBD_CEPDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_T::CEPDATCNT: DATCNT Position        */
#define USBD_CEPDATCNT_DATCNT_Msk        (0xfffful << USBD_CEPDATCNT_DATCNT_Pos)           /*!< USBD_T::CEPDATCNT: DATCNT Mask            */

#define USBD_SETUP1_0_SETUP0_Pos         (0)                                               /*!< USBD_T::SETUP1_0: SETUP0 Position         */
#define USBD_SETUP1_0_SETUP0_Msk         (0xfful << USBD_SETUP1_0_SETUP0_Pos)              /*!< USBD_T::SETUP1_0: SETUP0 Mask             */

#define USBD_SETUP1_0_SETUP1_Pos         (8)                                               /*!< USBD_T::SETUP1_0: SETUP1 Position         */
#define USBD_SETUP1_0_SETUP1_Msk         (0xfful << USBD_SETUP1_0_SETUP1_Pos)              /*!< USBD_T::SETUP1_0: SETUP1 Mask             */

#define USBD_SETUP3_2_SETUP2_Pos         (0)                                               /*!< USBD_T::SETUP3_2: SETUP2 Position         */
#define USBD_SETUP3_2_SETUP2_Msk         (0xfful << USBD_SETUP3_2_SETUP2_Pos)              /*!< USBD_T::SETUP3_2: SETUP2 Mask             */

#define USBD_SETUP3_2_SETUP3_Pos         (8)                                               /*!< USBD_T::SETUP3_2: SETUP3 Position         */
#define USBD_SETUP3_2_SETUP3_Msk         (0xfful << USBD_SETUP3_2_SETUP3_Pos)              /*!< USBD_T::SETUP3_2: SETUP3 Mask             */

#define USBD_SETUP5_4_SETUP4_Pos         (0)                                               /*!< USBD_T::SETUP5_4: SETUP4 Position         */
#define USBD_SETUP5_4_SETUP4_Msk         (0xfful << USBD_SETUP5_4_SETUP4_Pos)              /*!< USBD_T::SETUP5_4: SETUP4 Mask             */

#define USBD_SETUP5_4_SETUP5_Pos         (8)                                               /*!< USBD_T::SETUP5_4: SETUP5 Position         */
#define USBD_SETUP5_4_SETUP5_Msk         (0xfful << USBD_SETUP5_4_SETUP5_Pos)              /*!< USBD_T::SETUP5_4: SETUP5 Mask             */

#define USBD_SETUP7_6_SETUP6_Pos         (0)                                               /*!< USBD_T::SETUP7_6: SETUP6 Position         */
#define USBD_SETUP7_6_SETUP6_Msk         (0xfful << USBD_SETUP7_6_SETUP6_Pos)              /*!< USBD_T::SETUP7_6: SETUP6 Mask             */

#define USBD_SETUP7_6_SETUP7_Pos         (8)                                               /*!< USBD_T::SETUP7_6: SETUP7 Position         */
#define USBD_SETUP7_6_SETUP7_Msk         (0xfful << USBD_SETUP7_6_SETUP7_Pos)              /*!< USBD_T::SETUP7_6: SETUP7 Mask             */

#define USBD_CEPBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_T::CEPBUFSTART: SADDR Position       */
#define USBD_CEPBUFSTART_SADDR_Msk       (0xffful << USBD_CEPBUFSTART_SADDR_Pos)           /*!< USBD_T::CEPBUFSTART: SADDR Mask           */

#define USBD_CEPBUFEND_EADDR_Pos         (0)                                               /*!< USBD_T::CEPBUFEND: EADDR Position         */
#define USBD_CEPBUFEND_EADDR_Msk         (0xffful << USBD_CEPBUFEND_EADDR_Pos)             /*!< USBD_T::CEPBUFEND: EADDR Mask             */

#define USBD_DMACTL_EPNUM_Pos            (0)                                               /*!< USBD_T::DMACTL: EPNUM Position            */
#define USBD_DMACTL_EPNUM_Msk            (0xful << USBD_DMACTL_EPNUM_Pos)                  /*!< USBD_T::DMACTL: EPNUM Mask                */

#define USBD_DMACTL_DMARD_Pos            (4)                                               /*!< USBD_T::DMACTL: DMARD Position            */
#define USBD_DMACTL_DMARD_Msk            (0x1ul << USBD_DMACTL_DMARD_Pos)                  /*!< USBD_T::DMACTL: DMARD Mask                */

#define USBD_DMACTL_DMAEN_Pos            (5)                                               /*!< USBD_T::DMACTL: DMAEN Position            */
#define USBD_DMACTL_DMAEN_Msk            (0x1ul << USBD_DMACTL_DMAEN_Pos)                  /*!< USBD_T::DMACTL: DMAEN Mask                */

#define USBD_DMACTL_SGEN_Pos             (6)                                               /*!< USBD_T::DMACTL: SGEN Position             */
#define USBD_DMACTL_SGEN_Msk             (0x1ul << USBD_DMACTL_SGEN_Pos)                   /*!< USBD_T::DMACTL: SGEN Mask                 */

#define USBD_DMACTL_DMARST_Pos           (7)                                               /*!< USBD_T::DMACTL: DMARST Position           */
#define USBD_DMACTL_DMARST_Msk           (0x1ul << USBD_DMACTL_DMARST_Pos)                 /*!< USBD_T::DMACTL: DMARST Mask               */

#define USBD_DMACNT_DMACNT_Pos           (0)                                               /*!< USBD_T::DMACNT: DMACNT Position           */
#define USBD_DMACNT_DMACNT_Msk           (0xffffful << USBD_DMACNT_DMACNT_Pos)             /*!< USBD_T::DMACNT: DMACNT Mask               */

#define USBD_EPADAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPADAT_EPDAT_Msk            (0xfffffffful << USBD_EPADAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPAINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPAINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPAINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPAINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPAINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPAINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPAINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPAINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPAINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPAINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPAINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPAINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPAINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPAINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPAINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPAINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPAINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPAINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPAINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPAINTSTS_INTKIF_Msk        (0x1ul << USBD_EPAINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPAINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPAINTSTS_PINGIF_Msk        (0x1ul << USBD_EPAINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPAINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPAINTSTS_NAKIF_Msk         (0x1ul << USBD_EPAINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPAINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPAINTSTS_STALLIF_Msk       (0x1ul << USBD_EPAINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPAINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPAINTSTS_NYETIF_Msk        (0x1ul << USBD_EPAINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPAINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPAINTSTS_ERRIF_Msk         (0x1ul << USBD_EPAINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPAINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPAINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPAINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPAINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPAINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPAINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPAINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPAINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPAINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPAINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPAINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPAINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPAINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPAINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPAINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPAINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPAINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPAINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPAINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPAINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPAINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPAINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPAINTEN_INTKIEN_Msk        (0x1ul << USBD_EPAINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPAINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPAINTEN_PINGIEN_Msk        (0x1ul << USBD_EPAINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPAINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPAINTEN_NAKIEN_Msk         (0x1ul << USBD_EPAINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPAINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPAINTEN_STALLIEN_Msk       (0x1ul << USBD_EPAINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPAINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPAINTEN_NYETIEN_Msk        (0x1ul << USBD_EPAINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPAINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPAINTEN_ERRIEN_Msk         (0x1ul << USBD_EPAINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPAINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPAINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPAINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPADATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPADATCNT_DATCNT_Msk        (0xfffful << USBD_EPADATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPADATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPADATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPADATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPARSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPARSPCTL_FLUSH_Msk         (0x1ul << USBD_EPARSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPARSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPARSPCTL_MODE_Msk          (0x3ul << USBD_EPARSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPARSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPARSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPARSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPARSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPARSPCTL_HALT_Msk          (0x1ul << USBD_EPARSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPARSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPARSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPARSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPARSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPARSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPARSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPARSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPARSPCTL_DISBUF_Msk        (0x1ul << USBD_EPARSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPAMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPAMPS_EPMPS_Msk            (0x7fful << USBD_EPAMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPATXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPATXCNT_TXCNT_Msk          (0x7fful << USBD_EPATXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPACFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPACFG_EPEN_Msk             (0x1ul << USBD_EPACFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPACFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPACFG_EPTYPE_Msk           (0x3ul << USBD_EPACFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPACFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPACFG_EPDIR_Msk            (0x1ul << USBD_EPACFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPACFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPACFG_EPNUM_Msk            (0xful << USBD_EPACFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPABUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPABUFSTART_SADDR_Msk       (0xffful << USBD_EPABUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPABUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPABUFEND_EADDR_Msk         (0xffful << USBD_EPABUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPBDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPBDAT_EPDAT_Msk            (0xfffffffful << USBD_EPBDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPBINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPBINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPBINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPBINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPBINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPBINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPBINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPBINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPBINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPBINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPBINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPBINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPBINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPBINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPBINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPBINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPBINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPBINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPBINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPBINTSTS_INTKIF_Msk        (0x1ul << USBD_EPBINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPBINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPBINTSTS_PINGIF_Msk        (0x1ul << USBD_EPBINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPBINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPBINTSTS_NAKIF_Msk         (0x1ul << USBD_EPBINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPBINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPBINTSTS_STALLIF_Msk       (0x1ul << USBD_EPBINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPBINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPBINTSTS_NYETIF_Msk        (0x1ul << USBD_EPBINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPBINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPBINTSTS_ERRIF_Msk         (0x1ul << USBD_EPBINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPBINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPBINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPBINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPBINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPBINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPBINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPBINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPBINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPBINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPBINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPBINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPBINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPBINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPBINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPBINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPBINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPBINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPBINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPBINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPBINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPBINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPBINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPBINTEN_INTKIEN_Msk        (0x1ul << USBD_EPBINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPBINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPBINTEN_PINGIEN_Msk        (0x1ul << USBD_EPBINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPBINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPBINTEN_NAKIEN_Msk         (0x1ul << USBD_EPBINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPBINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPBINTEN_STALLIEN_Msk       (0x1ul << USBD_EPBINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPBINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPBINTEN_NYETIEN_Msk        (0x1ul << USBD_EPBINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPBINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPBINTEN_ERRIEN_Msk         (0x1ul << USBD_EPBINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPBINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPBINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPBINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPBDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPBDATCNT_DATCNT_Msk        (0xfffful << USBD_EPBDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPBDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPBDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPBDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPBRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPBRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPBRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPBRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPBRSPCTL_MODE_Msk          (0x3ul << USBD_EPBRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPBRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPBRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPBRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPBRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPBRSPCTL_HALT_Msk          (0x1ul << USBD_EPBRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPBRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPBRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPBRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPBRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPBRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPBRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPBRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPBRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPBRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPBMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPBMPS_EPMPS_Msk            (0x7fful << USBD_EPBMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPBTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPBTXCNT_TXCNT_Msk          (0x7fful << USBD_EPBTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPBCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPBCFG_EPEN_Msk             (0x1ul << USBD_EPBCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPBCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPBCFG_EPTYPE_Msk           (0x3ul << USBD_EPBCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPBCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPBCFG_EPDIR_Msk            (0x1ul << USBD_EPBCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPBCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPBCFG_EPNUM_Msk            (0xful << USBD_EPBCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPBBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPBBUFSTART_SADDR_Msk       (0xffful << USBD_EPBBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPBBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPBBUFEND_EADDR_Msk         (0xffful << USBD_EPBBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPCDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPCDAT_EPDAT_Msk            (0xfffffffful << USBD_EPCDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPCINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPCINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPCINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPCINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPCINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPCINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPCINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPCINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPCINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPCINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPCINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPCINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPCINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPCINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPCINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPCINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPCINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPCINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPCINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPCINTSTS_INTKIF_Msk        (0x1ul << USBD_EPCINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPCINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPCINTSTS_PINGIF_Msk        (0x1ul << USBD_EPCINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPCINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPCINTSTS_NAKIF_Msk         (0x1ul << USBD_EPCINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPCINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPCINTSTS_STALLIF_Msk       (0x1ul << USBD_EPCINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPCINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPCINTSTS_NYETIF_Msk        (0x1ul << USBD_EPCINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPCINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPCINTSTS_ERRIF_Msk         (0x1ul << USBD_EPCINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPCINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPCINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPCINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPCINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPCINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPCINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPCINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPCINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPCINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPCINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPCINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPCINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPCINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPCINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPCINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPCINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPCINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPCINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPCINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPCINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPCINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPCINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPCINTEN_INTKIEN_Msk        (0x1ul << USBD_EPCINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPCINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPCINTEN_PINGIEN_Msk        (0x1ul << USBD_EPCINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPCINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPCINTEN_NAKIEN_Msk         (0x1ul << USBD_EPCINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPCINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPCINTEN_STALLIEN_Msk       (0x1ul << USBD_EPCINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPCINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPCINTEN_NYETIEN_Msk        (0x1ul << USBD_EPCINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPCINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPCINTEN_ERRIEN_Msk         (0x1ul << USBD_EPCINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPCINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPCINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPCINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPCDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPCDATCNT_DATCNT_Msk        (0xfffful << USBD_EPCDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPCDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPCDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPCDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPCRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPCRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPCRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPCRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPCRSPCTL_MODE_Msk          (0x3ul << USBD_EPCRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPCRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPCRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPCRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPCRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPCRSPCTL_HALT_Msk          (0x1ul << USBD_EPCRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPCRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPCRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPCRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPCRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPCRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPCRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPCRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPCRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPCRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPCMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPCMPS_EPMPS_Msk            (0x7fful << USBD_EPCMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPCTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPCTXCNT_TXCNT_Msk          (0x7fful << USBD_EPCTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPCCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPCCFG_EPEN_Msk             (0x1ul << USBD_EPCCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPCCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPCCFG_EPTYPE_Msk           (0x3ul << USBD_EPCCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPCCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPCCFG_EPDIR_Msk            (0x1ul << USBD_EPCCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPCCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPCCFG_EPNUM_Msk            (0xful << USBD_EPCCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPCBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPCBUFSTART_SADDR_Msk       (0xffful << USBD_EPCBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPCBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPCBUFEND_EADDR_Msk         (0xffful << USBD_EPCBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPDDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPDDAT_EPDAT_Msk            (0xfffffffful << USBD_EPDDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPDINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPDINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPDINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPDINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPDINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPDINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPDINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPDINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPDINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPDINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPDINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPDINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPDINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPDINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPDINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPDINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPDINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPDINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPDINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPDINTSTS_INTKIF_Msk        (0x1ul << USBD_EPDINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPDINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPDINTSTS_PINGIF_Msk        (0x1ul << USBD_EPDINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPDINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPDINTSTS_NAKIF_Msk         (0x1ul << USBD_EPDINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPDINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPDINTSTS_STALLIF_Msk       (0x1ul << USBD_EPDINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPDINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPDINTSTS_NYETIF_Msk        (0x1ul << USBD_EPDINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPDINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPDINTSTS_ERRIF_Msk         (0x1ul << USBD_EPDINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPDINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPDINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPDINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPDINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPDINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPDINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPDINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPDINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPDINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPDINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPDINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPDINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPDINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPDINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPDINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPDINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPDINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPDINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPDINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPDINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPDINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPDINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPDINTEN_INTKIEN_Msk        (0x1ul << USBD_EPDINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPDINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPDINTEN_PINGIEN_Msk        (0x1ul << USBD_EPDINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPDINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPDINTEN_NAKIEN_Msk         (0x1ul << USBD_EPDINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPDINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPDINTEN_STALLIEN_Msk       (0x1ul << USBD_EPDINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPDINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPDINTEN_NYETIEN_Msk        (0x1ul << USBD_EPDINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPDINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPDINTEN_ERRIEN_Msk         (0x1ul << USBD_EPDINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPDINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPDINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPDINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPDDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPDDATCNT_DATCNT_Msk        (0xfffful << USBD_EPDDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPDDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPDDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPDDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPDRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPDRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPDRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPDRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPDRSPCTL_MODE_Msk          (0x3ul << USBD_EPDRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPDRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPDRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPDRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPDRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPDRSPCTL_HALT_Msk          (0x1ul << USBD_EPDRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPDRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPDRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPDRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPDRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPDRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPDRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPDRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPDRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPDRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPDMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPDMPS_EPMPS_Msk            (0x7fful << USBD_EPDMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPDTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPDTXCNT_TXCNT_Msk          (0x7fful << USBD_EPDTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPDCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPDCFG_EPEN_Msk             (0x1ul << USBD_EPDCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPDCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPDCFG_EPTYPE_Msk           (0x3ul << USBD_EPDCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPDCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPDCFG_EPDIR_Msk            (0x1ul << USBD_EPDCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPDCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPDCFG_EPNUM_Msk            (0xful << USBD_EPDCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPDBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPDBUFSTART_SADDR_Msk       (0xffful << USBD_EPDBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPDBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPDBUFEND_EADDR_Msk         (0xffful << USBD_EPDBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPEDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPEDAT_EPDAT_Msk            (0xfffffffful << USBD_EPEDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPEINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPEINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPEINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPEINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPEINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPEINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPEINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPEINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPEINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPEINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPEINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPEINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPEINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPEINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPEINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPEINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPEINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPEINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPEINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPEINTSTS_INTKIF_Msk        (0x1ul << USBD_EPEINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPEINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPEINTSTS_PINGIF_Msk        (0x1ul << USBD_EPEINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPEINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPEINTSTS_NAKIF_Msk         (0x1ul << USBD_EPEINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPEINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPEINTSTS_STALLIF_Msk       (0x1ul << USBD_EPEINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPEINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPEINTSTS_NYETIF_Msk        (0x1ul << USBD_EPEINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPEINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPEINTSTS_ERRIF_Msk         (0x1ul << USBD_EPEINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPEINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPEINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPEINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPEINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPEINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPEINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPEINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPEINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPEINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPEINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPEINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPEINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPEINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPEINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPEINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPEINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPEINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPEINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPEINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPEINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPEINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPEINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPEINTEN_INTKIEN_Msk        (0x1ul << USBD_EPEINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPEINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPEINTEN_PINGIEN_Msk        (0x1ul << USBD_EPEINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPEINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPEINTEN_NAKIEN_Msk         (0x1ul << USBD_EPEINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPEINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPEINTEN_STALLIEN_Msk       (0x1ul << USBD_EPEINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPEINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPEINTEN_NYETIEN_Msk        (0x1ul << USBD_EPEINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPEINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPEINTEN_ERRIEN_Msk         (0x1ul << USBD_EPEINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPEINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPEINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPEINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPEDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPEDATCNT_DATCNT_Msk        (0xfffful << USBD_EPEDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPEDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPEDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPEDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPERSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPERSPCTL_FLUSH_Msk         (0x1ul << USBD_EPERSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPERSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPERSPCTL_MODE_Msk          (0x3ul << USBD_EPERSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPERSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPERSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPERSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPERSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPERSPCTL_HALT_Msk          (0x1ul << USBD_EPERSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPERSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPERSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPERSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPERSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPERSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPERSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPERSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPERSPCTL_DISBUF_Msk        (0x1ul << USBD_EPERSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPEMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPEMPS_EPMPS_Msk            (0x7fful << USBD_EPEMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPETXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPETXCNT_TXCNT_Msk          (0x7fful << USBD_EPETXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPECFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPECFG_EPEN_Msk             (0x1ul << USBD_EPECFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPECFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPECFG_EPTYPE_Msk           (0x3ul << USBD_EPECFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPECFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPECFG_EPDIR_Msk            (0x1ul << USBD_EPECFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPECFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPECFG_EPNUM_Msk            (0xful << USBD_EPECFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPEBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPEBUFSTART_SADDR_Msk       (0xffful << USBD_EPEBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPEBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPEBUFEND_EADDR_Msk         (0xffful << USBD_EPEBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPFDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPFDAT_EPDAT_Msk            (0xfffffffful << USBD_EPFDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPFINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPFINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPFINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPFINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPFINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPFINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPFINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPFINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPFINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPFINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPFINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPFINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPFINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPFINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPFINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPFINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPFINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPFINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPFINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPFINTSTS_INTKIF_Msk        (0x1ul << USBD_EPFINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPFINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPFINTSTS_PINGIF_Msk        (0x1ul << USBD_EPFINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPFINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPFINTSTS_NAKIF_Msk         (0x1ul << USBD_EPFINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPFINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPFINTSTS_STALLIF_Msk       (0x1ul << USBD_EPFINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPFINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPFINTSTS_NYETIF_Msk        (0x1ul << USBD_EPFINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPFINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPFINTSTS_ERRIF_Msk         (0x1ul << USBD_EPFINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPFINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPFINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPFINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPFINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPFINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPFINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPFINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPFINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPFINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPFINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPFINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPFINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPFINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPFINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPFINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPFINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPFINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPFINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPFINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPFINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPFINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPFINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPFINTEN_INTKIEN_Msk        (0x1ul << USBD_EPFINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPFINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPFINTEN_PINGIEN_Msk        (0x1ul << USBD_EPFINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPFINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPFINTEN_NAKIEN_Msk         (0x1ul << USBD_EPFINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPFINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPFINTEN_STALLIEN_Msk       (0x1ul << USBD_EPFINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPFINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPFINTEN_NYETIEN_Msk        (0x1ul << USBD_EPFINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPFINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPFINTEN_ERRIEN_Msk         (0x1ul << USBD_EPFINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPFINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPFINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPFINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPFDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPFDATCNT_DATCNT_Msk        (0xfffful << USBD_EPFDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPFDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPFDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPFDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPFRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPFRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPFRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPFRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPFRSPCTL_MODE_Msk          (0x3ul << USBD_EPFRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPFRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPFRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPFRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPFRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPFRSPCTL_HALT_Msk          (0x1ul << USBD_EPFRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPFRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPFRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPFRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPFRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPFRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPFRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPFRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPFRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPFRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPFMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPFMPS_EPMPS_Msk            (0x7fful << USBD_EPFMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPFTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPFTXCNT_TXCNT_Msk          (0x7fful << USBD_EPFTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPFCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPFCFG_EPEN_Msk             (0x1ul << USBD_EPFCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPFCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPFCFG_EPTYPE_Msk           (0x3ul << USBD_EPFCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPFCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPFCFG_EPDIR_Msk            (0x1ul << USBD_EPFCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPFCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPFCFG_EPNUM_Msk            (0xful << USBD_EPFCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPFBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPFBUFSTART_SADDR_Msk       (0xffful << USBD_EPFBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPFBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPFBUFEND_EADDR_Msk         (0xffful << USBD_EPFBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPGDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPGDAT_EPDAT_Msk            (0xfffffffful << USBD_EPGDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPGINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPGINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPGINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPGINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPGINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPGINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPGINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPGINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPGINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPGINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPGINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPGINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPGINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPGINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPGINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPGINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPGINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPGINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPGINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPGINTSTS_INTKIF_Msk        (0x1ul << USBD_EPGINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPGINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPGINTSTS_PINGIF_Msk        (0x1ul << USBD_EPGINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPGINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPGINTSTS_NAKIF_Msk         (0x1ul << USBD_EPGINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPGINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPGINTSTS_STALLIF_Msk       (0x1ul << USBD_EPGINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPGINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPGINTSTS_NYETIF_Msk        (0x1ul << USBD_EPGINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPGINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPGINTSTS_ERRIF_Msk         (0x1ul << USBD_EPGINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPGINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPGINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPGINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPGINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPGINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPGINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPGINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPGINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPGINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPGINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPGINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPGINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPGINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPGINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPGINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPGINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPGINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPGINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPGINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPGINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPGINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPGINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPGINTEN_INTKIEN_Msk        (0x1ul << USBD_EPGINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPGINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPGINTEN_PINGIEN_Msk        (0x1ul << USBD_EPGINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPGINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPGINTEN_NAKIEN_Msk         (0x1ul << USBD_EPGINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPGINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPGINTEN_STALLIEN_Msk       (0x1ul << USBD_EPGINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPGINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPGINTEN_NYETIEN_Msk        (0x1ul << USBD_EPGINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPGINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPGINTEN_ERRIEN_Msk         (0x1ul << USBD_EPGINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPGINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPGINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPGINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPGDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPGDATCNT_DATCNT_Msk        (0xfffful << USBD_EPGDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPGDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPGDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPGDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPGRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPGRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPGRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPGRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPGRSPCTL_MODE_Msk          (0x3ul << USBD_EPGRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPGRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPGRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPGRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPGRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPGRSPCTL_HALT_Msk          (0x1ul << USBD_EPGRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPGRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPGRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPGRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPGRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPGRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPGRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPGRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPGRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPGRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPGMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPGMPS_EPMPS_Msk            (0x7fful << USBD_EPGMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPGTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPGTXCNT_TXCNT_Msk          (0x7fful << USBD_EPGTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPGCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPGCFG_EPEN_Msk             (0x1ul << USBD_EPGCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPGCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPGCFG_EPTYPE_Msk           (0x3ul << USBD_EPGCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPGCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPGCFG_EPDIR_Msk            (0x1ul << USBD_EPGCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPGCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPGCFG_EPNUM_Msk            (0xful << USBD_EPGCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPGBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPGBUFSTART_SADDR_Msk       (0xffful << USBD_EPGBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPGBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPGBUFEND_EADDR_Msk         (0xffful << USBD_EPGBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPHDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPHDAT_EPDAT_Msk            (0xfffffffful << USBD_EPHDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPHINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPHINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPHINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPHINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPHINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPHINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPHINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPHINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPHINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPHINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPHINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPHINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPHINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPHINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPHINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPHINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPHINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPHINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPHINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPHINTSTS_INTKIF_Msk        (0x1ul << USBD_EPHINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPHINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPHINTSTS_PINGIF_Msk        (0x1ul << USBD_EPHINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPHINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPHINTSTS_NAKIF_Msk         (0x1ul << USBD_EPHINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPHINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPHINTSTS_STALLIF_Msk       (0x1ul << USBD_EPHINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPHINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPHINTSTS_NYETIF_Msk        (0x1ul << USBD_EPHINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPHINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPHINTSTS_ERRIF_Msk         (0x1ul << USBD_EPHINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPHINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPHINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPHINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPHINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPHINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPHINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPHINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPHINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPHINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPHINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPHINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPHINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPHINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPHINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPHINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPHINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPHINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPHINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPHINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPHINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPHINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPHINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPHINTEN_INTKIEN_Msk        (0x1ul << USBD_EPHINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPHINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPHINTEN_PINGIEN_Msk        (0x1ul << USBD_EPHINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPHINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPHINTEN_NAKIEN_Msk         (0x1ul << USBD_EPHINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPHINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPHINTEN_STALLIEN_Msk       (0x1ul << USBD_EPHINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPHINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPHINTEN_NYETIEN_Msk        (0x1ul << USBD_EPHINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPHINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPHINTEN_ERRIEN_Msk         (0x1ul << USBD_EPHINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPHINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPHINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPHINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPHDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPHDATCNT_DATCNT_Msk        (0xfffful << USBD_EPHDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPHDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPHDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPHDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPHRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPHRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPHRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPHRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPHRSPCTL_MODE_Msk          (0x3ul << USBD_EPHRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPHRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPHRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPHRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPHRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPHRSPCTL_HALT_Msk          (0x1ul << USBD_EPHRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPHRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPHRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPHRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPHRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPHRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPHRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPHRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPHRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPHRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPHMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPHMPS_EPMPS_Msk            (0x7fful << USBD_EPHMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPHTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPHTXCNT_TXCNT_Msk          (0x7fful << USBD_EPHTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPHCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPHCFG_EPEN_Msk             (0x1ul << USBD_EPHCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPHCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPHCFG_EPTYPE_Msk           (0x3ul << USBD_EPHCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPHCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPHCFG_EPDIR_Msk            (0x1ul << USBD_EPHCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPHCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPHCFG_EPNUM_Msk            (0xful << USBD_EPHCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPHBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPHBUFSTART_SADDR_Msk       (0xffful << USBD_EPHBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPHBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPHBUFEND_EADDR_Msk         (0xffful << USBD_EPHBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPIDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPIDAT_EPDAT_Msk            (0xfffffffful << USBD_EPIDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPIINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPIINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPIINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPIINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPIINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPIINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPIINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPIINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPIINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPIINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPIINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPIINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPIINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPIINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPIINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPIINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPIINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPIINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPIINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPIINTSTS_INTKIF_Msk        (0x1ul << USBD_EPIINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPIINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPIINTSTS_PINGIF_Msk        (0x1ul << USBD_EPIINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPIINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPIINTSTS_NAKIF_Msk         (0x1ul << USBD_EPIINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPIINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPIINTSTS_STALLIF_Msk       (0x1ul << USBD_EPIINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPIINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPIINTSTS_NYETIF_Msk        (0x1ul << USBD_EPIINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPIINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPIINTSTS_ERRIF_Msk         (0x1ul << USBD_EPIINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPIINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPIINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPIINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPIINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPIINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPIINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPIINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPIINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPIINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPIINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPIINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPIINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPIINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPIINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPIINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPIINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPIINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPIINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPIINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPIINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPIINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPIINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPIINTEN_INTKIEN_Msk        (0x1ul << USBD_EPIINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPIINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPIINTEN_PINGIEN_Msk        (0x1ul << USBD_EPIINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPIINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPIINTEN_NAKIEN_Msk         (0x1ul << USBD_EPIINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPIINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPIINTEN_STALLIEN_Msk       (0x1ul << USBD_EPIINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPIINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPIINTEN_NYETIEN_Msk        (0x1ul << USBD_EPIINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPIINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPIINTEN_ERRIEN_Msk         (0x1ul << USBD_EPIINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPIINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPIINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPIINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPIDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPIDATCNT_DATCNT_Msk        (0xfffful << USBD_EPIDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPIDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPIDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPIDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPIRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPIRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPIRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPIRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPIRSPCTL_MODE_Msk          (0x3ul << USBD_EPIRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPIRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPIRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPIRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPIRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPIRSPCTL_HALT_Msk          (0x1ul << USBD_EPIRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPIRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPIRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPIRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPIRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPIRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPIRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPIRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPIRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPIRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPIMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPIMPS_EPMPS_Msk            (0x7fful << USBD_EPIMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPITXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPITXCNT_TXCNT_Msk          (0x7fful << USBD_EPITXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPICFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPICFG_EPEN_Msk             (0x1ul << USBD_EPICFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPICFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPICFG_EPTYPE_Msk           (0x3ul << USBD_EPICFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPICFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPICFG_EPDIR_Msk            (0x1ul << USBD_EPICFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPICFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPICFG_EPNUM_Msk            (0xful << USBD_EPICFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPIBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPIBUFSTART_SADDR_Msk       (0xffful << USBD_EPIBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPIBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPIBUFEND_EADDR_Msk         (0xffful << USBD_EPIBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPJDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPJDAT_EPDAT_Msk            (0xfffffffful << USBD_EPJDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPJINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPJINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPJINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPJINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPJINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPJINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPJINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPJINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPJINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPJINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPJINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPJINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPJINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPJINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPJINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPJINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPJINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPJINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPJINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPJINTSTS_INTKIF_Msk        (0x1ul << USBD_EPJINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPJINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPJINTSTS_PINGIF_Msk        (0x1ul << USBD_EPJINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPJINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPJINTSTS_NAKIF_Msk         (0x1ul << USBD_EPJINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPJINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPJINTSTS_STALLIF_Msk       (0x1ul << USBD_EPJINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPJINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPJINTSTS_NYETIF_Msk        (0x1ul << USBD_EPJINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPJINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPJINTSTS_ERRIF_Msk         (0x1ul << USBD_EPJINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPJINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPJINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPJINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPJINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPJINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPJINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPJINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPJINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPJINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPJINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPJINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPJINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPJINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPJINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPJINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPJINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPJINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPJINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPJINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPJINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPJINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPJINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPJINTEN_INTKIEN_Msk        (0x1ul << USBD_EPJINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPJINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPJINTEN_PINGIEN_Msk        (0x1ul << USBD_EPJINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPJINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPJINTEN_NAKIEN_Msk         (0x1ul << USBD_EPJINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPJINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPJINTEN_STALLIEN_Msk       (0x1ul << USBD_EPJINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPJINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPJINTEN_NYETIEN_Msk        (0x1ul << USBD_EPJINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPJINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPJINTEN_ERRIEN_Msk         (0x1ul << USBD_EPJINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPJINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPJINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPJINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPJDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPJDATCNT_DATCNT_Msk        (0xfffful << USBD_EPJDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPJDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPJDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPJDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPJRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPJRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPJRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPJRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPJRSPCTL_MODE_Msk          (0x3ul << USBD_EPJRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPJRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPJRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPJRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPJRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPJRSPCTL_HALT_Msk          (0x1ul << USBD_EPJRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPJRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPJRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPJRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPJRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPJRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPJRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPJRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPJRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPJRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPJMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPJMPS_EPMPS_Msk            (0x7fful << USBD_EPJMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPJTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPJTXCNT_TXCNT_Msk          (0x7fful << USBD_EPJTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPJCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPJCFG_EPEN_Msk             (0x1ul << USBD_EPJCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPJCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPJCFG_EPTYPE_Msk           (0x3ul << USBD_EPJCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPJCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPJCFG_EPDIR_Msk            (0x1ul << USBD_EPJCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPJCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPJCFG_EPNUM_Msk            (0xful << USBD_EPJCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPJBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPJBUFSTART_SADDR_Msk       (0xffful << USBD_EPJBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPJBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPJBUFEND_EADDR_Msk         (0xffful << USBD_EPJBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPKDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPKDAT_EPDAT_Msk            (0xfffffffful << USBD_EPKDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPKINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPKINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPKINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPKINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPKINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPKINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPKINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPKINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPKINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPKINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPKINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPKINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPKINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPKINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPKINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPKINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPKINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPKINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPKINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPKINTSTS_INTKIF_Msk        (0x1ul << USBD_EPKINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPKINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPKINTSTS_PINGIF_Msk        (0x1ul << USBD_EPKINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPKINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPKINTSTS_NAKIF_Msk         (0x1ul << USBD_EPKINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPKINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPKINTSTS_STALLIF_Msk       (0x1ul << USBD_EPKINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPKINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPKINTSTS_NYETIF_Msk        (0x1ul << USBD_EPKINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPKINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPKINTSTS_ERRIF_Msk         (0x1ul << USBD_EPKINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPKINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPKINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPKINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPKINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPKINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPKINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPKINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPKINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPKINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPKINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPKINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPKINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPKINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPKINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPKINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPKINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPKINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPKINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPKINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPKINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPKINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPKINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPKINTEN_INTKIEN_Msk        (0x1ul << USBD_EPKINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPKINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPKINTEN_PINGIEN_Msk        (0x1ul << USBD_EPKINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPKINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPKINTEN_NAKIEN_Msk         (0x1ul << USBD_EPKINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPKINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPKINTEN_STALLIEN_Msk       (0x1ul << USBD_EPKINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPKINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPKINTEN_NYETIEN_Msk        (0x1ul << USBD_EPKINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPKINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPKINTEN_ERRIEN_Msk         (0x1ul << USBD_EPKINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPKINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPKINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPKINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPKDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPKDATCNT_DATCNT_Msk        (0xfffful << USBD_EPKDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPKDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPKDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPKDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPKRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPKRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPKRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPKRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPKRSPCTL_MODE_Msk          (0x3ul << USBD_EPKRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPKRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPKRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPKRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPKRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPKRSPCTL_HALT_Msk          (0x1ul << USBD_EPKRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPKRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPKRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPKRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPKRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPKRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPKRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPKRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPKRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPKRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPKMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPKMPS_EPMPS_Msk            (0x7fful << USBD_EPKMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPKTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPKTXCNT_TXCNT_Msk          (0x7fful << USBD_EPKTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPKCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPKCFG_EPEN_Msk             (0x1ul << USBD_EPKCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPKCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPKCFG_EPTYPE_Msk           (0x3ul << USBD_EPKCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPKCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPKCFG_EPDIR_Msk            (0x1ul << USBD_EPKCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPKCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPKCFG_EPNUM_Msk            (0xful << USBD_EPKCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPKBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPKBUFSTART_SADDR_Msk       (0xffful << USBD_EPKBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPKBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPKBUFEND_EADDR_Msk         (0xffful << USBD_EPKBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_EPLDAT_EPDAT_Pos            (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPLDAT_EPDAT_Msk            (0xfffffffful << USBD_EPLDAT_EPDAT_Pos)           /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPLINTSTS_BUFFULLIF_Pos     (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPLINTSTS_BUFFULLIF_Msk     (0x1ul << USBD_EPLINTSTS_BUFFULLIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPLINTSTS_BUFEMPTYIF_Pos    (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPLINTSTS_BUFEMPTYIF_Msk    (0x1ul << USBD_EPLINTSTS_BUFEMPTYIF_Pos)          /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPLINTSTS_SHORTTXIF_Pos     (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPLINTSTS_SHORTTXIF_Msk     (0x1ul << USBD_EPLINTSTS_SHORTTXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPLINTSTS_TXPKIF_Pos        (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPLINTSTS_TXPKIF_Msk        (0x1ul << USBD_EPLINTSTS_TXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPLINTSTS_RXPKIF_Pos        (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPLINTSTS_RXPKIF_Msk        (0x1ul << USBD_EPLINTSTS_RXPKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPLINTSTS_OUTTKIF_Pos       (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPLINTSTS_OUTTKIF_Msk       (0x1ul << USBD_EPLINTSTS_OUTTKIF_Pos)             /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPLINTSTS_INTKIF_Pos        (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPLINTSTS_INTKIF_Msk        (0x1ul << USBD_EPLINTSTS_INTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPLINTSTS_PINGIF_Pos        (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPLINTSTS_PINGIF_Msk        (0x1ul << USBD_EPLINTSTS_PINGIF_Pos)              /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPLINTSTS_NAKIF_Pos         (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPLINTSTS_NAKIF_Msk         (0x1ul << USBD_EPLINTSTS_NAKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPLINTSTS_STALLIF_Pos       (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPLINTSTS_STALLIF_Msk       (0x1ul << USBD_EPLINTSTS_STALLIF_Pos)             /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPLINTSTS_NYETIF_Pos        (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPLINTSTS_NYETIF_Msk        (0x1ul << USBD_EPLINTSTS_NYETIF_Pos)              /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPLINTSTS_ERRIF_Pos         (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPLINTSTS_ERRIF_Msk         (0x1ul << USBD_EPLINTSTS_ERRIF_Pos)               /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPLINTSTS_SHORTRXIF_Pos     (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPLINTSTS_SHORTRXIF_Msk     (0x1ul << USBD_EPLINTSTS_SHORTRXIF_Pos)           /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPLINTEN_BUFFULLIEN_Pos     (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPLINTEN_BUFFULLIEN_Msk     (0x1ul << USBD_EPLINTEN_BUFFULLIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPLINTEN_BUFEMPTYIEN_Pos    (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPLINTEN_BUFEMPTYIEN_Msk    (0x1ul << USBD_EPLINTEN_BUFEMPTYIEN_Pos)          /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPLINTEN_SHORTTXIEN_Pos     (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPLINTEN_SHORTTXIEN_Msk     (0x1ul << USBD_EPLINTEN_SHORTTXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPLINTEN_TXPKIEN_Pos        (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPLINTEN_TXPKIEN_Msk        (0x1ul << USBD_EPLINTEN_TXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPLINTEN_RXPKIEN_Pos        (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPLINTEN_RXPKIEN_Msk        (0x1ul << USBD_EPLINTEN_RXPKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPLINTEN_OUTTKIEN_Pos       (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPLINTEN_OUTTKIEN_Msk       (0x1ul << USBD_EPLINTEN_OUTTKIEN_Pos)             /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPLINTEN_INTKIEN_Pos        (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPLINTEN_INTKIEN_Msk        (0x1ul << USBD_EPLINTEN_INTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPLINTEN_PINGIEN_Pos        (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPLINTEN_PINGIEN_Msk        (0x1ul << USBD_EPLINTEN_PINGIEN_Pos)              /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPLINTEN_NAKIEN_Pos         (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPLINTEN_NAKIEN_Msk         (0x1ul << USBD_EPLINTEN_NAKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPLINTEN_STALLIEN_Pos       (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPLINTEN_STALLIEN_Msk       (0x1ul << USBD_EPLINTEN_STALLIEN_Pos)             /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPLINTEN_NYETIEN_Pos        (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPLINTEN_NYETIEN_Msk        (0x1ul << USBD_EPLINTEN_NYETIEN_Pos)              /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPLINTEN_ERRIEN_Pos         (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPLINTEN_ERRIEN_Msk         (0x1ul << USBD_EPLINTEN_ERRIEN_Pos)               /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPLINTEN_SHORTRXIEN_Pos     (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPLINTEN_SHORTRXIEN_Msk     (0x1ul << USBD_EPLINTEN_SHORTRXIEN_Pos)           /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPLDATCNT_DATCNT_Pos        (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPLDATCNT_DATCNT_Msk        (0xfffful << USBD_EPLDATCNT_DATCNT_Pos)           /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPLDATCNT_DMALOOP_Pos       (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPLDATCNT_DMALOOP_Msk       (0x7ffful << USBD_EPLDATCNT_DMALOOP_Pos)          /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPLRSPCTL_FLUSH_Pos         (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPLRSPCTL_FLUSH_Msk         (0x1ul << USBD_EPLRSPCTL_FLUSH_Pos)               /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPLRSPCTL_MODE_Pos          (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPLRSPCTL_MODE_Msk          (0x3ul << USBD_EPLRSPCTL_MODE_Pos)                /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPLRSPCTL_TOGGLE_Pos        (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPLRSPCTL_TOGGLE_Msk        (0x1ul << USBD_EPLRSPCTL_TOGGLE_Pos)              /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPLRSPCTL_HALT_Pos          (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPLRSPCTL_HALT_Msk          (0x1ul << USBD_EPLRSPCTL_HALT_Pos)                /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPLRSPCTL_ZEROLEN_Pos       (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPLRSPCTL_ZEROLEN_Msk       (0x1ul << USBD_EPLRSPCTL_ZEROLEN_Pos)             /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPLRSPCTL_SHORTTXEN_Pos     (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPLRSPCTL_SHORTTXEN_Msk     (0x1ul << USBD_EPLRSPCTL_SHORTTXEN_Pos)           /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPLRSPCTL_DISBUF_Pos        (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPLRSPCTL_DISBUF_Msk        (0x1ul << USBD_EPLRSPCTL_DISBUF_Pos)              /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPLMPS_EPMPS_Pos            (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPLMPS_EPMPS_Msk            (0x7fful << USBD_EPLMPS_EPMPS_Pos)                /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPLTXCNT_TXCNT_Pos          (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPLTXCNT_TXCNT_Msk          (0x7fful << USBD_EPLTXCNT_TXCNT_Pos)              /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPLCFG_EPEN_Pos             (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPLCFG_EPEN_Msk             (0x1ul << USBD_EPLCFG_EPEN_Pos)                   /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPLCFG_EPTYPE_Pos           (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPLCFG_EPTYPE_Msk           (0x3ul << USBD_EPLCFG_EPTYPE_Pos)                 /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPLCFG_EPDIR_Pos            (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPLCFG_EPDIR_Msk            (0x1ul << USBD_EPLCFG_EPDIR_Pos)                  /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPLCFG_EPNUM_Pos            (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPLCFG_EPNUM_Msk            (0xful << USBD_EPLCFG_EPNUM_Pos)                  /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPLBUFSTART_SADDR_Pos       (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPLBUFSTART_SADDR_Msk       (0xffful << USBD_EPLBUFSTART_SADDR_Pos)           /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPLBUFEND_EADDR_Pos         (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPLBUFEND_EADDR_Msk         (0xffful << USBD_EPLBUFEND_EADDR_Pos)             /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */

#define USBD_UVCHDAT0_DAT_Pos            (0)                                               /*!< USBD_T::UVCHDAT0: DAT Position            */
#define USBD_UVCHDAT0_DAT_Msk            (0xfffffffful << USBD_UVCHDAT0_DAT_Pos)           /*!< USBD_T::UVCHDAT0: DAT Mask                */

#define USBD_UVCHDAT1_DAT_Pos            (0)                                               /*!< USBD_T::UVCHDAT1: DAT Position            */
#define USBD_UVCHDAT1_DAT_Msk            (0xfffffffful << USBD_UVCHDAT1_DAT_Pos)           /*!< USBD_T::UVCHDAT1: DAT Mask                */

#define USBD_UVCHDAT2_DAT_Pos            (0)                                               /*!< USBD_T::UVCHDAT2: DAT Position            */
#define USBD_UVCHDAT2_DAT_Msk            (0xfffffffful << USBD_UVCHDAT2_DAT_Pos)           /*!< USBD_T::UVCHDAT2: DAT Mask                */

#define USBD_UVCEPAHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPAHCNT: CNT Position          */
#define USBD_UVCEPAHCNT_CNT_Msk          (0xful << USBD_UVCEPAHCNT_CNT_Pos)                /*!< USBD_T::UVCEPAHCNT: CNT Mask              */

#define USBD_UVCEPBHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPBHCNT: CNT Position          */
#define USBD_UVCEPBHCNT_CNT_Msk          (0xful << USBD_UVCEPBHCNT_CNT_Pos)                /*!< USBD_T::UVCEPBHCNT: CNT Mask              */

#define USBD_UVCEPCHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPCHCNT: CNT Position          */
#define USBD_UVCEPCHCNT_CNT_Msk          (0xful << USBD_UVCEPCHCNT_CNT_Pos)                /*!< USBD_T::UVCEPCHCNT: CNT Mask              */

#define USBD_UVCEPDHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPDHCNT: CNT Position          */
#define USBD_UVCEPDHCNT_CNT_Msk          (0xful << USBD_UVCEPDHCNT_CNT_Pos)                /*!< USBD_T::UVCEPDHCNT: CNT Mask              */

#define USBD_UVCEPEHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPEHCNT: CNT Position          */
#define USBD_UVCEPEHCNT_CNT_Msk          (0xful << USBD_UVCEPEHCNT_CNT_Pos)                /*!< USBD_T::UVCEPEHCNT: CNT Mask              */

#define USBD_UVCEPFHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPFHCNT: CNT Position          */
#define USBD_UVCEPFHCNT_CNT_Msk          (0xful << USBD_UVCEPFHCNT_CNT_Pos)                /*!< USBD_T::UVCEPFHCNT: CNT Mask              */

#define USBD_UVCEPGHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPGHCNT: CNT Position          */
#define USBD_UVCEPGHCNT_CNT_Msk          (0xful << USBD_UVCEPGHCNT_CNT_Pos)                /*!< USBD_T::UVCEPGHCNT: CNT Mask              */

#define USBD_UVCEPHHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPHHCNT: CNT Position          */
#define USBD_UVCEPHHCNT_CNT_Msk          (0xful << USBD_UVCEPHHCNT_CNT_Pos)                /*!< USBD_T::UVCEPHHCNT: CNT Mask              */

#define USBD_UVCEPIHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPIHCNT: CNT Position          */
#define USBD_UVCEPIHCNT_CNT_Msk          (0xful << USBD_UVCEPIHCNT_CNT_Pos)                /*!< USBD_T::UVCEPIHCNT: CNT Mask              */

#define USBD_UVCEPJHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPJHCNT: CNT Position          */
#define USBD_UVCEPJHCNT_CNT_Msk          (0xful << USBD_UVCEPJHCNT_CNT_Pos)                /*!< USBD_T::UVCEPJHCNT: CNT Mask              */

#define USBD_UVCEPKHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPKHCNT: CNT Position          */
#define USBD_UVCEPKHCNT_CNT_Msk          (0xful << USBD_UVCEPKHCNT_CNT_Pos)                /*!< USBD_T::UVCEPKHCNT: CNT Mask              */

#define USBD_UVCEPLHCNT_CNT_Pos          (0)                                               /*!< USBD_T::UVCEPLHCNT: CNT Position          */
#define USBD_UVCEPLHCNT_CNT_Msk          (0xful << USBD_UVCEPLHCNT_CNT_Pos)                /*!< USBD_T::UVCEPLHCNT: CNT Mask              */

#define USBD_DMAADDR_DMAADDR_Pos         (0)                                               /*!< USBD_T::DMAADDR: DMAADDR Position         */
#define USBD_DMAADDR_DMAADDR_Msk         (0xfffffffful << USBD_DMAADDR_DMAADDR_Pos)        /*!< USBD_T::DMAADDR: DMAADDR Mask             */

#define USBD_PHYCTL_LOWPWREN_Pos         (3)                                               /*!< USBD_T::PHYCTL: LOWPWREN Position         */
#define USBD_PHYCTL_LOWPWREN_Msk         (0x1ul << USBD_PHYCTL_LOWPWREN_Pos)               /*!< USBD_T::PHYCTL: LOWPWREN Mask             */

#define USBD_PHYCTL_DPPUEN_Pos           (8)                                               /*!< USBD_T::PHYCTL: DPPUEN Position           */
#define USBD_PHYCTL_DPPUEN_Msk           (0x1ul << USBD_PHYCTL_DPPUEN_Pos)                 /*!< USBD_T::PHYCTL: DPPUEN Mask               */

#define USBD_PHYCTL_PHYEN_Pos            (9)                                               /*!< USBD_T::PHYCTL: PHYEN Position            */
#define USBD_PHYCTL_PHYEN_Msk            (0x1ul << USBD_PHYCTL_PHYEN_Pos)                  /*!< USBD_T::PHYCTL: PHYEN Mask                */

#define USBD_PHYCTL_WKEN_Pos             (24)                                              /*!< USBD_T::PHYCTL: WKEN Position             */
#define USBD_PHYCTL_WKEN_Msk             (0x1ul << USBD_PHYCTL_WKEN_Pos)                   /*!< USBD_T::PHYCTL: WKEN Mask                 */

#define USBD_PHYCTL_VBUSDET_Pos          (31)                                              /*!< USBD_T::PHYCTL: VBUSDET Position          */
#define USBD_PHYCTL_VBUSDET_Msk          (0x1ul << USBD_PHYCTL_VBUSDET_Pos)                /*!< USBD_T::PHYCTL: VBUSDET Mask              */

#define USBD_EPDAT_EPDAT_Pos             (0)                                               /*!< USBD_EP_T::EPDAT: EPDAT Position            */
#define USBD_EPDAT_EPDAT_Msk             (0xfffffffful << USBD_EPDAT_EPDAT_Pos)            /*!< USBD_EP_T::EPDAT: EPDAT Mask                */

#define USBD_EPINTSTS_BUFFULLIF_Pos      (0)                                               /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Position     */
#define USBD_EPINTSTS_BUFFULLIF_Msk      (0x1ul << USBD_EPINTSTS_BUFFULLIF_Pos)            /*!< USBD_EP_T::EPINTSTS: BUFFULLIF Mask         */

#define USBD_EPINTSTS_BUFEMPTYIF_Pos     (1)                                               /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Position    */
#define USBD_EPINTSTS_BUFEMPTYIF_Msk     (0x1ul << USBD_EPINTSTS_BUFEMPTYIF_Pos)           /*!< USBD_EP_T::EPINTSTS: BUFEMPTYIF Mask        */

#define USBD_EPINTSTS_SHORTTXIF_Pos      (2)                                               /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Position     */
#define USBD_EPINTSTS_SHORTTXIF_Msk      (0x1ul << USBD_EPINTSTS_SHORTTXIF_Pos)            /*!< USBD_EP_T::EPINTSTS: SHORTTXIF Mask         */

#define USBD_EPINTSTS_TXPKIF_Pos         (3)                                               /*!< USBD_EP_T::EPINTSTS: TXPKIF Position        */
#define USBD_EPINTSTS_TXPKIF_Msk         (0x1ul << USBD_EPINTSTS_TXPKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: TXPKIF Mask            */

#define USBD_EPINTSTS_RXPKIF_Pos         (4)                                               /*!< USBD_EP_T::EPINTSTS: RXPKIF Position        */
#define USBD_EPINTSTS_RXPKIF_Msk         (0x1ul << USBD_EPINTSTS_RXPKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: RXPKIF Mask            */

#define USBD_EPINTSTS_OUTTKIF_Pos        (5)                                               /*!< USBD_EP_T::EPINTSTS: OUTTKIF Position       */
#define USBD_EPINTSTS_OUTTKIF_Msk        (0x1ul << USBD_EPINTSTS_OUTTKIF_Pos)              /*!< USBD_EP_T::EPINTSTS: OUTTKIF Mask           */

#define USBD_EPINTSTS_INTKIF_Pos         (6)                                               /*!< USBD_EP_T::EPINTSTS: INTKIF Position        */
#define USBD_EPINTSTS_INTKIF_Msk         (0x1ul << USBD_EPINTSTS_INTKIF_Pos)               /*!< USBD_EP_T::EPINTSTS: INTKIF Mask            */

#define USBD_EPINTSTS_PINGIF_Pos         (7)                                               /*!< USBD_EP_T::EPINTSTS: PINGIF Position        */
#define USBD_EPINTSTS_PINGIF_Msk         (0x1ul << USBD_EPINTSTS_PINGIF_Pos)               /*!< USBD_EP_T::EPINTSTS: PINGIF Mask            */

#define USBD_EPINTSTS_NAKIF_Pos          (8)                                               /*!< USBD_EP_T::EPINTSTS: NAKIF Position         */
#define USBD_EPINTSTS_NAKIF_Msk          (0x1ul << USBD_EPINTSTS_NAKIF_Pos)                /*!< USBD_EP_T::EPINTSTS: NAKIF Mask             */

#define USBD_EPINTSTS_STALLIF_Pos        (9)                                               /*!< USBD_EP_T::EPINTSTS: STALLIF Position       */
#define USBD_EPINTSTS_STALLIF_Msk        (0x1ul << USBD_EPINTSTS_STALLIF_Pos)              /*!< USBD_EP_T::EPINTSTS: STALLIF Mask           */

#define USBD_EPINTSTS_NYETIF_Pos         (10)                                              /*!< USBD_EP_T::EPINTSTS: NYETIF Position        */
#define USBD_EPINTSTS_NYETIF_Msk         (0x1ul << USBD_EPINTSTS_NYETIF_Pos)               /*!< USBD_EP_T::EPINTSTS: NYETIF Mask            */

#define USBD_EPINTSTS_ERRIF_Pos          (11)                                              /*!< USBD_EP_T::EPINTSTS: ERRIF Position         */
#define USBD_EPINTSTS_ERRIF_Msk          (0x1ul << USBD_EPINTSTS_ERRIF_Pos)                /*!< USBD_EP_T::EPINTSTS: ERRIF Mask             */

#define USBD_EPINTSTS_SHORTRXIF_Pos      (12)                                              /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Position     */
#define USBD_EPINTSTS_SHORTRXIF_Msk      (0x1ul << USBD_EPINTSTS_SHORTRXIF_Pos)            /*!< USBD_EP_T::EPINTSTS: SHORTRXIF Mask         */

#define USBD_EPINTEN_BUFFULLIEN_Pos      (0)                                               /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Position     */
#define USBD_EPINTEN_BUFFULLIEN_Msk      (0x1ul << USBD_EPINTEN_BUFFULLIEN_Pos)            /*!< USBD_EP_T::EPINTEN: BUFFULLIEN Mask         */

#define USBD_EPINTEN_BUFEMPTYIEN_Pos     (1)                                               /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Position    */
#define USBD_EPINTEN_BUFEMPTYIEN_Msk     (0x1ul << USBD_EPINTEN_BUFEMPTYIEN_Pos)           /*!< USBD_EP_T::EPINTEN: BUFEMPTYIEN Mask        */

#define USBD_EPINTEN_SHORTTXIEN_Pos      (2)                                               /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Position     */
#define USBD_EPINTEN_SHORTTXIEN_Msk      (0x1ul << USBD_EPINTEN_SHORTTXIEN_Pos)            /*!< USBD_EP_T::EPINTEN: SHORTTXIEN Mask         */

#define USBD_EPINTEN_TXPKIEN_Pos         (3)                                               /*!< USBD_EP_T::EPINTEN: TXPKIEN Position        */
#define USBD_EPINTEN_TXPKIEN_Msk         (0x1ul << USBD_EPINTEN_TXPKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: TXPKIEN Mask            */

#define USBD_EPINTEN_RXPKIEN_Pos         (4)                                               /*!< USBD_EP_T::EPINTEN: RXPKIEN Position        */
#define USBD_EPINTEN_RXPKIEN_Msk         (0x1ul << USBD_EPINTEN_RXPKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: RXPKIEN Mask            */

#define USBD_EPINTEN_OUTTKIEN_Pos        (5)                                               /*!< USBD_EP_T::EPINTEN: OUTTKIEN Position       */
#define USBD_EPINTEN_OUTTKIEN_Msk        (0x1ul << USBD_EPINTEN_OUTTKIEN_Pos)              /*!< USBD_EP_T::EPINTEN: OUTTKIEN Mask           */

#define USBD_EPINTEN_INTKIEN_Pos         (6)                                               /*!< USBD_EP_T::EPINTEN: INTKIEN Position        */
#define USBD_EPINTEN_INTKIEN_Msk         (0x1ul << USBD_EPINTEN_INTKIEN_Pos)               /*!< USBD_EP_T::EPINTEN: INTKIEN Mask            */

#define USBD_EPINTEN_PINGIEN_Pos         (7)                                               /*!< USBD_EP_T::EPINTEN: PINGIEN Position        */
#define USBD_EPINTEN_PINGIEN_Msk         (0x1ul << USBD_EPINTEN_PINGIEN_Pos)               /*!< USBD_EP_T::EPINTEN: PINGIEN Mask            */

#define USBD_EPINTEN_NAKIEN_Pos          (8)                                               /*!< USBD_EP_T::EPINTEN: NAKIEN Position         */
#define USBD_EPINTEN_NAKIEN_Msk          (0x1ul << USBD_EPINTEN_NAKIEN_Pos)                /*!< USBD_EP_T::EPINTEN: NAKIEN Mask             */

#define USBD_EPINTEN_STALLIEN_Pos        (9)                                               /*!< USBD_EP_T::EPINTEN: STALLIEN Position       */
#define USBD_EPINTEN_STALLIEN_Msk        (0x1ul << USBD_EPINTEN_STALLIEN_Pos)              /*!< USBD_EP_T::EPINTEN: STALLIEN Mask           */

#define USBD_EPINTEN_NYETIEN_Pos         (10)                                              /*!< USBD_EP_T::EPINTEN: NYETIEN Position        */
#define USBD_EPINTEN_NYETIEN_Msk         (0x1ul << USBD_EPINTEN_NYETIEN_Pos)               /*!< USBD_EP_T::EPINTEN: NYETIEN Mask            */

#define USBD_EPINTEN_ERRIEN_Pos          (11)                                              /*!< USBD_EP_T::EPINTEN: ERRIEN Position         */
#define USBD_EPINTEN_ERRIEN_Msk          (0x1ul << USBD_EPINTEN_ERRIEN_Pos)                /*!< USBD_EP_T::EPINTEN: ERRIEN Mask             */

#define USBD_EPINTEN_SHORTRXIEN_Pos      (12)                                              /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Position     */
#define USBD_EPINTEN_SHORTRXIEN_Msk      (0x1ul << USBD_EPINTEN_SHORTRXIEN_Pos)            /*!< USBD_EP_T::EPINTEN: SHORTRXIEN Mask         */

#define USBD_EPDATCNT_DATCNT_Pos         (0)                                               /*!< USBD_EP_T::EPDATCNT: DATCNT Position        */
#define USBD_EPDATCNT_DATCNT_Msk         (0xfffful << USBD_EPDATCNT_DATCNT_Pos)            /*!< USBD_EP_T::EPDATCNT: DATCNT Mask            */

#define USBD_EPDATCNT_DMALOOP_Pos        (16)                                              /*!< USBD_EP_T::EPDATCNT: DMALOOP Position       */
#define USBD_EPDATCNT_DMALOOP_Msk        (0x7ffful << USBD_EPDATCNT_DMALOOP_Pos)           /*!< USBD_EP_T::EPDATCNT: DMALOOP Mask           */

#define USBD_EPRSPCTL_FLUSH_Pos          (0)                                               /*!< USBD_EP_T::EPRSPCTL: FLUSH Position         */
#define USBD_EPRSPCTL_FLUSH_Msk          (0x1ul << USBD_EPRSPCTL_FLUSH_Pos)                /*!< USBD_EP_T::EPRSPCTL: FLUSH Mask             */

#define USBD_EPRSPCTL_MODE_Pos           (1)                                               /*!< USBD_EP_T::EPRSPCTL: MODE Position          */
#define USBD_EPRSPCTL_MODE_Msk           (0x3ul << USBD_EPRSPCTL_MODE_Pos)                 /*!< USBD_EP_T::EPRSPCTL: MODE Mask              */

#define USBD_EPRSPCTL_TOGGLE_Pos         (3)                                               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Position        */
#define USBD_EPRSPCTL_TOGGLE_Msk         (0x1ul << USBD_EPRSPCTL_TOGGLE_Pos)               /*!< USBD_EP_T::EPRSPCTL: TOGGLE Mask            */

#define USBD_EPRSPCTL_HALT_Pos           (4)                                               /*!< USBD_EP_T::EPRSPCTL: HALT Position          */
#define USBD_EPRSPCTL_HALT_Msk           (0x1ul << USBD_EPRSPCTL_HALT_Pos)                 /*!< USBD_EP_T::EPRSPCTL: HALT Mask              */

#define USBD_EPRSPCTL_ZEROLEN_Pos        (5)                                               /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Position       */
#define USBD_EPRSPCTL_ZEROLEN_Msk        (0x1ul << USBD_EPRSPCTL_ZEROLEN_Pos)              /*!< USBD_EP_T::EPRSPCTL: ZEROLEN Mask           */

#define USBD_EPRSPCTL_SHORTTXEN_Pos      (6)                                               /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Position     */
#define USBD_EPRSPCTL_SHORTTXEN_Msk      (0x1ul << USBD_EPRSPCTL_SHORTTXEN_Pos)            /*!< USBD_EP_T::EPRSPCTL: SHORTTXEN Mask         */

#define USBD_EPRSPCTL_DISBUF_Pos         (7)                                               /*!< USBD_EP_T::EPRSPCTL: DISBUF Position        */
#define USBD_EPRSPCTL_DISBUF_Msk         (0x1ul << USBD_EPRSPCTL_DISBUF_Pos)               /*!< USBD_EP_T::EPRSPCTL: DISBUF Mask            */

#define USBD_EPMPS_EPMPS_Pos             (0)                                               /*!< USBD_EP_T::EPMPS: EPMPS Position            */
#define USBD_EPMPS_EPMPS_Msk             (0x7fful << USBD_EPMPS_EPMPS_Pos)                 /*!< USBD_EP_T::EPMPS: EPMPS Mask                */

#define USBD_EPTXCNT_TXCNT_Pos           (0)                                               /*!< USBD_EP_T::EPTXCNT: TXCNT Position          */
#define USBD_EPTXCNT_TXCNT_Msk           (0x7fful << USBD_EPTXCNT_TXCNT_Pos)               /*!< USBD_EP_T::EPTXCNT: TXCNT Mask              */

#define USBD_EPCFG_EPEN_Pos              (0)                                               /*!< USBD_EP_T::EPCFG: EPEN Position             */
#define USBD_EPCFG_EPEN_Msk              (0x1ul << USBD_EPCFG_EPEN_Pos)                    /*!< USBD_EP_T::EPCFG: EPEN Mask                 */

#define USBD_EPCFG_EPTYPE_Pos            (1)                                               /*!< USBD_EP_T::EPCFG: EPTYPE Position           */
#define USBD_EPCFG_EPTYPE_Msk            (0x3ul << USBD_EPCFG_EPTYPE_Pos)                  /*!< USBD_EP_T::EPCFG: EPTYPE Mask               */

#define USBD_EPCFG_EPDIR_Pos             (3)                                               /*!< USBD_EP_T::EPCFG: EPDIR Position            */
#define USBD_EPCFG_EPDIR_Msk             (0x1ul << USBD_EPCFG_EPDIR_Pos)                   /*!< USBD_EP_T::EPCFG: EPDIR Mask                */

#define USBD_EPCFG_EPNUM_Pos             (4)                                               /*!< USBD_EP_T::EPCFG: EPNUM Position            */
#define USBD_EPCFG_EPNUM_Msk             (0xful << USBD_EPCFG_EPNUM_Pos)                   /*!< USBD_EP_T::EPCFG: EPNUM Mask                */

#define USBD_EPBUFSTART_SADDR_Pos        (0)                                               /*!< USBD_EP_T::EPBUFSTART: SADDR Position       */
#define USBD_EPBUFSTART_SADDR_Msk        (0xffful << USBD_EPBUFSTART_SADDR_Pos)            /*!< USBD_EP_T::EPBUFSTART: SADDR Mask           */

#define USBD_EPBUFEND_EADDR_Pos          (0)                                               /*!< USBD_EP_T::EPBUFEND: EADDR Position         */
#define USBD_EPBUFEND_EADDR_Msk          (0xffful << USBD_EPBUFEND_EADDR_Pos)              /*!< USBD_EP_T::EPBUFEND: EADDR Mask             */


/**@}*/ /* USBD_CONST */
/**@}*/ /* end of USBD register group */


/*---------------------- USB Host Controller -------------------------*/
/**
    @addtogroup USBH USB Host Controller(USBH)
    Memory Mapped Structure for USBH Controller
@{ */

typedef struct
{


    /**
     * HcRevision
     * ===================================================================================================
     * Offset: 0x00  Host Controller Revision Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[7:0]   |REV       |Revision
     * |        |          |Indicates the Open HCI Specification revision number implemented by the Hardware.
     * |        |          |Host Controller supports 1.1 specificatio.
     * |        |          |(X.Y = XYh).
    */
    __I  uint32_t HcRevision;

    /**
     * HcControl
     * ===================================================================================================
     * Offset: 0x04  Host Controller Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |CBSR      |Control Bulk Service Ratio
     * |        |          |This specifies the service ratio between Control and Bulk EDs.
     * |        |          |Before processing any of the non-periodic lists, HC must compare the ratio specified with its internal count on how many nonempty Control EDs have been processed, in determining whether to continue serving another Control ED or switching to Bulk EDs.
     * |        |          |The internal count will be retained when crossing the frame boundary.
     * |        |          |In case of reset, HCD is responsible for restoring this.
     * |        |          |Value.
     * |        |          |00 = Number of Control EDs over Bulk EDs served is 1:1.
     * |        |          |01 = Number of Control EDs over Bulk EDs served is 2:1.
     * |        |          |10 = Number of Control EDs over Bulk EDs served is 3:1.
     * |        |          |11 = Number of Control EDs over Bulk EDs served is 4:1.
     * |[2]     |PLE       |Periodic List Enable Bit
     * |        |          |When set, this bit enables processing of the Periodic (interrupt and isochronous) list.
     * |        |          |The Host Controller checks this bit prior to attempting any periodic transfers in a fram.
     * |        |          |0 = Processing of the Periodic (Interrupt and Isochronous) list after next SOF (Start-Of-Frame) Disabled.
     * |        |          |1 = Processing of the Periodic (Interrupt and Isochronous) list in the next frame Enabled.
     * |        |          |To enable the processing of the Isochronous list, user has to set both PLE and IE (HcControl[3]) high.
     * |[3]     |IE        |Isochronous Enable Bit
     * |        |          |Both ISOEn and PLE (HcControl[2]) high enables Host Controller to process the Isochronous list.
     * |        |          |Either ISOEn or PLE (HcControl[2]) is low disables Host Controller to process the Isochronous lis.
     * |        |          |0 = Processing of the Isochronous list after next SOF (Start-Of-Frame) Disabled.
     * |        |          |1 = Processing of the Isochronous list in the next frame Enabled,