/**************************************************************************//**
 * @file     tamper.h
 * @version  V3.00
 * @brief    M2354 series TAMPER driver header file
 *
 * @note
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
 *****************************************************************************/
#ifndef __TAMPER_H__
#define __TAMPER_H__

#ifdef __cplusplus
extern "C"
{
#endif


/** @addtogroup Standard_Driver Standard Driver
  @{
*/

/** @addtogroup TAMPER_Driver Tamper Driver
  @{
*/

/** @addtogroup TAMPER_EXPORTED_CONSTANTS Tamper Exported Constants
  @{
*/

#define TAMPER_TAMPER0_SELECT      (0x1UL << 0)    /*!< Select Tamper 0 */
#define TAMPER_TAMPER1_SELECT      (0x1UL << 1)    /*!< Select Tamper 1 */
#define TAMPER_TAMPER2_SELECT      (0x1UL << 2)    /*!< Select Tamper 2 */
#define TAMPER_TAMPER3_SELECT      (0x1UL << 3)    /*!< Select Tamper 3 */
#define TAMPER_TAMPER4_SELECT      (0x1UL << 4)    /*!< Select Tamper 4 */
#define TAMPER_TAMPER5_SELECT      (0x1UL << 5)    /*!< Select Tamper 5 */
#define TAMPER_MAX_TAMPER_PIN_NUM  6UL             /*!< Tamper Pin number */

#define TAMPER_TAMPER_HIGH_LEVEL_DETECT    1UL     /*!< Tamper pin detect voltage level is high */
#define TAMPER_TAMPER_LOW_LEVEL_DETECT     0UL     /*!< Tamper pin detect voltage level is low */

#define TAMPER_TAMPER_DEBOUNCE_ENABLE      1UL     /*!< Enable tamper pin de-bounce function */
#define TAMPER_TAMPER_DEBOUNCE_DISABLE     0UL     /*!< Disable tamper pin de-bounce function */

#define TAMPER_PAIR0_SELECT        (0x1UL << 0)    /*!< Select Pair 0 */
#define TAMPER_PAIR1_SELECT        (0x1UL << 1)    /*!< Select Pair 1 */
#define TAMPER_PAIR2_SELECT        (0x1UL << 2)    /*!< Select Pair 2 */
#define TAMPER_MAX_PAIR_NUM        3UL             /*!< Pair number */

#define TAMPER_2POW6_CLK           (0x0UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 RTC clock cycles */
#define TAMPER_2POW7_CLK           (0x1UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 2 RTC clock cycles */
#define TAMPER_2POW8_CLK           (0x2UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 4 RTC clock cycles */
#define TAMPER_2POW9_CLK           (0x3UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 6 RTC clock cycles */
#define TAMPER_2POW10_CLK          (0x4UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 8 RTC clock cycles */
#define TAMPER_2POW11_CLK          (0x5UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 10 RTC clock cycles */
#define TAMPER_2POW12_CLK          (0x6UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 12 RTC clock cycles */
#define TAMPER_2POW13_CLK          (0x7UL << TAMPER_TIOCTL_DYNRATE_Pos) /*!< 64 x 14 RTC clock cycles */

#define TAMPER_ACTS_2POW10_CLK     (0x0UL << 5)    /*!< 1024 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW11_CLK     (0x1UL << 5)    /*!< 1024 x 2 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW12_CLK     (0x2UL << 5)    /*!< 1024 x 4 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW13_CLK     (0x3UL << 5)    /*!< 1024 x 6 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW14_CLK     (0x4UL << 5)    /*!< 1024 x 8 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW15_CLK     (0x5UL << 5)    /*!< 1024 x 16 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW16_CLK     (0x6UL << 5)    /*!< 1024 x 32 LIRC32K clock cycles */
#define TAMPER_ACTS_2POW17_CLK     (0x7UL << 5)    /*!< 1024 x 64 LIRC32K clock cycles */

#define TAMPER_REF_RANDOM_PATTERN        0x0UL     /*!< The new reference pattern is generated by random number generator when the reference pattern run out */
#define TAMPER_REF_SEED                  0x1UL     /*!< The new reference pattern is repeated from SEED (TAMPER_SEED[31:0]) when the reference pattern run out */

#define TAMPER_PWRG_96M_SAMPLE           0x0UL     /*!< Select power glitch 96M sampleing rate */
#define TAMPER_PWRG_192M_SAMPLE          0x1UL     /*!< Select power glitch 192M sampleing rate */

/*@}*/ /* end of group TAMPER_EXPORTED_CONSTANTS */


/** @addtogroup TAMPER_EXPORTED_FUNCTIONS Tamper Exported Functions
  @{
*/

/**
  * @brief      Reset Tamper Coreblock
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER INIT control register to reset the tamper coreblock.
  *
  */
#define TAMPER_CORE_RESET()                 ((uint32_t)(TAMPER->INIT = 0x55AA))

/**
  * @brief      Release Tamper Coreblock
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER INIT control register to release the tamper coreblock.
  *
  */
#define TAMPER_CORE_RELEASE()               ((uint32_t)(TAMPER->INIT = 0x5500))

/**
  * @brief      Get the TLDO Power Ready Status
  *
  * @param      None
  *
  * @retval     0   The power status of TLDO is not ready.
  * @retval     1   The power status of TLDO is ready.
  *
  * @details    This macro will return the power status of TLDO.
  *
  */
#define TAMPER_TLDO_IS_READY()              (TAMPER->INIT & TAMPER_INIT_TLDORDY_Msk ? 1:0)

/**
  * @brief      Enable LXT Clock Detection
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to enable LXT clock detection.
  *
  */
#define TAMPER_ENABLE_LXTDET()              ((uint32_t)(TAMPER->FUNEN = (TAMPER->FUNEN & ~0xFFUL) | 0x44))

/**
  * @brief      Disable LXT Clock Detection
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to disable LXT clock detection.
  *
  */
#define TAMPER_DISABLE_LXTDET()             ((uint32_t)(TAMPER->FUNEN = (TAMPER->FUNEN & ~0xFFUL) | 0x40))

/**
  * @brief      Enable HXT Clock Detection
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to enable HXT clock detection.
  *
  */
#define TAMPER_ENABLE_HXTDET()              ((uint32_t)(TAMPER->FUNEN = (TAMPER->FUNEN & ~0xFFUL) | 0x34))

/**
  * @brief      Disable HXT Clock Detection
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to disable HXT clock detection.
  *
  */
#define TAMPER_DISABLE_HXTDET()             ((uint32_t)(TAMPER->FUNEN = (TAMPER->FUNEN & ~0xFFUL) | 0x30))

/**
  * @brief      Tamper I/O TAMPER Block Detection Selection
  *
  * @param[in]  u32TamperSelect   Tamper pin select. Possible options are
  *                               - \ref TAMPER_TAMPER0_SELECT
  *                               - \ref TAMPER_TAMPER1_SELECT
  *                               - \ref TAMPER_TAMPER2_SELECT
  *                               - \ref TAMPER_TAMPER3_SELECT
  *                               - \ref TAMPER_TAMPER4_SELECT
  *                               - \ref TAMPER_TAMPER5_SELECT
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to select tamper I/O 0~5 and its function is detected through TAMPER block.
  *
  */
__STATIC_INLINE void TAMPER_IOSEL_TAMPER(uint32_t u32TamperSelect)
{
    uint32_t i;

    for(i = 0UL; i < (uint32_t)TAMPER_MAX_TAMPER_PIN_NUM; i++)
    {
        if(u32TamperSelect & (0x1UL << i))
        {
            TAMPER->FUNEN = (TAMPER->FUNEN & ~0xFFUL) | (0x94 + i * 0x10UL);
        }
    }
}

/**
  * @brief      Tamper I/O RTC Block Detection Selection
  *
  * @param[in]  u32TamperSelect   Tamper pin select. Possible options are
  *                               - \ref TAMPER_TAMPER0_SELECT
  *                               - \ref TAMPER_TAMPER1_SELECT
  *                               - \ref TAMPER_TAMPER2_SELECT
  *                               - \ref TAMPER_TAMPER3_SELECT
  *                               - \ref TAMPER_TAMPER4_SELECT
  *                               - \ref TAMPER_TAMPER5_SELECT
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to select tamper I/O 0~5 and its function is detected through RTC block.
  *
  */
__STATIC_INLINE void TAMPER_IOSEL_RTC(uint32_t u32TamperSelect)
{
    uint32_t i;

    for(i = 0UL; i < (uint32_t)TAMPER_MAX_TAMPER_PIN_NUM; i++)
    {
        if(u32TamperSelect & (0x1UL << i))
        {
            TAMPER->FUNEN = (TAMPER->FUNEN & ~0xFFUL) | (0x90 + i * 0x10UL);
        }
    }
}

/**
  * @brief      Enable HIRC48M
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to enable HIRC48M.
  *
  */
#define TAMPER_ENABLE_HIRC48M()             ((uint32_t)(TAMPER->FUNEN &= (~TAMPER_FUNEN_HIRC48MEN_Msk)))

/**
  * @brief      Disable HIRC48M
  *
  * @param      None
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to disable HIRC48M.
  *
  */
#define TAMPER_DISABLE_HIRC48M()            ((uint32_t)(TAMPER->FUNEN = (TAMPER->FUNEN & (~TAMPER_FUNEN_HIRC48MEN_Msk)) | (0x5A << TAMPER_FUNEN_HIRC48MEN_Pos)))

/**
  * @brief      Power Glitch Sampling Rate Selection
  *
  * @param[in]  u32PWRGSampleRate Power Glitch sampling rate select. Possible options are
  *                               - \ref TAMPER_PWRG_96M_SAMPLE
  *                               - \ref TAMPER_PWRG_192M_SAMPLE
  *
  * @return     None
  *
  * @details    To set TAMPER FUNEN control register to enable power glitch channel 0~3 to select power glitch sampling rate.
  *
  */
__STATIC_INLINE void TAMPER_PWRG_SAMPLE_SEL(uint32_t u32PWRGSampleRate)
{
    TAMPER->FUNEN &= ~0xF000000UL;

    if(u32PWRGSampleRate == TAMPER_PWRG_96M_SAMPLE)
    {
        TAMPER->FUNEN |= TAMPER_FUNEN_PWRGCHEN0_Msk | TAMPER_FUNEN_PWRGCHEN1_Msk;
    }
    else if(u32PWRGSampleRate == TAMPER_PWRG_192M_SAMPLE)
    {
        TAMPER->FUNEN |= TAMPER_FUNEN_PWRGCHEN0_Msk | TAMPER_FUNEN_PWRGCHEN1_Msk | TAMPER_FUNEN_PWRGCHEN2_Msk | TAMPER_FUNEN_PWRGCHEN3_Msk;
    }
}

/**
  * @brief      Enable to Trigger FMC
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Set FMCTRIGEN bit of TAMPER TRIEN control register to trigger FMC when Tamper event is detected.
  *
  */
#define TAMPER_ENABLE_FMC_TRIG()            ((uint32_t)(TAMPER->TRIEN |= TAMPER_TRIEN_FMCTRIGEN_Msk))

/**
  * @brief      Disable to Trigger FMC
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Clear FMCTRIGEN bit of TAMPER TRIEN control register to not trigger FMC when Tamper event is detected.
  *
  */
#define TAMPER_DISABLE_FMC_TRIG()           ((uint32_t)(TAMPER->TRIEN &= (~TAMPER_TRIEN_FMCTRIGEN_Msk)))

/**
  * @brief      Enable to Trigger Key Store
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Set KSTRIGEN bit of TAMPER TRIEN control register to trigger Key Store when Tamper event is detected.
  *
  */
#define TAMPER_ENABLE_KS_TRIG()             ((uint32_t)(TAMPER->TRIEN |= TAMPER_TRIEN_KSTRIGEN_Msk))

/**
  * @brief      Disable to Trigger Key Store
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Clear KSTRIGEN bit of TAMPER TRIEN control register to not trigger Key Store when Tamper event is detected.
  *
  */
#define TAMPER_DISABLE_KS_TRIG()            ((uint32_t)(TAMPER->TRIEN &= (~TAMPER_TRIEN_KSTRIGEN_Msk)))

/**
  * @brief      Enable Wake-up Function
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Set WAKEUPEN bit of TAMPER TRIEN control register to wake-up the system when Tamper event is detected.
  *
  */
#define TAMPER_ENABLE_WAKEUP()              ((uint32_t)(TAMPER->TRIEN |= TAMPER_TRIEN_WAKEUPEN_Msk))

/**
  * @brief      Disable Wake-up Function
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Clear WAKEUPEN bit of TAMPER TRIEN control register to not wake-up the system when Tamper event is detected.
  *
  */
#define TAMPER_DISABLE_WAKEUP()             ((uint32_t)(TAMPER->TRIEN &= (~TAMPER_TRIEN_WAKEUPEN_Msk)))

/**
  * @brief      Enable to Clear Crypto Function
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Set CRYPTOEN bit of TAMPER TRIEN control register to reset Crypto when Tamper event is detected.
  *
  */
#define TAMPER_ENABLE_CRYPTO()              ((uint32_t)(TAMPER->TRIEN |= TAMPER_TRIEN_CRYPTOEN_Msk))

/**
  * @brief      Disable to Clear Crypto Function
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Clear CRYPTOEN bit of TAMPER TRIEN control register to not reset Crypto when Tamper event is detected.
  *
  */
#define TAMPER_DISABLE_CRYPTO()             ((uint32_t)(TAMPER->TRIEN &= (~TAMPER_TRIEN_CRYPTOEN_Msk)))

/**
  * @brief      Enable to Trigger Chip Reset
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Set CHIPRSTEN bit of TAMPER TRIEN control register to reset the system when Tamper event is detected.
  *
  */
#define TAMPER_ENABLE_CHIPRST()              ((uint32_t)(TAMPER->TRIEN |= TAMPER_TRIEN_CHIPRSTEN_Msk))

/**
  * @brief      Disable to Trigger Chip Reset
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Clear CHIPRSTEN bit of TAMPER TRIEN control register to not reset the system when Tamper event is detected.
  *
  */
#define TAMPER_DISABLE_CHIPRST()             ((uint32_t)(TAMPER->TRIEN &= (~TAMPER_TRIEN_CHIPRSTEN_Msk)))

/**
  * @brief      Enable to Clear RTC Spare Register
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Set RTCSPCLREN bit of TAMPER TRIEN control register to reset RTC spare register when Tamper event is detected.
  *
  */
#define TAMPER_ENABLE_RTCSPCLR()              ((uint32_t)(TAMPER->TRIEN |= TAMPER_TRIEN_RTCSPCLREN_Msk))

/**
  * @brief      Disable to Clear RTC Spare Register
  *
  * @param      None
  *
  * @return     None
  *
  * @details    Clear RTCSPCLREN bit of TAMPER TRIEN control register to not reset RTC spare register when Tamper event is detected.
  *
  */
#define TAMPER_DISABLE_RTCSPCLR()             ((uint32_t)(TAMPER->TRIEN &= (~TAMPER_TRIEN_RTCSPCLREN_Msk)))

/**
  * @brief      Get Tamper Interrupt Flag
  *
  * @param      None
  *
  * @retval     0   Tamper event Interrupt did not occur
  * @retval     1   Tamper event Interrupt occurred
  *
  * @details    This macro indicates Tamper event intertupt occurred or not.
  *
  */
#define TAMPER_GET_INT_FLAG()      ((TAMPER->INTSTS & (0xAA7FAFFF))? 1:0)

/**
  * @brief      Clear Tamper Interrupt Status
  *
  * @param[in]  u32TamperFlag   Tamper event interrupt flag. It consists of:
  *                             - \ref TAMPER_INTSTS_TAMP0IF_Msk
  *                             - \ref TAMPER_INTSTS_TAMP1IF_Msk
  *                             - \ref TAMPER_INTSTS_TAMP2IF_Msk
  *                             - \ref TAMPER_INTSTS_TAMP3IF_Msk
  *                             - \ref TAMPER_INTSTS_TAMP4IF_Msk
  *                             - \ref TAMPER_INTSTS_TAMP5IF_Msk
  *                             - \ref TAMPER_INTSTS_CLKFAILIF_Msk
  *                             - \ref TAMPER_INTSTS_CLKSTOPIF_Msk
  *                             - \ref TAMPER_INTSTS_OVPOUTIF_Msk
  *                             - \ref TAMPER_INTSTS_PWRPEVIF_Msk
  *                             - \ref TAMPER_INTSTS_PWRNEVIF_Msk
  *                             - \ref TAMPER_INTSTS_ACTSEIF_Msk
  *                             - \ref TAMPER_INTSTS_ACTST5IF_Msk
  *                             - \ref TAMPER_INTSTS_ACTST25IF_Msk
  *                             - \ref TAMPER_INTSTS_HXTERRIF_Msk
  *                             - \ref TAMPER_INTSTS_VBATLOSSIF_Msk
  *                             - \ref TAMPER_INTSTS_SECWDTIF_Msk
  *                             - \ref TAMPER_INTSTS_TLDOBODIF_Msk
  *                             - \ref TAMPER_INTSTS_ACTST1IF_Msk
  *                             - \ref TAMPER_INTSTS_ACTST3IF_Msk
  *                             - \ref TAMPER_INTSTS_ACTST21IF_Msk
  *                             - \ref TAMPER_INTSTS_ACTST23IF_Msk
  *
  * @return     None
  *
  * @details    This macro is used to clear Tamper event flag.
  *
  */
#define TAMPER_CLR_INT_STATUS(u32TamperFlag)   (TAMPER->INTSTS = (u32TamperFlag))

/**
  * @brief      Get Tamper Interrupt Status
  *
  * @param      None
  *
  * @retval     TAMPER_INTSTS_TAMP0IF_Msk    Tamper 0 event flag is generated
  * @retval     TAMPER_INTSTS_TAMP1IF_Msk    Tamper 1 event flag is generated
  * @retval     TAMPER_INTSTS_TAMP2IF_Msk    Tamper 2 event flag is generated
  * @retval     TAMPER_INTSTS_TAMP3IF_Msk    Tamper 3 event flag is generated
  * @retval     TAMPER_INTSTS_TAMP4IF_Msk    Tamper 4 event flag is generated
  * @retval     TAMPER_INTSTS_TAMP5IF_Msk    Tamper 5 event flag is generated
  * @retval     TAMPER_INTSTS_CLKFAILIF_Msk  LXT frequency is abnormal
  * @retval     TAMPER_INTSTS_CLKSTOPIF_Msk  LXT frequency is almost stopped
  * @retval     TAMPER_INTSTS_OVPOUTIF_Msk   Core_SB over voltage detected
  * @retval     TAMPER_INTSTS_PWRPEVIF_Msk   Core Power positive glitch is detected
  * @retval     TAMPER_INTSTS_PWRNEVIF_Msk   Core Power negative glitch is detected
  * @retval     TAMPER_INTSTS_ACTSEFIF_Msk   Active shield event is detected including the VDD and GND attack
  * @retval     TAMPER_INTSTS_ACTST5IF_Msk   Active shiled Tamper 5 event flag is generated
  * @retval     TAMPER_INTSTS_ACTST25IF_Msk  2th Active shiled Tamper 5 event flag is generated
  * @retval     TAMPER_INTSTS_RTCLVRIF_Msk   RTC Power low voltage detection event is detected
  * @retval     TAMPER_INTSTS_RIOTRIGIF_Msk  There is RTC tamper I/O detection event
  * @retval     TAMPER_INTSTS_RCLKTRIGIF_Msk There is RTC clock monitor detection event
  * @retval     TAMPER_INTSTS_HXTERRIF_Msk   Core HXT fail event
  * @retval     TAMPER_INTSTS_VBATLOSSIF_Msk RTC VBAT Power loss detection event is detected
  * @retval     TAMPER_INTSTS_SECWDTIF_Msk   Security WDT event is detected
  * @retval     TAMPER_INTSTS_TLDOBODIF_Msk  Brown out is detected
  * @retval     TAMPER_INTSTS_ACTST1IF_Msk   Active shiled Tamper 1 event flag is generated
  * @retval     TAMPER_INTSTS_ACTST3IF_Msk   Active shiled Tamper 3 event flag is generated
  * @retval     TAMPER_INTSTS_ACTST21IF_Msk  2th Active shiled Tamper 1 event flag is generated
  * @retval     TAMPER_INTSTS_ACTST23IF_Msk  2th Active shiled Tamper 3 event flag is generated
  *
  * @details    This macro indicates Tamper event status.
  *
  */
#define TAMPER_GET_INT_STATUS()    ((TAMPER->INTSTS & (0xAA7FAFFF)))

void TAMPER_EnableInt(uint32_t u32IntFlagMask);
void TAMPER_DisableInt(uint32_t u32IntFlagMask);
void TAMPER_StaticTamperEnable(uint32_t u32TamperSelect, uint32_t u32DetecLevel, uint32_t u32DebounceEn);
void TAMPER_StaticTamperDisable(uint32_t u32TamperSelect);
void TAMPER_DynamicTamperEnable(uint32_t u32PairSel, uint32_t u32DebounceEn, uint32_t u32Pair1Source, uint32_t u32Pair2Source);
void TAMPER_DynamicTamperDisable(uint32_t u32PairSel);
void TAMPER_DynamicTamperConfig(uint32_t u32ChangeRate, uint32_t u32SeedReload, uint32_t u32RefPattern, uint32_t u32Seed);
void TAMPER_ActiveShieldDynamicTamperEnable(uint32_t u32PairSel1, uint32_t u32Pair1Source1, uint32_t u32PairSel2, uint32_t u32Pair1Source2);
void TAMPER_ActiveShieldDynamicTamperDisable(uint32_t u32PairSel1, uint32_t u32PairSe2);
void TAMPER_ActiveShieldDynamicTamperConfig(uint32_t u32ChangeRate1, uint32_t u32SeedReload1, uint32_t u32RefPattern1, uint32_t u32Seed,
                                            uint32_t u32ChangeRate2, uint32_t u32SeedReload2, uint32_t u32RefPattern2, uint32_t u32Seed2);


/*@}*/ /* end of group TAMPER_EXPORTED_FUNCTIONS */

/*@}*/ /* end of group TAMPER_Driver */

/*@}*/ /* end of group Standard_Driver */

#ifdef __cplusplus
}
#endif

#endif /* __TAMPER_H__ */

/*** (C) COPYRIGHT 2020 Nuvoton Technology Corp. ***/
