/**************************************************************************//**
 * @file     prng_reg.h
 * @version  V1.00
 * @brief    PRNG register definition header file
 *
 * SPDX-License-Identifier: Apache-2.0
 * @copyright (C) 2020 Nuvoton Technology Corp. All rights reserved.
 *****************************************************************************/
#ifndef __PRNG_REG_H__
#define __PRNG_REG_H__

#if defined ( __CC_ARM   )
#pragma anon_unions
#endif

/**
   @addtogroup REGISTER Control Register
   @{
*/

/**
    @addtogroup PRNG Pseudo Random Number Generator Controller(PRNG)
    Memory Mapped Structure for PRNG Controller
@{ */

typedef struct
{

    /**
     * @var PRNG_T::INTEN
     * Offset: 0x00  PRNG Interrupt Enable Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[16]    |PRNGIEN   |PRNG Interrupt Enable Bit
     * |        |          |0 = PRNG interrupt Disabled.
     * |        |          |1 = PRNG interrupt Enabled.
     * @var PRNG_T::INTSTS
     * Offset: 0x04  PRNG Interrupt Flag
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[16]    |PRNGIF    |PRNG Finish Interrupt Flag
     * |        |          |0 = No PRNG interrupt.
     * |        |          |1 = PRNG key generation done interrupt.
     * |        |          |Note: This bit is cleared by writing 1, and it has no effect by writing 0.
     * @var PRNG_T::CTL
     * Offset: 0x08  PRNG Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |START     |Start PRNG Engine
     * |        |          |0 = Stop PRNG engine.
     * |        |          |1 = Generate a new key and store the new key to the register PRNG_KEYx , which will be cleared when the new key is generated.
     * |[1]     |SEEDRLD   |Reload New Seed for PRNG Engine
     * |        |          |0 = Generating key based on the current seed.
     * |        |          |1 = Reload new seed.
     * |[3:2]   |KEYSZ     |PRNG Generate Key Size
     * |        |          |00 = 64 bits.
     * |        |          |01 = 128 bits.
     * |        |          |10 = 192 bits.
     * |        |          |11 = 256 bits.
     * |[8]     |BUSY      |PRNG Busy (Read Only)
     * |        |          |0 = PRNG engine is idle.
     * |        |          |1 = PRNG engine is generating PRNG_KEYx.
     * @var PRNG_T::SEED
     * Offset: 0x0C  Seed for PRNG
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |SEED      |Seed for PRNG (Write Only)
     * |        |          |The bits store the seed for PRNG engine.
     * @var PRNG_T::KEY[8]
     * Offset: 0x10 ~ 0x2C  PRNG Generated Key0 ~ Key7
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[31:0]  |KEY       |Store PRNG Generated Key (Read Only)
     * |        |          |The bits store the key that is generated by PRNG.
     */
    __IO uint32_t INTEN;                 /*!< [0x0000] PRNG Interrupt Enable Control Register                           */
    __IO uint32_t INTSTS;                /*!< [0x0004] PRNG Interrupt Flag                                              */
    __IO uint32_t CTL;                   /*!< [0x0008] PRNG Control Register                                            */
    __O  uint32_t SEED;                  /*!< [0x000c] Seed for PRNG                                                    */
    __I  uint32_t KEY[8];                /*!< [0x0010] ~ [0x002c] PRNG Generated Key0 ~ Key7                            */

} PRNG_T;

/**
    @addtogroup PRNG_CONST PRNG Bit Field Definition
    Constant Definitions for PRNG Controller
@{ */

#define PRNG_INTEN_PRNGIEN_Pos           (16)                                              /*!< PRNG_T::INTEN: PRNGIEN Position        */
#define PRNG_INTEN_PRNGIEN_Msk           (0x1ul << PRNG_INTEN_PRNGIEN_Pos)                 /*!< PRNG_T::INTEN: PRNGIEN Mask            */

#define PRNG_INTSTS_PRNGIF_Pos           (16)                                              /*!< PRNG_T::INTSTS: PRNGIF Position        */
#define PRNG_INTSTS_PRNGIF_Msk           (0x1ul << PRNG_INTSTS_PRNGIF_Pos)                 /*!< PRNG_T::INTSTS: PRNGIF Mask            */

#define PRNG_CTL_START_Pos               (0)                                               /*!< PRNG_T::CTL: START Position            */
#define PRNG_CTL_START_Msk               (0x1ul << PRNG_CTL_START_Pos)                     /*!< PRNG_T::CTL: START Mask                */

#define PRNG_CTL_SEEDRLD_Pos             (1)                                               /*!< PRNG_T::CTL: SEEDRLD Position          */
#define PRNG_CTL_SEEDRLD_Msk             (0x1ul << PRNG_CTL_SEEDRLD_Pos)                   /*!< PRNG_T::CTL: SEEDRLD Mask              */

#define PRNG_CTL_KEYSZ_Pos               (2)                                               /*!< PRNG_T::CTL: KEYSZ Position            */
#define PRNG_CTL_KEYSZ_Msk               (0x3ul << PRNG_CTL_KEYSZ_Pos)                     /*!< PRNG_T::CTL: KEYSZ Mask                */

#define PRNG_CTL_BUSY_Pos                (8)                                               /*!< PRNG_T::CTL: BUSY Position             */
#define PRNG_CTL_BUSY_Msk                (0x1ul << PRNG_CTL_BUSY_Pos)                      /*!< PRNG_T::CTL: BUSY Mask                 */

#define PRNG_SEED_SEED_Pos               (0)                                               /*!< PRNG_T::SEED: SEED Position            */
#define PRNG_SEED_SEED_Msk               (0xfffffffful << PRNG_SEED_SEED_Pos)              /*!< PRNG_T::SEED: SEED Mask                */

#define PRNG_KEYx_KEY_Pos                (0)                                               /*!< PRNG_T::KEY[8]: KEY Position           */
#define PRNG_KEYx_KEY_Msk                (0xfffffffful << PRNG_KEYx_KEY_Pos)               /*!< PRNG_T::KEY[8]: KEY Mask               */

/**@}*/ /* PRNG_CONST */
/**@}*/ /* end of PRNG register group */
/**@}*/ /* end of REGISTER group */

#if defined ( __CC_ARM   )
#pragma no_anon_unions
#endif

#endif /* __PRNG_REG_H__ */

