#include <stdio.h>
#include <stdarg.h>
#include <arm_cmse.h>
#include "NuMicro.h"
#include "partition_M2351.h"

#include "tee_driver.h"

#include "apps/xor/xor.h"
#include "apps/strcat/strcat.h"

#define NEXT_BOOT_BASE  FMC_NON_SECURE_BASE
#define JUMP_HERE       0xe7fee7ff      /* Instruction Code of "B ." */

/* typedef for NonSecure callback functions */
typedef __NONSECURE_CALL int32_t (*NonSecure_funcptr)(uint32_t);

/**
 * 用于跳转到下一 boot 阶段 (Non-Secure 区域代码) 的 Boot_Init() 函数
 */
void Boot_Init(uint32_t u32BootBase)
{
    NonSecure_funcptr fp;

    /* SCB_NS.VTOR points to the Non-secure vector table base address. */
    SCB_NS->VTOR = u32BootBase;

    /* 1st Entry in the vector table is the Non-secure Main Stack Pointer. */
    __TZ_set_MSP_NS(*((uint32_t *)SCB_NS->VTOR));      /* Set up MSP in Non-secure code */

    /* 2nd entry contains the address of the Reset_Handler (CMSIS-CORE) function */
    fp = ((NonSecure_funcptr)(*(((uint32_t *)SCB_NS->VTOR) + 1)));

    /* Clear the LSB of the function address to indicate the function-call
       will cause a state switch from Secure to Non-secure */
    fp = cmse_nsfptr_create(fp);

    /* Check if the Reset_Handler address is in Non-secure space */
    if(cmse_is_nsfptr(fp) && (((uint32_t)fp & 0xf0000000) == 0x10000000))
    {
        printf("Execute non-secure code ...\r\n");
        fp(0); /* Non-secure function call */
    }
    else
    {
        /* Something went wrong */
        printf("No code in non-secure region!\r\n");
        printf("CPU will halted at non-secure state\r\n");

        /* Set nonsecure MSP in nonsecure region */
        __TZ_set_MSP_NS(NON_SECURE_SRAM_BASE + 512);

        /* Try to halted in non-secure state (SRAM) */
        M32(NON_SECURE_SRAM_BASE) = JUMP_HERE;
        fp = (NonSecure_funcptr)(NON_SECURE_SRAM_BASE + 1);
        fp(0);

        while(1);
    }
}

void SYS_Init(void)
{
    //////////////// 初始化系统时钟 ////////////////

    // 使能 PLL (128MHz PLL output with HIRC)
    CLK->PLLCTL = CLK_PLLCTL_128MHz_HIRC;

    // 等待 PLL 稳定
    while ((CLK->STATUS & CLK_STATUS_PLLSTB_Msk) == 0);

    // 设置 HCLK 分频系数为 2
    CLK->CLKDIV0 = (CLK->CLKDIV0 & (~CLK_CLKDIV0_HCLKDIV_Msk)) | 1;

    // 切换 HCLK 的时钟源为 PLL
    CLK->CLKSEL0 = (CLK->CLKSEL0 & (~CLK_CLKSEL0_HCLKSEL_Msk)) | CLK_CLKSEL0_HCLKSEL_PLL;

    // 使能 UART0 时钟
    CLK->APBCLK0 |= CLK_APBCLK0_UART0CKEN_Msk;

    // 切换 UART0 时钟源为 HIRC
    CLK->CLKSEL1 = (CLK->CLKSEL1 & (~CLK_CLKSEL1_UART0SEL_Msk)) | CLK_CLKSEL1_UART0SEL_HIRC;

    //////////////// 初始化 IO Multi-function ////////////////

    // 设置 UART0 的 RXD (PB.12) 和 TXD (PB.13) 引脚的 multi-function
    SYS->GPB_MFPH = (SYS->GPB_MFPH & (~(UART0_RXD_PB12_Msk | UART0_TXD_PB13_Msk))) | UART0_RXD_PB12 | UART0_TXD_PB13;
}

/**
 * 初始化调试用 UART 端口
 */
void DEBUG_PORT_Init(void)
{
    // 配置 UART0
    DEBUG_PORT->BAUD = UART_BAUD_MODE2 | UART_BAUD_MODE2_DIVIDER(__HIRC, 115200);
    DEBUG_PORT->LINE = UART_WORD_LEN_8 | UART_PARITY_NONE | UART_STOP_BIT_1;
}

static uint32_t _ticks = 0;

void SysTick_Handler(void)
{
    _ticks++;
}

int main(void)
{
    SYS_UnlockReg();

    SYS_Init();

    DEBUG_PORT_Init();

    printf("In Secure main()\r\n");

    // 配置 Secure 的 SysTick 为每 10ms 产生一次中断
    SystemCoreClockUpdate();
    SysTick_Config(SystemCoreClock / 100);

    // 注册 TA
    teeDriverTaRegister(&g_ta_reference_xor);
    teeDriverTaRegister(&g_ta_reference_strcat);

    // 引导到下一阶段 (Non-Secure) 代码
    Boot_Init(NEXT_BOOT_BASE);

    while (1)
    {
        __WFI();
    }
}
