/**
 * 参考文档:
 * [TEE_CLIENT_API_SPEC]        TEE Client API Specification v1.0 (July 2010)
 */

#ifndef __TEE_CLIENT_API_H__
#define __TEE_CLIENT_API_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stddef.h>
#include <stdbool.h>
#include <limits.h>

extern void TEEC_Log(const char *format, ...);

/*
 * 定义在 TEEC_OpenSession() 或 TEEC_InvokeCommand() 操作中可用的 memory reference 数量
 */
#define TEEC_CONFIG_PAYLOAD_REF_COUNT 4

//////////////////////////////// 4. Specification ////////////////////////////////

//////////////// 4.2. Header File ////////////////

/**
 * The header file for the TEE Client API must have the name "tee_client_api.h"
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.2. Header File
 */

//////////////// 4.3. Data Types ////////////////

/**
 * 基础数据类型
 *
 * uint32_t:    a 32-bit unsigned integer
 * uint16_t:    a 16-bit unsigned integer
 * uint8_t:     an 8-bit unsigned integer
 * char:        a character
 * size_t:      an unsigned integer large enough to hold the size of an object in memory
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.1. Basic Types
 */

/**
 * TEEC_Result
 *
 * 该类型用于表示调用 TEE Client API 的结果返回码
 *
 * @see     API 返回码定义
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.2. TEEC_Result
 */
typedef uint32_t TEEC_Result;

/**
 * TEEC_UUID 结构体
 *
 * 该结构体包含 RFC4122 定义的 UUID (Universally Unique Resource Identifier) 类型，用于标识 TA (Trusted Application)
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.3. TEEC_UUID
 */
typedef struct {
    uint32_t timeLow;
    uint16_t timeMid;
    uint16_t timeHiAndVersion;
    uint8_t clockSeqAndNode[8];
} TEEC_UUID;

/**
 * TEEC_Context 结构体
 *
 * 存储 CA 和 TEE 之间的连接相关的上下文信息
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.4. TEEC_Context
 */
typedef struct {
    // 整体都是 implementation-defined 的

    /** 该 context 是否有效 */
    bool valid;
} TEEC_Context;

/**
 * TEEC_Session 结构体
 *
 * 表示 CA 和特定 TA 间连接的 TEE Session.
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.5. TEEC_Session
 */
typedef struct {
    // 整体都是 implementation-defined 的

    /** 指向打开该会话时所指定的 TEEC_Context 结构体的指针 (该结构体在 TEEC_InitializeContext() 中被初始化) */
    TEEC_Context *context;
    /** 打开会话时所产生的代表该会话的 session id */
    uint32_t sessionId;
} TEEC_Session;

/**
 * TEEC_SharedMemory 结构体
 *
 * 用于在 CA 和 TA 间传输数据的内存
 *
 * This type denotes a Shared Memory block which has either been registered with the Implementation or allocated by it
 *
 * A shared memory block is a region of memory allocated in the context of the
 * client application memory space that can be used to transfer data between
 * that client application and a trusted application. The user of this struct
 * is responsible to populate the buffer pointer.
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.6. TEEC_SharedMemory
 */
typedef struct {
    /** 指向与 TEE 共享的内存缓冲区的指针 */
    void *buffer;
    /** 内存缓冲区 buffer 的大小，以字节为单位 */
    size_t size;
    /** a bit-vector which can contain the following flags: TEEC_MEM_INPUT, TEEC_MEM_OUTPUT */
    uint32_t flags;

    // 以下部分为 implementation-defined 的

    /** 已分配空间大小 */
    size_t allocatedSize;
    /** 是否已为 buffer 分配空间 */
    bool bufferIsAllocated;
} TEEC_SharedMemory;

/**
 * TEEC_TempMemoryReference 结构体
 *
 * 一个临时的内存引用，仅在 operation 期间有效。
 *
 * 用于 TEEC_Operation 参数中，当对应的参数类型是 TEEC_MEMREF_TEMP_INPUT, TEEC_MEMREF_TEMP_OUTPUT,
 * 或 TEEC_MEMREF_TEMP_INOUT 时，
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.7. TEEC_TempMemoryReference
 */
typedef struct {
    /** 指向在 Operation 期间需要临时注册的内存缓冲区的指针 (可以为 NULL 来指定一个空的 Memory Reference) */
    void *buffer;
    /** buffer 所指向内存缓冲区的大小，以字节为单位 */
    size_t size;
} TEEC_TempMemoryReference;

/**
 * TEEC_RegisteredMemoryReference 结构体
 *
 * 共享内存块整体或其中一部分的引用。
 *
 * Registered Memory Reference, 即 pre-registered 或 pre-allocated 的 Shared Memory 块。
 * 用于 TEEC_Operation 参数中，当对应的参数类型是 TEEC_MEMREF_WHOLE, TEEC_MEMREF_PARTIAL_INPUT,
 * TEEC_MEMREF_PARTIAL_OUTPUT, 或 TEEC_MEMREF_PARTIAL_INOUT 时。
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.8. TEEC_RegisteredMemoryReference
 */
typedef struct {
    /** 指向一个 TEEC_SharedMemory 结构体的指针 (不能为 NULL)，所引用的内存是其整体或一部分 */
    TEEC_SharedMemory *parent;
    /** 所引用内存区域的大小，以字节为单位 */
    size_t size;
    /** 所引用内存区域在 parent 中起始位置的偏移量，以字节为单位 */
    size_t offset;
} TEEC_RegisteredMemoryReference;

/**
 * TEEC_Value 结构体
 *
 * 少量以值传递的原始数据。
 *
 * 用于 TEEC_Operation 参数中，当对应的参数类型是 TEEC_VALUE_INPUT, TEEC_VALUE_OUTPUT, 或 TEEC_VALUE_INOUT 时。
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.9. TEEC_Value
 */
typedef struct {
    /** 第 1 个值, 无特定含义，取决于 CA 和 TA 之间的协议 */
    uint32_t a;
    /** 第 2 个值，无特定含义，取决于 CA 和 TA 之间的协议 */
    uint32_t b;
} TEEC_Value;

/**
 * TEEC_Parameter 联合体
 *
 * 内容可以是 Temporary Memory Reference, Registered Memory Reference 或 Value Parameter.
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.10. TEEC_Parameter
 */
typedef union {
    /** 参数类型为 TEEC_MEMREF_TEMP_INPUT, TEEC_MEMREF_TEMP_OUTPUT 或 TEEC_MEMREF_TEMP_INOUT 时使用 */
    TEEC_TempMemoryReference tmpref;
    /** 参数类型为 TEEC_MEMREF_WHOLE, TEEC_MEMREF_PARTIAL_INPUT, TEEC_MEMREF_PARTIAL_OUTPUT 或 TEEC_MEMREF_PARTIAL_INOUT 时使用 */
    TEEC_RegisteredMemoryReference memref;
    /** 参数类型为 TEEC_VALUE_INPUT, TEEC_VALUE_OUTPUT 或 TEEC_VALUE_INOUT 时使用 */
    TEEC_Value value;
} TEEC_Parameter;

/**
 * TEEC_Operation 结构体
 *
 * TEEC_OpenSession() 或 TEEC_InvokeCommand() 操作的负载 (payload)
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.3.11. TEEC_Operation
 */
typedef struct {
    /** 如果需要取消要执行的操作，CA 必须初始化该 field 为 0 */
    uint32_t started;
    /** 使用 TEEC_PARAM_TYPES() 宏创建的参数类型编码后的值 */
    uint32_t paramTypes;
    /** TEEC_Parameter 数组 */
    TEEC_Parameter params[TEEC_CONFIG_PAYLOAD_REF_COUNT];

    // 以下部分为 implementation-defined 的

    /** 指向 TEEC_InvokeCommand() 最后一次使用的 TEEC_Session 的指针 */
    TEEC_Session *session;
} TEEC_Operation;

//////////////// 4.4. Constants ////////////////

/**
 * API 返回码定义
 *
 * 数据类型是 TEEC_Result
 *
 * 由 [TEE_CLIENT_API_SPEC] 中 4.4.2. Return Codes 的
 * Table 4-2: API Return Code Constants 定义
 */
/** The operation was successful */
#define TEEC_SUCCESS                0x00000000
/** Non-specific cause */
#define TEEC_ERROR_GENERIC          0xFFFF0000
/** Access privileges are not sufficient */
#define TEEC_ERROR_ACCESS_DENIED    0xFFFF0001
/** The operation was cancelled */
#define TEEC_ERROR_CANCEL           0xFFFF0002
/** Concurrent accesses caused conflict */
#define TEEC_ERROR_ACCESS_CONFLICT  0xFFFF0003
/** Too much data for the requested operation was passed */
#define TEEC_ERROR_EXCESS_DATA      0xFFFF0004
/** Input data was of invalid format */
#define TEEC_ERROR_BAD_FORMAT       0xFFFF0005
/** Input parameters were invalid */
#define TEEC_ERROR_BAD_PARAMETERS   0xFFFF0006
/** Operation is not valid in the current state */
#define TEEC_ERROR_BAD_STATE        0xFFFF0007
/** The requested data item is not found */
#define TEEC_ERROR_ITEM_NOT_FOUND   0xFFFF0008
/** The requested operation should exist but is not yet implemented */
#define TEEC_ERROR_NOT_IMPLEMENTED  0xFFFF0009
/** The requested operation is valid but is not supported in this Implementation */
#define TEEC_ERROR_NOT_SUPPORTED    0xFFFF000A
/** Expected data was missing */
#define TEEC_ERROR_NO_DATA          0xFFFF000B
/** System ran out of resources */
#define TEEC_ERROR_OUT_OF_MEMORY    0xFFFF000C
/** The system is busy working on something else */
#define TEEC_ERROR_BUSY             0xFFFF000D
/** Communication with a remote party failed */
#define TEEC_ERROR_COMMUNICATION    0xFFFF000E
/** A security fault was detected */
#define TEEC_ERROR_SECURITY         0xFFFF000F
/** The supplied buffer is too short for the generated output */
#define TEEC_ERROR_SHORT_BUFFER     0xFFFF0010
#define TEEC_ERROR_EXTERNAL_CANCEL  0xFFFF0011
/** Trusted Application has panicked during the operation */
#define TEEC_ERROR_TARGET_DEAD      0xFFFF3024

/**
 * 返回码的起源 (origin) 定义
 *
 * 用于表明特定的返回错误码是从软件中的哪部分产生的。数据类型是 uint32_t
 *
 * 由 [TEE_CLIENT_API_SPEC] 中 4.4.3. Return Code Origins 的
 * Table 4-3: API Return Code Origin Constants 定义
 */
/** The return code is an error that originated within the TEE Client API implementation */
#define TEEC_ORIGIN_API          0x00000001
/** The return code is an error that originated within the underlying communications stack linking the rich OS with the TEE */
#define TEEC_ORIGIN_COMMS        0x00000002
/**  The return code is an error that originated within the common TEE code */
#define TEEC_ORIGIN_TEE          0x00000003
/** The return code originated within the Trusted Application code. This includes the case where the return code is a success */
#define TEEC_ORIGIN_TRUSTED_APP  0x00000004

/**
 * Memory Reference 类型定义
 *
 * 用于 TEEC_Operation 内的 TEEC_Parameter 的编码，数据类型是 uint32_t
 *
 * 由 [TEE_CLIENT_API_SPEC] 中 4.4.4. Shared Memory Control 的
 * Table 4-5: API Memory Reference Types 定义
 */

/** The Shared Memory can carry data from the client application to the Trusted Application */
#define TEEC_MEM_INPUT   0x00000001
/** The Shared Memory can carry data from the Trusted Application to the client application */
#define TEEC_MEM_OUTPUT  0x00000002

/** The parameter is not used */
#define TEEC_NONE                   0x00000000
/** The parameter is a TEEC_Value tagged as input */
#define TEEC_VALUE_INPUT            0x00000001
/** The parameter is a TEEC_Value tagged as output */
#define TEEC_VALUE_OUTPUT           0x00000002
/** The parameter is a TEEC_Value tagged as both as input and output */
#define TEEC_VALUE_INOUT            0x00000003
/** A Temporary Memory Reference tagged as both input */
#define TEEC_MEMREF_TEMP_INPUT      0x00000005
/** A Temporary Memory Reference tagged as both output */
#define TEEC_MEMREF_TEMP_OUTPUT     0x00000006
/** A Temporary Memory Reference tagged as both input and output */
#define TEEC_MEMREF_TEMP_INOUT      0x00000007
/** A Registered Memory Reference that refers to the entirety of its parent Shared Memory block */
#define TEEC_MEMREF_WHOLE           0x0000000C
/** A Registered Memory Reference structure that refers to a partial region of its parent Shared Memory block and is tagged as input */
#define TEEC_MEMREF_PARTIAL_INPUT   0x0000000D
/** A Registered Memory Reference structure that refers to a partial region of its parent Shared Memory block and is tagged as output */
#define TEEC_MEMREF_PARTIAL_OUTPUT  0x0000000E
/** A Registered Memory Reference structure that refers to a partial region of its parent Shared Memory block and is tagged as both input and output */
#define TEEC_MEMREF_PARTIAL_INOUT   0x0000000F

/**
 * Session login methods, for use in TEEC_OpenSession() as parameter
 * connectionMethod. Type is uint32_t.
 *
 * TEEC_LOGIN_PUBLIC                No login data is provided.
 * TEEC_LOGIN_USER                  Login data about the user running the Client
 *                                  Application process is provided.
 * TEEC_LOGIN_GROUP                 Login data about the group running the Client
 *                                  Application process is provided.
 * TEEC_LOGIN_APPLICATION           Login data about the running Client Application
 *                                  itself is provided.
 * TEEC_LOGIN_USER_APPLICATION      Login data about the user and the running
 *                                  Client Application itself is provided.
 * TEEC_LOGIN_GROUP_APPLICATION     Login data about the group and the running
 *                                  Client Application itself is provided.
 */
#define TEEC_LOGIN_PUBLIC               0x00000000
#define TEEC_LOGIN_USER                 0x00000001
#define TEEC_LOGIN_GROUP                0x00000002
#define TEEC_LOGIN_APPLICATION          0x00000004
#define TEEC_LOGIN_USER_APPLICATION     0x00000005
#define TEEC_LOGIN_GROUP_APPLICATION    0x00000006

//////////////// 4.5. Functions ////////////////

/**
 * 初始化一个持有连接信息的 context
 *
 * @param name      不支持指定 TEE, 应该设置为 NULL 以连接到默认 TEE
 * @param context   将会被初始化的 context 结构体
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.2. TEEC_InitializeContext
 *
 * @return  初始化成功时返回 TEEC_SUCCESS, 否则返回错误码
 */
TEEC_Result TEEC_InitializeContext(const char *name, TEEC_Context *context);

/**
 * 销毁一个 context
 *
 * 该函数销毁一个已初始化的 TEE context, 关闭 CA 和 TEE 之间的连接。该函数必须仅在
 * 所有 session 都已被关闭，且所有共享内存块都已被释放后调用。
 *
 * @param context   要销毁的 context
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.3. TEEC_FinalizeContext
 */
void TEEC_FinalizeContext(TEEC_Context *context);

/**
 * TEEC_RegisterSharedMemory()
 *
 * Register a block of existing memory as a
 * shared block within the scope of the specified context.
 *
 * @param context    The initialized TEE context structure in which scope to
 *                   open the session.
 * @param sharedMem  pointer to the shared memory structure to register.
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.4. TEEC_RegisterSharedMemory
 *
 * @return TEEC_SUCCESS              The registration was successful.
 * @return TEEC_ERROR_OUT_OF_MEMORY  Memory exhaustion.
 * @return TEEC_Result               Something failed.
 */
TEEC_Result TEEC_RegisterSharedMemory(
        TEEC_Context *context,
        TEEC_SharedMemory *sharedMem);

/**
 * 分配一块内存空间
 *
 * 在调用该函数前，CA 必须设置好 sharedMem 所指向 TEEC_SharedMemory 结构体的 size 和 flags 域。
 *
 * @param   context     指向 TEEC_Context 结构体的指针
 * @param   sharedMem   指向要分配空间的 TEEC_SharedMemory 结构体的指针
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.5. TEEC_AllocateSharedMemory
 *
 * @return  内存空间分配成功时返回 TEEC_SUCCESS, 否则返回错误码
 */
TEEC_Result TEEC_AllocateSharedMemory(
        TEEC_Context *context,
        TEEC_SharedMemory *sharedMem);

/**
 * 释放之前已申请的内存空间
 *
 * @param   sharedMem   指向已申请空间的 TEEC_SharedMemory 结构体的指针
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.6. TEEC_ReleaseSharedMemory
 */
void TEEC_ReleaseSharedMemory(TEEC_SharedMemory *sharedMem);

/**
 * 打开 CA 与指定的 TA 间的一个新的会话
 *
 * @param   context             指向一个已初始化的 TEEC_Context 结构体的指针，在该 context 中打开会话
 * @param   session             指向一个 Session 结构体的指针，打开会话时将初始化该结构体
 * @param   destination         指向存放了目标 TA 的 UUID 的结构体的指针
 * @param   connectionMethod    要使用的连接方法
 * @param   connectionData      所选择连接方法要使用的数据 (不支持，应指定为 NULL)
 * @param   operation           指向一个 TEEC_Operation 结构体的指针，该结构体中包含一系列
 *                              用于与 TA 进行交互使用的 buffer 或其他变量。如果在打开会话时 CA 和 TA
 *                              不需要交换数据，则可以指定为 NULL。
 * @param   returnOrigin        指向包含 return origin 的变量的指针，如果不需要 return origin 则可以指定为 NULL
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.7. TEEC_OpenSession
 *
 * @return  会话打开成功时返回 TEEC_SUCCESS, 否则返回错误码
 */
TEEC_Result TEEC_OpenSession(
        TEEC_Context        *context,
        TEEC_Session        *session,
        const TEEC_UUID     *destination,
        uint32_t            connectionMethod,
        const void          *connectionData,
        TEEC_Operation      *operation,
        uint32_t            *returnOrigin);

/**
 * 关闭一个已打开的会话
 *
 * @param session       指向一个 Session 结构体的指针，将关闭该会话
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.8. TEEC_CloseSession
 */
void TEEC_CloseSession(TEEC_Session *session);

/**
 * 在指定会话内调用命令
 *
 * @param session           指向会话 TEEC_Context 结构体的指针
 * @param commandID         要调用命令的 ID (由 TA 定义)
 * @param operation         指向由 CA 初始化的 TEEC_Operation 结构体的指针，
 *                          如果不需要交换数据，或命令不需要支持取消，则可以指定为 NULL。
 * @param returnOrigin      指向包含 return origin 的变量的指针，如果不需要 return origin 则可以指定为 NULL
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.9. TEEC_InvokeCommand
 *
 * @return  命令执行成功时返回 TEEC_SUCCESS, 否则返回错误码
 */
TEEC_Result TEEC_InvokeCommand(TEEC_Session *session,
        uint32_t commandID,
        TEEC_Operation *operation,
        uint32_t *returnOrigin);

/**
 * TEEC_RequestCancellation() - Request the cancellation of a pending open
 *                              session or command invocation.
 *
 * @param operation Pointer to an operation previously passed to open session
 *                  or invoke.
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.10. TEEC_RequestCancellation
 */
void TEEC_RequestCancellation(TEEC_Operation *operation);


/**
 * 用于将多个 TEEC_Parameter 的类型编码为 TEEC_Operation 结构体的 paramTypes 所需要的 uint32_t 值
 *
 * @param p0    参数 0 的类型
 * @param p1    参数 1 的类型
 * @param p2    参数 2 的类型
 * @param p3    参数 3 的类型
 *
 * @see     [TEE_CLIENT_API_SPEC] 中的 4.5.11. Function-Like Macro: TEEC_PARAM_TYPES
 */
#define TEEC_PARAM_TYPES(p0, p1, p2, p3) \
    ((p0) | ((p1) << 4) | ((p2) << 8) | ((p3) << 12))

/**
 * 从 paramType 中获取参数 i 的类型
 *
 * @param p     paramType 值
 * @param i     取值为 0, 1, 2, ...
 */
#define TEEC_PARAM_TYPE_GET(p, i)       (((p) >> (i * 4)) & 0xF)


#ifdef __cplusplus
}
#endif

#endif  // __TEE_CLIENT_API_H__
