#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include "NuMicro.h"
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "iot_interface.h"
#include "wifi_interface.h"
#include "wifi_module.h"

#define PIN_GREEN_LED_OFF           PA11_NS

#define GPIO_MODE_INIT()            do {            \
    PA_NS->MODE = (GPIO_MODE_OUTPUT << 11*2);       \
} while (0)


// 配置信息 (NonSecure)
static struct iot_context iot_ctx =
{
    // 平台生成，全网唯一产品ID，请替换井号中间的ID，并将2个井号删除
    // 注意: 需与 Secure 项目中 iot_context_secure 结构体中的值一致
    #1002781074#,
    // 提示: M2351 平台上，产品密钥值在 Secure 项目中的 iot_context_secure 结构体中配置
    // MCU 固件版本, 格式为 "xx.xx", 其中 x 为 0 - 9 的数字
    "01.01",
    // 数据接收缓冲区大小
    2048,
    // 数据发送缓冲区大小
    2048,
    // 云服务器地址
    "dispatch.qinglianyun.com",
    // 云服务器端口
    8990
};


extern int cloud_is_connected;

const static iot_u8* test_tx   = "this is linux unvarnished transmission data, cloud do not save this message";
const static char* test_fault  = "device battery is low.";
const static char* test_str    = "linux string data";
const static iot_u8 test_bin[] = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10};

static char _taskListBuffer[256];

static void sleep(int secs)
{
    vTaskDelay(secs * 1000);
}

static void mainTask(void *pvParameters)
{
    iot_u32 seq;
    float pm25 = 0.0;

    vTaskDelay(500);

    printf("reset WiFi module...\r\n");
    if (!wifi_esp8266_at_reset()) {
        printf("failed to reset ESP8266 WiFi module\r\n");

        while (1) {
            // do nothing
        }
    }

//    wifi_esp8266_connect_to_ap("clear_ssid", "clear_password");

    // 等待一段时间，以便 WiFi 模块自动连接到 AP
    printf("wait for auto connecting to AP...\r\n");
    for (int i = 0; i < 5; i++) {
        vTaskDelay(1000);

        if (wifi_esp8266_is_connected()) {
            break;
        } else {
            if (i < 4) {
                printf("%d seconds to wait...\r\n", (4 - i));
            }
        }
    }

    if (wifi_esp8266_is_connected()) {
        printf("connected to AP\r\n");
    } else {
        printf("not connect to AP\r\n");

#if 0
        printf("connecting to specified AP...\r\n");
        if (!wifi_esp8266_connect_to_ap("ssid", "password")) {
            printf("connect to AP failed\r\n");
        }
#else
        printf("start smartconfig...\r\n");
        bool smartconfig_success = false;
        if (wifi_esp8266_smartconfig(3 /* ESP-TOUCH + AirKiss */, 60 /* timeout in secs */, &smartconfig_success)) {
            printf("smartconfig %s\r\n", (smartconfig_success ? "succeeded" : "failed"));
        } else {
            printf("smartconfig failed\r\n");
        }
#endif

        if (!wifi_esp8266_is_connected()) {
            printf("WiFi module failed to connect\r\n");

            while (1) {

            }
        }

        printf("connected to ap\r\n");
    }

    printf("iot_start()\r\n");
    iot_start(&iot_ctx);

    printf("iot_ota_option_set()\r\n");
    iot_ota_option_set(121, 2048);

    while (1)
    {
        printf("loop begin\r\n");

        printf("FreeRTOS free heap size: %u\r\n", xPortGetFreeHeapSize());
        printf("FreeRTOS task list:\r\n");
        vTaskList(_taskListBuffer);
        printf("%s", _taskListBuffer);

        if (!cloud_is_connected)
        {
            printf("not connected cloud!\r\n");
            sleep(1);
        }
        else
        {
            //上报“硬件单品”或“网关”自身的数据
            pm25 = (float)(rand() % 1000) / 100.0f + 50;
            dp_up_add_float(DP_ID_PM25, pm25);                                 //添加浮点型数据，对应数据点DP_ID_PM25
            dp_up_add_bool(DP_ID_DP_SWITCH, rand() % 2);                     //添加布尔数据，对应数据点DP_ID_DP_SWITCH
            dp_up_add_string(DP_ID_CONFIG, test_str, strlen(test_str));        //添加字符串数据，对应数据点DP_ID_CONFIG
            dp_up_add_binary(DP_ID_BIN, test_bin, sizeof(test_bin));           //添加二进制数据，对应数据点DP_ID_BIN
            dp_up_add_int(DP_ID_TEMP, rand() % 20);                          //添加整型数据，对应数据点DP_ID_TEMP
            iot_upload_dps(NULL, &seq);                                        //上传至云端，保存数据

            dp_up_add_fault(DP_ID_FAULT, test_fault, strlen(test_fault));      //添加故障数据，对应数据点DP_ID_FAULT
            iot_upload_dps(NULL, &seq);                                        //上传至云端，保存数据
            printf("fault data seq:%d\r\n", seq);                              //打印故障类型数据的序号

            sleep(5);

            iot_tx_data(NULL, &seq, test_tx, strlen(test_tx));                 //透传一条数据至手机端，云端不保存
            sleep(5);

            iot_get_info(INFO_TYPE_USER_MASTER);    //获取绑定用户信息
            sleep(5);
            iot_get_info(INFO_TYPE_USER_SHARE);     //获取分享用户信息
            sleep(5);
            iot_get_info(INFO_TYPE_DEVICE);         //获取设备信息
            sleep(5);

            //如开发硬件单品，可删除以下宏定义内子设备相关的代码
#ifdef IOT_DEVICE_IS_GATEWAY
            //上报子设备温湿度传感器TEMP000001的数据
            dp_up_add_int(DP_ID_TEMP, rand() % 20);
            dp_up_add_int(DP_ID_HUMIDITY, rand() % 20);
            iot_upload_dps("TEMP000001", &seq);

            sleep(5);

            //上报子设备温湿度传感器TEMP000002的数据
            dp_up_add_int(DP_ID_TEMP, rand() % 20);
            dp_up_add_int(DP_ID_HUMIDITY, rand() % 20);
            iot_upload_dps("TEMP000002", &seq);

            sleep(5);

            //上报子设备风扇FAN0000001的风速
            dp_up_add_enum(DP_ID_WIND_SPD, rand() % 5);
            iot_upload_dps("FAN0000001", &seq);

            sleep(5);
#endif
        }
    }

    // 不应该执行到这里
    printf("MainTask ended unexpectedly!");
    while (1) {
        // do nothing
    }
}

int main(void)
{
    printf("Entered NonSecure main()\r\n");

//    printf("It is new firmware!!!\r\n");

    // 配置 GPIO 的模式
    GPIO_MODE_INIT();

    printf("WiFi module initializing...\r\n");

    wifi_module_init();

    printf("WiFi module power on...\r\n");

    PIN_GREEN_LED_OFF = 1;
    wifi_module_shutdown_and_power_on();
    PIN_GREEN_LED_OFF = 0;

    printf("create FreeRTOS main task...\r\n");

    xTaskCreate(mainTask,                           // The function that implements the task.
                "MainTask",                         // The text name assigned to the task - for debug only as it is not used by the kernel.
                1024,                               // The size of the stack to allocate to the task.
                NULL,                               // The parameter passed to the task - just to check the functionality.
                (tskIDLE_PRIORITY + 1),             // The priority assigned to the task.
                NULL);                              // The task handle is not required, so NULL is passed.

    printf("start FreeRTOS scheduler...\r\n");

    // Start the tasks and timer running.
    vTaskStartScheduler();

    printf("FreeRTOS scheduler ended...\r\n");

    /* If all is well, the scheduler will now be running, and the following
    line will never be reached.  If the following line does execute, then
    there was insufficient FreeRTOS heap memory available for the idle and/or
    timer tasks to be created.  See the memory management section on the
    FreeRTOS web site for more details. */
    while (1)
    {
        // do nothing
    }

    return 0;
}

void vApplicationMallocFailedHook(void)
{
    /* vApplicationMallocFailedHook() will only be called if
    configUSE_MALLOC_FAILED_HOOK is set to 1 in FreeRTOSConfig.h.  It is a hook
    function that will get called if a call to pvPortMalloc() fails.
    pvPortMalloc() is called internally by the kernel whenever a task, queue,
    timer or semaphore is created.  It is also called by various parts of the
    demo application.  If heap_1.c or heap_2.c are used, then the size of the
    heap available to pvPortMalloc() is defined by configTOTAL_HEAP_SIZE in
    FreeRTOSConfig.h, and the xPortGetFreeHeapSize() API function can be used
    to query the size of free heap space that remains (although it does not
    provide information on how the remaining heap might be fragmented). */
    taskDISABLE_INTERRUPTS();
    for (;;);
}

void vApplicationIdleHook(void)
{
    /* vApplicationIdleHook() will only be called if configUSE_IDLE_HOOK is set
    to 1 in FreeRTOSConfig.h.  It will be called on each iteration of the idle
    task.  It is essential that code added to this hook function never attempts
    to block in any way (for example, call xQueueReceive() with a block time
    specified, or call vTaskDelay()).  If the application makes use of the
    vTaskDelete() API function (as this demo application does) then it is also
    important that vApplicationIdleHook() is permitted to return to its calling
    function, because it is the responsibility of the idle task to clean up
    memory allocated by the kernel to any task that has since been deleted. */
}

void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName)
{
    (void)pcTaskName;
    (void)pxTask;

    /* Run time stack overflow checking is performed if
    configCHECK_FOR_STACK_OVERFLOW is defined to 1 or 2.  This hook
    function is called if a stack overflow is detected. */
    taskDISABLE_INTERRUPTS();

    // __BKPT();

    printf("Stack overflow task name=%s\n", pcTaskName);

    for (;;);
}

void vApplicationTickHook(void)
{
    /* This function will be called by each tick interrupt if
    configUSE_TICK_HOOK is set to 1 in FreeRTOSConfig.h.  User code can be
    added here, but the tick hook is called from an interrupt context, so
    code must not attempt to block, and only the interrupt safe FreeRTOS API
    functions can be used (those that end in FromISR()).  The code in this
    tick hook implementation is for demonstration only - it has no real
    purpose.  It just gives a semaphore every 50ms.  The semaphore unblocks a
    task that then toggles an LED.  Additionally, the call to
    vQueueSetAccessQueueSetFromISR() is part of the "standard demo tasks"
    functionality. */

    /* The semaphore and associated task are not created when the simple blinky
    demo is used. */
}
