#include <stdio.h>
#include <string.h>
#include "NuMicro.h"
#include "wifi_interface.h"

#define WIFI_MODULE_GPIO_PIN_PWR_OFF        PD7_NS
#define WIFI_MODULE_GPIO_PIN_GPIO0          PD6_NS

#define WIFI_MODULE_GPIO_MODE_INIT()        do {                            \
    PD_NS->MODE = (GPIO_MODE_OUTPUT << 6*2) | (GPIO_MODE_OUTPUT << 7*2);    \
} while (0)

#define WIFI_MODULE_UART_PORT               UART3_NS

static void _enable_uart_rda_interrupt(void)
{
    // 使能 WIFI_MODULE_UART_PORT 的 RDA 中断
    UART_EnableInt(WIFI_MODULE_UART_PORT, UART_INTEN_RDAIEN_Msk);
}

static void _disable_uart_rda_interrupt(void)
{
    // 禁止 WIFI_MODULE_UART_PORT 的 RDA 中断
    UART_DisableInt(WIFI_MODULE_UART_PORT, UART_INTEN_RDAIEN_Msk);
}

/**
 * UART3 的中断处理函数
 */
void UART3_IRQHandler(void)
{
    uint8_t inputChar = 0xFF;
    uint32_t intSts = WIFI_MODULE_UART_PORT->INTSTS;

    // 接收数据可用中断
    if (intSts & UART_INTSTS_RDAINT_Msk)
    {
        // 处理所有接收到的字符
        while (UART_IS_RX_READY(WIFI_MODULE_UART_PORT))
        {
            // 读取一个字符
            inputChar = UART_READ(WIFI_MODULE_UART_PORT);

            // 输出字符到调试串口 (调试时使用)
//            SendChar(inputChar);

            wifi_esp8266_at_write_received_data(&inputChar, 1);
        }
    }

    // 清除中断标志位
    if (WIFI_MODULE_UART_PORT->FIFOSTS & (UART_FIFOSTS_BIF_Msk | UART_FIFOSTS_FEF_Msk | UART_FIFOSTS_PEF_Msk | UART_FIFOSTS_RXOVIF_Msk))
    {
        WIFI_MODULE_UART_PORT->FIFOSTS = (UART_FIFOSTS_BIF_Msk | UART_FIFOSTS_FEF_Msk | UART_FIFOSTS_PEF_Msk | UART_FIFOSTS_RXOVIF_Msk);
    }
}

void wifi_esp8266_at_send_data_cb(const void *data, int data_length)
{
    char *p = data;

    for (int i = 0; i < data_length; i++)
    {
        while (WIFI_MODULE_UART_PORT->FIFOSTS & UART_FIFOSTS_TXFULL_Msk)
        {
            // do nothing
        }
        WIFI_MODULE_UART_PORT->DAT = (uint32_t)(*p++);
    }
}

void wifi_esp8266_at_smartconfig_status_changed_cb(int new_status) {
    switch (new_status) {
        case WIFI_ESP8266_SMART_CONFIG_STATUS_STOPPED:
            printf("[smartconfig] stopped\r\n");
            break;

        case WIFI_ESP8266_SMART_CONFIG_STATUS_STARTED:
            printf("[smartconfig] started\r\n");
            break;

        case WIFI_ESP8266_SMART_CONFIG_STATUS_IN_PROGRESS:
            printf("[smartconfig] in progress\r\n");
            break;

        case WIFI_ESP8266_SMART_CONFIG_STATUS_GOT_WIFI_INFO:
            printf("[smartconfig] got wifi info, start to connect\r\n");
            break;

        case WIFI_ESP8266_SMART_CONFIG_STATUS_CONNECTED:
            printf("[smartconfig] connected to AP\r\n");
            break;
            
        default:
            printf("[smartconfig] unknown status = %d\r\n", new_status);
            break;
    }
}

void wifi_module_init(void)
{
    // 配置 GPIO
    WIFI_MODULE_GPIO_MODE_INIT();

    // 配置 WIFI_MODULE_UART_PORT
    WIFI_MODULE_UART_PORT->LINE = UART_PARITY_NONE | UART_STOP_BIT_1 | UART_WORD_LEN_8;
    WIFI_MODULE_UART_PORT->BAUD = UART_BAUD_MODE2 | UART_BAUD_MODE2_DIVIDER(__HIRC, 115200);
//    WIFI_MODULE_UART_PORT->BAUD = UART_BAUD_MODE2 | UART_BAUD_MODE2_DIVIDER(__HIRC, 74880);       // for viewing boot log
}

int wifi_module_shutdown_and_power_on(void)
{
    // 禁止 UART 的 RDA 中断
    _disable_uart_rda_interrupt();

    // 关闭 WiFi 模块的电源
    WIFI_MODULE_GPIO_PIN_PWR_OFF = 1;
    WIFI_MODULE_GPIO_PIN_GPIO0 = 1;

    // 等待一段时间
    CLK_SysTickLongDelay(3000000);

    // 在为 WiFi 模块上电前先设置好 ESP8266 GPIO0 引脚的电平，以确定是否可对其升级 (下载固件)
    // WIFI_MODULE_GPIO_PIN_GPIO0 = 0; // Set 0 to enable WIFI module firmware update.
    WIFI_MODULE_GPIO_PIN_GPIO0 = 1; // Set 1 to Disable WIFI module firmware update.

    // 等待一段时间后上电
    CLK_SysTickLongDelay(1000000);
    WIFI_MODULE_GPIO_PIN_PWR_OFF = 0;

    // 使能 UART 的 RDA 中断
    _enable_uart_rda_interrupt();
}
