#include <stdio.h>
#include <stdarg.h>
#include <arm_cmse.h>
#include "NuMicro.h"
#include "partition_M2351.h"
#include "iot_interface_secure.h"
#include "ota.h"


// 配置信息 (Secure)
static struct iot_context_secure iot_ctx_secure = {
    // 平台生成，全网唯一产品ID，请替换井号中间的ID，并将2个井号删除
    // 注意: 需与 NonSecure 项目中 iot_context 结构体中的值一致
    #1002781074#,
    // 平台生成，全网唯一产品密钥，请替换井号中间的密钥，并将2个井号删除
    // 请将产品密钥字符串转换成相应的十六进制编码，即在每个字节前增加 0x 作为开头。如密钥是 5668，替换时应改为 0x56，0x68
    {#0x78, 0x79, 0x52, 0x72, 0x26, 0x4a, 0x35, 0x78, 0x23, 0x37, 0x24, 0xd1, 0x88, 0x68, 0x7d, 0x68#},
};


// 调试阶段可启用 ota_log() 函数，以输出 OTA 相关 log
#if 1
void ota_log(const char *format, ...)
{
    va_list args;
    va_start(args, format);

    printf("[OTA] ");
    vprintf(format, args);

    va_end(args);
}
#endif

// typedef for NonSecure callback functions
typedef __attribute__((cmse_nonsecure_call)) int32_t (*NonSecure_funcptr)(uint32_t);

#define NEXT_BOOT_BASE  FMC_NON_SECURE_BASE
#define JUMP_HERE       0xe7fee7ff      /* Instruction Code of "B ." */

/**
 * 用于跳转到下一 boot 阶段 (Non-Secure 区域代码) 的 Boot_Init() 函数
 *
 * 注意: 不可变更函数名和参数，otaCheckFirmwareAndBoot() 内会调用该函数
 */
void Boot_Init(uint32_t u32BootBase)
{
    NonSecure_funcptr fp;

    /* SCB_NS.VTOR points to the Non-secure vector table base address. */
    SCB_NS->VTOR = u32BootBase;

    /* 1st Entry in the vector table is the Non-secure Main Stack Pointer. */
    __TZ_set_MSP_NS(*((uint32_t *)SCB_NS->VTOR));      /* Set up MSP in Non-secure code */

    /* 2nd entry contains the address of the Reset_Handler (CMSIS-CORE) function */
    fp = ((NonSecure_funcptr)(*(((uint32_t *)SCB_NS->VTOR) + 1)));

    /* Clear the LSB of the function address to indicate the function-call
       will cause a state switch from Secure to Non-secure */
    fp = cmse_nsfptr_create(fp);

    /* Check if the Reset_Handler address is in Non-secure space */
    if (cmse_is_nsfptr(fp) && (((uint32_t)fp & 0xf0000000) == 0x10000000))
    {
        printf("Execute non-secure code ...\r\n");
        fp(0); /* Non-secure function call */
    }
    else
    {
        /* Something went wrong */
        printf("No code in non-secure region!\r\n");
        printf("CPU will halted at non-secure state\r\n");

        /* Set nonsecure MSP in nonsecure region */
        __TZ_set_MSP_NS(NON_SECURE_SRAM_BASE+512);

        /* Try to halted in non-secure state (SRAM) */
        M32(NON_SECURE_SRAM_BASE) = JUMP_HERE;
        fp = (NonSecure_funcptr)(NON_SECURE_SRAM_BASE+1);
        fp(0);

        while (1);
    }
}

/**
 * 系统初始化
 */
void SYS_Init(void)
{
    //////////////// 初始化系统时钟 ////////////////

    // 使能 PLL (128MHz PLL output with HIRC)
    CLK->PLLCTL = CLK_PLLCTL_128MHz_HIRC;

    // 等待 PLL 稳定
    while ((CLK->STATUS & CLK_STATUS_PLLSTB_Msk) == 0);

    // 设置 HCLK 分频系数为 2
    CLK->CLKDIV0 = (CLK->CLKDIV0 & (~CLK_CLKDIV0_HCLKDIV_Msk)) | 1;

    // 切换 HCLK 的时钟源为 PLL
    CLK->CLKSEL0 = (CLK->CLKSEL0 & (~CLK_CLKSEL0_HCLKSEL_Msk)) | CLK_CLKSEL0_HCLKSEL_PLL;

    // 使能 AHB 和 APB 总线上所有设备的时钟
    CLK->AHBCLK = 0xFFFFFFFFul;
    CLK->APBCLK0 = 0xFFFFFFFFul;
    CLK->APBCLK1 = 0xFFFFFFFFul;

    // 设置 UART0 和 UART5 的时钟源为 HIRC
    CLK->CLKSEL1 = CLK_CLKSEL1_UART0SEL_HIRC;
    CLK->CLKSEL3 = CLK_CLKSEL3_UART5SEL_HIRC;

    // 设置 UART3 的时钟源为 HIRC
    CLK->CLKSEL3 = (CLK->CLKSEL3 & (~CLK_CLKSEL3_UART3SEL_Msk)) | CLK_CLKSEL3_UART3SEL_HIRC;

    // 更新 SystemCoreClock 等变量的值
    // 用户可以使用 SystemCoreClockUpdate() 自动地计算 SystemCoreClock 和 CyclesPerUs 的值
    SystemCoreClockUpdate();

    //////////////// 初始化 IO Multi-function ////////////////

    // 设置 UART0 的 RXD (PB.12) 和 TXD (PB.13) 引脚的 multi-function
    SYS->GPB_MFPH = (SYS->GPB_MFPH & (~(UART0_RXD_PB12_Msk | UART0_TXD_PB13_Msk))) | UART0_RXD_PB12 | UART0_TXD_PB13;

    // 设置 UART3 的 RXD (PD.0) 和 TXD (PD.1) 引脚的 multi-function
    SYS->GPD_MFPL = (SYS->GPD_MFPL & (~(UART3_RXD_PD0_Msk | UART3_TXD_PD1_Msk))) | UART3_RXD_PD0 | UART3_TXD_PD1;
}

/**
 * 初始化调试用 UART 端口
 */
void DEBUG_PORT_Init(void)
{
    // 配置 UART0
    DEBUG_PORT->BAUD = UART_BAUD_MODE2 | UART_BAUD_MODE2_DIVIDER(__HIRC, 115200);
    DEBUG_PORT->LINE = UART_WORD_LEN_8 | UART_PARITY_NONE | UART_STOP_BIT_1;
}

/**
 * 获取 SystemCoreClock 的值
 *
 * 用于 NonSecure 区的 FreeRTOS 等程序通过该 Non-secure Callable 获取 SystemCoreClock 的值
 */
__NONSECURE_ENTRY
uint32_t GetSystemCoreClock(void)
{
    return SystemCoreClock;
}

void CRPT_IRQHandler()
{
    if (AES_GET_INT_FLAG(CRPT))
    {
        iot_secure_crpt_aes_is_done();
        AES_CLR_INT_FLAG(CRPT);
    }
}

int main(void)
{
    SYS_UnlockReg();
    SYS_Init();

    DEBUG_PORT_Init();

    NVIC_EnableIRQ(CRPT_IRQn);
    AES_ENABLE_INT(CRPT);

    // 初始化 Secure 区域的配置
    iot_secure_config(&iot_ctx_secure);

    // 注册 SDK 和 OTA 的 TA
    teeDriverTaRegister(&g_ta_reference_iotsdk);
    teeDriverTaRegister(&g_ta_reference_iotota);

    // 检查当前是否有需要进行的 OTA 操作，
    // 并根据情况引导到下一阶段 (Non-Secure 代码) 或 reboot
    otaCheckFirmwareAndBoot();

    while (1)
    {
        __WFI();
    }

    return 0;
}
