#ifndef __TINY_TEE_H__
#define __TINY_TEE_H__

#include "NuMicro.h"

#include "tee_internal_api.h"

extern void TEE_Log(const char *format, ...);

typedef struct {
    TA_CreateEntryPoint_FuncPtr         createEntryPoint;
    TA_DestroyEntryPoint_FuncPtr        destroyEntryPoint;
    TA_OpenSessionEntryPoint_FuncPtr    openSessionEntryPoint;
    TA_CloseSessionEntryPoint_FuncPtr   closeSessionEntryPoint;
    TA_InvokeCommandEntryPoint_FuncPtr  invokeCommandEntryPoint;
} TA_EntryPointTable;

typedef struct {
    /** TA 的 UUID 标识符 */
    TEE_UUID                *appId;
    /*
    (尚不支持)
    // 是否对所有会话使用该 TA 的单一实例
    bool                    singleInstance;
    // 该 TA 的实例是否支持多个会话 (当 singleInstance 属性被设置为 false 时，该属性被忽略)
    bool                    multiSession;
    // 在没有任何连接到该 TA 实例的会话时，TA 的实例是否应当被保留 (该属性仅在 singleInstance 设置为 true 时有意义)
    bool                    instanceKeepAlive;
    */
    /** TA 的入口函数表 */
    TA_EntryPointTable      entryPointTable;
} TA_Reference;

/** 可注册 TA 最大数量 */
#define TA_REFERENCE_MAX_COUNT      10

/** 可同时打开会话的最大数量 */
#define TA_SESSION_MAX_COUNT        10

#define TEE_DRIVER_INVOKE_TYPE_BASE             0

/** 打开会话操作的 request code */
#define TEE_DRIVER_INVOKE_TYPE_OPEN_SESSION     (TEE_DRIVER_INVOKE_TYPE_BASE + 2)
/** 调用命令操作的 request code */
#define TEE_DRIVER_INVOKE_TYPE_INVOKE           (TEE_DRIVER_INVOKE_TYPE_BASE + 3)
/** 关闭会话操作的 request code */
#define TEE_DRIVER_INVOKE_TYPE_CLOSE_SESSION    (TEE_DRIVER_INVOKE_TYPE_BASE + 5)

typedef enum {
    TEE_DRIVER_PARAMETER_TYPE_NONE = 0,

    // for TeeDriver_Parameter_Value
    TEE_DRIVER_PARAMETER_TYPE_VALUE_INPUT = 1,
    TEE_DRIVER_PARAMETER_TYPE_VALUE_OUTPUT = 2,
    TEE_DRIVER_PARAMETER_TYPE_VALUE_INOUT = 3,

    // for TeeDriver_Parameter_MemoryReference
    TEE_DRIVER_PARAMETER_TYPE_MEMREF_INPUT = 5,
    TEE_DRIVER_PARAMETER_TYPE_MEMREF_OUTPUT = 6,
    TEE_DRIVER_PARAMETER_TYPE_MEMREF_INOUT = 7
} TeeDriver_Parameter_Type;

/**
 * 与 TEE Client API 中的 TEEC_LOGIN_* 系列常量定义匹配
 */
#define TEE_DRIVER_LOGIN_PUBLIC                 0
#define TEE_DRIVER_LOGIN_USER                   1
#define TEE_DRIVER_LOGIN_GROUP                  2
#define TEE_DRIVER_LOGIN_APPLICATION            4
#define TEE_DRIVER_LOGIN_USER_APPLICATION       5
#define TEE_DRIVER_LOGIN_GROUP_APPLICATION      6

typedef struct {
    void        *buffer;
    uint32_t    size;
} TeeDriver_Parameter_MemoryReference;

typedef struct {
    uint32_t a;
    uint32_t b;
} TeeDriver_Parameter_Value;

typedef struct {
    /** 参数类型 */
    TeeDriver_Parameter_Type type;

    /** 参数值 (联合体共用空间) */
    union {
        TeeDriver_Parameter_MemoryReference memref;
        TeeDriver_Parameter_Value value;
    } unionValue;
} TeeDriver_Parameter;

#define TEE_IOCTL_UUID_LEN      16

/**
 * TeeDriver_OpenSessionArgument 结构体
 *
 * 通过 TEEC_OpenSession() 打开会话时所用的参数
 */
typedef struct {
    /** [in] TA (Trusted Application) 的 UUID */
    uint8_t uuid[TEE_IOCTL_UUID_LEN];
    /** [in] client 的 login class (TEE_IOCTL_LOGIN_*) */
    uint32_t loginMethod;
    /** [in] 参数数量，即该结构体后所跟 TeeDriver_Parameter 结构体的数量 */
    uint32_t paramCount;
    /** [in] cancellation id (尚未实现) */
    uint32_t cancelId;
    /** [out] session id */
    uint32_t sessionId;
    /** [out] 返回值 */
    uint32_t returnValue;
    /** [out] 返回值的 origin (returnOrigin) */
    uint32_t returnOrigin;
} TeeDriver_OpenSessionArgument;

/**
 * TeeDriver_InvokeArgument 结构体
 *
 * 通过 TEEC_InvokeCommand() 调用命令时所用的参数
 */
typedef struct {
    /** [in] command ID */
    uint32_t commandId;
    /** [in] 参数数量，即该结构体后所跟 TeeDriver_Parameter 结构体的数量 */
    uint32_t paramCount;
    /** [in] session id */
    uint32_t sessionId;
    /** [in] cancellation id (尚未实现) */
    uint32_t cancelId;
    /** [out] 返回值 */
    uint32_t returnValue;
    /** [out] 返回值的 origin (returnOrigin) */
    uint32_t returnOrigin;
} TeeDriver_InvokeArgument;

/**
 * TeeDriver_CancelArgument 结构体
 *
 * 通过 TEEC_RequestCancellation() 取消正在执行的打开会话或调用命令操作时所用的参数
 */
typedef struct {
    uint32_t cancelId;
    uint32_t session;
} TeeDriver_CancelArgument;

/**
 * TeeDriver_CloseSessionArgument 结构体
 *
 * 通过 TEEC_CloseSession() 关闭会话时所用的参数
 */
typedef struct {
    /** [in] session id */
    uint32_t sessionId;
} TeeDriver_CloseSessionArgument;

/**
 * 注册 TA
 *
 * @param   taReference     指向包含 TA 的 UUID 和入口函数信息的 TA_Reference 结构体的指针
 *
 * @return  注册成功时返回 TEE_SUCCESS, 否则返回错误码
 */
uint32_t teeDriverTaRegister(TA_Reference *taReference);

uint32_t teeDriverInvokeInternal(uint32_t invokeType, void *bufferData, size_t bufferDataLength);

/**
 * TEE Driver 的调用接口
 *
 * 需由用户在 Secure 区域定义，并在函数内调用 teeDriverInvokeInternal() 函数
 */
__NONSECURE_ENTRY
uint32_t teeDriverInvoke(uint32_t invokeType, void *bufferData, size_t bufferDataLength);

#endif  // __TINY_TEE_H__
