/**
 * 参考文档:
 * [TEE_INTERNAL_CORE_API_SPEC]     TEE Internal Core API Specification Version 1.1.1 (June 2016)
 */

#ifndef __TEE_INTERNAL_API_H__
#define __TEE_INTERNAL_API_H__

#include <inttypes.h>
#include <stdbool.h>
#include <stddef.h>

//////////////////////////////// 3 Common Definitions ////////////////////////////////

//////////////// 3.1 Header File ////////////////

/**
 * The header file for the TEE Internal Core API MUST have the name "tee_internal_api.h"
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 3.1 Header File
 */

//////////////// 3.2 Data Types ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 3.2.3 TEE_Result, TEEC_Result
 */
typedef uint32_t TEE_Result;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 3.2.4 TEE_UUID, TEEC_UUID
 */
typedef struct {
    uint32_t timeLow;
    uint16_t timeMid;
    uint16_t timeHiAndVersion;
    uint8_t clockSeqAndNode[8];
} TEE_UUID;

//////////////// 3.3 Constants ////////////////

/* API Error Codes */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 Table 3-3: API Return Codes
 */
#define TEE_SUCCESS                       0x00000000
#define TEE_ERROR_CORRUPT_OBJECT          0xF0100001
#define TEE_ERROR_CORRUPT_OBJECT_2        0xF0100002
#define TEE_ERROR_STORAGE_NOT_AVAILABLE   0xF0100003
#define TEE_ERROR_STORAGE_NOT_AVAILABLE_2 0xF0100004
#define TEE_ERROR_GENERIC                 0xFFFF0000
#define TEE_ERROR_ACCESS_DENIED           0xFFFF0001
#define TEE_ERROR_CANCEL                  0xFFFF0002
#define TEE_ERROR_ACCESS_CONFLICT         0xFFFF0003
#define TEE_ERROR_EXCESS_DATA             0xFFFF0004
#define TEE_ERROR_BAD_FORMAT              0xFFFF0005
#define TEE_ERROR_BAD_PARAMETERS          0xFFFF0006
#define TEE_ERROR_BAD_STATE               0xFFFF0007
#define TEE_ERROR_ITEM_NOT_FOUND          0xFFFF0008
#define TEE_ERROR_NOT_IMPLEMENTED         0xFFFF0009
#define TEE_ERROR_NOT_SUPPORTED           0xFFFF000A
#define TEE_ERROR_NO_DATA                 0xFFFF000B
#define TEE_ERROR_OUT_OF_MEMORY           0xFFFF000C
#define TEE_ERROR_BUSY                    0xFFFF000D
#define TEE_ERROR_COMMUNICATION           0xFFFF000E
#define TEE_ERROR_SECURITY                0xFFFF000F
#define TEE_ERROR_SHORT_BUFFER            0xFFFF0010
#define TEE_ERROR_EXTERNAL_CANCEL         0xFFFF0011
#define TEE_ERROR_OVERFLOW                0xFFFF300F
#define TEE_ERROR_TARGET_DEAD             0xFFFF3024
#define TEE_ERROR_STORAGE_NO_SPACE        0xFFFF3041
#define TEE_ERROR_MAC_INVALID             0xFFFF3071
#define TEE_ERROR_SIGNATURE_INVALID       0xFFFF3072
#define TEE_ERROR_TIME_NOT_SET            0xFFFF5000
#define TEE_ERROR_TIME_NEEDS_RESET        0xFFFF5001

//////////////////////////////// 4 Trusted Core Framework API ////////////////////////////////

//////////////// 4.1 Data Types ////////////////

/**
 * 该 TEE_Identity 结构体描述 the full identity of a Client:
 *
 * login 的值是 TEE_LOGIN_* 系列常量中的一个。
 * uuid 包含客户端的 UUID, 或者为 Nil (as defined in [RFC 4122]) if not applicable
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.1.1 TEE_Identity
 */
typedef struct {
    uint32_t    login;
    TEE_UUID    uuid;
} TEE_Identity;

/*
 * 该联合体描述一个由框架传递给 TA_OpenSessionEntryPoint 或 TA_InvokeCommandEntryPoint 入口函数的参数，
 * 或由 TA 传递给 TEE_OpenTASession() 或 TEE_InvokeTACommand() 的参数。
 *
 * 应该使用 memref 还是 value 域由传递给入口函数的 paramTypes 参数中指定的参数类型决定。
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.1.2 TEE_Param
 */
typedef union {
    struct {
        void        *buffer;
        size_t      size;
    } memref;

    struct {
        uint32_t    a;
        uint32_t    b;
    } value;
} TEE_Param;

/*
 * The type of opaque handles on TA Session. These handles are returned by
 * the function TEE_OpenTASession.
 */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.1.3 TEE_TASessionHandle
 */
typedef struct __TEE_TASessionHandle *TEE_TASessionHandle;

/*
 * The type of opaque handles on property sets or enumerators. These
 * handles are either one of the pseudo handles TEE_PROPSET_XXX or are
 * returned by the function TEE_AllocatePropertyEnumerator.
*/
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.1.4 TEE_PropSetHandle
 */
typedef struct __TEE_PropSetHandle *TEE_PropSetHandle;

//////////////// 4.2 Constants ////////////////

/**
 * 参数类型常量
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 Table 4-1: Parameter Type Constants
 */
#define TEE_PARAM_TYPE_NONE             0
#define TEE_PARAM_TYPE_VALUE_INPUT      1
#define TEE_PARAM_TYPE_VALUE_OUTPUT     2
#define TEE_PARAM_TYPE_VALUE_INOUT      3
#define TEE_PARAM_TYPE_MEMREF_INPUT     5
#define TEE_PARAM_TYPE_MEMREF_OUTPUT    6
#define TEE_PARAM_TYPE_MEMREF_INOUT     7

/* Login Type Constants */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 Table 4-2: Login Type Constants
 */
#define TEE_LOGIN_PUBLIC                0x00000000
#define TEE_LOGIN_USER                  0x00000001
#define TEE_LOGIN_GROUP                 0x00000002
#define TEE_LOGIN_APPLICATION           0x00000004
#define TEE_LOGIN_APPLICATION_USER      0x00000005
#define TEE_LOGIN_APPLICATION_GROUP     0x00000006
#define TEE_LOGIN_TRUSTED_APP           0xF0000000

/* Origin Code Constants */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 Table 4-3: Origin Code Constants
 */
#define TEE_ORIGIN_API                  0x00000001
#define TEE_ORIGIN_COMMS                0x00000002
#define TEE_ORIGIN_TEE                  0x00000003
#define TEE_ORIGIN_TRUSTED_APP          0x00000004

/* Property Sets pseudo handles */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 Table 4-4: Property Set Pseudo-Handle Constants
 */
#define TEE_PROPSET_TEE_IMPLEMENTATION  (TEE_PropSetHandle)0xFFFFFFFD
#define TEE_PROPSET_CURRENT_CLIENT      (TEE_PropSetHandle)0xFFFFFFFE
#define TEE_PROPSET_CURRENT_TA          (TEE_PropSetHandle)0xFFFFFFFF

/* Memory Access Rights Constants */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 Table 4-5: Memory Access Rights Constants
 */
#define TEE_MEMORY_ACCESS_READ             0x00000001
#define TEE_MEMORY_ACCESS_WRITE            0x00000002
#define TEE_MEMORY_ACCESS_ANY_OWNER        0x00000004

//////////////// 4.3 TA Interface ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.1 TA_CreateEntryPoint
 */
typedef TEE_Result (*TA_CreateEntryPoint_FuncPtr)(void);

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.2 TA_DestroyEntryPoint
 */
typedef void (*TA_DestroyEntryPoint_FuncPtr)(void);

/**
 * 当客户端请求打开与 TA 间的会话时，框架会调用 TA_OpenSessionEntryPoint 入口函数
 *
 * @param   paramTypes      调用时所传递 4 个参数的类型，通过 TEE_PARAM_TYPES() 宏获得
 * @param   params          指向 4 个参数的数组的指针
 * @param   sessionContext  指向一个 void * 类型指针的指针，TA 程序会设置其所指向的 void * 指针
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.3 TA_OpenSessionEntryPoint
 */
typedef TEE_Result (*TA_OpenSessionEntryPoint_FuncPtr)(uint32_t paramTypes, TEE_Param params[4], void **sessionContext);

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.4 TA_CloseSessionEntryPoint
 */
typedef void (*TA_CloseSessionEntryPoint_FuncPtr)(void *sessionContext);

/**
 * 当客户端在给定的会话中调用命令时，框架会调用 TA_InvokeCommandEntryPoint 入口函数
 *
 * @param   sessionContext  由 TA 程序在 TA_OpenSessionEntryPoint() 中设置的 void * 类型指针
 * @param   commandID       TA 程序中所指定的要调用的命令的 ID
 * @param   paramTypes      调用时所传递 4 个参数的类型，通过 TEE_PARAM_TYPES() 宏获得
 * @param   params          指向 4 个参数的数组的指针
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.5 TA_InvokeCommandEntryPoint
 */
typedef TEE_Result (*TA_InvokeCommandEntryPoint_FuncPtr)(void *sessionContext, uint32_t commandID, uint32_t paramTypes, TEE_Param params[4]);

/*
 * The macro TEE_PARAM_TYPES can be used to construct a value that you can
 * compare against an incoming paramTypes to check the type of all the
 * parameters in one comparison, like in the following example:
 * if (paramTypes != TEE_PARAM_TYPES(TEE_PARAM_TYPE_MEMREF_INPUT,
 *                                  TEE_PARAM_TYPE_MEMREF_OUPUT,
 *                                  TEE_PARAM_TYPE_NONE, TEE_PARAM_TYPE_NONE)) {
 *      return TEE_ERROR_BAD_PARAMETERS;
 *  }
 */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.6.1 Content of paramTypes Argument
 */
#define TEE_PARAM_TYPES(t0, t1, t2, t3)     ((t0) | ((t1) << 4) | ((t2) << 8) | ((t3) << 12))

/*
 * The macro TEE_PARAM_TYPE_GET can be used to extract the type of a given
 * parameter from paramTypes if you need more fine-grained type checking.
 */
/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.3.6.1 Content of paramTypes Argument
 */
#define TEE_PARAM_TYPE_GET(t, i)            ((((uint32_t)(t)) >> ((i) * 4)) & 0xF)

/*
 * The macro TEE_PARAM_TYPE_SET can be used to load the type of a given
 * parameter from paramTypes without specifying all types (TEE_PARAM_TYPES)
 */
#define TEE_PARAM_TYPE_SET(t, i)            (((uint32_t)(t) & 0xF) << ((i) * 4))


//////////////// 4.11 Memory Management Functions ////////////////

/**
 * Table 4-17: Valid Hint Values
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 4.11.4 TEE_Malloc
 */
#define TEE_MALLOC_FILL_ZERO               0x00000000


//////////////////////////////// 5 Trusted Storage API for Data and Keys ////////////////////////////////

//////////////// 5.3 Data Types ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.3.1 TEE_Attribute
 */
typedef struct {
    uint32_t attributeID;
    union {
        struct {
            void *buffer;
            uint32_t length;
        } ref;
        struct {
            uint32_t a, b;
        } value;
    } content;
} TEE_Attribute;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.3.2 TEE_ObjectInfo
 */
typedef struct {
    uint32_t objectType;
    uint32_t objectSize;
    uint32_t maxObjectSize;
    uint32_t objectUsage;
    uint32_t dataSize;
    uint32_t dataPosition;
    uint32_t handleFlags;
} TEE_ObjectInfo;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.3.3 TEE_Whence
 */
typedef enum {
    TEE_DATA_SEEK_SET = 0,
    TEE_DATA_SEEK_CUR = 1,
    TEE_DATA_SEEK_END = 2
} TEE_Whence;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.3.4 TEE_ObjectHandle
 */
typedef struct __TEE_ObjectHandle *TEE_ObjectHandle;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.3.5 TEE_ObjectEnumHandle
 */
typedef struct __TEE_ObjectEnumHandle *TEE_ObjectEnumHandle;

//////////////// 5.4 Constants ////////////////

/**
 * Table 5-2: Object Storage Constants
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.4.1 Constants Used in Trusted Storage API for Data and Keys
 */
#define TEE_STORAGE_PRIVATE                0x00000001

/**
 * Table 5-3: Data Flag Constants
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.4.1 Constants Used in Trusted Storage API for Data and Keys
 */
#define TEE_DATA_FLAG_ACCESS_READ          0x00000001
#define TEE_DATA_FLAG_ACCESS_WRITE         0x00000002
#define TEE_DATA_FLAG_ACCESS_WRITE_META    0x00000004
#define TEE_DATA_FLAG_SHARE_READ           0x00000010
#define TEE_DATA_FLAG_SHARE_WRITE          0x00000020
#define TEE_DATA_FLAG_OVERWRITE            0x00000400

/**
 * Table 5-4: Usage Constants
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.4.1 Constants Used in Trusted Storage API for Data and Keys
 */
#define TEE_USAGE_EXTRACTABLE              0x00000001
#define TEE_USAGE_ENCRYPT                  0x00000002
#define TEE_USAGE_DECRYPT                  0x00000004
#define TEE_USAGE_MAC                      0x00000008
#define TEE_USAGE_SIGN                     0x00000010
#define TEE_USAGE_VERIFY                   0x00000020
#define TEE_USAGE_DERIVE                   0x00000040

/**
 * Table 5-4b: Miscellaneous Constants
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.4.1 Constants Used in Trusted Storage API for Data and Keys
 */
#define TEE_DATA_MAX_POSITION              0xFFFFFFFF
#define TEE_OBJECT_ID_MAX_LEN              64

/**
 * Table 5-5: Handle Flag Constants
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.4.2 Constants Used in Cryptographic Operations API
 */
#define TEE_HANDLE_FLAG_PERSISTENT         0x00010000
#define TEE_HANDLE_FLAG_INITIALIZED        0x00020000
#define TEE_HANDLE_FLAG_KEY_SET            0x00040000
#define TEE_HANDLE_FLAG_EXPECT_TWO_KEYS    0x00080000

/**
 * Table 5-6: Operation Constants
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 5.4.2 Constants Used in Cryptographic Operations API
 */
#define TEE_OPERATION_CIPHER               1
#define TEE_OPERATION_MAC                  3
#define TEE_OPERATION_AE                   4
#define TEE_OPERATION_DIGEST               5
#define TEE_OPERATION_ASYMMETRIC_CIPHER    6
#define TEE_OPERATION_ASYMMETRIC_SIGNATURE 7
#define TEE_OPERATION_KEY_DERIVATION       8
#define TEE_OPERATION_EXTENSION            0xF
#define TEE_OPERATION_STATE_INITIAL        0x00000000
#define TEE_OPERATION_STATE_ACTIVE         0x00000001


//////////////////////////////// 6 Cryptographic Operations API ////////////////////////////////

//////////////// 6.1 Data Types ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.1.1 TEE_OperationMode
 */
typedef enum {
    TEE_MODE_ENCRYPT = 0,
    TEE_MODE_DECRYPT = 1,
    TEE_MODE_SIGN = 2,
    TEE_MODE_VERIFY = 3,
    TEE_MODE_MAC = 4,
    TEE_MODE_DIGEST = 5,
    TEE_MODE_DERIVE = 6
} TEE_OperationMode;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.1.2 TEE_OperationInfo
 */
typedef struct {
    uint32_t algorithm;
    uint32_t operationClass;
    uint32_t mode;
    uint32_t digestLength;
    uint32_t maxKeySize;
    uint32_t keySize;
    uint32_t requiredKeyUsage;
    uint32_t handleState;
} TEE_OperationInfo;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.1.3 TEE_OperationInfoMultiple
 */
typedef struct {
    uint32_t keySize;
    uint32_t requiredKeyUsage;
} TEE_OperationInfoKey;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.1.3 TEE_OperationInfoMultiple
 */
typedef struct {
    uint32_t algorithm;
    uint32_t operationClass;
    uint32_t mode;
    uint32_t digestLength;
    uint32_t maxKeySize;
    uint32_t handleState;
    uint32_t operationState;
    uint32_t numberOfKeys;
    TEE_OperationInfoKey keyInformation[];
} TEE_OperationInfoMultiple;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.1.4 TEE_OperationHandle
 */
typedef struct __TEE_OperationHandle *TEE_OperationHandle;

//////////////// 6.10 Cryptographic Algorithms Specification ////////////////

/**
 * Table 6-11: List of Algorithm Identifiers
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.10.1 List of Algorithm Identifiers
 */
#define TEE_ALG_AES_ECB_NOPAD                   0x10000010
#define TEE_ALG_AES_CBC_NOPAD                   0x10000110
#define TEE_ALG_AES_CTR                         0x10000210
#define TEE_ALG_AES_CTS                         0x10000310
#define TEE_ALG_AES_XTS                         0x10000410
#define TEE_ALG_AES_CBC_MAC_NOPAD               0x30000110
#define TEE_ALG_AES_CBC_MAC_PKCS5               0x30000510
#define TEE_ALG_AES_CMAC                        0x30000610
#define TEE_ALG_AES_CCM                         0x40000710
#define TEE_ALG_AES_GCM                         0x40000810
#define TEE_ALG_DES_ECB_NOPAD                   0x10000011
#define TEE_ALG_DES_CBC_NOPAD                   0x10000111
#define TEE_ALG_DES_CBC_MAC_NOPAD               0x30000111
#define TEE_ALG_DES_CBC_MAC_PKCS5               0x30000511
#define TEE_ALG_DES3_ECB_NOPAD                  0x10000013
#define TEE_ALG_DES3_CBC_NOPAD                  0x10000113
#define TEE_ALG_DES3_CBC_MAC_NOPAD              0x30000113
#define TEE_ALG_DES3_CBC_MAC_PKCS5              0x30000513
#define TEE_ALG_RSASSA_PKCS1_V1_5_MD5           0x70001830
#define TEE_ALG_RSASSA_PKCS1_V1_5_SHA1          0x70002830
#define TEE_ALG_RSASSA_PKCS1_V1_5_SHA224        0x70003830
#define TEE_ALG_RSASSA_PKCS1_V1_5_SHA256        0x70004830
#define TEE_ALG_RSASSA_PKCS1_V1_5_SHA384        0x70005830
#define TEE_ALG_RSASSA_PKCS1_V1_5_SHA512        0x70006830
#define TEE_ALG_RSASSA_PKCS1_V1_5_MD5SHA1       0x7000F830
#define TEE_ALG_RSASSA_PKCS1_PSS_MGF1_SHA1      0x70212930
#define TEE_ALG_RSASSA_PKCS1_PSS_MGF1_SHA224    0x70313930
#define TEE_ALG_RSASSA_PKCS1_PSS_MGF1_SHA256    0x70414930
#define TEE_ALG_RSASSA_PKCS1_PSS_MGF1_SHA384    0x70515930
#define TEE_ALG_RSASSA_PKCS1_PSS_MGF1_SHA512    0x70616930
#define TEE_ALG_RSAES_PKCS1_V1_5                0x60000130
#define TEE_ALG_RSAES_PKCS1_OAEP_MGF1_SHA1      0x60210230
#define TEE_ALG_RSAES_PKCS1_OAEP_MGF1_SHA224    0x60310230
#define TEE_ALG_RSAES_PKCS1_OAEP_MGF1_SHA256    0x60410230
#define TEE_ALG_RSAES_PKCS1_OAEP_MGF1_SHA384    0x60510230
#define TEE_ALG_RSAES_PKCS1_OAEP_MGF1_SHA512    0x60610230
#define TEE_ALG_RSA_NOPAD                       0x60000030
#define TEE_ALG_DSA_SHA1                        0x70002131
#define TEE_ALG_DSA_SHA224                      0x70003131
#define TEE_ALG_DSA_SHA256                      0x70004131
#define TEE_ALG_DH_DERIVE_SHARED_SECRET         0x80000032
#define TEE_ALG_MD5                             0x50000001
#define TEE_ALG_SHA1                            0x50000002
#define TEE_ALG_SHA224                          0x50000003
#define TEE_ALG_SHA256                          0x50000004
#define TEE_ALG_SHA384                          0x50000005
#define TEE_ALG_SHA512                          0x50000006
#define TEE_ALG_MD5SHA1                         0x5000000F
#define TEE_ALG_HMAC_MD5                        0x30000001
#define TEE_ALG_HMAC_SHA1                       0x30000002
#define TEE_ALG_HMAC_SHA224                     0x30000003
#define TEE_ALG_HMAC_SHA256                     0x30000004
#define TEE_ALG_HMAC_SHA384                     0x30000005
#define TEE_ALG_HMAC_SHA512                     0x30000006

/**
 * Table 6-13: List of Object Types
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.10.2 Object Types
 */
#define TEE_TYPE_AES                        0xA0000010
#define TEE_TYPE_DES                        0xA0000011
#define TEE_TYPE_DES3                       0xA0000013
#define TEE_TYPE_HMAC_MD5                   0xA0000001
#define TEE_TYPE_HMAC_SHA1                  0xA0000002
#define TEE_TYPE_HMAC_SHA224                0xA0000003
#define TEE_TYPE_HMAC_SHA256                0xA0000004
#define TEE_TYPE_HMAC_SHA384                0xA0000005
#define TEE_TYPE_HMAC_SHA512                0xA0000006
#define TEE_TYPE_RSA_PUBLIC_KEY             0xA0000030
#define TEE_TYPE_RSA_KEYPAIR                0xA1000030
#define TEE_TYPE_DSA_PUBLIC_KEY             0xA0000031
#define TEE_TYPE_DSA_KEYPAIR                0xA1000031
#define TEE_TYPE_DH_KEYPAIR                 0xA1000032
#define TEE_TYPE_ECDSA_PUBLIC_KEY           0xA0000041
#define TEE_TYPE_ECDSA_KEYPAIR              0xA1000041
#define TEE_TYPE_ECDH_PUBLIC_KEY            0xA0000042
#define TEE_TYPE_ECDH_KEYPAIR               0xA1000042
#define TEE_TYPE_GENERIC_SECRET             0xA0000000
#define TEE_TYPE_CORRUPTED_OBJECT           0xA00000BE
#define TEE_TYPE_DATA                       0xA00000BF

/**
 * Table 6-14: List of Supported ECC Curves
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.10.3 Elliptic Curve Types
 */
#define TEE_ECC_CURVE_NIST_P192             0x00000001
#define TEE_ECC_CURVE_NIST_P224             0x00000002
#define TEE_ECC_CURVE_NIST_P256             0x00000003
#define TEE_ECC_CURVE_NIST_P384             0x00000004
#define TEE_ECC_CURVE_NIST_P521             0x00000005

/**
 * Table 6-15: Object or Operation Attributes
 *
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 6.11 Object or Operation Attributes
 */
#define TEE_ATTR_SECRET_VALUE               0xC0000000
#define TEE_ATTR_RSA_MODULUS                0xD0000130
#define TEE_ATTR_RSA_PUBLIC_EXPONENT        0xD0000230
#define TEE_ATTR_RSA_PRIVATE_EXPONENT       0xC0000330
#define TEE_ATTR_RSA_PRIME1                 0xC0000430
#define TEE_ATTR_RSA_PRIME2                 0xC0000530
#define TEE_ATTR_RSA_EXPONENT1              0xC0000630
#define TEE_ATTR_RSA_EXPONENT2              0xC0000730
#define TEE_ATTR_RSA_COEFFICIENT            0xC0000830
#define TEE_ATTR_DSA_PRIME                  0xD0001031
#define TEE_ATTR_DSA_SUBPRIME               0xD0001131
#define TEE_ATTR_DSA_BASE                   0xD0001231
#define TEE_ATTR_DSA_PUBLIC_VALUE           0xD0000131
#define TEE_ATTR_DSA_PRIVATE_VALUE          0xC0000231
#define TEE_ATTR_DH_PRIME                   0xD0001032
#define TEE_ATTR_DH_SUBPRIME                0xD0001132
#define TEE_ATTR_DH_BASE                    0xD0001232
#define TEE_ATTR_DH_X_BITS                  0xF0001332
#define TEE_ATTR_DH_PUBLIC_VALUE            0xD0000132
#define TEE_ATTR_DH_PRIVATE_VALUE           0xC0000232
#define TEE_ATTR_RSA_OAEP_LABEL             0xD0000930
#define TEE_ATTR_RSA_PSS_SALT_LENGTH        0xF0000A30
#define TEE_ATTR_ECC_PUBLIC_VALUE_X         0xD0000141
#define TEE_ATTR_ECC_PUBLIC_VALUE_Y         0xD0000241
#define TEE_ATTR_ECC_PRIVATE_VALUE          0xC0000341
#define TEE_ATTR_ECC_CURVE                  0xF0000441

#define TEE_ATTR_BIT_PROTECTED              (1 << 28)
#define TEE_ATTR_BIT_VALUE                  (1 << 29)




//////////////////////////////// 7 Time API ////////////////////////////////

//////////////// 7.1 Data Types ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 7.1.1 TEE_Time
 */
typedef struct {
    uint32_t seconds;
    uint32_t millis;
} TEE_Time;

//////////////// 7.2 Time Functions ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 7.2.2 TEE_Wait
 */
#define TEE_TIMEOUT_INFINITE              0xFFFFFFFF


//////////////////////////////// 8 TEE Arithmetical API ////////////////////////////////

//////////////// 8.3 Data Types ////////////////

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 8.3.1 TEE_BigInt
 */
typedef uint32_t TEE_BigInt;

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 8.3.2 TEE_BigIntFMMContext
 */
// typedef uint32_t TEE_BigIntFMMContext __aligned(__alignof__(void *));

/**
 * @see     [TEE_INTERNAL_CORE_API_SPEC] 中的 8.3.3 TEE_BigIntFMM
 */
typedef uint32_t TEE_BigIntFMM;




/* Other definitions */
typedef uint32_t TEE_ErrorOrigin;
typedef void *TEE_Session;


/* Not specified in the standard */
#define TEE_NUM_PARAMS  4

/* TEE Arithmetical APIs */

#define TEE_BigIntSizeInU32(n) ((((n)+31)/32)+2)




#endif /* __TEE_INTERNAL_API_H__ */
