#include <stdio.h>
#include <string.h>
#include "ota.h"
#include "iot_interface.h"

int cloud_is_connected = 0;

//  第一部分 [系统函数]  /////////////////////////////////////////////////////

/*
 * 打印日志函数，请自行实现，可根据需要重定向到串口、屏幕、文件等位置
 */
void iot_print(const char *str)
{
    printf(str);
}

/*
 * 当设备连接状态发生改变时，sdk自动调用此函数
 */
void iot_status_cb( DEV_STATUS_T dev_status, iot_u32 timestamp )
{
    iot_u32 seq = 0;
    printf("device state:%d, ts:%d\r\n", dev_status, timestamp);

    if( dev_status == DEV_STA_CONNECTED_CLOUD )
    {
    //如开发硬件单品，可删除以下宏定义内子设备相关的代码
    #ifdef IOT_DEVICE_IS_GATEWAY
        //连接云端成功后，应将网关下已经连接的子设备上报一次
        //有4个设备已连接到网关，这里添加4个子设备
        sub_dev_add("TEMP000001", "Thermometer", "01.01", 1);
        sub_dev_add("TEMP000002", "Thermometer", "01.01", 1);
        sub_dev_add("FAN0000001", "electric fan", "01.01", 2);
        sub_dev_add("FAN0000002", "electric fan", "01.01", 2);
        iot_sub_dev_active(&seq);
        
        //连接云端成功后，应将设备最新的各数据点数据全部上报一次
        
        //上报子设备温湿度传感器TEMP000001的数据
        dp_up_add_int(DP_ID_TEMP, 26);
        dp_up_add_int(DP_ID_HUMIDITY, 35);
        iot_upload_dps("TEMP000001", &seq);
        //上报子设备温湿度传感器TEMP000002的数据
        dp_up_add_int(DP_ID_TEMP, 28);
        dp_up_add_int(DP_ID_HUMIDITY, 33);
        iot_upload_dps("TEMP000002", &seq);
        
        //上报子设备风扇FAN0000001的风速
        dp_up_add_enum(DP_ID_WIND_SPD, 3);
        iot_upload_dps("FAN0000001", &seq);
        //上报子设备风扇FAN0000002的风速
        dp_up_add_enum(DP_ID_WIND_SPD, 4);
        iot_upload_dps("FAN0000002", &seq);
    #endif
        //上报网关自身的开关值
        dp_up_add_bool(DP_ID_DP_SWITCH, 1);
        iot_upload_dps(NULL, &seq);
        
        //连接云端成功
        cloud_is_connected = 1;

        //允许APP绑定，60秒有效
        printf("permit_bind\r\n");
        iot_status_set (DEV_STA_BIND_PERMIT,60);
    }
    else if( dev_status == DEV_STA_FACTORY_RESET )
    {
        //恢复出厂设置完成，请重启设备重新连接云端
        /* 示例
        system_reboot();
        */
    }
    else if(dev_status == DEV_STA_BIND_FAILED)
    {
        //用户绑定失败，允许绑定超时或者设备未联网
        printf("\r\nbind user timeout cb!\r\n");
    }
    else if(dev_status == DEV_STA_BIND_SUCCESS)
    {
        //用户绑定成功
        printf("\r\nbind user cb!\r\n");
    }
    else if(dev_status == DEV_STA_UNBIND)
    {
        //用户解绑成功
        printf("\r\n unbound user cb!\r\n");
    }
    else if(dev_status == DEV_STA_USER_SHARE_INC)
    {
        //分享用户增加
        printf("\r\n share user add cb!\r\n");
    } 
    else if(dev_status == DEV_STA_USER_SHARE_DEC)
    {
        //分享用户减少
        printf("\r\n share user delete cb!\r\n");
    }  
    else
    {
        //断开云端连接
        cloud_is_connected = 0;
    }
}

/*
 * 发送数据成功后，sdk自动调用下面这个回调函数
 */
void iot_data_cb ( iot_u32 data_seq )
{
    printf("iot_data_cb :%d.\r\n", data_seq);
}

/*
 * 当设备接收到云端下发的设备信息数据，sdk自动调用此函数
 */
void iot_info_cb(INFO_TYPE_T info_type, void * info)
{
    int i = 0;
    user_info_t master_info;
    share_info_t share_info;
    dev_info_t dev_info;
    ota_info_t ota_info;

    printf("\r\ninfo_type:%d\r\n", info_type);

    switch(info_type)
    {
        case INFO_TYPE_USER_MASTER:
            memcpy(&master_info, info, sizeof(user_info_t));
            printf("id:%s\r\n", master_info.id);
            printf("email:%s\r\n", master_info.email);
            printf("country_code:%s\r\n", master_info.country_code);
            printf("phone:%s\r\n", master_info.phone);
            printf("name:%s\r\n", master_info.name);
            break;
        case INFO_TYPE_USER_SHARE:
            memcpy(&share_info, info, sizeof(share_info_t));
            printf("count:%d\r\n", share_info.count);
            for(i = 0; i < share_info.count; i++)
            {
                printf("share user %d\r\n", i);
                printf("id:%s\r\n", share_info.user[i].id);
                printf("email:%s\r\n", share_info.user[i].email);
                printf("country_code:%s\r\n", share_info.user[i].country_code);
                printf("phone:%s\r\n", share_info.user[i].phone);
                printf("name:%s\r\n", share_info.user[i].name);
                printf("\r\n");
            }
            break;
        case INFO_TYPE_DEVICE:
            memcpy(&dev_info, info, sizeof(dev_info_t));
            printf("is_bind:%d,sdk_version:%s\r\n", dev_info.is_bind,dev_info.sdk_ver);
            break;
        case INFO_TYPE_OTA:
            memcpy(&ota_info, info, sizeof(ota_info_t));
            printf("owner:%d\r\n", ota_info.owner);
            printf("flen:%d\r\n", ota_info.flen);
            printf("ver:%s\r\n", ota_info.ota_ver);
            break;
        default:
            printf("info_type is error\r\n");
    }
}

//  第一部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第二部分 [传输数据]  /////////////////////////////////////////////////////

//  各下发数据点处理函数，用户请按需自行实现  /////////////////////////////////

void dp_down_handle_switch ( char sub_id[16], iot_u8* in_data, iot_u32 in_len )
{
    iot_u32 seq = 0;
    iot_u8 dp_switch  = bytes_to_bool(in_data);
    
    printf("dp_switch:%d\r\n", dp_switch);

    if(dp_switch == 0)
    {
        //关闭开关
    }
    else
    {
        //开启开关
    }
    //硬件操作完成后，应将更新后的状态上报一次
    dp_up_add_bool(DP_ID_DP_SWITCH, dp_switch);
    //上报网关自身的开关值
    iot_upload_dps(NULL, &seq);
}

void dp_down_handle_temp( char sub_id[16], iot_u8* in_data, iot_u32 in_len )
{
    int temp = bytes_to_int(in_data);
    printf("dp_temp:%d\r\n", temp);
}

void dp_down_handle_wind_spd ( char sub_id[16], iot_u8* in_data, iot_u32 in_len )
{
    iot_u32 seq;
    iot_u8 wind_spd = bytes_to_enum(in_data);
    printf("wind_spd:%d\r\n", wind_spd);
    
    //如开发硬件单品，可删除以下宏定义内子设备相关的代码
    #ifdef IOT_DEVICE_IS_GATEWAY
        if(sub_id == NULL)
        {
            //网关自身数据
            printf("gateway wind_spd:%d\r\n", wind_spd);
        }
        else
        {
            //比较sub id，控制相应的子设备
            if ( strncmp( sub_id, "FAN0000001", 16 ) == 0 )
            {
                printf("get FAN0000001 cmd\r\n");
            }
            else if ( strncmp( sub_id, "FAN0000002", 16 ) == 0 )
            {
                printf("get FAN0000002 cmd\r\n");
            }
            else
            {
                printf("subid err\r\n");
            }
        }
    #endif
   
    //风速切换操作
    switch(wind_spd)
    {
    case 0:
        break;
    case 1:
        break;
    case 2:
        break;
    case 3:
        break;            
    case 4:
        break;
    case 5:
        break;
    default:break;
    }
    //硬件操作完成后，应将更新后的状态上报一次
    dp_up_add_enum(DP_ID_WIND_SPD, wind_spd);
    
    iot_upload_dps(sub_id, &seq);
}
void dp_down_handle_pm25( char sub_id[16], iot_u8* in_data, iot_u32 in_len )
{
    float dp_pm25 = bytes_to_float(in_data);
    
    printf("dp_pm25:%f\r\n", dp_pm25);
}

void dp_down_handle_config ( char sub_id[16], iot_u8* in_data, iot_u32 in_len )
{
    printf("config:%s\r\n", in_data);
}

void dp_down_handle_bin ( char sub_id[16], iot_u8* in_data, iot_u32 in_len )
{
    int i = 0;
    
    printf("recv binary data[%d]:\r\n", in_len);
    
    for(i = 0; i < in_len;i++) 
    {
        printf("%02x ",in_data[i]);
    }
    printf("\r\n");
}

//  下发数据点处理函数结束  ////////////////////////////////////////////////////


//  下发数据点数据结构定义  ////////////////////////////////////////////////////

/*****************************************************************************
结构名称 : 下发数据点结构体数组
功能描述 : 用户请将需要处理的下发数据点，按数据点ID、数据点类型、
           数据点对应处理函数填写下面这个数组，并将处理函数实现
*****************************************************************************/
iot_download_dps_t iot_down_dps[] = 
{   /*  数据点ID     , 数据点类型  ,   处理函数               */
    { DP_ID_DP_SWITCH, DP_TYPE_BOOL,   dp_down_handle_switch   },
    { DP_ID_TEMP,      DP_TYPE_INT,    dp_down_handle_temp     },
    { DP_ID_PM25,      DP_TYPE_FLOAT,  dp_down_handle_pm25     },
    { DP_ID_WIND_SPD,  DP_TYPE_ENUM,   dp_down_handle_wind_spd },
    { DP_ID_CONFIG,    DP_TYPE_STRING, dp_down_handle_config   },
    { DP_ID_BIN,       DP_TYPE_BIN,    dp_down_handle_bin      }
};
/*****************************************************************************
结构名称 : DOWN_DPS_CNT整数
功能描述 : 下发数据点结构体数组中数据点数量，用户请勿更改
*****************************************************************************/
iot_u32 DOWN_DPS_CNT = sizeof(iot_down_dps)/sizeof(iot_download_dps_t);

//  下发数据点数据结构定义结束  /////////////////////////////////////////////////


//  第二部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第三部分 [固件升级]  /////////////////////////////////////////////////////

iot_s32 iot_ota_chunk_cb ( iot_u8  chunk_is_last, iot_u32 chunk_offset, iot_u32 chunk_size, const iot_s8*  chunk )
{
    printf("get chunk offset = %d, size = %d\r\n", chunk_offset, chunk_size);

    // 第一个数据包，准备升级空间
    if (chunk_offset == 0)
    {
        // 擦除flash，准备写入固件
        if (iotota_ca_write_new_firmware_begin() != 0)
        {
            return -1;
        }
    }

    //将固件包写入升级空间
    if (iotota_ca_write_new_firmware_data(chunk_offset, chunk, chunk_size) != 0)
    {
        return -1;
    }

    //最后一个数据包
    if (chunk_is_last)
    {
        //一般需要校验整个固件的合法性(具体看硬件平台实现)
        printf("get ota chunks over, wait to reboot...\r\n");

        if (iotota_ca_write_new_firmware_end() != 0)
        {
            return -1;
        }
    }

    return ACK_OK;
}

void iot_ota_upgrade_cb ( void )
{
    printf("get ota cmd success,reboot right now\r\n");
    // 处理收到的升级指令，需重启，运行新版本固件

    if (iotota_ca_reboot() != 0)
    {
        printf("failed to reboot\r\n");
    }
}

//  第三部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第四部分 [高级功能]  /////////////////////////////////////////////////////

void iot_rx_data_cb ( char sub_id[16], iot_u64 data_timestamp, iot_u8* data, iot_u32 data_len )
{
    iot_u32 seq = 0;
    int i = 0;

    printf("receive custom data[%d], timestamp:[%llu]\r\n", data_len, data_timestamp);

    //如开发硬件单品，可删除以下宏定义内子设备相关的代码
    #ifdef IOT_DEVICE_IS_GATEWAY
        if(sub_id == NULL)
        {
            //网关自身数据
            printf("gateway data :\r\n");
        }
        else
        {
            printf("sub device [%s] data :\r\n", sub_id);
        }
    #endif
    
    //以十六进制格式打印数据，如数据类型为字符串，请用%s自行打印
    for(i = 0; i < data_len; i++) 
    {
        printf("%02x ",data[i]);
    }
    printf("\r\n");
    
    //透传一条数据至手机端，云端不保存
    iot_tx_data(sub_id, &seq, data, data_len);
}

//  第四部分结束  ////////////////////////////////////////////////////////////

//end iot_interface.c

