#ifndef __IOT_INTERFACE_H__
#define __IOT_INTERFACE_H__

#ifdef __cplusplus
extern "C" {
#endif

/** 返回值 **/
#define ACK_OK              0
#define ACK_ERR             -1

//如开发网关类产品，可开启此宏定义注释，测试代码中的样例
#define IOT_DEVICE_IS_GATEWAY

/*****************************************************************************
* 数据点ID，由云平台生成，请将对应ID设置成宏定义，方便使用
*****************************************************************************/
#define DP_ID_DP_SWITCH     1
#define DP_ID_TEMP          2
#define DP_ID_PM25          3
#define DP_ID_WIND_SPD      4
#define DP_ID_FAULT         5
#define DP_ID_CONFIG        6
#define DP_ID_BIN           7
#define DP_ID_HUMIDITY      8

typedef unsigned char       iot_u8;
typedef signed char         iot_s8;
typedef unsigned short      iot_u16;
typedef signed short        iot_s16;
typedef unsigned int        iot_u32;
typedef signed int          iot_s32;
typedef unsigned long long  iot_u64;
typedef signed long long    iot_s64;
typedef float               iot_f32;
typedef double              iot_f64;

typedef enum DP_TYPE {
    DP_TYPE_INT,
    DP_TYPE_BOOL,
    DP_TYPE_ENUM,
    DP_TYPE_STRING,
    DP_TYPE_FLOAT,
    DP_TYPE_FAULT,
    DP_TYPE_BIN
}DP_TYPE_t;

// product_key 的配置见 iot_interface_secure.h 文件中的 iot_context_secure 结构体
struct iot_context {
    iot_u32 product_id;         // 平台生成，4 字节的无符号整型数字，全网唯一产品 ID
    char    *mcu_version;       // MCU 固件版本，"xx.xx", 0≤x≤9
    iot_s32 recvbuf_size;       // 接收数据 buffer 大小，范围 1024-4096, 默认 1024
    iot_s32 sendbuf_size;       // 发送数据 buffer 大小，范围 1024-4096, 默认 1024
    char    *server_addr;       // 云服务器地址
    iot_u16 server_port;        // 云服务器端口
};

typedef enum DEV_STATUS {
    DEV_STA_CONNECTED_CLOUD   = 0,  //连接云端成功
    DEV_STA_DISCONN_CLOUD     = 1,  //网络异常连接断开
    DEV_STA_FACTORY_RESET     = 2,  //恢复出厂设置
    DEV_STA_BIND_PERMIT       = 3,  //允许APP用户绑定设备
    DEV_STA_BIND_FAILED       = 4,  //绑定失败（允许绑定状态超时或者设备未联网）
    DEV_STA_BIND_SUCCESS      = 5,  //设备与APP用户绑定成功
    DEV_STA_UNBIND            = 6,  //设备与APP用户解绑成功
    DEV_STA_USER_SHARE_INC    = 7,  //分享用户增加
    DEV_STA_USER_SHARE_DEC    = 8,  //分享用户减少
    DEV_STA_AUTH_LIMIT        = 11, //设备授权数达上限，无法通过云端认证
    DEV_STA_MAC_BIND_OTHER_ID = 12  //设备mac地址已绑定其他产品，无法通过云端认证
}DEV_STATUS_T;

typedef enum INFO_TYPE {
    INFO_TYPE_USER_MASTER,          //绑定用户信息
    INFO_TYPE_USER_SHARE,           //分享用户信息
    INFO_TYPE_DEVICE,               //设备信息
    INFO_TYPE_OTA,                  //OTA信息
}INFO_TYPE_T;

typedef enum CONFIG {
    CONFIG_ENCRYPT_TYPE,            //加解密类型
    CONFIG_SN,                      //设备序列号
}CONFIG_T;

typedef struct user_info {
    char * id;                      //用户ID
    char * email;                   //用户email
    char * country_code;            //用户手机号的国家代码，比如中国是0086
    char * phone;                   //用户手机号
    char * name;                    //用户昵称
}user_info_t;

typedef struct share_info {
    iot_s32      count;             //分享用户的数量
    user_info_t* user;              //分享用户信息结构体的列表
}share_info_t;

typedef struct dev_info {
    iot_u8 is_bind;                 //设备是否被绑定
    char   sdk_ver[6];              //SDK版本号
}dev_info_t;

typedef struct ota_info {
    iot_u8  owner;                  //固件类型，0：wifi固件，1：MCU固件
    iot_u32 flen;                   //固件文件大小
    char    ota_ver[6];             //OTA固件版本号
}ota_info_t;

/*获取时间的结构体*/
struct s_time {
	int sec;                        //秒
    int min;                        //分
    int hour;                       //小时
    int day;                        //日
    int mon;                        //月
    int year;                       //年
    int week;
};

//  第一部分 [系统函数]  /////////////////////////////////////////////////////

/*****************************************************************************
函数名称 : iot_start
功能描述 : 初始化设备与云端交互的上下文环境，sdk与云端会建立长连接
传入参数 : ctx : 云服务上下文
传出参数 : 无
返回参数 : -1 : 失败
           0  : 成功
*****************************************************************************/
extern iot_s32 iot_start(struct iot_context* ctx);

/*****************************************************************************
函数名称 : iot_status_set
功能描述 : 设置设备运行状态
传入参数 : 无
传出参数 : dev_status : DEV_STA_FACTORY_RESET    : 恢复出厂设置，设备会删除云端配置，解除与APP端的绑定关系
                        DEV_STA_BIND_PERMIT      : 允许用户通过APP搜索绑定设备
                        DEV_STA_UNBIND           ：主动解除与APP的绑定关系
           timeout    : 设备运行状态保持时长，单位为秒;timeout = 0 表示不设置超时状态.
返回参数 : 无
*****************************************************************************/
extern void iot_status_set( DEV_STATUS_T dev_status, iot_u32 timeout );
/*****************************************************************************
函数名称 : iot_status_cb
功能描述 : 当设备连接状态发生改变时，sdk自动调用此函数
传入参数 : 无
传出参数 : dev_status : DEV_STA_CONNECTED_CLOUD   : 连接云端成功
                        DEV_STA_DISCONN_CLOUD     : 网络异常连接断开
                        DEV_STA_FACTORY_RESET     : 恢复出厂设置，设备会解除与APP端的绑定关系
                        DEV_STA_BIND_FAILED       : 绑定失败（绑定超时或者设备未联网）
                        DEV_STA_BIND_SUCCESS      : 绑定成功
                        DEV_STA_UNBIND            : 解绑成功
                        DEV_STA_USER_SHARE_INC    : 分享用户增加
                        DEV_STA_USER_SHARE_DEC    : 分享用户减少
                        DEV_STA_AUTH_LIMIT        : 设备授权数达上限，无法通过云端认证
                        DEV_STA_MAC_BIND_OTHER_ID : 设备mac地址已绑定其他产品，无法通过云端认证
           timestamp  : 状态改变的时间点
返回参数 : 无
*****************************************************************************/
extern void iot_status_cb( DEV_STATUS_T dev_status, iot_u32 timestamp );
/*****************************************************************************
函数名称 : iot_get_onlinetime
功能描述 : 获取网络时间
传入参数 : 无
传出参数 : 无
返回参数 : 0  : 时间无效
           >0 : 实时网络时间戳
*****************************************************************************/
extern iot_u32 iot_get_onlinetime( void );
/*****************************************************************************
函数名称 : iot_print
功能描述 : 打印输出函数，可根据需要重定向到串口、屏幕、文件等位置
传入参数 : str : 输出的日志信息
传出参数 : 无
返回参数 : 无
*****************************************************************************/
extern void iot_print(const char* str);
/*****************************************************************************
函数名称 : iot_get_info
功能描述 : 请求获取设备信息
传入参数 : info_type : INFO_TYPE_USER_MASTER       : 绑定用户的信息，包括用户ID、手机号、邮箱等
                       INFO_TYPE_USER_SHARE        : 分享用户的信息，包括用户ID、手机号、邮箱等
                       INFO_TYPE_DEVICE            : 设备的详细信息，包括绑定状态,版本信息等
传出参数 : 无
返回参数 : -1 : 失败
           0  : 成功
*****************************************************************************/
extern iot_s32 iot_get_info(INFO_TYPE_T info_type);
/*****************************************************************************
函数名称 : iot_info_cb
功能描述 : 收到云端设备信息数据后，sdk自动调用下面这个回调函数
传入参数 : 无
传出参数 : info_type : 内容同上个函数，额外还有一个消息类型
                       INFO_TYPE_OTA               ：OTA下载固件的信息，包括固件类型、固件总大小、固件版本号
           info      : 请求消息类型的数据内容
返回参数 : -1 : 失败
           0  : 成功
*****************************************************************************/
extern void iot_info_cb( INFO_TYPE_T info_type, void* info );
/*****************************************************************************
函数名称 : iot_data_cb
功能描述 : 发送数据成功后，sdk自动调用下面这个回调函数
传入参数 : 无
传出参数 : data_seq : 某条数据的序列号
返回参数 : 无
*****************************************************************************/
extern void iot_data_cb ( iot_u32 data_seq );

/*****************************************************************************
函数名称 : iot_parse_timestamp
功能描述 : 将时间戳转化为时间结构体
传入参数 : 无
传出参数 : tick:时间戳  st:时间结构体
返回参数 : 无
*****************************************************************************/
extern void iot_parse_timestamp(iot_u32 tick, struct s_time* st);
//  第一部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第二部分 [传输数据]  /////////////////////////////////////////////////////

/*****************************************************************************
函数名称 : dp_up_add_int
功能描述 : 添加整型数据点到发送队列
传入参数 : dpid  : 范围1-200，请与云端创建的数据点对应
           value : 范围-2147483647到2147483647，数据点对应的整数值
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_int ( iot_u8 dpid, iot_s32 value );
/*****************************************************************************
函数名称 : dp_up_add_bool
功能描述 : 添加布尔数据点到发送队列
传入参数 : dpid  : 范围1-200，请与云端创建的数据点对应
           value : 范围0-1，数据点对应的布尔数值
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_bool ( iot_u8 dpid, iot_u8 value );
/*****************************************************************************
函数名称 : dp_up_add_enum
功能描述 : 添加枚举型数据点到发送队列
传入参数 : dpid  : 范围1-200，请与云端创建的数据点对应
           value : 范围0-255，数据点对应的枚举数值
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_enum ( iot_u8 dpid, iot_u8 value );
/*****************************************************************************
函数名称 : dp_up_add_float
功能描述 : 添加浮点型数据点到发送队列
传入参数 : dpid  : 范围1-200，请与云端创建的数据点对应
           value : 数据点对应的浮点数值，32位浮点数
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_float ( iot_u8 dpid, iot_f32 value );
/*****************************************************************************
函数名称 : dp_up_add_string
功能描述 : 添加字符串数据点到发送队列
传入参数 : dpid    : 范围1-200，请与云端创建的数据点对应
           str     : 数据点对应的字符串数据
           str_len : 字符串的长度
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_string ( iot_u8 dpid, const char* str, iot_u32 str_len );
/*****************************************************************************
函数名称 : dp_up_add_fault
功能描述 : 添加故障数据点到发送队列
传入参数 : dpid      : 范围1-200，请与云端创建的数据点对应
           fault     : 数据点对应的故障字符串数据
           fault_len : 故障字符串的长度
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_fault ( iot_u8 dpid, const char* fault, iot_u32 fault_len );
/*****************************************************************************
函数名称 : dp_up_add_binary
功能描述 : 添加二进制数据点到发送队列
传入参数 : dpid    : 范围1-200，请与云端创建的数据点对应
           bin     : 数据点对应的二进制数据
           bin_len : 二进制数据的长度
传出参数 : 无
返回参数 :  0 : 添加成功
           -1 : 添加失败
*****************************************************************************/
extern iot_s32 dp_up_add_binary( iot_u8 dpid, const iot_u8* bin, iot_u32 bin_len );
/*****************************************************************************
函数名称 : iot_upload_dps
功能描述 : 上传一条数据，目前一条数据支持同时上传15个数据点
传入参数 : sub_id   : 自定义的子设备id，仅限字母数字组合
传出参数 : data_seq : 本条数据的序列号
返回参数 : 无
*****************************************************************************/
extern iot_s32 iot_upload_dps ( const char sub_id[16], iot_u32* data_seq );
/*****************************************************************************
结构名称 : dp_down_handle_t类型函数指针
功能描述 : 处理下发数据点函数类型
传入参数 : sub_id  : 自定义的子设备id，仅限字母数字组合
           in_data : 收到云端下发的数据
           in_len  : 数据长度
传出参数 : 无
返回参数 : 无
*****************************************************************************/
typedef void ( *dp_down_handle_t ) ( char sub_id[16], iot_u8* in_data, iot_u32 in_len );

/*****************************************************************************
结构名称 : struct iot_download_dps
功能描述 : 下发数据点结构体
参    数 : dpid           : 范围1-200，请与云端创建的数据点对应
           dptype         : 某数据点对应的数据类型
           dp_down_handle : dp_down_handle_t类型函数指针，某数据点对应的处理函数
返回参数 : 无
*****************************************************************************/
typedef struct iot_download_dps
{
    iot_u8 dpid;
    iot_u8 dptype;
    dp_down_handle_t dp_down_handle;
}iot_download_dps_t;
/*****************************************************************************
结构名称 : iot_down_dps数组
功能描述 : 下发数据点结构体数组
*****************************************************************************/
extern iot_download_dps_t iot_down_dps[];
/*****************************************************************************
结构名称 : DOWN_DPS_CNT整数
功能描述 : 下发数据点结构体数组中数据点数量
*****************************************************************************/
extern iot_u32 DOWN_DPS_CNT;
/*****************************************************************************
函数名称 : bytes_to_int
功能描述 : 将4字节数组转换成整型
传入参数 : bytes[4] : 字节数据
传出参数 : 无
返回参数 : 整数值
*****************************************************************************/
extern iot_s32 bytes_to_int ( const iot_u8 bytes[4] );
/*****************************************************************************
函数名称 : bytes_to_bool
功能描述 : 将1字节数组转换成布尔数值
传入参数 : bytes[1] : 字节数据
传出参数 : 无
返回参数 : 布尔数值
*****************************************************************************/
extern iot_u8 bytes_to_bool ( const iot_u8 bytes[1] );
/*****************************************************************************
函数名称 : bytes_to_enum
功能描述 : 将1字节数组转换成枚举
传入参数 : bytes[1] : 字节数据
传出参数 : 无
返回参数 : 枚举数值
*****************************************************************************/
extern iot_u8 bytes_to_enum ( const iot_u8 bytes[1] );
/*****************************************************************************
函数名称 : bytes_to_float
功能描述 : 将4字节数组转换成浮点数
传入参数 : bytes[4] : 字节数据
传出参数 : 无
返回参数 : 浮点数值
*****************************************************************************/
extern iot_f32 bytes_to_float ( const iot_u8 bytes[4] );


//  第二部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第三部分 [固件升级]  /////////////////////////////////////////////////////

/*****************************************************************************
函数名称 : iot_ota_option_set
功能描述 : 设置ota属性
传入参数 : expect_time : 期望升级倒计时，单位秒，范围120-3600
           chunk_size  : 固件块大小，2的n次幂，范围1024-1048576，默认1024
传出参数 : 无
返回参数 :  0 : 设置成功
           -1 : 设置失败，参数错误
*****************************************************************************/
extern iot_s32 iot_ota_option_set ( iot_u32 expect_time, iot_u32 chunk_size );
/*****************************************************************************
函数名称 : iot_ota_chunk_cb
功能描述 : 上传数据成功后，sdk自动调用下面这个回调函数
传入参数 : 无
传出参数 : chunk_is_last : 0 : 不是最后一个分块
                           1 : 是最后一个分块，固件传输结束
           chunk_offset  : 本数据块相对于完整固件的偏移量
           chunk_size    : 本数据块的长度
           chunk         : 固件数据块
返回参数 :  0 : 写入固件块成功
           -1 : 写入固件块失败
*****************************************************************************/
extern iot_s32 iot_ota_chunk_cb ( iot_u8  chunk_is_last,
                                  iot_u32 chunk_offset,
                                  iot_u32 chunk_size,
                                  const iot_s8*  chunk );
/*****************************************************************************
函数名称 : iot_ota_upgrade_cb
功能描述 : 收到此命令，重启，运行新版本固件
传入参数 : 无
传出参数 : 无
返回参数 : 无
*****************************************************************************/
extern void iot_ota_upgrade_cb ( void );

//  第三部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第四部分 [高级功能]  /////////////////////////////////////////////////////

/*****************************************************************************
函数名称 : iot_tx_data
功能描述 : 透传自定义数据
传入参数 : sub_id   : 自定义的子设备id，仅限字母数字组合
           data     : 自定义字符，支持字符串、二进制数据，特殊字符数据
           data_len : 自定义数据长度
传出参数 : data_seq : 本条数据的序列号
返回参数 :  0 : 成功
           -1 : 失败
*****************************************************************************/
extern iot_s32 iot_tx_data ( const char sub_id[16], iot_u32* data_seq, const iot_u8* data, iot_u32 data_len );
/*****************************************************************************
函数名称 : iot_rx_data_cb
功能描述 : 收到云端透传数据后，sdk自动调用下面这个回调函数
传入参数 : 无
传出参数 : sub_id         : 自定义的子设备id，仅限字母数字组合
           data_timestamp : 时间戳，云端收到APP数据时的时间点
           data           : 自定义数据，仅支持字符型
           data_len       : 自定义数据长度
返回参数 : 无
*****************************************************************************/
extern void iot_rx_data_cb ( char sub_id[16], iot_u64 data_timestamp, iot_u8* data, iot_u32 data_len );
/*****************************************************************************
函数名称 : iot_local_save
功能描述 : 保存数据到本地存储器
传入参数 : data_len       : 自定义数据长度
           data           : 需要保存的自定义数据
传出参数 : 无
返回参数 :  0 : 成功
           -1 : 失败
*****************************************************************************/
extern iot_s32 iot_local_save( iot_u32 data_len, const void* data );
/*****************************************************************************
函数名称 : iot_local_load
功能描述 : 从本地存储器加载数据
传入参数 : 无
传出参数 : data_len       : 需要加载的数据长度
           data           : 需要加载的自定义数据
返回参数 :  0 : 成功
           -1 : 失败，加载数据出错或data_len大于实际数据长度
*****************************************************************************/
extern iot_s32 iot_local_load( iot_u32 data_len, void* data );
/*****************************************************************************
函数名称 : iot_local_reset
功能描述 : 将本地存储器保存的数据清空
传入参数 : 无
传出参数 : 无
返回参数 :  0 : 成功
           -1 : 失败
*****************************************************************************/
extern iot_s32 iot_local_reset( void );
/*****************************************************************************
函数名称 : iot_config
功能描述 : 配置设备信息，请在iot_start之前调用
传入参数 : config : CONFIG_ENCRYPT_TYPE : 配置加密算法
                    CONFIG_SN           : 配置设备序列号，厂商自定义，每种类型产品下须保证唯一
           data   : 根据config的不同，data请传入不同值
                    CONFIG_ENCRYPT_TYPE : "AES" : AES算法（默认）
                                          "SSL" : SSL算法
                                          "SM4" : SM4算法
                    CONFIG_SN : 自定义字符串，数字字母组合，最长16字节
传出参数 : 无
返回参数 : -1 : 失败
           0  : 成功
*****************************************************************************/
extern iot_s32 iot_config(CONFIG_T config, void* data);

//  第四部分结束  ////////////////////////////////////////////////////////////


/*##########################################################################*/


//  第五部分 [子设备功能]  /////////////////////////////////////////////////////

/*****************************************************************************
函数名称 : sub_dev_add
功能描述 : 添加子设备信息到发送队列
传入参数 : sub_id      : 自定义的子设备id，仅限字母数字组合
           sub_name    : 子设备名称，仅限字母数字组合
           sub_version : 子设备固件版本，"xx.xx"，0≤x≤9
           sub_type    : 子设备类型，请与APP端自行约定
传出参数 : 无
返回参数 :  0 : 成功
           -1 : 失败
*****************************************************************************/
extern iot_s32 sub_dev_add (const char  sub_id[16], 
                            const char  sub_name[32],
                            const char  sub_version[5], 
                            iot_u16     sub_type);
/*****************************************************************************
函数名称 : iot_sub_dev_active
功能描述 : 成功添加子设备后，调用此接口告知云端子设备已上线
传出参数 : data_seq : 本条数据的序列号
返回参数 :  0 : 成功
           -1 : 失败
*****************************************************************************/
extern iot_s32 iot_sub_dev_active ( iot_u32* data_seq );
/*****************************************************************************
函数名称 : iot_sub_dev_inactive
功能描述 : 子设备离线时，调用此接口
传入参数 : sub_id      : 自定义的子设备id，仅限字母数字组合
传出参数 : 无
返回参数 :  0 : 成功
           -1 : 失败
*****************************************************************************/
extern iot_s32 iot_sub_dev_inactive ( const char sub_id[16], iot_u32* data_seq );

//  第五部分结束  ////////////////////////////////////////////////////////////

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* __IOT_INTERFACE_H__ */
