#ifndef __WIFI_INTERFACE_H__
#define __WIFI_INTERFACE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdbool.h>

#define WIFI_ESP8266_SMART_CONFIG_STATUS_STOPPED        0
#define WIFI_ESP8266_SMART_CONFIG_STATUS_STARTED        1
#define WIFI_ESP8266_SMART_CONFIG_STATUS_IN_PROGRESS    2
#define WIFI_ESP8266_SMART_CONFIG_STATUS_GOT_WIFI_INFO  3
#define WIFI_ESP8266_SMART_CONFIG_STATUS_CONNECTED      4

/**
 * 用于通知 smartconfig 进度发生变化的回调函数，需用户实现
 *
 * @param   new_status      新的状态 (可能值见上方 WIFI_ESP8266_SMART_CONFIG_STATUS_* 系列宏定义)
 */
void wifi_esp8266_at_smartconfig_status_changed_cb(int new_status);

/**
 * 用于向 ESP8266 AT 固件模块发送数据的回调函数，需用户实现
 *
 * @param   data            指向要发送数据起始位置的指针
 * @param   data_length     要发送数据的长度
 */
void wifi_esp8266_at_send_data_cb(const void *data, int data_length);

/**
 * 将收到的来自 ESP8266 AT 固件模块的数据写入处理程序的缓冲区
 *
 * @param   data            指向收到数据起始位置的指针
 * @param   data_length     收到数据的长度
 *
 * @return  写入成功时返回 true; 失败时返回 false (缓冲区已满)
 */
bool wifi_esp8266_at_write_received_data(const char *data, int data_length);

/**
 * 发送 AT+RST 指令复位 ESP8266 AT 固件模块，并进行必要的初始化
 *
 * @return  若指令执行成功则返回 true, 否则返回 false
 */
bool wifi_esp8266_at_reset(void);

/**
 * 检查 ESP8266 AT 固件模块是否已连接到 AP
 *
 * @return  若已连接到 AP 则返回 true, 若未连接到 AP 或查询指令执行失败则返回 false
 */
bool wifi_esp8266_is_connected(void);

/**
 * 连接到指定 AP
 *
 * @param   ssid        SSID (最长 32 个字符)
 * @param   password    密码 (最长 64 个字符)
 *
 * @return  若指令执行成功则返回 true, 否则返回 false
 */
bool wifi_esp8266_connect_to_ap(const char *ssid, const char *password);

/**
 * 开始 smartconfig 过程
 *
 * @param   type                    smartconfig 类型 (1: ESP-TOUCH, 2: AirKiss, 3: ESP-TOUCH + AirKiss)
 * @param   timeout_secs            超时时间，即从开始到完成 smartconfig 所允许使用的最长时间 (单位为秒)
 * @param   smartconfig_success     指向一个 bool 类型变量的指针，通过该变量返回 smartconfig 是否成功完成
 *
 * @return  若所有相关指令执行成功则返回 true, 否则返回 false
 */
bool wifi_esp8266_smartconfig(char type, unsigned int timeout_secs, bool *smartconfig_success);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* __WIFI_INTERFACE_H__ */
