/* ----------------------------------------------------------------------
* Copyright (C) 2010-2014 ARM Limited. All rights reserved.
*
* $Date: 16/10/14 6:01p $Revision:  V.1.4.5
*
* Project:      CMSIS DSP Library
* Title:        arm_fir_decimate_fast_q31.c
*
* Description:  Fast Q31 FIR Decimator.
*
* Target Processor: Cortex-M4/Cortex-M3
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*   - Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*   - Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the
*     distribution.
*   - Neither the name of ARM LIMITED nor the names of its contributors
*     may be used to endorse or promote products derived from this
*     software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
* COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
* ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
* -------------------------------------------------------------------- */

#include "arm_math.h"

/**
 * @ingroup groupFilters
 */

/**
 * @addtogroup FIR_decimate
 * @{
 */

/**
 * @brief Processing function for the Q31 FIR decimator (fast variant) for Cortex-M3 and Cortex-M4.
 * @param[in] *S points to an instance of the Q31 FIR decimator structure.
 * @param[in] *pSrc points to the block of input data.
 * @param[out] *pDst points to the block of output data
 * @param[in] blockSize number of input samples to process per call.
 * @return none
 *
 * <b>Scaling and Overflow Behavior:</b>
 *
 * \par
 * This function is optimized for speed at the expense of fixed-point precision and overflow protection.
 * The result of each 1.31 x 1.31 multiplication is truncated to 2.30 format.
 * These intermediate results are added to a 2.30 accumulator.
 * Finally, the accumulator is saturated and converted to a 1.31 result.
 * The fast version has the same overflow behavior as the standard version and provides less precision since it discards the low 32 bits of each multiplication result.
 * In order to avoid overflows completely the input signal must be scaled down by log2(numTaps) bits (where log2 is read as log to the base 2).
 *
 * \par
 * Refer to the function <code>arm_fir_decimate_q31()</code> for a slower implementation of this function which uses a 64-bit accumulator to provide higher precision.
 * Both the slow and the fast versions use the same instance structure.
 * Use the function <code>arm_fir_decimate_init_q31()</code> to initialize the filter structure.
 */

void arm_fir_decimate_fast_q31(
    arm_fir_decimate_instance_q31 * S,
    q31_t * pSrc,
    q31_t * pDst,
    uint32_t blockSize)
{
    q31_t *pState = S->pState;                     /* State pointer */
    q31_t *pCoeffs = S->pCoeffs;                   /* Coefficient pointer */
    q31_t *pStateCurnt;                            /* Points to the current sample of the state */
    q31_t x0, c0;                                  /* Temporary variables to hold state and coefficient values */
    q31_t *px;                                     /* Temporary pointers for state buffer */
    q31_t *pb;                                     /* Temporary pointers for coefficient buffer */
    q31_t sum0;                                    /* Accumulator */
    uint32_t numTaps = S->numTaps;                 /* Number of taps */
    uint32_t i, tapCnt, blkCnt, outBlockSize = blockSize / S->M;  /* Loop counters */
    uint32_t blkCntN2;
    q31_t x1;
    q31_t acc0, acc1;
    q31_t *px0, *px1;

    /* S->pState buffer contains previous frame (numTaps - 1) samples */
    /* pStateCurnt points to the location where the new input data should be written */
    pStateCurnt = S->pState + (numTaps - 1u);

    /* Total number of output samples to be computed */

    blkCnt = outBlockSize / 2;
    blkCntN2 = outBlockSize - (2 * blkCnt);

    while(blkCnt > 0u)
    {
        /* Copy decimation factor number of new input samples into the state buffer */
        i = 2 * S->M;

        do
        {
            *pStateCurnt++ = *pSrc++;

        }
        while(--i);

        /* Set accumulator to zero */
        acc0 = 0;
        acc1 = 0;

        /* Initialize state pointer */
        px0 = pState;
        px1 = pState + S->M;

        /* Initialize coeff pointer */
        pb = pCoeffs;

        /* Loop unrolling.  Process 4 taps at a time. */
        tapCnt = numTaps >> 2;

        /* Loop over the number of taps.  Unroll by a factor of 4.
         ** Repeat until we've computed numTaps-4 coefficients. */
        while(tapCnt > 0u)
        {
            /* Read the b[numTaps-1] coefficient */
            c0 = *(pb);

            /* Read x[n-numTaps-1] for sample 0 sample 1 */
            x0 = *(px0);
            x1 = *(px1);

            /* Perform the multiply-accumulate */
            acc0 = (q31_t)((((q63_t) acc0 << 32) + ((q63_t) x0 * c0)) >> 32);
            acc1 = (q31_t)((((q63_t) acc1 << 32) + ((q63_t) x1 * c0)) >> 32);

            /* Read the b[numTaps-2] coefficient */
            c0 = *(pb + 1u);

            /* Read x[n-numTaps-2]  for sample 0 sample 1  */
            x0 = *(px0 + 1u);
            x1 = *(px1 + 1u);

            /* Perform the multiply-accumulate */
            acc0 = (q31_t)((((q63_t) acc0 << 32) + ((q63_t) x0 * c0)) >> 32);
            acc1 = (q31_t)((((q63_t) acc1 << 32) + ((q63_t) x1 * c0)) >> 32);

            /* Read the b[numTaps-3] coefficient */
            c0 = *(pb + 2u);

            /* Read x[n-numTaps-3]  for sample 0 sample 1 */
            x0 = *(px0 + 2u);
            x1 = *(px1 + 2u);
            pb += 4u;

            /* Perform the multiply-accumulate */
            acc0 = (q31_t)((((q63_t) acc0 << 32) + ((q63_t) x0 * c0)) >> 32);
            acc1 = (q31_t)((((q63_t) acc1 << 32) + ((q63_t) x1 * c0)) >> 32);

            /* Read the b[numTaps-4] coefficient */
            c0 = *(pb - 1u);

            /* Read x[n-numTaps-4] for sample 0 sample 1 */
            x0 = *(px0 + 3u);
            x1 = *(px1 + 3u);


            /* Perform the multiply-accumulate */
            acc0 = (q31_t)((((q63_t) acc0 << 32) + ((q63_t) x0 * c0)) >> 32);
            acc1 = (q31_t)((((q63_t) acc1 << 32) + ((q63_t) x1 * c0)) >> 32);

            /* update state pointers */
            px0 += 4u;
            px1 += 4u;

            /* Decrement the loop counter */
            tapCnt--;
        }

        /* If the filter length is not a multiple of 4, compute the remaining filter taps */
        tapCnt = numTaps % 0x4u;

        while(tapCnt > 0u)
        {
            /* Read coefficients */
            c0 = *(pb++);

            /* Fetch 1 state variable */
            x0 = *(px0++);
            x1 = *(px1++);

            /* Perform the multiply-accumulate */
            acc0 = (q31_t)((((q63_t) acc0 << 32) + ((q63_t) x0 * c0)) >> 32);
            acc1 = (q31_t)((((q63_t) acc1 << 32) + ((q63_t) x1 * c0)) >> 32);

            /* Decrement the loop counter */
            tapCnt--;
        }

        /* Advance the state pointer by the decimation factor
         * to process the next group of decimation factor number samples */
        pState = pState + S->M * 2;

        /* The result is in the accumulator, store in the destination buffer. */
        *pDst++ = (q31_t)(acc0 << 1);
        *pDst++ = (q31_t)(acc1 << 1);

        /* Decrement the loop counter */
        blkCnt--;
    }

    while(blkCntN2 > 0u)
    {
        /* Copy decimation factor number of new input samples into the state buffer */
        i = S->M;

        do
        {
            *pStateCurnt++ = *pSrc++;

        }
        while(--i);

        /* Set accumulator to zero */
        sum0 = 0;

        /* Initialize state pointer */
        px = pState;

        /* Initialize coeff pointer */
        pb = pCoeffs;

        /* Loop unrolling.  Process 4 taps at a time. */
        tapCnt = numTaps >> 2;

        /* Loop over the number of taps.  Unroll by a factor of 4.
         ** Repeat until we've computed numTaps-4 coefficients. */
        while(tapCnt > 0u)
        {
            /* Read the b[numTaps-1] coefficient */
            c0 = *(pb++);

            /* Read x[n-numTaps-1] sample */
            x0 = *(px++);

            /* Perform the multiply-accumulate */
            sum0 = (q31_t)((((q63_t) sum0 << 32) + ((q63_t) x0 * c0)) >> 32);

            /* Read the b[numTaps-2] coefficient */
            c0 = *(pb++);

            /* Read x[n-numTaps-2] sample */
            x0 = *(px++);

            /* Perform the multiply-accumulate */
            sum0 = (q31_t)((((q63_t) sum0 << 32) + ((q63_t) x0 * c0)) >> 32);

            /* Read the b[numTaps-3] coefficient */
            c0 = *(pb++);

            /* Read x[n-numTaps-3] sample */
            x0 = *(px++);

            /* Perform the multiply-accumulate */
            sum0 = (q31_t)((((q63_t) sum0 << 32) + ((q63_t) x0 * c0)) >> 32);

            /* Read the b[numTaps-4] coefficient */
            c0 = *(pb++);

            /* Read x[n-numTaps-4] sample */
            x0 = *(px++);

            /* Perform the multiply-accumulate */
            sum0 = (q31_t)((((q63_t) sum0 << 32) + ((q63_t) x0 * c0)) >> 32);

            /* Decrement the loop counter */
            tapCnt--;
        }

        /* If the filter length is not a multiple of 4, compute the remaining filter taps */
        tapCnt = numTaps % 0x4u;

        while(tapCnt > 0u)
        {
            /* Read coefficients */
            c0 = *(pb++);

            /* Fetch 1 state variable */
            x0 = *(px++);

            /* Perform the multiply-accumulate */
            sum0 = (q31_t)((((q63_t) sum0 << 32) + ((q63_t) x0 * c0)) >> 32);

            /* Decrement the loop counter */
            tapCnt--;
        }

        /* Advance the state pointer by the decimation factor
         * to process the next group of decimation factor number samples */
        pState = pState + S->M;

        /* The result is in the accumulator, store in the destination buffer. */
        *pDst++ = (q31_t)(sum0 << 1);

        /* Decrement the loop counter */
        blkCntN2--;
    }

    /* Processing is complete.
     ** Now copy the last numTaps - 1 samples to the satrt of the state buffer.
     ** This prepares the state buffer for the next function call. */

    /* Points to the start of the state buffer */
    pStateCurnt = S->pState;

    i = (numTaps - 1u) >> 2u;

    /* copy data */
    while(i > 0u)
    {
        *pStateCurnt++ = *pState++;
        *pStateCurnt++ = *pState++;
        *pStateCurnt++ = *pState++;
        *pStateCurnt++ = *pState++;

        /* Decrement the loop counter */
        i--;
    }

    i = (numTaps - 1u) % 0x04u;

    /* copy data */
    while(i > 0u)
    {
        *pStateCurnt++ = *pState++;

        /* Decrement the loop counter */
        i--;
    }
}

/**
 * @} end of FIR_decimate group
 */
